/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr.h                                                     */
/*  DESCRIPTION:  Nonpareil expressions (header file)                        */
/*                                                                           */
/*****************************************************************************/


/*****************************************************************************/
/*                                                                           */
/*  EXPR                                                                     */
/*                                                                           */
/*  Object of this type are expressions.  The basic form of the record, as   */
/*  given below, is all that's required for some of the simpler expression   */
/*  types:  EKIND_LIT_BOOLEAN, EKIND_LIT_CHARACTER, EKIND_LIT_STRING,        */
/*  EKIND_LIT_INTEGER, and EKIND_LIT_REAL.  For the literals,                */
/*  e->token->value holds the value.  The other kinds of expressions have    */
/*  more elaborate records, with this one as a prefix to allow casting.      */
/*  These more elaborate records are private to the various expr_*.c files.  */
/*                                                                           */
/*****************************************************************************/

struct expr_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* the identifying token             */
  USTRING		param_name;	/* param name when := present        */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* invoke this instead when non-NULL */
};


/*****************************************************************************/
/*                                                                           */
/*  ExprNew(name, type, kind, pos)                                           */
/*                                                                           */
/*  Set name to a new EXPR of type type with the given attributes, and with  */
/*  param_name and type set to NULL.                                         */
/*                                                                           */
/*****************************************************************************/

#define ExprNew(name, xtype, kind, pos, pname)				\
  GetMemory((name), xtype);						\
  (name)->kind_tag = kind;						\
  (name)->file_pos = pos;						\
  (name)->param_name = pname;						\
  (name)->type = NULL;							\
  (name)->large_scale = FALSE;						\
  (name)->be_var = NULL;	


/*****************************************************************************/
/*                                                                           */
/*  ExprKindCheck(e, knd, fn_name)                                           */
/*  ExprKindCheck2(e, knd1, knd2, fn_name)                                   */
/*                                                                           */
/*****************************************************************************/

#define ExprKindCheck(e, kind, fn_name)					\
if( (e)->kind_tag != kind )						\
{									\
  fprintf(stderr,							\
    "internal error in %s: expected expr kind %s but got %s\n",		\
    fn_name, KindShow(kind), KindShow((e)->kind_tag));			\
  abort();								\
}

#define ExprKindCheck2(e, kind1, kind2, fn_name)			\
if( (e)->kind_tag != kind1 && (e)->kind_tag != kind2 )			\
{									\
  fprintf(stderr,							\
    "internal error in %s: expected expr kind %s or %s but got %s\n",	\
    fn_name, KindShow(kind1), KindShow(kind2),KindShow((e)->kind_tag));	\
  abort();								\
}


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_lit.c                                          */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_lit_rec *EXPR_LIT;

extern BOOLEAN ExprLitParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern EXPR ExprLitCopyUninstantiated(EXPR_LIT expr_lit,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprLitManifest(EXPR_LIT *e, CONTEXT context,
  TYPE self_type, BEFN encl_befn);
extern void ExprLitDebug(EXPR_LIT expr_lit, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style);
extern void ExprLitCodeGen(EXPR_LIT expr_lit, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be);
extern int ExprLitConstIntValue(EXPR e);
extern int ExprLitConstCmp(EXPR e1, EXPR e2);
extern BOOLEAN ExprLitInitOrder(EXPR_LIT expr_lit, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_paren.c                                        */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_paren_rec * EXPR_PAREN;

extern EXPR ExprParenNew(FILE_POS file_pos, EXPR sub_expr);
extern EXPR ExprParenCopyUninstantiated(EXPR_PAREN expr_paren,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprParenManifest(EXPR_PAREN *e, CONTEXT context,
  TYPE self_type, BEFN encl_befn);
extern void ExprParenDebug(EXPR_PAREN expr_paren, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style);
extern void ExprParenCodeGen(EXPR_PAREN expr_paren, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be);
extern BOOLEAN ExprParenInitOrder(EXPR_PAREN expr_paren, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_tuple.c                                        */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_tuple_rec * EXPR_TUPLE;

extern BOOLEAN ExprParenOrTupleParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern EXPR ExprTupleCopyUninstantiated(EXPR_TUPLE expr_tuple,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprTupleManifest(EXPR_TUPLE *e, CONTEXT context,
  TYPE self_type, BEFN encl_befn);
extern void ExprTupleDebug(EXPR_TUPLE expr_tuple, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_list.c                                         */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_list_rec * EXPR_LIST;

extern BOOLEAN ExprListParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern EXPR ExprListCopyUninstantiated(EXPR_LIST expr_list,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprListManifest(EXPR_LIST *e, CONTEXT context,
  TYPE self_type, BEFN encl_befn);
extern void ExprListDebug(EXPR_LIST expr_list, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_array.c                                        */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_array_rec *EXPR_ARRAY;

extern BOOLEAN ExprArrayParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern EXPR ExprArrayCopyUninstantiated(EXPR_ARRAY expr_array,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprArrayManifest(EXPR_ARRAY *e, CONTEXT context,
  TYPE self_type, BEFN encl_befn);
extern void ExprArrayAddToArray(EXPR e, EXPR elem);
extern void ExprArrayCodeGen(EXPR_ARRAY expr_array, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be);
extern void ExprManifestArrayDebug(EXPR_ARRAY expr_array, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style);
extern BOOLEAN ExprArrayInitOrder(EXPR_ARRAY expr_array, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_call.c                                         */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_call_rec * EXPR_CALL;

/* parsing */
extern void ExprCallAddToCloneBody(EXPR e, EXPR actual_param);
extern BOOLEAN ExprCallParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern BOOLEAN ExprBuiltinParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern BOOLEAN ExprCallFeatureParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern BOOLEAN ExprCallApplyParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern BOOLEAN ExprCallPrefixOpParse(TOKEN *t, SYSTEM_VIEW sv,
  NAME nd, EXPR *res);
extern BOOLEAN ExprCallPostfixOpParse(TOKEN *t, SYSTEM_VIEW sv,
  NAME nd, EXPR *res);
extern BOOLEAN ExprCallInfixSeqParse(TOKEN *t, SYSTEM_VIEW sv,EXPR *res);

/* uninstantiated copy */
extern EXPR ExprCallCopyUninstantiated(EXPR_CALL expr_call,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);

/* manifesting */
extern BOOLEAN ExprCallManifest(EXPR_CALL *e, CONTEXT context,
  TYPE self_type, BEFN encl_befn);

/* code generation */
extern EXPR ExprCallSoleParameter(EXPR e);
extern BOOLEAN ExprCallIsConst(EXPR e, EXPR *root);
extern int ExprCallConstIntValue(EXPR e);
extern void ExprCallCodeGen(EXPR_CALL expr_call, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be);
extern BOOLEAN ExprCallInitOrder(EXPR_CALL expr_call, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);
extern BOOLEAN ExprCallWrapsBuiltin(EXPR e, ARRAY_BEFN_PARAM parameters,
  CODEGEN_OBJ *be_obj);

/* debug */
extern void ExprCallDebug(EXPR_CALL expr_call, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_let.c                                          */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_let_rec * EXPR_LET;

extern BOOLEAN ExprLetParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern BOOLEAN ExprFunParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern EXPR ExprLetCopyUninstantiated(EXPR_LET expr_let,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprLetManifest(EXPR_LET *e, CONTEXT context, TYPE self_type,
  BEFN encl_befn);
extern void ExprLetDebug(EXPR_LET expr_let, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style);
extern void ExprLetCodeGen(EXPR_LET expr_let, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be);
extern BOOLEAN ExprLetInitOrder(EXPR_LET expr_let, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_if.c                                           */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_if_rec * EXPR_IF;

extern BOOLEAN ExprIfParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern EXPR ExprIfCopyUninstantiated(EXPR_IF expr_if,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprIfManifest(EXPR_IF *e, CONTEXT context, TYPE self_type,
  BEFN encl_befn);
extern void ExprIfDebug(EXPR_IF expr_if, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style);
extern void ExprIfCodeGen(EXPR_IF expr_if, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be);
extern BOOLEAN ExprIfInitOrder(EXPR_IF expr_if, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_case.c                                         */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_case_rec * EXPR_CASE;

extern BOOLEAN ExprCaseParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern EXPR ExprCaseCopyUninstantiated(EXPR_CASE expr_case,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprCaseManifest(EXPR_CASE *e, CONTEXT context,
  TYPE self_type, BEFN encl_befn);
extern void ExprCaseDebug(EXPR_CASE expr_case, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style);
extern void ExprCaseCodeGen(EXPR_CASE expr_case, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be);
extern BOOLEAN ExprCaseInitOrder(EXPR_CASE expr_case, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_fnhead.c                                       */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_fnhead_rec *EXPR_FNHEAD;

extern EXPR ExprFnHeadCopyUninstantiated(EXPR_FNHEAD expr_fnhead,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprFnHeadManifest(EXPR_FNHEAD *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn);
extern void ExprFnHeadCodeGen(EXPR_FNHEAD expr_fnhead, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be);
extern void ExprFnHeadDebug(EXPR_FNHEAD expr_fnhead, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style);
extern BOOLEAN ExprFnHeadInitOrder(EXPR_FNHEAD expr_fnhead, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_precond.c                                      */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_precond_rec *EXPR_PRECOND;

extern EXPR ExprPrecondCopyUninstantiated(EXPR_PRECOND expr_precond,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprPrecondManifest(EXPR_PRECOND *e, CONTEXT cxt,
  TYPE self_type, BEFN encl_befn);
extern BOOLEAN ExprPrecondInitOrder(EXPR_PRECOND expr_precond, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);
extern void ExprPrecondCodeGen(EXPR_PRECOND expr_precond,
  CODEGEN_OBJ res_be_var, CODEGEN_TYPE res_be_type, CODEGEN be);
extern void ExprPrecondDebug(EXPR_PRECOND expr_precond, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style);


/*****************************************************************************/
/*                                                                           */
/*  External functions - expr_default.c                                      */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_default_rec * EXPR_DEFAULT;

extern EXPR ExprDefaultNew(FILE_POS file_pos, USTRING param_name, TYPE type);
extern EXPR ExprDefaultCopyUninstantiated(EXPR_DEFAULT expr_default,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
extern BOOLEAN ExprDefaultManifest(EXPR_DEFAULT *e, CONTEXT context,
  TYPE self_type, BEFN encl_befn);
extern void ExprDefaultCodeGen(EXPR_DEFAULT expr_default,
  CODEGEN_OBJ res_be_var, CODEGEN_TYPE res_be_type, CODEGEN be);
extern BOOLEAN ExprDefaultInitOrder(EXPR_DEFAULT expr_default,
  int visit_num, BOOLEAN *report, BEFN_SYSTEM_INIT fun);
extern void ExprDefaultDebug(EXPR_DEFAULT expr_default, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style);
