/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_array.c                                               */
/*  DESCRIPTION:  Nonpareil manifest array expressions                       */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"
#define DEBUG1 1


/*****************************************************************************/
/*                                                                           */
/*  EXPR_ARRAY                                                               */
/*                                                                           */
/*  An array of zero or more expressions, held in e->subexpressions.         */
/*                                                                           */
/*****************************************************************************/

struct expr_array_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* file position of expression       */
  FILE_POS		param_name;	/* param name when := present        */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */
  ARRAY_EXPR		subexpressions;	/* the subexpressions                */
};


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprArrayParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)              */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprArrayParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  EXPR sub_expr;  EXPR_ARRAY res_array;

  /* set up res_array and *res */
  ExprNew(res_array, EXPR_ARRAY, KIND_EXPR_ARRAY, LexFilePos(curr_token), NULL);
  ArrayInit(&res_array->subexpressions);
  *res = (EXPR) res_array;

  /* skip initial left bar bracket, known to be there */
  next_token;

  /* parse optional expression sequence */
  if( LexType(curr_token) != TK_RIGHT_BAR_BRACKET )
  {
    /* nonempty list case */
    if( !ExprParse(t, sv, &sub_expr) )
      return FALSE;
    ArrayAddLast(res_array->subexpressions, sub_expr);
    while( LexType(curr_token) == TK_COMMA )
    {
      next_token;
      if( !ExprParse(t, sv, &sub_expr) )
        return FALSE;
      ArrayAddLast(res_array->subexpressions, sub_expr);
    }
    skip(TK_RIGHT_BAR_BRACKET, "\",\" or closing \"|]\" of manifest array");
  }
  else
  {
    /* empty list case */
    next_token;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprArrayEmpty(FILE_POS file_pos)                                   */
/*                                                                           */
/*  Return a new, empty array expression with the given file position.       */
/*                                                                           */
/*****************************************************************************/

EXPR ExprArrayEmpty(FILE_POS file_pos)
{
  EXPR_ARRAY res_array;
  ExprNew(res_array, EXPR_ARRAY, KIND_EXPR_ARRAY, file_pos, NULL);
  ArrayInit(&res_array->subexpressions);
  return (EXPR) res_array;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprArrayFromString(FILE_POS file_pos, USTRING str)                 */
/*                                                                           */
/*  Return a new array expression whose elements are taken from str.         */
/*                                                                           */
/*****************************************************************************/

EXPR ExprArrayFromString(FILE_POS file_pos, USTRING str)
{
  EXPR_ARRAY res_array;  int i, len;  UCHAR ch;
  ExprNew(res_array, EXPR_ARRAY, KIND_EXPR_ARRAY, file_pos, NULL);
  ArrayInit(&res_array->subexpressions);
  len = UStringLength(str);
  for( i = 0;  i < len;  i++ )
  {
    ch = UStringGet(str, i);
    ArrayAddLast(res_array->subexpressions, ExprLitFromChar(file_pos, ch));
  }
  res_array->type = ClassType(ClassString);
  return (EXPR) res_array;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprArrayCopyUninstantiated(EXPR_ARRAY expr_array,                  */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on manifest        */
/*  array expression e.                                                      */
/*                                                                           */
/*****************************************************************************/

EXPR ExprArrayCopyUninstantiated(EXPR_ARRAY expr_array,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_ARRAY res;  EXPR e;
  ExprNew(res, EXPR_ARRAY, KIND_EXPR_ARRAY, expr_array->file_pos,
    expr_array->param_name);
  if( expr_array->subexpressions != NULL )
  {
    ArrayInit(&res->subexpressions);
    ArrayForEach(expr_array->subexpressions, e)
      ArrayAddLast(res->subexpressions,
	ExprCopyUninstantiated(e, orig_params, copy_params));
  }
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprArrayManifest(EXPR_ARRAY *e, CONTEXT cxt, TYPE self_type,    */
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Carry out the specification of ExprManifest on an array expression.      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprArrayManifest(EXPR_ARRAY *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn)
{
  EXPR_ARRAY expr_array;  EXPR sub_expr;  int i;
  ARRAY_TYPE generics;  TYPE joined_type;  TYPE_VARS array_vars;
  expr_array = *e;

  /* manifest the elements and find the join of their types */
  joined_type = NULL;
  for( i = 0;  i < ArraySize(expr_array->subexpressions);  i++ )
  {
    if( !ExprManifest(&ArrayGet(expr_array->subexpressions, i),
	cxt, self_type, encl_befn) )
      return FALSE;
    sub_expr = ArrayGet(expr_array->subexpressions, i);
    if( !TypeJoin(&joined_type, sub_expr->type, cxt) )
    {
      fprintf(stderr, "%s: incompatible type of manifest array element\n",
	FilePosShow(sub_expr->file_pos));
      return FALSE; 
    }
  }

  /* type is array{joined_type}, or array{..} if empty */
  if( joined_type == NULL )
  {
    array_vars = ClassVars(ClassArray);
    assert(ArraySize(array_vars) == 1);
    joined_type = TypeMakeUnconstrainedRange(expr_array->file_pos,
      TypeVarName(ArrayFirst(array_vars)));
  }
  ArrayInit(&generics);
  ArrayAddLast(generics, joined_type);
  expr_array->type = TypeMakeInstantiatedClassType2(expr_array->file_pos,
    ClassArray, generics);

  expr_array->large_scale = TRUE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprArrayAddToArray(EXPR e, EXPR elem)                              */
/*                                                                           */
/*  Add elem to the end of the array of array_expr e.                        */
/*                                                                           */
/*****************************************************************************/

void ExprArrayAddToArray(EXPR e, EXPR elem)
{
  EXPR_ARRAY expr_array;
  ExprKindCheck(e, KIND_EXPR_ARRAY, "ExprArrayAddToArray");
  expr_array = (EXPR_ARRAY) e;
  ArrayAddLast(expr_array->subexpressions, elem);
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprArrayCodeGen(EXPR_ARRAY expr_array, CODEGEN_OBJ res_be_var,     */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Code gen manifest array expression e.                                    */
/*                                                                           */
/*****************************************************************************/

void ExprArrayCodeGen(EXPR_ARRAY expr_array, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  int i;  EXPR sub_expr;
  CODEGEN_OBJ be_var, be_var2, create1, create2;
  CODEGEN_TYPE be_type, arr_be_type;

  assert(res_be_var != NULL);
  if( expr_array->type == ClassType(ClassString) &&
      ArraySize(expr_array->subexpressions) <= 5 )
  {
    /* use shortcut string creation calls */
    be_type = ClassBEType(ClassChar);
    switch( ArraySize(expr_array->subexpressions) )
    {
      case 0:

	VarAssign(res_be_var, Call0(NPBack_String_Make0));
	break;

      case 1:

	VarAssign(res_be_var, Call1(NPBack_String_Make1,
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,0),NULL,be_type,be)));
	break;

      case 2:

	VarAssign(res_be_var, Call2(NPBack_String_Make2,
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,0),NULL,be_type,be),
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,1),NULL,be_type,be)));
	break;

      case 3:

	VarAssign(res_be_var, Call3(NPBack_String_Make3,
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,0),NULL,be_type,be),
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,1),NULL,be_type,be),
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,2),NULL,be_type,be)));
	break;

      case 4:

	VarAssign(res_be_var, Call4(NPBack_String_Make4,
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,0),NULL,be_type,be),
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,1),NULL,be_type,be),
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,2),NULL,be_type,be),
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,3),NULL,be_type,be)));
	break;

      case 5:

	VarAssign(res_be_var, Call5(NPBack_String_Make5,
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,0),NULL,be_type,be),
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,1),NULL,be_type,be),
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,2),NULL,be_type,be),
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,3),NULL,be_type,be),
	  ExprCodeGen(ArrayGet(expr_array->subexpressions,4),NULL,be_type,be)));
	break;
    }
  }
  else
  {
    /* ordinary calls, enclosed in a block */
    be->BlockBegin();
    arr_be_type = TypeBEType(expr_array->type, be);
    be_var = be->VarMakeAndDeclare("arr", NULL, arr_be_type);

    if( expr_array->type == ClassType(ClassString) )
    {
      create1 = NPBack_String_Create_Fn1;
      create2 = NPBack_String_Create_Fn2;
      be_type = ClassBEType(ClassChar);
    }
    else
    {
      create1 = NPBack_Array_Create_Fn1;
      create2 = NPBack_Array_Create_Fn2;
      be_type = be->voidp_type;
    }

    /* <be_var> = create1(<length>) */
    VarAssign(be_var,
      Call1(create1, Int(ArraySize(expr_array->subexpressions))));

    /* assignment of array elements */
    be_var2 = be->VarIndexedAttributeMake(be_var, "elems", 0);
    for( i = 0;  i < ArraySize(expr_array->subexpressions);  i++ )
    {
      be->VarIndexedAttributeUpdateIndex(be_var2, i);
      sub_expr = ArrayGet(expr_array->subexpressions, i);
      ExprCodeGen(sub_expr, be_var2, be_type, be);
    }

    /* <res_be_var> = create2(<be_var>) */
    VarAssign(res_be_var, Call1(create2, Var(be_var)));
    be->VarUnDeclare(be_var);
    be->BlockEnd();
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprManifestArrayDebug(EXPR_ARRAY expr_array, CONTEXT cxt,          */
/*    BOOLEAN show_types, FILE *fp, int print_style)                         */
/*                                                                           */
/*  Debug print of array expr e on *fp.                                      */
/*                                                                           */
/*****************************************************************************/

void ExprManifestArrayDebug(EXPR_ARRAY expr_array, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style)
{
  EXPR child;
  fprintf(fp, "[|");
  ArrayForEach(expr_array->subexpressions, child)
  {
    if( child != ArrayFirst(expr_array->subexpressions) )
      fprintf(fp, ", ");
    ExprDebug(child, cxt, show_types, fp, SINGLE_LINE);
  }
  fprintf(fp, "|]");
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprArrayInitOrder(EXPR_ARRAY expr_array, int visit_num,         */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of ExprInitOrder on array expression e.      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprArrayInitOrder(EXPR_ARRAY expr_array, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  EXPR sub_expr;  int i;
  for( i = 0;  i < ArraySize(expr_array->subexpressions);  i++ )
  {
    sub_expr = ArrayGet(expr_array->subexpressions, i);
    if( !ExprInitOrder(sub_expr, visit_num, report, fun) )
      return FALSE;
  }
  return TRUE;
}
