/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_call.c                                                */
/*  DESCRIPTION:  Nonpareil function call expressions                        */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0
#define DEBUG6 0
#define DEBUG7 0
#define DEBUG8 0
#define DEBUG9 0
#define DEBUG10 0
#define DEBUG11 0
#define DEBUG12 0
#define DEBUG13 0
#define DEBUG14 0
#define DEBUG15 0
#define DEBUG16 0


/*****************************************************************************/
/*                                                                           */
/*  EXPR_CALL                                                                */
/*                                                                           */
/*  Each entry on the following list is one legal syntactic type of          */
/*  call expression.  In the grammar, "generics" stands for an optional      */
/*  sequence of actual generic parameters enclosed in {}; "parameters"       */
/*  stands for an optional sequence of parameters enclosed in (); id is      */
/*  an identifier; and name is an identifier or punctseq.  The part of       */
/*  the call that is taken as the call's file position is marked with a ^.   */
/*                                                                           */
/*    SYNTAX_VANILLA (a call with nothing distinctive about it)              */
/*                                                                           */
/*        ^id generics parameters                                            */
/*                                                                           */
/*    SYNTAX_EXCLAM (an unclassed feature call, with !)                      */
/*                                                                           */
/*        [ id generics ] "!" ^name generics parameters                      */
/*                                                                           */
/*    SYNTAX_EXCLAM2 (an unclassed feature call, with !!)                    */
/*                                                                           */
/*        [ id generics ] "!!" ^name generics parameters                     */
/*                                                                           */
/*    SYNTAX_BUILTIN (a call on a builtin function):                         */
/*                                                                           */
/*        builtin ^id generics parameters                                    */
/*                                                                           */
/*    SYNTAX_DOTTED (a feature call, with .)                                 */
/*                                                                           */
/*        expr "." ^id generics parameters                                   */
/*                                                                           */
/*    SYNTAX_APPLY (a function application, parameters *not* optional):      */
/*                                                                           */
/*        expr ^parameters                                                   */
/*                                                                           */
/*    SYNTAX_PREFIX (a call on a prefix operator):                           */
/*                                                                           */
/*        ^name generics parameters expr                                     */
/*                                                                           */
/*    SYNTAX_POSTFIX (a call on a postfix operator):                         */
/*                                                                           */
/*        expr ^name generics parameters                                     */
/*                                                                           */
/*    SYNTAX_INFIX (a call on an infix operator):                            */
/*                                                                           */
/*        expr ^name generics parameters expr                                */
/*                                                                           */
/*  The SYNTAX_APPLY option is taken only if the list of parameters cannot   */
/*  be added to some other kind of call expression.                          */
/*                                                                           */
/*  The elements of these grammatical expressions are stored in various      */
/*  fields of the expression call record:  the identifier/name in e->value,  */
/*  generics in e->generics, and parameters in e->parameters.  If ! or !!    */
/*  is present and there is a type to its left, it is stored in e->uc_type.  */
/*                                                                           */
/*  Actual parameters are uniformly stored in the order they are read, left  */
/*  to right.  This includes isolated parameters (targets, left and right    */
/*  operands of operators, and the stems of apply expressions).  This rule   */
/*  implies that the target of a feature will almost always be the           */
/*  leftmost actual parameter.  The only exceptions are features that are    */
/*  prefix operators, when the target is the rightmost actual parameter.     */
/*  This differs from formal parameters, where the "self" parameter is       */
/*  the leftmost formal parameter in all cases; however, there is no         */
/*  problem, because the "self" parameter is the sole compulsory parameter   */
/*  of any prefix operator feature, so there is no doubt in matching it.     */
/*                                                                           */
/*  In the operator cases, e->op_name is a name definition from the          */
/*  operator table which proves that this token really is an operator.       */
/*  It is used during parsing (or rather, its precedence value is), and      */
/*  also its key (which might have an initial underscore) is used to         */
/*  look up names in contexts and feature names in types.                    */
/*                                                                           */
/*  In each actual parameter p, if the parameter was named by "name :=",     */
/*  then p->param_name records this name.                                    */
/*                                                                           */
/*  After manifesting, e->function contains the function that this call      */
/*  invokes.  Also, e->impl_function contains the function that implements   */
/*  that function:  if e->function is a class view creation function, then   */
/*  e->impl_function is the corresponding class creation function; if        */
/*  e->function is a feature view function, then e->impl_function is the     */
/*  corresponding feature function; else e->impl_function == e->function.    */
/*                                                                           */
/*  There is only an indirect connection between the syntactic form of a     */
/*  call expression and its semantics.  Vanilla and operator syntax can      */
/*  conceal creation function calls, feature calls (on self), calls on       */
/*  letdefs, etc.  So e->function is polymorphic and not highly correlated   */
/*  with e->syntax_kind.                                                     */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  SYNTAX_VANILLA,
  SYNTAX_EXCLAM,
  SYNTAX_EXCLAM2,
  SYNTAX_DOTTED,
  SYNTAX_PREFIX,
  SYNTAX_POSTFIX,
  SYNTAX_INFIX,
  SYNTAX_BUILTIN,
  SYNTAX_APPLY
} SYNTAX_TYPE;

struct expr_call_rec {
  KIND_TAG		kind_tag;	/* KIND_EXPR_CALL                    */
  FILE_POS		file_pos;	/* file position of expression       */
  USTRING		param_name;	/* param name when := present        */
  TYPE			type;		/* type when manifested              */
  BOOLEAN		large_scale;	/* if contains let or case           */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */

  SYNTAX_TYPE		syntax_type;	/* syntactic classification          */
  USTRING		value;		/* value of identifier or punctseq   */
  ARRAY_TYPE		generics;	/* actual generic parameters         */
  TYPE			uc_type;	/* type preceding ! or !! if any     */
  NAME			op_name;	/* name def for operators            */
  ARRAY_EXPR		parameters;	/* parameters (NULL if none)         */
  ARRAY_EXPR		sorted_actuals;	/* parameters after signature match  */
  FEFN			function;	/* the function the call is on       */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "parsing".                                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  EXPR_CALL ExprCallNew(FILE_POS file_pos, USTRING value,                  */
/*    SYNTAX_TYPE syntax_type)                                               */
/*                                                                           */
/*  Return a new expression call with these attributes.                      */
/*                                                                           */
/*****************************************************************************/

static EXPR_CALL ExprCallNew(FILE_POS file_pos, USTRING value,
  SYNTAX_TYPE syntax_type)
{
  EXPR_CALL res;
  ExprNew(res, EXPR_CALL, KIND_EXPR_CALL, file_pos, NULL);
  res->syntax_type = syntax_type;
  res->value = value;
  res->generics = NULL;
  res->uc_type = NULL;
  res->op_name = NULL;
  res->parameters = NULL;
  res->sorted_actuals = NULL;
  res->function = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprMakeRawCall(FILE_POS file_pos, USTRING value,                   */
/*    ARRAY_TYPE generics, ARRAY_EXPR parameters)                            */
/*                                                                           */
/*  Make a raw call expression with these attributes.                        */
/*                                                                           */
/*****************************************************************************/

EXPR ExprMakeRawCall(FILE_POS file_pos, USTRING value,
  ARRAY_TYPE generics, ARRAY_EXPR parameters)
{
  EXPR_CALL res;
  res = ExprCallNew(file_pos, value, SYNTAX_VANILLA);
  res->generics = generics;
  res->parameters = parameters;
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprMakeDottedCall(USTRING value, EXPR param)                       */
/*                                                                           */
/*  Return an expression denoting (param).value.                             */
/*                                                                           */
/*****************************************************************************/

EXPR ExprMakeDottedCall(USTRING value, EXPR param)
{
  EXPR_CALL expr_call;
  expr_call = ExprCallNew(param->file_pos, value, SYNTAX_DOTTED);
  expr_call->large_scale = param->large_scale;
  ArrayInit(&expr_call->parameters);
  ArrayAddLast(expr_call->parameters, param);
  return (EXPR) expr_call;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprMakeBuiltinCall(FILE_POS file_pos, USTRING value,               */
/*    ARRAY_EXPR parameters)                                                 */
/*                                                                           */
/*  Return an expression denoting "builtin value(parameters)".               */
/*                                                                           */
/*****************************************************************************/

EXPR ExprMakeBuiltinCall(FILE_POS file_pos, USTRING value,
  ARRAY_EXPR parameters)
{
  EXPR_CALL expr_call;
  expr_call = ExprCallNew(file_pos, value, SYNTAX_BUILTIN);
  expr_call->parameters = parameters;
  return (EXPR) expr_call;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCallAddToCloneBody(EXPR e, EXPR actual_param)                   */
/*                                                                           */
/*  Here e is an expr_call; it is in fact the creation call that makes up    */
/*  the entire body of a clone feature.  Add actual_param to this call.      */
/*                                                                           */
/*****************************************************************************/

void ExprCallAddToCloneBody(EXPR e, EXPR actual_param)
{
  EXPR_CALL expr_call;
  ExprKindCheck(e, KIND_EXPR_CALL, "ExprCallAddToCloneBody");
  expr_call = (EXPR_CALL) e;
  if( expr_call->parameters == NULL )
    ArrayInit(&expr_call->parameters);
  ArrayAddLast(expr_call->parameters, actual_param);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ParseOneActualParameter(TOKEN *t, SYSTEM_VIEW sv,                */
/*    BOOLEAN named, EXPR *res)                                              */
/*                                                                           */
/*  Parse one actual parameter.  If named is TRUE it must be a named         */
/*  parameter, otherwise being a named parameter is optional.                */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ParseOneActualParameter(TOKEN *t, SYSTEM_VIEW sv,
  BOOLEAN named, EXPR *res)
{
  USTRING name;

  /* parse optional name and := */
  if( LexIsName(curr_token) && LexType(LexNext(curr_token))==TK_COLON_EQUALS )
  {
    name = LexValue(curr_token);
    next_token;
    next_token;
  }
  else if( named )
  {
    fprintf(stderr, "%s: optional parameter of operator must be named\n",
      LexPos(curr_token));
    return FALSE;
  }
  else
    name = NULL;

  /* parse the expression which is the actual parameter value */
  if( !ExprParse(t, sv, res) )
    return FALSE;

  /* record any name := seen earlier */
  (*res)->param_name = name;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallActualParametersParse(TOKEN *t, BOOLEAN named,           */
/*    SYSTEM_VIEW sv, ARRAY_EXPR *res)                                       */
/*                                                                           */
/*  Parse an optional list of actual parameters.  If named is true then      */
/*  they must all be named parameters, otherwise they don't have to be.      */
/*                                                                           */
/*  IMPORTANT POINT: *res is assumed to be already initialized, either to    */
/*  NULL (meaning no parameters yet) or to an array, which will contain one  */
/*  element; and the current token is assumed to be an lpar.                 */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ExprCallActualParametersParse(TOKEN *t, SYSTEM_VIEW sv,
  BOOLEAN named, ARRAY_EXPR *res)
{
  EXPR sub_expr;
  skip(TK_LEFT_PAREN, "internal error in ExprCallActualParametersParse");
  if( !ParseOneActualParameter(t, sv, named, &sub_expr) )
    return FALSE;
  if( *res == NULL )
    ArrayInit(res);
  ArrayAddLast(*res, sub_expr);
  while( LexType(curr_token) == TK_COMMA )
  {
    next_token;
    if( !ParseOneActualParameter(t, sv, named, &sub_expr) )
      return FALSE;
    ArrayAddLast(*res, sub_expr);
  }
  skip(TK_RIGHT_PAREN, "\",\" or closing \")\" of actual parameter list");
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ParseOperator(TOKEN *t, SYSTEM_VIEW sv, NAME op_name, EXPR *res) */
/*                                                                           */
/*  The current token is known to be an operator name; op_name is a name     */
/*  definition from sv's operator table which proves this.  Parse the        */
/*  operator and its following optional actual generic parameters and        */
/*  optional actual parameters.                                              */
/*                                                                           */
/*  If *res != NULL it is a first parameter, to be added to parameters.      */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ParseOperator(TOKEN *t, SYSTEM_VIEW sv, NAME op_name, EXPR *res)
{
  EXPR_CALL res_call;

  /* set up res_call */
  res_call = ExprCallNew(LexFilePos(curr_token), LexValue(curr_token),
    NameIsPrefixOperator(op_name) ? SYNTAX_PREFIX :
    NameIsPostfixOperator(op_name) ? SYNTAX_POSTFIX : SYNTAX_INFIX);
  res_call->op_name = op_name;

  /* set up parameters, possibly add *res to them, and set final *res */
  if( *res != NULL )
  {
    ArrayInit(&res_call->parameters);
    ArrayAddLast(res_call->parameters, *res);
  }
  else
    res_call->parameters = NULL;
  *res = (EXPR) res_call;

  /* skip operator name token, which is known to be there */
  next_token;

  /* parse optional actual generic parameters */
  if( !TypeOptionalActualGenericsParse(t, &res_call->generics))
    return FALSE;

  /* parse optional actual parameters; requires a lot of lookahead to  */
  /* distinguish this case from the start of the following factor      */
  /* NB tokens always end with TK_END_FILE so no danger of falling off */
  if( LexType(curr_token) == TK_LEFT_PAREN && LexIsName(LexNext(curr_token)) &&
      LexType(LexNext(LexNext(curr_token))) == TK_COLON_EQUALS )
  {
    /* found actual parameters; now parse them */
    if( !ExprCallActualParametersParse(t, sv, TRUE, &res_call->parameters) )
      return FALSE;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)               */
/*                                                                           */
/*  Parse one vanilla or unclassed call, returning the result in *res.       */
/*  Current token is TK_IDENTIFIER, TK_EXCLAM, TK_EXCLAM_EXCLAM, or TK_SELF. */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCallParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  EXPR_CALL res_call;  BOOLEAN exclam2;
  TOKEN prec_token, token;  ARRAY_TYPE prec_generics;

  /* get initial "id generics" in prec_token and prec_generics */
  prec_token = NULL;
  if( LexType(curr_token) == TK_IDENTIFIER || LexType(curr_token) == TK_SELF )
  {
    prec_token = curr_token;
    next_token;
    if( !TypeOptionalActualGenericsParse(t, &prec_generics) )
      return FALSE;
  }

  if( LexType(curr_token)==TK_EXCLAM || LexType(curr_token)==TK_EXCLAM_EXCLAM )
  {
    /* unclassed feature call: "! name generics" */
    exclam2 = LexType(curr_token) == TK_EXCLAM_EXCLAM;
    next_token;
    if( !NameParse(t, &token) )
      return FALSE;
    res_call = ExprCallNew(LexFilePos(token), LexValue(token), 
      exclam2 ? SYNTAX_EXCLAM2 : SYNTAX_EXCLAM);
    if( !TypeOptionalActualGenericsParse(t, &res_call->generics) )
      return FALSE;

    /* save the type expression preceding the ! or !! */
    res_call->uc_type = (prec_token == NULL ? NULL :
      TypeMakeUninstantiatedClassType(LexFilePos(prec_token),
	LexValue(prec_token), prec_generics));
  }
  else
  {
    /* not an unclassed feature call: "id generics" */
    res_call = ExprCallNew(LexFilePos(prec_token), LexValue(prec_token),
      SYNTAX_VANILLA);
    res_call->generics = prec_generics;
  }

  /* optional actual parameters */
  if( LexType(curr_token) == TK_LEFT_PAREN )
    if( !ExprCallActualParametersParse(t, sv, FALSE, &res_call->parameters) )
      return FALSE;

  *res = (EXPR) res_call;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprAbbreviatedCallParse(TOKEN *t, USTRING str, SYSTEM_VIEW sv,  */
/*    EXPR *res)                                                             */
/*                                                                           */
/*  Parse one abbreviated call, that is, a vanilla call with the initial     */
/*  name not present and instead supplied by the "name" parameter.           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprAbbreviatedCallParse(TOKEN *t, USTRING str, SYSTEM_VIEW sv,
  EXPR *res)
{
  EXPR_CALL res_call;
  res_call = ExprCallNew(LexFilePos(curr_token), str, SYNTAX_VANILLA);
  if( LexType(curr_token) == TK_LEFT_PAREN )
    if( !ExprCallActualParametersParse(t, sv, FALSE, &res_call->parameters) )
      return FALSE;
  *res = (EXPR) res_call;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprBuiltinParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)            */
/*                                                                           */
/*  Parse one builtin expression, assuming the current token is "builtin".   */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprBuiltinParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  EXPR_CALL res_call;

  /* skip "builtin" keyword, known to be there */
  next_token;

  /* make call expression and skip name token */
  res_call = ExprCallNew(LexFilePos(curr_token), LexValue(curr_token),
    SYNTAX_BUILTIN);
  skip(TK_LIT_STRING, "string name following \"builtin\" keyword");
  
  /* parse optional generic parameters */
  if( !TypeOptionalActualGenericsParse(t, &res_call->generics) )
    return FALSE;

  /* optional actual parameters */
  if( LexType(curr_token) == TK_LEFT_PAREN )
    if( !ExprCallActualParametersParse(t, sv, FALSE, &res_call->parameters) )
      return FALSE; 

  *res = (EXPR) res_call;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallFeatureParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)        */
/*                                                                           */
/*  Parse feature call, returning the resulting EXPR in *res.                */
/*  The current token is a dot.                                              */
/*                                                                           */
/*  Unlike most ExprParse calls, for this one *res is defined at entry,      */
/*  and it holds the expression to the left of the dot.                      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCallFeatureParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  EXPR_CALL res_call;

  /* skip the dot */
  next_token;

  /* make the call object, recording the token containing the feature name */
  res_call = ExprCallNew(LexFilePos(curr_token), LexValue(curr_token),
    SYNTAX_DOTTED);
  ArrayInit(&res_call->parameters);
  ArrayAddLast(res_call->parameters, *res);

  /* skip feature name and parse optional actual generics */
  skip(TK_IDENTIFIER, "identifier (feature name)");
  if( !TypeOptionalActualGenericsParse(t, &res_call->generics) )
    return FALSE;

  /* optional actual parameters */
  if( LexType(curr_token) == TK_LEFT_PAREN )
    if( !ExprCallActualParametersParse(t, sv, FALSE, &res_call->parameters) )
      return FALSE; 

  /* new result has old result as target */
  *res = (EXPR) res_call;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallApplyParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)          */
/*                                                                           */
/*  Parse function application, returning the resulting EXPR in *res.        */
/*  The current token is an lpar.                                            */
/*                                                                           */
/*  Unlike most ExprParse calls, for this one *res is defined at entry,      */
/*  and it holds the function expression to the left of this application.    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCallApplyParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  EXPR_CALL res_call;
  if( DEBUG1 )
  {
    fprintf(stderr, "[ ExprCallApplyParse(t, sv, ");
    ExprDebug(*res, NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, "\n");
  }

  /* new result with existing *res as target */
  res_call = ExprCallNew(LexFilePos(curr_token), LexValue(curr_token),
    SYNTAX_APPLY);
  ArrayInit(&res_call->parameters);
  ArrayAddLast(res_call->parameters, *res);
  *res = (EXPR) res_call;

  /* get parameters; lpar is known to be first token */
  if( !ExprCallActualParametersParse(t, sv, FALSE, &res_call->parameters) )
    db_return(DEBUG1, "] ExprCallApplyParse", FALSE);

  /* return result */
  if( DEBUG1 )
  {
    fprintf(stderr, "] ExprCallApplyParse returning TRUE; *res = ");
    ExprDebug(*res, NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, "\n");
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallPrefixOpParse(TOKEN *t, SYSTEM_VIEW sv, NAME op_name,    */
/*    EXPR *res)                                                             */
/*                                                                           */
/*  The current token is a prefix operator, as op_name proves.  Parse the    */
/*  operator including the following factor which is its actual parameter.   */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCallPrefixOpParse(TOKEN *t, SYSTEM_VIEW sv, NAME op_name,
  EXPR *res)
{
  EXPR sub_expr;  EXPR_CALL res_call;
  *res = NULL;
  if( !ParseOperator(t, sv, op_name, res) )
    return FALSE;
  res_call = (EXPR_CALL) *res;
  if( !ExprDecoratedFactorParse(t, sv, &sub_expr) )
    return FALSE;
  if( res_call->parameters == NULL )
    ArrayInit(&res_call->parameters);
  ArrayAddLast(res_call->parameters, sub_expr);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallPostfixOpParse(TOKEN *t, SYSTEM_VIEW sv, NAME op_name,   */
/*    EXPR *res)                                                             */
/*                                                                           */
/*  The current token is a postfix operator, as op_name proves.  Parse the   */
/*  operator and set *res to the result.                                     */
/*                                                                           */
/*  Unlike most ExprParse calls, for this one *res is defined at entry,      */
/*  and it holds the expression to the left of the operator.                 */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCallPostfixOpParse(TOKEN *t, SYSTEM_VIEW sv, NAME op_name,
  EXPR *res)
{
  return ParseOperator(t, sv, op_name, res);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallInfixSeqParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)       */
/*                                                                           */
/*  Parse one infix sequence expression.  Consult sv's operator table        */
/*  to find whether a name encountered is an operator or not, and if so      */
/*  which kind.                                                              */
/*                                                                           */
/*  This code is rather tricky because it implements infix operator          */
/*  precedences.  As each new operator is discovered, it has to be           */
/*  inserted into the right shell of the growing result tree at the point    */
/*  dictated by its precedence.  The right shell is always a sequence of     */
/*  zero or more infix operators, sorted into non-decreasing order           */
/*  by precedence as we go down, and terminated by a node which is           */
/*  not an infix operator.                                                   */
/*                                                                           */
/*  This algorithm depends on the fact that we store parenthesized           */
/*  expressions as EXPR_PAREN nodes: we don't forget the parentheses.        */
/*                                                                           */
/*  Examples                                                                 */
/*  --------                                                                 */
/*           ___           ___                ___                    ___     */
/*  a      a + b     a + b + c      a + b + c * d      a + b + c * d * e     */
/*                                                                           */
/*  a  =>    +    =>       +    =>        +        =>        +               */
/*          / \          /  \           /   \              /   \             */
/*         a   b       +     c        +       *          +       *           */
/*                    / \            / \     / \        / \     / \          */
/*                   a   b          a   b   c   d      a   b   *   e         */
/*                                                            / \            */
/*                                                           c   d           */
/*                                                                           */
/*  Or if * is right-associative, it goes like this:                         */
/*                                                                           */
/*           ___           ___                ___                    ___     */
/*  a      a + b     a + b + c      a + b + c * d      a + b + c * d * e     */
/*                                                                           */
/*  a  =>    +    =>       +    =>        +        =>        +               */
/*          / \          /  \           /   \              /   \             */
/*         a   b       +     c        +       *          +       *           */
/*                    / \            / \     / \        / \     / \          */
/*                   a   b          a   b   c   d      a   b   c   *         */
/*                                                                / \        */
/*                                                               d   e       */
/*                                                                           */
/*  At each step, given an infix operator and its immediate right            */
/*  parameter, we insert the infix operator into its place in the right      */
/*  shell, move the node it replaces to be the left child of the new         */
/*  operator, and make its immediate right parameter be its right child.     */
/*  Non-infix nodes are considered to have very high precedence.             */
/*                                                                           */
/*  The difference between left-associative and right-associative infix      */
/*  operators is that for a left-associative operator we do the insert       */
/*  above the first shell node with equal or higher precedence, whereas      */
/*  for a right-associative operator we do the insert above the first        */
/*  shell node with strictly higher precedence.  In fact, to make this       */
/*  happen we simply add 1 to the precedence of right-associative infix      */
/*  operators.                                                               */
/*                                                                           */
/*****************************************************************************/

static int get_prec(EXPR e)
{
  if( e->kind_tag == KIND_EXPR_CALL &&
      ((EXPR_CALL) e)->syntax_type == SYNTAX_INFIX )
    return NamePrecedence(((EXPR_CALL) e)->op_name);
  else
    return MAX_INFIX_PRECEDENCE + 2;
}

BOOLEAN ExprCallInfixSeqParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  NAMED op_name;  EXPR sub_expr, curr_op;  EXPR_CALL x;
  int adjusted_prec;

  /* get first decorated factor */
  if( !ExprDecoratedFactorParse(t, sv, res) )
    return FALSE;

  /* keep looping while an infix operator is next */
  while( LexIsName(curr_token) &&
     SymRetrieve(SystemViewOpTable(sv), LexValue(curr_token), &op_name) &&
     NameIsInfixOperator(NamedName(op_name)) )
  {
    adjusted_prec = NamePrecedence(NamedName(op_name));
    adjusted_prec += adjusted_prec % 2; /* handles right associative */

    /* find curr_op, the insert position of the infix op, and insert it */
    if( adjusted_prec <= get_prec(*res) )
    {
      /* operator has lowest precedence, so insert as new root */
      curr_op = *res;
      if( !ParseOperator(t, sv, NamedName(op_name), &curr_op) )
        return FALSE;
      *res = curr_op;
    }
    else
    {
      /* *res is infix; insert within right shell, just below x */
      x = (EXPR_CALL) *res;
      while( adjusted_prec > get_prec(ArrayLast(x->parameters)) )
	x = (EXPR_CALL) ArrayLast(x->parameters);
      curr_op = ArrayRemoveLast(x->parameters);
      if( !ParseOperator(t, sv, NamedName(op_name), &curr_op) )
        return FALSE;
      ArrayAddLast(x->parameters, curr_op);
    }

    /* parse expression to right and attach to curr_op as last parameter */
    if( !ExprDecoratedFactorParse(t, sv, &sub_expr) )
      return FALSE;
    ArrayAddLast(((EXPR_CALL) curr_op)->parameters, sub_expr);
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "uninstantiated copy"                                          */
/*                                                                           */
/*  EXPR ExprCallCopyUninstantiated(EXPR_CALL expr_call,                     */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on this call expr. */
/*                                                                           */
/*****************************************************************************/

EXPR ExprCallCopyUninstantiated(EXPR_CALL expr_call,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_CALL res;  TYPE type;  EXPR e;
  ExprNew(res, EXPR_CALL, KIND_EXPR_CALL, expr_call->file_pos,
    expr_call->param_name);
  res->syntax_type = expr_call->syntax_type;
  res->value = expr_call->value;
  res->generics = NULL;
  if( expr_call->generics != NULL )
  {
    ArrayInit(&res->generics);
    ArrayForEach(expr_call->generics, type)
      ArrayAddLast(res->generics, TypeCopyUninstantiated(type));
  }
  res->uc_type = expr_call->uc_type == NULL ? NULL :
    TypeCopyUninstantiated(expr_call->uc_type);
  res->op_name = expr_call->op_name;
  res->parameters = NULL;
  if( expr_call->parameters != NULL )
  {
    ArrayInit(&res->parameters);
    ArrayForEach(expr_call->parameters, e)
      ArrayAddLast(res->parameters,
	ExprCopyUninstantiated(e, orig_params, copy_params));
  }
  res->sorted_actuals = NULL;
  res->function = NULL;
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "manifesting".                                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  USTRING ApplyString(int n)                                               */
/*                                                                           */
/*  Return the string "applyn" for n >= 1.  Cache them, since used a lot.    */
/*                                                                           */
/*****************************************************************************/

static USTRING ApplyString(int n)
{
  static ARRAY_USTRING vals = NULL;
  int i;  char buff[10];
  assert(n >= 1);
  if( vals == NULL )
    ArrayInit(&vals);
  for( i = ArraySize(vals);  i <= n;  i++ )
  {
    sprintf(buff, "apply%d", i);
    ArrayAddLast(vals, AStringToUString(buff));  /* this will be vals[i] */
  }
  return ArrayGet(vals, n);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FindInContext(EXPR_CALL expr_call, CONTEXT cxt,                  */
/*    TYPE self_type, FEFN *fun, FEFN_FEATURE_SET *fvs)                      */
/*                                                                           */
/*  Search for expr_call->value in cxt and return TRUE if found, setting     */
/*  either *fun or *fvs to the result, depending on which it was.  If        */
/*  the call turns out to be a call on a feature of the current class        */
/*  (and here it must be with an implicit target), add an explicit target    */
/*  and manifest it.                                                         */
/*                                                                           */
/*  If the call turns out to be a *fun rather than an *fvs, also set         */
/*  *impl_fun, to the function that implements *fun.                         */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FindInContext(EXPR_CALL expr_call, CONTEXT cxt,
  TYPE self_type, FEFN *fun, FEFN_FEATURE_SET *fvs)
{
  NAMED named;  EXPR self_target;  USTRING key;  FEFN_FEATURE fv;
  key = expr_call->op_name!=NULL ? NameKey(expr_call->op_name):expr_call->value;
  if( DEBUG9 )
    fprintf(stderr, "[ FindInContext(%s)\n", UStringToUTF8(key));
  *fun = NULL;
  *fvs = NULL;
  if( !ContextRetrieve(cxt, key, &named) )
  {
    db_return(DEBUG9, "FindInContext cxt", FALSE);
  }
  else switch( NamedKind(named) )
  {
    case KIND_FEFN_PARAM:

      if( DEBUG9 )
	fprintf(stderr, "  param %s\n", FEFnParamShow((FEFN_PARAM)named, cxt));
      *fun = (FEFN) named;
      db_return(DEBUG9, "FindInContext param", TRUE);
	  

    case KIND_FEFN_LETDEF:
    case KIND_FEFN_DOWNDEF:

      *fun = (FEFN) named;
      db_return(DEBUG9, "FindInContext letdef or downdef", TRUE);


    case KIND_CLASS_VIEW:

      *fun = (FEFN) ClassViewFEFnCreation((CLASS_VIEW) named);
      db_return(DEBUG9, "FindInContext class view", TRUE);


    case KIND_FEFN_FEATURE_SET:

      if( NameIsOperator(NamedName(named)) )
	return FALSE; /* implicit calls on operator features are forbidden */
      *fvs = (FEFN_FEATURE_SET) named;
      fv = FEFnFeatureSetSoleUncoercedMember(*fvs);
      if( fv == NULL || FEFnFeatureFType(fv) < FEATURE_PREDEF )
      {
	/* not predefined, so add "self" actual parameter to call */
	self_target = (EXPR) ExprCallNew(expr_call->file_pos,
	  AStringToUString("self"), SYNTAX_VANILLA);
	if( expr_call->parameters == NULL )
	  ArrayInit(&expr_call->parameters);
	ArrayInsert(expr_call->parameters, 0, self_target);
	expr_call->syntax_type = SYNTAX_DOTTED;
	if( !ExprManifest(&ArrayGet(expr_call->parameters, 0), cxt, 
	      self_type, NULL) )
	  return FALSE;
      }
      db_return(DEBUG9, "FindInContext feature view set", TRUE);


    case KIND_TYPE_VAR:

      fprintf(stderr,"%s: type variable %s where expression expected\n",
	FilePosShow(expr_call->file_pos), UStringToUTF8(expr_call->value));
      fprintf(stderr, "  (%s is defined at %s)\n",
	UStringToUTF8(expr_call->value), FilePosShow(NamedFilePos(named)));
      db_return(DEBUG9, "FindInContext type_var", TRUE);


    default:

      assert(FALSE);
  }
  db_return(DEBUG9, "FindInContext", FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FindParameter(ARRAY_FEFN_PARAM sig_params, USTRING str,int *pos) */
/*                                                                           */
/*  Search for a parameter named str among sig_params.  If found, return     */
/*  TRUE and set *pos to its position in the array.                          */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FindParameter(ARRAY_FEFN_PARAM sig_params, USTRING str, int *pos)
{
  int i;  FEFN_PARAM param;
  if( sig_params != NULL ) 
    for( i = 0;  i < ArraySize(sig_params);  i++ )
    {
      param = ArrayGet(sig_params, i);
      if( UStringEqual(NameKey(FEFnParamName(param)), str) )
      {
	*pos = i;
	return TRUE;
      }
    }
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN MatchActual(EXPR *actual, FEFN_PARAM formal, BOOLEAN confirm,    */
/*    CONTEXT cxt, TYPE self_type, ASTRING param_type)                       */
/*                                                                           */
/*  Match *actual against formal and return TRUE if successful.  If          */
/*  confirm is TRUE we really mean this, so print error messages on          */
/*  failure, and insert coercions on success.                                */
/*                                                                           */
/*  Implementation note.  The call to CoercionInsert may be slightly         */
/*  premature here, in that the types involved are not stabilized until      */
/*  after the call to TypeFixRange.  This issue needs further analysis.      */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN MatchActual(EXPR *actual, FEFN_PARAM formal, BOOLEAN confirm,
  CONTEXT cxt, TYPE self_type, ASTRING param_type)
{
  COERCION c;
  if( DEBUG16 )
    fprintf(stderr, "[ MatchActual(%s, %s, %s, cxt, self_type, \"%s\")\n",
      TypeShow(ExprType(*actual), cxt), FEFnParamShow(formal, cxt),
      bool(confirm), param_type);

  /* do the subtype check */
  if( !TypeIsSubType(ExprType(*actual), FEFnParamType(formal), &c, cxt) )
  {
    if( confirm )
    {
      fprintf(stderr, "%s: actual %s has wrong type\n",
	FilePosShow(ExprFilePos(*actual)), param_type);
      fprintf(stderr, "  (corresponding formal parameter is %s at %s)\n",
	NameShow(FEFnParamName(formal)), FilePosShow(FEFnParamFilePos(formal)));
      if( DEBUG5 )
      {
	fprintf(stderr, "formal parameter %s type: %s\n",
	  NameShow(FEFnParamName(formal)), TypeShow(FEFnParamType(formal),cxt));
	fprintf(stderr, "actual parameter type: %s\n",
	  TypeShow(ExprType(*actual), cxt));
      }
    }
    db_return(DEBUG16, "MatchActual", FALSE);
  }

  /* add coerced actual expr_call->excess_params */
  if( confirm )
    CoercionInsert(c, actual, cxt, self_type);
  db_return(DEBUG16, "MatchActual", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  void InferEnd(BOOLEAN inferring, TYPE_VARS tv, ARRAY_TYPE *actuals,      */
/*    BOOLEAN confirm)                                                       */
/*                                                                           */
/*  Convenience function for ending type inference bracket.                  */
/*                                                                           */
/*****************************************************************************/

static void InferEnd(BOOLEAN inferring, TYPE_VARS tv, ARRAY_TYPE *actuals,
  BOOLEAN confirm)
{
  if( inferring )
  {
    TypeRangeMarkEnd(confirm);
    TypeVarsInferEnd(tv, actuals, confirm);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN MatchExcessActuals(EXPR_CALL expr_call,                          */
/*    ARRAY_EXPR excess_actuals, TYPE result_type, BOOLEAN confirm,          */
/*    CONTEXT cxt, TYPE self_type, BOOLEAN inferring,                        */
/*    TYPE *excess_result_type, FEFN_FEATURE *excess_fv)                     */
/*                                                                           */
/*  There are excess actual parameters excess_actuals left over from         */
/*  matching expr_call.  If these can be matched with an "apply" feature of  */
/*  result_type, do so, set *excess_result_type to the result type of the    */
/*  apply, set *excess_fv to the "apply" feature view, and return TRUE.      */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN MatchExcessActuals(EXPR_CALL expr_call,
  ARRAY_EXPR excess_actuals, TYPE result_type, BOOLEAN confirm,
  CONTEXT cxt, TYPE self_type, BOOLEAN inferring,
  TYPE *excess_result_type, FEFN_FEATURE *excess_fv)
{
  EXPR actual;  FEFN_FEATURE_SET fvs;  FEFN_FEATURE fv;  COERCION c;
  TYPE_VARS sig_vars;  ARRAY_FEFN_PARAM sig_params;  TYPE sig_result_type;
  FEFN fun;  BEFN impl_fun;
  FEFN_PARAM formal;  ARRAY_TYPE generics;  int i;

  /* search for an "applyn" feature in result_type, for n excess actuals */
  if( DEBUG10 )
  {
    fprintf(stderr, "[ MatchExcessActuals(");
    ExprCallDebug(expr_call, cxt, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ")\n");
  }
  assert(excess_actuals != NULL && ArraySize(excess_actuals) > 0);
  *excess_result_type = NULL;
  *excess_fv = NULL;
  actual = ArrayFirst(excess_actuals);
  if( !TypeRetrieveFEFnFeatureSet(result_type, cxt, actual->file_pos,
      ApplyString(ArraySize(excess_actuals)), &fvs) ||
      (fv = FEFnFeatureSetSoleUncoercedMember(fvs)) == NULL )
  {
    if( confirm )
    {
      if( ArraySize(excess_actuals) == 1 )
	fprintf(stderr, "%s: excess actual parameter\n",
	  FilePosShow(ExprFilePos(actual)));
      else
	fprintf(stderr, "%s: excess actual parameter (first of %d)\n",
	  FilePosShow(ExprFilePos(actual)),
	  ArraySize(excess_actuals));
    }
    db_return(DEBUG10, "MatchExcessActuals (excess no fun)", FALSE);
  }
  if( DEBUG11 )
    fprintf(stderr, "  excess %d: %s\n", ArraySize(excess_actuals),
      FEFnSignatureShow((FEFN) fv, cxt));

  /* get signature */
  fun = (FEFN) fv;
  impl_fun = (BEFN) FEFnBEFn((FEFN) fv);
  /* BEFnSortParameters(impl_fun); */
  FEFnSignature(fun, &sig_vars, &sig_params, &sig_result_type);
  assert(sig_params != NULL);
  assert(ArraySize(sig_params) == ArraySize(excess_actuals) + 1);

  /* insert unification variables for missing generics */
  generics = NULL;
  if( inferring )
  {
    if( !TypeVarsInferBegin(sig_vars, &generics, expr_call->file_pos,
	  confirm, cxt) )
      db_return(DEBUG10, "MatchExcessActuals (infer begin)", FALSE);
    TypeRangeMarkBegin();
  }

  /* verify that first parameter matches *result_type */
  formal = ArrayFirst(sig_params);
  if( !TypeIsSubType(FEFnParamType(formal), result_type, &c, cxt) )
    assert(FALSE);

  /* match the other parameters with the excess actuals */
  for( i = 0;  i < ArraySize(excess_actuals);  i++ )
  {
    formal = ArrayGet(sig_params, i+1);
    actual = ArrayGet(excess_actuals, i);
    if( !MatchActual(&actual, formal, confirm, cxt, self_type,
	"named parameter") )
    {
      InferEnd(inferring, sig_vars, &generics, confirm);
      db_return(DEBUG10, "MatchExcessActuals (not subtype)", FALSE);
    }
    ArrayPut(excess_actuals, i, actual);
  }

  /* complete the operation of resolving generics */
  InferEnd(inferring, sig_vars, &generics, confirm);

  /* all in order */
  *excess_result_type = sig_result_type;
  *excess_fv = fv;
  db_return(DEBUG10, "MatchExcessActuals (excess)", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN Match(EXPR_CALL expr_call, FEFN fun, BOOLEAN confirm,            */
/*  CONTEXT cxt, TYPE self_type, BOOLEAN inferring, TYPE *excess_result_type,*/
/*    ARRAY_EXPR *excess_actuals, FEFN_FEATURE *excess_fv)                   */
/*                                                                           */
/*  Match fun against expr_call and return TRUE if successful.               */
/*                                                                           */
/*  If confirm is TRUE, this function is definitely the one we want to       */
/*  use, so leave things in a state indicating that, or if there are         */
/*  problems with the match, print error messages saying so.                 */
/*                                                                           */
/*  If confirm is FALSE, we are just testing one function signature out of   */
/*  a set of overloaded signatures, so make sure there are no state changes  */
/*  on exit, and no error messages printed.                                  */
/*                                                                           */
/*  Although signature matching occurs against fun, the layout of            */
/*  expr_call->sorted_actuals matches impl_fun.  This is useful since        */
/*  code generation does not have to work out again which actual parameter   */
/*  matches with which formal parameter of the implementation function.      */
/*                                                                           */
/*  There may be more actual parameters than formals, in which case          */
/*  an attempt is made to insert an "applyn" call.  If this happens,         */
/*  a successful match will return the excess parameters in *excess_actuals, */
/*  and the "applyn" feature view in *excess_fv.  Otherwise these two will   */
/*  be NULL.                                                                 */
/*                                                                           */
/*****************************************************************************/
static BOOLEAN FindFunction(EXPR_CALL expr_call, CONTEXT cxt, TYPE self_type,
  BOOLEAN inferring, TYPE *excess_result_type, ARRAY_EXPR *excess_actuals,
  FEFN_FEATURE *excess_fv);

static BOOLEAN Match(EXPR_CALL expr_call, FEFN fun, BOOLEAN confirm,
  CONTEXT cxt, TYPE self_type, BOOLEAN inferring, TYPE *excess_result_type,
  ARRAY_EXPR *excess_actuals, FEFN_FEATURE *excess_fv)
{
  TYPE result_type;  TYPE_VARS sig_vars;
  ARRAY_FEFN_PARAM sig_params;  TYPE sig_result_type;
  FEFN_PARAM formal;  BEFN_PARAM cformal;
  EXPR actual, actual2, expr_dft;  EXPR_CALL hidden_expr;
  int pos, prev_used, gap_count, i, index, sig_params_len;
  int last_compulsory, total_compulsory;
  USTRING pname;  ARRAY_BEFN_PARAM impl_params; BEFN impl_fun;

  if( DEBUG4 )
  {
    /* static USTRING nlist_str = NULL; */
    fprintf(stderr, "[ Match(expr_call, %s%s, %s, use_pp, cxt)\n",
      NameShow(FEFnName((FEFN) fun)),
      FEFnSignatureShow((FEFN) fun, cxt), bool(confirm));
    /* ***
    if( nlist_str == NULL )
      nlist_str = AStringToUString("nlist");
    if( UStringEqual(expr_call->value, nlist_str) )
      TypeDebugOn();
    *** */
  }

  impl_fun = FEFnBEFn(fun);
  BEFnSetUtilized(impl_fun);
  *excess_result_type = NULL;
  *excess_actuals = NULL;
  *excess_fv = NULL;

  /* get the signature and fail if no result type */
  FEFnSignature(fun, &sig_vars, &sig_params, &sig_result_type);
  if( sig_result_type == NULL )
  {
    fprintf(stderr, "%s: result type missing from recursive function\n",
      FilePosShow(FEFnFilePos(fun)));
    /* if( DEBUG4 ) TypeDebugOff(); */
    return FALSE;
  }

  /* resolve generics and insert unification variables for missing ones */
  if( inferring )
  {
    if( !TypeVarsInferBegin(sig_vars, &expr_call->generics,
	  expr_call->file_pos, confirm, cxt) )
      db_return(DEBUG4, "Match", FALSE);
    TypeRangeMarkBegin();
  }

  result_type = NULL;

  /* sorted_actuals has one (possibly null) entry for each param of impl_fun */
  ArrayFresh(expr_call->sorted_actuals);
  if( BEFnParamsCount(impl_fun) > 0 )
    ArrayEnsureLegalIndex(expr_call->sorted_actuals,
      BEFnParamsCount(impl_fun) - 1, 1000, NULL);

  /* handle every hidden formal parameter */
  last_compulsory = -1;  /* index of last (rightmost) compulsory parameter */
  total_compulsory = 0;  /* total number of compulsory parameters */
  if( sig_params != NULL )
  {
    for( i = 0;  i < ArraySize(sig_params);  i++ )
    {
      formal = ArrayGet(sig_params, i);
      if( FEFnParamKind(formal) == PARAM_HIDDEN )
      {
	index = BEFnParamIndex(FEFnParamBEFnParam(formal));
	assert(ArrayInRange(expr_call->sorted_actuals, index));
	assert(ArrayGet(expr_call->sorted_actuals, index) == NULL);
        hidden_expr = ExprCallNew(expr_call->file_pos,
	  NameRep(FEFnParamName(formal)), SYNTAX_VANILLA);
	if( !FindFunction(hidden_expr, cxt, self_type, FALSE,
	  excess_result_type, excess_actuals, excess_fv) )
	  assert(FALSE);
	assert(*excess_actuals == NULL);
	ArrayPut(expr_call->sorted_actuals, index, (EXPR) hidden_expr);
	if( DEBUG14 )
	{
	  fprintf(stderr, "  inserting hidden actual parameter ");
	  ExprDebug((EXPR) hidden_expr, cxt, FALSE, stderr, SINGLE_LINE);
	  fprintf(stderr, " referencing %p\n", (void *) hidden_expr->function);
	}
	if( index > last_compulsory )
	  last_compulsory = index;
	total_compulsory++;
      }
    }
  }

  if( expr_call->parameters != NULL )
  {
    /* handle every named actual parameter (may be compulsory or optional) */
    for( i = 0;  i < ArraySize(expr_call->parameters);  i++ )
    {
      actual = ArrayGet(expr_call->parameters, i);
      pname = ExprParamName(actual);
      if( pname != NULL )
      {
	/* retrieve corresponding formal parameter from function signature */
	if( !FindParameter(sig_params, pname, &pos) )
	{
	  if( confirm )
	  {
	    fprintf(stderr, "%s: named actual parameter %s unknown\n",
	      FilePosShow(ExprFilePos(actual)), UStringToUTF8(pname));
	    /* ** have lost this capacity (for now?) ***
	    fprintf(stderr, "  (unknown in function signature defined at %s)\n",
	      LexPos(FunSigPos(fsig)));
	    *** */
	  }
	  InferEnd(inferring, sig_vars, &expr_call->generics, confirm);
	  /* if( DEBUG4 ) TypeDebugOff(); */
	  db_return(DEBUG4, "Match (FindParameter)", FALSE);
	}
	formal = ArrayGet(sig_params, pos);
	index = BEFnParamIndex(FEFnParamBEFnParam(formal));
	assert(ArrayInRange(expr_call->sorted_actuals, index));

	/* make sure spot intended for actual is not already occupied */
	actual2 = ArrayGet(expr_call->sorted_actuals, index);
	if( actual2 != NULL )
	{
	  /* this error is so bad we print it whether confirm or not */
	  fprintf(stderr, "%s: named actual parameter %s occurs twice\n",
	    FilePosShow(ExprFilePos(actual)), UStringToUTF8(pname));
	  fprintf(stderr, "  (previous occurrence was at %s)\n",
	    FilePosShow(ExprFilePos(actual2)));
	  InferEnd(inferring, sig_vars, &expr_call->generics, confirm);
	  /* if( DEBUG4 ) TypeDebugOff(); */
	  db_return(DEBUG4, "Match (occurs twice)", FALSE);
	}

	/* check subtype relation between actual and formal */
        if( !MatchActual(&actual, formal, confirm, cxt, self_type,
	      "named parameter") )
	{
	  InferEnd(inferring, sig_vars, &expr_call->generics, confirm);
	  /* if( DEBUG4 ) TypeDebugOff(); */
	  db_return(DEBUG4, "Match (not subtype)", FALSE);
	}
	ArrayPut(expr_call->sorted_actuals, index, actual);

	/* keep last_compulsory and total_compulsory up to date */
	if( !FEFnParamIsOptional(formal) )
	{
	  if( index > last_compulsory )
	    last_compulsory = index;
	  total_compulsory++;
	}
      }
    }

    /* handle every positional (and therefore compulsory) parameter */
    prev_used = -1;
    sig_params_len = sig_params == NULL ? 0 : ArraySize(sig_params);
    for( i = 0;  i < ArraySize(expr_call->parameters);  i++ )
    {
      actual = ArrayGet(expr_call->parameters, i);
      if( ExprParamName(actual) == NULL )
      {
	/* find the next free compulsory parameter slot */
	for( prev_used++;  prev_used < sig_params_len;  prev_used++ )
	{
	  formal = ArrayGet(sig_params, prev_used);
	  if( !FEFnParamIsOptional(formal) )
	  {
	    if( DEBUG4 )
	      fprintf(stderr, "  at formal %s\n", FEFnParamShow(formal, cxt));
	    index = BEFnParamIndex(FEFnParamBEFnParam(formal));
	    assert(ArrayInRange(expr_call->sorted_actuals, index));
	    if( ArrayGet(expr_call->sorted_actuals, index) == NULL )
	      break;
	  }
	}

	/* if no slot, move to *excess_actuals and keep going */
	if( prev_used >= sig_params_len )
	{
	  if( *excess_actuals == NULL )
	    ArrayInit(excess_actuals);
	  ArrayAddLast(*excess_actuals, actual);
	  continue;
	}

	/* check subtype relation */
        if( !MatchActual(&actual, formal, confirm, cxt, self_type,
	      "parameter") )
	{
	  InferEnd(inferring, sig_vars, &expr_call->generics, confirm);
	  /* if( DEBUG4 ) TypeDebugOff(); */
	  db_return(DEBUG4, "Match (not subtype)", FALSE);
	}
	ArrayPut(expr_call->sorted_actuals, index, actual);

	/* keep last_compulsory and total_compulsory up to date */
	if( index > last_compulsory )
	  last_compulsory = index;
	total_compulsory++;
      }
    }
  }

  /* check for gaps in the sequence of optional parameters; insert defaults */
  impl_params = BEFnParameters(impl_fun);
  if( impl_params != NULL )
    for( i = 0;  i < ArraySize(impl_params);  i++ )
    {
      cformal = ArrayGet(impl_params, i);
      if( BEFnParamIsOptional(cformal) )
      {
	if( ArrayGet(expr_call->sorted_actuals, i) == NULL )
	{
	  expr_dft = ExprDefaultNew(expr_call->file_pos, NULL, NULL);
	  ArrayPut(expr_call->sorted_actuals, i, expr_dft);
	}
      }
    }

  /* check for missing compulsory parameters before last_compulsory */
  gap_count = 0;
  for( i = 0;  i < last_compulsory;  i++ )
    if( ArrayGet(expr_call->sorted_actuals, i) == NULL )
	gap_count++;
  if( gap_count > 0 )
  {
    if( confirm )
    {
      fprintf(stderr, "%s: %d missing actual compulsory %s\n",
	FilePosShow(expr_call->file_pos), gap_count,
	gap_count > 1 ? "parameters" : "parameter");
      /* lost capacity to do this (for now?) ***
      fprintf(stderr, "  (with respect to function signature defined at %s)\n",
	LexPos(FunSigPos(fsig)));
      *** */
    }
    InferEnd(inferring, sig_vars, &expr_call->generics, confirm);
    /* if( DEBUG4 ) TypeDebugOff(); */
    db_return(DEBUG4, "Match (gaps)", FALSE);
  }

  /* result type, not counting any excess actuals */
  result_type = FEFnCallResultType(sig_params, sig_result_type,
    total_compulsory, expr_call->file_pos);

  /* complete the operation of resolving generics of the main call */
  InferEnd(inferring, sig_vars, &expr_call->generics, confirm);

  /* handle excess actuals */
  if( *excess_actuals != NULL )
  {
    if( !MatchExcessActuals(expr_call, *excess_actuals, result_type, confirm,
	  cxt, self_type, inferring, excess_result_type, excess_fv) )
    {
      /* if( DEBUG4 ) TypeDebugOff(); */
      db_return(DEBUG4, "Match (excess)", FALSE);
    }
  }

  /* return, saving the result type if confirm and success */
  if( confirm )
  {
    if( DEBUG4 )
      fprintf(stderr, "  about to confirm %s\n", TypeShow(result_type, cxt));
    expr_call->type = TypeFixRange(result_type);
    expr_call->function = fun;
  }
  /* if( DEBUG4 ) TypeDebugOff(); */
  db_return(DEBUG4, "Match", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  void InsertApply(EXPR *e, TYPE result_type, ARRAY_EXPR actuals,          */
/*    FEFN_FEATURE fv)                                                       */
/*                                                                           */
/*  Replace *expr_call with a new call which is a call on fv with the        */
/*  old call as first parameter and the given actuals as the others,         */
/*  whose result type is as given.                                           */
/*                                                                           */
/*****************************************************************************/

static void InsertApply(EXPR *e, TYPE result_type, ARRAY_EXPR actuals,
  FEFN_FEATURE fv)
{
  EXPR old_call = *e;  EXPR_CALL expr_call;
  if( DEBUG12 )
  {
    fprintf(stderr, "InsertApply(");
    ExprDebug(*e, NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ", ");
    ExprArrayDebug(actuals, NULL, FALSE, stderr);
    fprintf(stderr, ", %s)\n", NameShow(FEFnFeatureName(fv)));
  }
  expr_call = ExprCallNew(old_call->file_pos, NameRep(FEFnFeatureName(fv)),
    SYNTAX_DOTTED);
  ArrayInsert(actuals, 0, old_call);
  expr_call->type = result_type;
  expr_call->parameters = expr_call->sorted_actuals = actuals;
  expr_call->function = (FEFN) fv;
  *e = (EXPR) expr_call;
  if( DEBUG12 )
  {
    fprintf(stderr, "InsertApply returning, *e = ");
    ExprDebug(*e, NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, "\n");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FindFunction(EXPR_CALL expr_call, CONTEXT cxt, TYPE self_type,   */
/*    BOOLEAN inferring, TYPE *excess_result_type, ARRAY_EXPR *excess_actuals*/
/*    FEFN_FEATURE *excess_fv)                                               */
/*                                                                           */
/*  Set expr_call->function to the function matching expr_call, and set      */
/*  e->sorted_actuals to the sorted parameters (minus any curried ones)      */
/*  that correspond to the parameters of e->impl_function.  Return TRUE if   */
/*  successful; otherwise print an error message and return FALSE.           */
/*                                                                           */
/*  If inferring is TRUE we are doing generic actual parameter type          */
/*  inference.  We do this on ordinary calls but not when inserting          */
/*  coercions, since they are not needed and the type system cannot          */
/*  handle nested inferences.                                                */
/*                                                                           */
/*  A successful match may have excess actual parameters.  If this           */
/*  happens, *excess_actuals will be set to those, *excess_fv to the         */
/*  "apply" feature that successfully handles them, and *excess_result_type  */
/*  to the result type of that apply.  We're only interested in using        */
/*  these values when confirming.                                            */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FindFunction(EXPR_CALL expr_call, CONTEXT cxt, TYPE self_type,
  BOOLEAN inferring, TYPE *excess_result_type, ARRAY_EXPR *excess_actuals,
  FEFN_FEATURE *excess_fv)
{
  FEFN_FEATURE_SET fvs;  FEFN fun;
  FEFN_FEATURE fv, fv2;  USTRING str;  EXPR target;  FEFN_BUILTIN bif;
  int i, j, clen, clen2, matches_count, count;

  if( DEBUG13 )
  {
    fprintf(stderr, "[ FindFunction(");
    ExprDebug((EXPR) expr_call, NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ")\n");
  }

  /* find either a matching function (fun) or feature view set (fvs) */
  fun = NULL;
  fvs = NULL;
  switch( expr_call->syntax_type )
  {

    case SYNTAX_VANILLA:

      /* f(a1, ... , an) */
      if( !FindInContext(expr_call, cxt, self_type, &fun, &fvs) )
      {
	fprintf(stderr, "%s: %s not found\n", FilePosShow(expr_call->file_pos),
	  UStringToUTF8(expr_call->value));
	db_return(DEBUG13, "FindFunction", FALSE);
      }
      if( DEBUG13 && fun != NULL )
	fprintf(stderr, "  FindFunction found vanilla %p: %s%s\n", (void *) fun,
	  NameShow(FEFnName(fun)), FEFnSignatureShow(fun, cxt));
      break;


    case SYNTAX_BUILTIN:

      /* builtin id(a1, ... , an) */
      count =
	expr_call->parameters != NULL ? ArraySize(expr_call->parameters) : 0;
      if( !FEFnBuiltinFind(expr_call->value, expr_call->file_pos, count, &bif) )
	db_return(DEBUG13, "FindFunction", FALSE);
      fun = (FEFN) bif;
      break;


    case SYNTAX_EXCLAM:
    case SYNTAX_EXCLAM2:

      /* [class]!f(a1, ... , an) or [class]!!f(a1, ... , an) */
      if( expr_call->uc_type == NULL )
	expr_call->uc_type = self_type;
      else if( !TypeManifest(expr_call->uc_type, cxt, FALSE) )
	db_return(DEBUG13, "FindFunction", FALSE);
      str = expr_call->syntax_type == SYNTAX_EXCLAM ? expr_call->value :
	UStringCat(AStringToUString("_"), expr_call->value);
      if( TypeRetrieveFEFnFeatureSet(expr_call->uc_type, cxt,
	    expr_call->file_pos, expr_call->value, &fvs) )
	fv = FEFnFeatureSetSoleUncoercedMember(fvs);
      else
	fv = NULL;
      if( fv == NULL )
      {
	fprintf(stderr, "%s: feature %s unknown\n",
	  FilePosShow(expr_call->file_pos), UStringToUTF8(expr_call->value));
        fprintf(stderr, "  (target type is %s)\n",
	  TypeShow(expr_call->uc_type, cxt));
	db_return(DEBUG13, "FindFunction", FALSE);
      }
      fun = (FEFN) fv;
      fvs = NULL;
      break;


    case SYNTAX_DOTTED:

      /* expr.f(a1, ... , an) */
      target = ArrayFirst(expr_call->parameters);
      if( !TypeRetrieveFEFnFeatureSet(target->type, cxt,
	  expr_call->file_pos, expr_call->value, &fvs) )
      {
	fprintf(stderr, "%s: feature %s unknown\n",
	  FilePosShow(expr_call->file_pos), UStringToUTF8(expr_call->value));
        fprintf(stderr, "  (target type is %s)\n", TypeShow(target->type, cxt));
	db_return(DEBUG13, "FindFunction", FALSE);
      }
      break;


    case SYNTAX_PREFIX:
    case SYNTAX_POSTFIX:
    case SYNTAX_INFIX:

      /* [ expr ] f(n1 := a1, ..., nn := an) [ expr ] */
      /* take from cxt if present, otherwise take positional expr as target */
      if( !FindInContext(expr_call, cxt, self_type, &fun, &fvs) )
      {
	/* only other possibility is a feature call */
	target = expr_call->syntax_type == SYNTAX_PREFIX ?
	  ArrayLast(expr_call->parameters) : ArrayFirst(expr_call->parameters);
	if( !TypeRetrieveFEFnFeatureSet(target->type, cxt,
	  expr_call->file_pos, NameKey(expr_call->op_name), &fvs) )
	{
	  fprintf(stderr, "%s: %s not found\n",FilePosShow(expr_call->file_pos),
	    UStringToUTF8(NameKey(expr_call->op_name)));
	  fprintf(stderr, "  (%s expression has type %s)\n",
	    expr_call->syntax_type==SYNTAX_PREFIX ? "following" : "preceding",
	    TypeShow(target->type, cxt));
	  db_return(DEBUG13, "FindFunction", FALSE);
	}
      }
      break;


    case SYNTAX_APPLY:

      /* expr(a1, ..., an) -> expr.applyn(a1, ..., an) */
      target = ArrayFirst(expr_call->parameters);
      expr_call->value = ApplyString(ArraySize(expr_call->parameters) - 1);
      /* expr_call->syntax_type = SYNTAX_DOTTED; */
      if( !TypeRetrieveFEFnFeatureSet(target->type, cxt, expr_call->file_pos,
	expr_call->value, &fvs) )
      {
	fprintf(stderr, "%s: preceding expression type has no %s feature\n",
	  FilePosShow(expr_call->file_pos), UStringToUTF8(expr_call->value));
	fprintf(stderr, "  (type of preceding expression is %s)\n",
	  TypeShow(target->type, cxt));
	db_return(DEBUG13, "FindFunction", FALSE);
      }
      break;


    default:

      assert(FALSE);
      break;

  }

  /* if fvs has only one member, move it across to fun */
  assert( (fun != NULL) != (fvs != NULL));
  if( fvs != NULL && FEFnFeatureSetSize(fvs) == 1 )
  {
    fv = FEFnFeatureSetFEFnFeature(fvs, 0);
    fun = (FEFN) fv;
    fvs = NULL;
  }

  /* match fun or fvs against the call */
  if( fun != NULL )
  {
    if( !Match(expr_call, fun, TRUE, cxt, self_type, inferring,
	  excess_result_type, excess_actuals, excess_fv) )
      db_return(DEBUG13, "FindFunction", FALSE);
  }
  else
  {
    /* find first matching signature */
    for( i = 0;  i < FEFnFeatureSetSize(fvs);  i++ )
    {
      fv = FEFnFeatureSetFEFnFeature(fvs, i);
      if( Match(expr_call, (FEFN) fv, FALSE, cxt, self_type,
	  inferring, excess_result_type, excess_actuals, excess_fv) )
	break;
    }

    /* error if no matching signature */
    if( i >= FEFnFeatureSetSize(fvs) )
    {
      fprintf(stderr, "%s: no matching signature for %s; choices were:\n",
	FilePosShow(expr_call->file_pos), UStringToUTF8(expr_call->value));
      for( i = 0;  i < FEFnFeatureSetSize(fvs);  i++ )
      {
	fv = FEFnFeatureSetFEFnFeature(fvs, i);
	fprintf(stderr, "  %d: %s\n", i+1,
	  FEFnSignatureShow((FEFN) fv, cxt));
      }
      db_return(DEBUG13, "FindFunction", FALSE);
    }

    /* make sure there is no subsequent matching signature of equal length */
    matches_count = 1;  /* i.e. fv */
    clen = CoercionLength(FEFnFeatureCoercion(fv));
    for( j = i+1;  j < FEFnFeatureSetSize(fvs);  j++ )
    {
      fv2 = FEFnFeatureSetFEFnFeature(fvs, j);
      clen2 = CoercionLength(FEFnFeatureCoercion(fv2));
      if( clen2 > clen )
	break;
      if( Match(expr_call, (FEFN) fv2, FALSE, cxt, self_type,
	    inferring, excess_result_type, excess_actuals, excess_fv) )
      {
	matches_count++;
	if( matches_count == 2 )
	{
	  fprintf(stderr, "%s: ambiguous overloading of %s; choices were:\n",
	    FilePosShow(expr_call->file_pos), UStringToUTF8(expr_call->value));
	  fprintf(stderr, "  1: %s [%s]\n",
	    FEFnSignatureShow((FEFN)fv, cxt),
	    CoercionShow(FEFnFeatureCoercion(fv)));
	}
	fprintf(stderr, "  %d: %s [%s]\n", matches_count,
	  FEFnSignatureShow((FEFN) fv2, cxt),
	  CoercionShow(FEFnFeatureCoercion(fv2)));
      }
    }
    if( matches_count >= 2 )
      db_return(DEBUG13, "FindFunction", FALSE);

    /* confirm the ith match and coerce the target if required */
    if( !Match(expr_call, (FEFN) fv, TRUE, cxt, self_type,
	  inferring, excess_result_type, excess_actuals, excess_fv ) )
      assert(FALSE);
    /* target coercion still to do */
  }

  if( DEBUG13 )
    fprintf(stderr, "] FindFunction returning %s%s\n",
      NameShow(FEFnName(expr_call->function)),
      FEFnSignatureShow(expr_call->function, cxt));
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallManifest(EXPR_CALL *e, CONTEXT cxt, TYPE self_type,      */
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Carry out the specification of ExprManifest on a call expression.        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCallManifest(EXPR_CALL *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn)
{
  EXPR_CALL expr_call;  TYPE gen;  int i;
  TYPE excess_result_type;  ARRAY_EXPR excess_actuals;
  FEFN_FEATURE excess_fv;

  expr_call = *e;
  expr_call->large_scale = FALSE;
  if( DEBUG8 )
  {
    fprintf(stderr, "[ ExprCallManifest(");
    ExprCallDebug(*e, cxt, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ")\n");
  }

  /* instantiate actual generic parameters if any */
  if( expr_call->generics != NULL )
    ArrayForEach(expr_call->generics, gen)
      if( !TypeManifest(gen, cxt, FALSE) )
	db_return(DEBUG8, "ExprCallManifest generics", FALSE);

  /* manifest actual parameters if any */
  if( expr_call->parameters != NULL )
    for( i = 0;  i < ArraySize(expr_call->parameters);  i++ )
    {
      if( !ExprManifest(&ArrayGet(expr_call->parameters, i), cxt, self_type,
	    encl_befn) )
	db_return(DEBUG8, "ExprCallManifest parameter", FALSE);
      if( ArrayGet(expr_call->parameters, i)->large_scale )
	expr_call->large_scale = TRUE;
    }

  /* set expr_call->function and expr_call->impl_function */
  if( !FindFunction(expr_call, cxt, self_type, TRUE, &excess_result_type,
	&excess_actuals, &excess_fv) )
    db_return(DEBUG8, "ExprCallManifest find function", FALSE);

  /* insert an apply node if needed */
  if( excess_actuals != NULL )
    InsertApply((EXPR *) e, excess_result_type, excess_actuals, excess_fv);

  if( DEBUG8 )
  {
    fprintf(stderr, "] ExprCallManifest returning TRUE: ");
    ExprCallDebug(*e, cxt, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ": %s\n", TypeShow((*e)->type, cxt));
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCallInsertCoercion(EXPR *e, FEFN_FEATURE fv, CONTEXT cxt,       */
/*    TYPE self_type)                                                        */
/*                                                                           */
/*  Expression *e has to be coerced by fv.  So if expr is the expression     */
/*  referred to by *e, replace it by expr.fv.  Expression *e is assumed      */
/*  here to be already manifested.                                           */
/*                                                                           */
/*****************************************************************************/

void ExprCallInsertCoercion(EXPR *e, FEFN_FEATURE fv, CONTEXT cxt,
  TYPE self_type)
{
  TYPE excess_result_type;  ARRAY_EXPR excess_actuals;
  FEFN_FEATURE excess_fv;

  /* make the call expression and insert it in place of *e */
  *e = ExprMakeDottedCall(NameRep(FEFnFeatureName(fv)), *e);

  /* the following is all that ExprCallManifest does after params manifested */
  if( !FindFunction((EXPR_CALL) *e, cxt, self_type, FALSE, &excess_result_type,
	&excess_actuals, &excess_fv) )
    assert(FALSE);
  assert(excess_actuals == NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprMakeInstantiatedFeatureCall(FEFN_FEATURE fv)                    */
/*                                                                           */
/*  Return an instantiated call on predefined feature fv (no parameters).    */
/*                                                                           */
/*****************************************************************************/

EXPR ExprMakeInstantiatedFeatureCall(FEFN_FEATURE fv)
{
  EXPR_CALL res;
  res = ExprCallNew(FEFnFeatureFilePos(fv), NameRep(FEFnFeatureName(fv)),
    SYNTAX_EXCLAM);
  res->uc_type = FEFnResultType((FEFN) fv);
  res->function = (FEFN) fv;
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprEncloseInInstantiatedAssert(EXPR e)                             */
/*                                                                           */
/*  Return assert(e), where assert is the builtin assert function.           */
/*                                                                           */
/*****************************************************************************/

EXPR ExprEncloseInInstantiatedAssert(EXPR e)
{
  EXPR_CALL res;
  assert(BuiltinAssertFn != NULL);
  res = ExprCallNew(e->file_pos,
    NameRep(FEFnName((FEFN) BuiltinAssertFn)), SYNTAX_VANILLA);
  res->function = (FEFN) BuiltinAssertFn;
  res->large_scale = e->large_scale;
  ArrayInit(&res->sorted_actuals);
  ArrayAddLast(res->sorted_actuals, e);
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "code generation".                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprCallSoleParameter(EXPR expr_call)                               */
/*                                                                           */
/*  Assuming that expr_call has exactly one actual parameter, return that    */
/*  actual parameter.                                                        */
/*                                                                           */
/*****************************************************************************/

EXPR ExprCallSoleParameter(EXPR e)
{
  EXPR_CALL expr_call;
  assert(e->kind_tag == KIND_EXPR_CALL);
  expr_call = (EXPR_CALL) e;
  assert(expr_call->parameters != NULL);
  assert(ArraySize(expr_call->parameters) == 1);
  return ArrayFirst(expr_call->parameters);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallIsConst(EXPR e, EXPR *root)                              */
/*                                                                           */
/*  Carry out the specification of ExprIsConst on call expression e.         */
/*                                                                           */
/*  If e denotes a coercion, look at its parameter.  Otherwise, if e         */
/*  denotes a predefined object, that's OK.  Else not constant.              */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCallIsConst(EXPR e, EXPR *root)
{
  EXPR_CALL expr_call;  FEFN_FEATURE fv;
  assert(e->kind_tag == KIND_EXPR_CALL);
  expr_call = (EXPR_CALL) e;
  if( NamedKind((NAMED) expr_call->function) == KIND_FEFN_FEATURE )
  {
    fv = (FEFN_FEATURE) expr_call->function;
    if( FEFnFeatureFType(fv) == FEATURE_PREDEF &&
	ClassIsEnum(ClassViewClass(FEFnFeatureClassView(fv))) )
    {
      /* we have a call on a predefined object feature */
      *root = e;
      return TRUE;
    }
    else if( FEFnFeatureIsCoerce(fv) )
    {
      /* we have a coercion; try its sole parameter (?) */
      return ExprIsConst(ArrayFirst(expr_call->parameters), root);
    }
  }
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  int ExprCallConstIntValue(EXPR e)                                        */
/*                                                                           */
/*  Carry out the specification of ExprConstIntValue on e, which is a root   */
/*  expression so can only be a call on a predefined object feature of an    */
/*  enumerated type.                                                         */
/*                                                                           */
/*****************************************************************************/

int ExprCallConstIntValue(EXPR e)
{
  EXPR_CALL expr_call;  FEFN_FEATURE fv;

  /* cast e into an expression call object */
  assert(e->kind_tag == KIND_EXPR_CALL);
  expr_call = (EXPR_CALL) e;
  
  /* must be a call on a predefined feature */
  assert(NamedKind((NAMED) expr_call->function) == KIND_FEFN_FEATURE);
  fv = (FEFN_FEATURE) expr_call->function;
  assert(FEFnFeatureFType(fv) == FEATURE_PREDEF);

  /* result is the code number */
  return FEFnFeatureCode(fv);
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCallCodeGen(EXPR_CALL expr_call, CODEGEN_OBJ res_be_var,        */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Carry out the specification of ExprCodeGen on call expression expr_call. */
/*  If the call is large-scale, variables representing its large-scale       */
/*  parameters need to be declared and evaluated before the actual           */
/*  function call is carried out.  Otherwise, just the call is needed.       */
/*                                                                           */
/*****************************************************************************/

void ExprCallCodeGen(EXPR_CALL expr_call, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  EXPR param;  CODEGEN_OBJ save_be_var;

  if( DEBUG6 )
  {
    fprintf(stderr, "[ ExprCallCodeGen(%s: ", FilePosShow(expr_call->file_pos));
    ExprCallDebug(expr_call, NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ")\n");
  }

  if( expr_call->large_scale )
  {
    /* print declarations for variables for the large-scale parameters */
    ArrayForEach(expr_call->parameters, param)
      if( param->large_scale )
	param->be_var =
	  be->VarMakeAndDeclare("tmp", NULL, TypeBEType(param->type, be));

    /* evaluate those same parameters */
    ArrayForEach(expr_call->parameters, param)
      if( param->large_scale )
      {
	save_be_var = param->be_var;
	param->be_var = NULL;
	ExprCodeGen(param, save_be_var, TypeBEType(param->type, be), be);
	param->be_var = save_be_var;
	param->large_scale = FALSE;  /* since just need be_var from now on */
      }

    /* generate the appropriate assignment and the function call */
    assert(res_be_var != NULL);
    be->VarAsstBegin(res_be_var);
    expr_call->large_scale = FALSE;
    BEFnCallCodeGen(FEFnBEFn(expr_call->function),
      expr_call->sorted_actuals, res_be_type, be);
    expr_call->large_scale = TRUE;
    be->AsstEnd();

    /* undo changes to params, to return to original state */
    ArrayForEach(expr_call->parameters, param)
      if( param->be_var != NULL )
      {
	be->ObjFree(param->be_var);
	param->be_var = NULL;
	param->large_scale = TRUE;
      }
  }
  else
  {
    /* just the function call */
    if( res_be_var != NULL )
      be->VarAsstBegin(res_be_var);
    BEFnCallCodeGen(FEFnBEFn(expr_call->function),
      expr_call->sorted_actuals, res_be_type, be);
    if( res_be_var != NULL )
      be->AsstEnd();
  }

  if( DEBUG6 )
    fprintf(stderr, "] ExprCallCodeGen returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallInitOrder(EXPR_CALL expr_call, int visit_num,            */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of ExprInitOrder on expr_call.               */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCallInitOrder(EXPR_CALL expr_call, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  EXPR param;  int i;

  if( DEBUG15 )
  {
    fprintf(stderr, "[ ExprCallInitOrder(%s: ",
      FilePosShow(expr_call->file_pos));
    ExprCallDebug(expr_call, NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ")\n");
  }

  /* check parameters */
  if( expr_call->parameters != NULL )
    for( i = 0;  i < ArraySize(expr_call->parameters);  i++ )
    {
      param = ArrayGet(expr_call->parameters, i);
      if( !ExprInitOrder(param, visit_num, report, fun) )
	db_return(DEBUG15, "ExprCallInitOrder (param)", FALSE);
    }

  /* check whatever function is called */
  if( !BEFnInitOrder(FEFnBEFn(expr_call->function), visit_num, report, fun) )
  {
    if( *report )
      fprintf(stderr, "  \"%s\" (defined at %s) called from %s\n",
	NameShow(FEFnName(expr_call->function)),
	FilePosShow(FEFnFilePos(expr_call->function)),
	FilePosShow(expr_call->file_pos));
    db_return(DEBUG15, "ExprCallInitOrder", FALSE);
  }
  db_return(DEBUG15, "ExprCallInitOrder", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCallWrapsBuiltin(EXPR e, ARRAY_BEFN_PARAM parameters,        */
/*    CODEGEN_OBJ *be_obj)                                                   */
/*                                                                           */
/*  Return TRUE if e is a call expression on a builtin function with         */
/*  the same parameters as the supplied list.                                */
/*                                                                           */
/*  This function is only called if e is a call expression.                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCallWrapsBuiltin(EXPR e, ARRAY_BEFN_PARAM parameters,
  CODEGEN_OBJ *be_obj)
{
  EXPR_CALL expr_call, actual_call;  EXPR actual;  BEFN_PARAM formal;
  int formal_param_count, actual_param_count, i;
  ExprKindCheck(e, KIND_EXPR_CALL, "ExprCallWrapsBuiltin");
  expr_call = (EXPR_CALL) e;

  /* function must be builtin */
  if( NamedKind((NAMED) expr_call->function) != KIND_FEFN_BUILTIN )
    return FALSE;

  /* must have same number of parameters */
  formal_param_count = parameters == NULL ? 0 : ArraySize(parameters);
  actual_param_count =
    expr_call->parameters == NULL ? 0 : ArraySize(expr_call->parameters);
  if( formal_param_count != actual_param_count )
    return FALSE;

  /* parameters must agree item by item */
  for( i = 0;  i < formal_param_count;  i++ )
  {
    formal = ArrayGet(parameters, i);
    actual = ArrayGet(expr_call->parameters, i);
    if( actual->kind_tag != KIND_EXPR_CALL )
      return FALSE;
    actual_call = (EXPR_CALL) actual;
    if( FEFnBEFn(actual_call->function) != (BEFN) formal )
      return FALSE;
  }

  /* parameters match, so we have a wrapped builtin */
  *be_obj = BEFnBEObj(FEFnBEFn(expr_call->function));
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "debug".                                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void ExprCallOptionalParamsDebug(ARRAY_EXPR params, CONTEXT cxt,         */
/*    BOOLEAN show_types, FILE *fp, EXPR skip1, EXPR skip2)                  */
/*                                                                           */
/*  If parameter array params is non-empty, print it in the usual format     */
/*  (parentheses and commas), with optional "name :=" in front of each.      */
/*                                                                           */
/*  Ignore children equal to skip1 and skip2 (these will be                  */
/*  operator left and/or right parameters, to be printed separately).        */
/*  The list of non-ignorable parameters could be empty, in which case       */
/*  print nothing.                                                           */
/*                                                                           */
/*****************************************************************************/

static void ExprCallOptionalParamsDebug(ARRAY_EXPR params, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, EXPR skip1, EXPR skip2)
{
  EXPR child;  BOOLEAN started = FALSE;
  if( params != NULL )
  {
    ArrayForEach(params, child)
    {
      if( child != skip1 && child != skip2 )
      {
        /* found something to print; ensure preceding ( or , is there */
        fprintf(fp, started ? ", " : "(");
        started = TRUE;
  
        /* now print the actual parameter, including any param_names */
        ExprDebug(child, NULL, show_types, fp, SINGLE_LINE);
      }
    }
    if( started )
      fprintf(fp, ")");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCallDebug(EXPR_CALL expr_call, CONTEXT cxt, BOOLEAN show_types, */
/*    FILE *fp, int print_style)                                             */
/*                                                                           */
/*  Debug print of expr_call on *fp.  After manifesting there may be         */
/*  the odd peculiarity, such as "self" inserted etc.                        */
/*                                                                           */
/*****************************************************************************/
#define DETAILS 0

void ExprCallDebug(EXPR_CALL expr_call, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style)
{
  EXPR e1, e2;  ARRAY_EXPR params;
  params = expr_call->sorted_actuals != NULL ?
    expr_call->sorted_actuals : expr_call->parameters;
  switch( expr_call->syntax_type )
  {
    case SYNTAX_VANILLA:

      fprintf(fp, "%s%s", UStringToUTF8(expr_call->value),
	TypeArrayShow(expr_call->generics, cxt));
      ExprCallOptionalParamsDebug(params, cxt, show_types, fp, NULL, NULL);
      break;


    case SYNTAX_BUILTIN:

      fprintf(fp, "builtin \"%s\"%s", UStringToUTF8(expr_call->value),
	TypeArrayShow(expr_call->generics, cxt));
      ExprCallOptionalParamsDebug(params, cxt, show_types, fp, NULL, NULL);
      break;


    case SYNTAX_EXCLAM:
    case SYNTAX_EXCLAM2:

      if( expr_call->uc_type != NULL )
        fprintf(fp, (ASTRING) TypeShow(expr_call->uc_type, cxt));
      fprintf(fp, "%s%s%s",
	expr_call->syntax_type == SYNTAX_EXCLAM ? "!" : "!!",
	UStringToUTF8(expr_call->value),
	TypeArrayShow(expr_call->generics,cxt));
      ExprCallOptionalParamsDebug(params, cxt, show_types, fp, NULL, NULL);
      break;


    case SYNTAX_DOTTED:

      ExprDebug(ArrayFirst(params), cxt, show_types, fp, SINGLE_LINE);
      fprintf(fp, ".%s%s", UStringToUTF8(expr_call->value),
	TypeArrayShow(expr_call->generics, cxt));
      ExprCallOptionalParamsDebug(params, cxt, show_types, fp,
	ArrayFirst(params), NULL);
      break;


    case SYNTAX_PREFIX:
    case SYNTAX_INFIX:
    case SYNTAX_POSTFIX:

      e1 = expr_call->syntax_type == SYNTAX_PREFIX ? NULL : ArrayFirst(params);
      e2 = expr_call->syntax_type == SYNTAX_POSTFIX ? NULL : ArrayLast(params);
      fprintf(fp, "(");
      if( e1 != NULL )
      {
	ExprDebug(e1, cxt, show_types, fp, SINGLE_LINE);
	fprintf(fp, " ");
      }
      if( DETAILS )
	fprintf(fp, "%s%s%s", UStringToUTF8(NameKey(expr_call->op_name)),
          expr_call->syntax_type == SYNTAX_PREFIX ? "<pre>" :
          expr_call->syntax_type == SYNTAX_INFIX ? "<inf>" : "<post>",
	  TypeArrayShow(expr_call->generics, cxt));
      else
	fprintf(fp, "%s%s", UStringToUTF8(expr_call->value),
	  TypeArrayShow(expr_call->generics, cxt));
      ExprCallOptionalParamsDebug(params, cxt, show_types, fp, e1, e2);
      if( e2 != NULL )
      {
	fprintf(fp, " ");
	ExprDebug(e2, cxt, show_types, fp, SINGLE_LINE);
      }
      fprintf(fp, ")");
      break;


    case SYNTAX_APPLY:

      e1 = ArrayFirst(params);
      ExprDebug(e1, cxt, show_types, fp, SINGLE_LINE);
      ExprCallOptionalParamsDebug(params, cxt, show_types, fp, e1, NULL);
      break;


    default:

      assert(FALSE);
      break;

  }
}
