/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_case.c                                                */
/*  DESCRIPTION:  Nonpareil case expressions                                 */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0


typedef struct case_label_rec *			CASE_LABEL;
typedef ARRAY(CASE_LABEL)			ARRAY_CASE_LABEL;
typedef struct case_entry_rec *			CASE_ENTRY;
typedef ARRAY(CASE_ENTRY)			ARRAY_CASE_ENTRY;

/*****************************************************************************/
/*                                                                           */
/*  CASE_INTERVAL, and CASE_PARTITION - private, for codegen.                */
/*                                                                           */
/*****************************************************************************/
#define ExprIsString(e) ((e)->kind_tag == KIND_EXPR_LIT_STRING)

typedef struct case_interval_rec {
  FILE_POS	file_pos;
  CASE_ENTRY	action;
  EXPR		start_point;
  BOOLEAN	start_closed;
  EXPR		end_point;
  BOOLEAN	end_closed;
} *CASE_INTERVAL;

typedef ARRAY(CASE_INTERVAL) ARRAY_CASE_INTERVAL;
#define CASE_PARTITION ARRAY_CASE_INTERVAL


/*****************************************************************************/
/*                                                                           */
/*  EXPR_CASE                                                                */
/*                                                                           */
/*  A case expression.                                                       */
/*                                                                           */
/*****************************************************************************/

struct case_label_rec {
  EXPR			first_label;	/* the first label of the pair       */
  EXPR			second_label;	/* the optional second label         */
};

struct case_entry_rec {
  ARRAY_CASE_LABEL	labels;		/* labels (empty for "else" entry)   */
  EXPR			sub_expr;	/* expression of entry               */
  ARRAY_CASE_INTERVAL	case_intervals;	/* all intervals holding sub_expr    */
  CODEGEN_OBJ		jump_label;	/* jump label when sharing code      */
};

struct expr_case_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* file position of expression       */
  USTRING		param_name;	/* param name token when := present  */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */
  EXPR			test_expr;	/* the test expression               */
  CASE_TYPE		case_type;	/* classification of test type       */
  ARRAY_CASE_ENTRY	entries;	/* the entries of the case           */
  BOOLEAN		error_printed;	/* TRUE if "else" error mess printed */
};


/*****************************************************************************/
/*                                                                           */
/*  CASE_ENTRY CaseEntryNew(ARRAY_CASE_LABEL labels, EXPR sub_expr)          */
/*                                                                           */
/*  Return a new case entry with these attributes.                           */
/*                                                                           */
/*****************************************************************************/

static CASE_ENTRY CaseEntryNew(ARRAY_CASE_LABEL labels, EXPR sub_expr)
{
  CASE_ENTRY res;
  GetMemory(res, CASE_ENTRY);
  res->labels = labels;
  res->sub_expr = sub_expr;
  ArrayInit(&res->case_intervals);
  res->jump_label = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCaseParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)               */
/*                                                                           */
/*  Parse a case expression.                                                 */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ParseCaseLabel(TOKEN *t, SYSTEM_VIEW sv, CASE_LABEL *res)
{
  if( DEBUG1 )
    fprintf(stderr, "[ ParseCaseLabel\n");
  GetMemory(*res, CASE_LABEL);
  (*res)->second_label = NULL;
  if( !ExprParse(t, sv, &(*res)->first_label) )
    db_return(DEBUG1, "ParseCaseLabel (1)", FALSE);
  if( LexType(curr_token) == TK_DOT_DOT )
  {
    next_token;
    if( !ExprParse(t, sv, &(*res)->second_label) )
      db_return(DEBUG1, "ParseCaseLabel (2)", FALSE);
  }
  db_return(DEBUG1, "ParseCaseLabel", TRUE);
}

BOOLEAN ExprCaseParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  EXPR_CASE res_case;  CASE_LABEL case_label;  CASE_ENTRY entry;
  ARRAY_CASE_LABEL labels;  EXPR sub_expr;

  /* set up res_case and *res */
  if( DEBUG1 )
    fprintf(stderr, "[ ExprCaseParse\n");
  ExprNew(res_case, EXPR_CASE, KIND_EXPR_CASE, LexFilePos(curr_token), NULL);
  ArrayInit(&res_case->entries);
  /* res_case->case_partition = NULL; */
  res_case->error_printed = FALSE;
  *res = (EXPR) res_case;

  /* skip initial case keyword, known to be there */
  next_token;

  /* parse test expression */
  if( !ExprParse(t, sv, &res_case->test_expr) )
    db_return(DEBUG1, "ExprCaseParse (1)", FALSE);

  /* parse optional when-yield pairs */
  while( LexType(curr_token) == TK_WHEN )
  {
    /* skip initial when */
    next_token;

    /* parse sequence of expression optional pairs */
    ArrayInit(&labels);
    if( !ParseCaseLabel(t, sv, &case_label) )
      db_return(DEBUG1, "ExprCaseParse (2)", FALSE);
    ArrayAddLast(labels, case_label);
    while( LexType(curr_token) == TK_COMMA )
    {
      next_token;
      if( !ParseCaseLabel(t, sv, &case_label) )
	db_return(DEBUG1, "ExprCaseParse (3)", FALSE);
      ArrayAddLast(labels, case_label);
    }

    /* parse concluding yield and expression */
    skip(TK_YIELD, "\"yield\" in \"case\" expression");
    if( !ExprParse(t, sv, &sub_expr) )
      db_return(DEBUG1, "ExprCaseParse (4)", FALSE);

    /* make case entry and add to entries */
    entry = CaseEntryNew(labels, sub_expr);
    ArrayAddLast(res_case->entries, entry);
  }

  /* parse optional concluding else part and make a case entry for it */
  if( LexType(curr_token) == TK_ELSE )
  {
    next_token;
    if( !ExprParse(t, sv, &sub_expr) )
      db_return(DEBUG1, "ExprCaseParse (5)", FALSE);
    entry = CaseEntryNew(NULL, sub_expr);
    ArrayAddLast(res_case->entries, entry);
  }
  skip(TK_END, "concluding \"end\" of \"case\" expression");
  db_return(DEBUG1, "ExprCaseParse", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  CASE_LABEL LabelCopyUninstantiated(CASE_LABEL label,                     */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Helper function for EntryCopyUninstantiated just below.  Make a copy     */
/*  of an uninstantiated case label.                                         */
/*                                                                           */
/*****************************************************************************/

static CASE_LABEL LabelCopyUninstantiated(CASE_LABEL label,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  CASE_LABEL res;
  GetMemory(res, CASE_LABEL);
  res->first_label =
    ExprCopyUninstantiated(label->first_label, orig_params, copy_params);
  res->second_label =
    ExprCopyUninstantiated(label->second_label, orig_params, copy_params);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  CASE_ENTRY EntryCopyUninstantiated(CASE_ENTRY entry,                     */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Helper function for ExprCaseCopyUninstantiated just below.  Make a       */
/*  copy of an uninstantiated case entry.                                    */
/*                                                                           */
/*****************************************************************************/

static CASE_ENTRY EntryCopyUninstantiated(CASE_ENTRY entry,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  CASE_ENTRY res;  CASE_LABEL label;
  GetMemory(res, CASE_ENTRY);
  ArrayInit(&res->labels);
  ArrayForEach(entry->labels, label)
    ArrayAddLast(res->labels,
      LabelCopyUninstantiated(label, orig_params, copy_params));
  res->sub_expr =
    ExprCopyUninstantiated(entry->sub_expr, orig_params, copy_params);
  assert(ArraySize(entry->case_intervals) == 0);
  ArrayInit(&res->case_intervals);
  res->jump_label = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprCaseCopyUninstantiated(EXPR_CASE expr_case,                     */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on case            */
/*  expression expr_case.                                                    */
/*                                                                           */
/*****************************************************************************/

EXPR ExprCaseCopyUninstantiated(EXPR_CASE expr_case,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_CASE res;  CASE_ENTRY entry;
  ExprNew(res, EXPR_CASE, KIND_EXPR_CASE, expr_case->file_pos,
    expr_case->param_name);
  res->test_expr = ExprCopyUninstantiated(expr_case->test_expr,
    orig_params, copy_params);
  res->entries = NULL;
  if( expr_case->entries != NULL )
  {
    ArrayInit(&res->entries);
    ArrayForEach(expr_case->entries, entry)
      ArrayAddLast(res->entries,
	EntryCopyUninstantiated(entry, orig_params, copy_params));
  }
  res->error_printed = FALSE;
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprLabelManifest(EXPR *label, CONTEXT cxt, TYPE self_type,      */
/*    BEFN encl_befn, TYPE test_type, ASTRING second)                        */
/*                                                                           */
/*  Manifest a case label.  Parameter test_type is the type of the test      */
/*  expression that the label has to match; second is "" if this is a first  */
/*  label, or "second " if it is a second one (i.e. after ..); the other     */
/*  parameters are as for ExprManifest.                                      */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ExprLabelManifest(EXPR *label, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn, TYPE test_type, ASTRING second)
{
  COERCION c;  EXPR root;

  /* manifest the label */
  if( !ExprManifest(label, cxt, self_type, encl_befn) )
    return FALSE;

  /* check that the label is a constant expression */
  if( !ExprIsConst(*label, &root) )
  {
    fprintf(stderr, "%s: %scase label is not a constant expression\n",
      FilePosShow((*label)->file_pos), second);
    return FALSE;
  }

  /* check that the label type matches the test type, and return */
  if( !TypeIsSubType((*label)->type, test_type, &c, cxt) )
  {
    fprintf(stderr, "%s: type of %scase label (%s) does not match\n",
      FilePosShow((*label)->file_pos), second, TypeShow((*label)->type, cxt));
    return FALSE;
  }
  CoercionInsert(c, label, cxt, self_type);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCaseManifest(EXPR_CASE *e, CONTEXT cxt, TYPE self_type,      */
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Carry out specification of ExprManifest on a case expression.            */
/*  Also check that the case expressions are constant expressions.           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCaseManifest(EXPR_CASE *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn)
{
  EXPR_CASE e_case;  TYPE test_type, joined_type;
  CASE_LABEL label;  CASE_ENTRY entry;

  /* manifest test expression */
  e_case = (EXPR_CASE) *e;
  e_case->large_scale = TRUE;
  if( !ExprManifest(&e_case->test_expr, cxt, self_type, encl_befn) )
    return FALSE;
  test_type = e_case->test_expr->type;

  /* record case_type and check that it is suitable */
  e_case->case_type = TypeCaseType(test_type);
  if( e_case->case_type == CASE_TYPE_OTHER )
  {
    fprintf(stderr, "%s: case test expression has unsuitable type %s\n",
      FilePosShow(e_case->test_expr->file_pos), TypeShow(test_type, cxt));
    return FALSE;
  }

  /* manifest the entries */
  joined_type = NULL;
  ArrayForEach(e_case->entries, entry)
  {
    /* manifest the ith labels (unless last subexpr), not if result_type_only */
    if( entry->labels != NULL ) ArrayForEach(entry->labels, label)
    {
      /* manifest compulsory first label; type must equal test_expr */
      if( !ExprLabelManifest(&label->first_label, cxt, self_type, encl_befn,
	  test_type, "") )
	return FALSE;

      /* manifest optional second label; type must equal test_expr */
      if( label->second_label != NULL &&
	  !ExprLabelManifest(&label->second_label, cxt, self_type, encl_befn,
	  test_type, "second ") )
	return FALSE;
    }

    /* manifest the ith subexpression */
    if( !ExprManifest(&entry->sub_expr, cxt, self_type, encl_befn) )
      return FALSE;

    /* join the ith subexpression's type into joined_type */
    if( !TypeJoin(&joined_type, entry->sub_expr->type, cxt) )
    {
      fprintf(stderr, "%s: incompatible type of \"case\" branch\n",
	FilePosShow(entry->sub_expr->file_pos));
      return FALSE; 
    }
  }

  e_case->type = joined_type;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCaseDebug(EXPR_CASE expr_case, CONTEXT cxt, BOOLEAN show_types, */
/*    FILE *fp, int print_style)                                             */
/*                                                                           */
/*  Debug print of case expr e on *fp.                                       */
/*                                                                           */
/*****************************************************************************/

void ExprCaseDebug(EXPR_CASE expr_case, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style)
{
  CASE_ENTRY entry;  CASE_LABEL label;
  fprintf(fp, "case ");
  ExprDebug(expr_case->test_expr, cxt, show_types, fp, SINGLE_LINE);
  ArrayForEach(expr_case->entries, entry)
  {
    next_line;
    if( entry->labels != NULL )
    {
      /* when-yield */
      fprintf(fp, "when ");
      ArrayForEach(entry->labels, label)
      {
	if( label != ArrayFirst(entry->labels) )
	  fprintf(fp, ", ");
	ExprDebug(label->first_label, cxt, show_types, fp, SINGLE_LINE);
	if( label->second_label != NULL )
	{
	  fprintf(fp, " .. ");
	  ExprDebug(label->second_label, cxt, show_types, fp, SINGLE_LINE);
	}
      }
      fprintf(fp, " yield");
    }
    else
    {
      /* else */
      fprintf(fp, "else");
    }
    begin_indent;
    next_line;
    ExprDebug(entry->sub_expr, cxt, show_types, fp, print_style);
    end_indent;
  }
  next_line;
  fprintf(fp, "end");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "code generation (case intervals)".                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void CaseIntervalDebug(CASE_INTERVAL ci, FILE *fp)                       */
/*                                                                           */
/*  Debug print of case interval ci onto fp.                                 */
/*                                                                           */
/*****************************************************************************/

static void CaseIntervalDebug(CASE_INTERVAL ci, FILE *fp)
{
  if( ci == NULL )
    fprintf(fp, "NULL");
  else
  {
    fprintf(fp, "%s", ci->start_closed ? "[" : "(");
    ExprDebug(ci->start_point, NULL, FALSE, fp, SINGLE_LINE);
    fprintf(fp, ", ");
    ExprDebug(ci->end_point, NULL, FALSE, fp, SINGLE_LINE);
    fprintf(fp, "%s", ci->end_closed ? "]" : ")");
  }
}


/*****************************************************************************/
/*                                                                           */
/*  CASE_INTERVAL CaseIntervalNew(FILE_POS file_pos, CASE_ENTRY action,      */
/*    EXPR start_point, BOOLEAN start_closed,                                */
/*    EXPR end_point, BOOLEAN end_closed)                                    */
/*                                                                           */
/*  Make and return a new case interval with these attributes.               */
/*                                                                           */
/*****************************************************************************/

CASE_INTERVAL CaseIntervalNew(FILE_POS file_pos, CASE_ENTRY action,
  EXPR start_point, BOOLEAN start_closed,
  EXPR end_point, BOOLEAN end_closed)
{
  CASE_INTERVAL res;
  GetMemory(res, CASE_INTERVAL);
  res->file_pos = file_pos;
  res->action = action;
  res->start_point = start_point;
  res->start_closed = start_closed;
  res->end_point = end_point;
  res->end_closed = end_closed;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int CaseIntervalIncreasingCmp(const void *t1, const void *t2)            */
/*                                                                           */
/*  Comparison function for sorting case intervals by increasing             */
/*  start_point.                                                             */
/*                                                                           */
/*****************************************************************************/

static int CaseIntervalIncreasingCmp(const void *t1, const void *t2)
{
  CASE_INTERVAL cs1 = * (CASE_INTERVAL *) t1;
  CASE_INTERVAL cs2 = * (CASE_INTERVAL *) t2;
  return ExprConstCmp(cs1->start_point, cs2->start_point);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN IntervalIsEmpty(EXPR start_point, BOOLEAN start_closed,          */
/*    EXPR end_point, BOOLEAN end_closed)                                    */
/*                                                                           */
/*  Return true if the interval from start_point to end_point is empty.      */
/*  Endpoints are open or closed as given by start_closed and end_closed.    */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN IntervalIsEmpty(EXPR start_point, BOOLEAN start_closed,
  EXPR end_point, BOOLEAN end_closed)
{
  int cmp, first_included, last_included;
  cmp = ExprConstCmp(start_point, end_point);
  if( cmp > 0 )
  {
    /* start point is to the right of end point, so interval is empty */
    return TRUE;
  }
  else if( cmp == 0 )
  {
    /* end points are equal, so empty if either endpoint is open */
    return !start_closed || !end_closed;
  }
  else if( ExprIsString(end_point) )
  {
    /* start_point strictly left of end_point, and type is string */
    return FALSE;
  }
  else
  {
    /* start_point strictly left of end_point, and type is integer */
    /* these operations are safe even near INT_MAX and INT_MIN     */
    first_included = start_closed ?
      ExprConstIntValue(start_point) : ExprConstIntValue(start_point) + 1;
    last_included = end_closed ?
      ExprConstIntValue(end_point) : ExprConstIntValue(end_point) - 1;
    return first_included > last_included;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN CaseIntervalsMergeable(CASE_INTERVAL ci1, CASE_INTERVAL ci2)     */
/*                                                                           */
/*  Return TRUE if these two case intervals may be merged into one.          */
/*  This is the same question as whether the interval separating             */
/*  them is empty - the interval from ci1's end point to ci2's start point,  */
/*  with the ends open or closed as the boundaries are closed or open.       */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN CaseIntervalsMergeable(CASE_INTERVAL ci1, CASE_INTERVAL ci2)
{
  return IntervalIsEmpty(ci1->end_point, !ci1->end_closed,
    ci2->start_point, !ci2->start_closed);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN CaseIntervalsIntersect(CASE_INTERVAL ci1, CASE_INTERVAL ci2)     */
/*                                                                           */
/*  Return TRUE if ci1 and ci2 have at least one case point in common.       */
/*  Both intervals are assumed to be non-empty.                              */
/*                                                                           */
/*  The first case is when ci2's start point is equal to or to the right     */
/*  of ci1's end point:                                                      */
/*                                                                           */
/*      [ ci1 ]           or   [ ci1 ]                                       */
/*            [ ci2 ]                  [ ci2 ]                               */
/*                                                                           */
/*  In this case the result is FALSE if ci2's start point is strictly to     */
/*  the right; if it is equal, it is TRUE iff both intervals are closed,     */
/*  using the assumption that they are non-empty.                            */
/*                                                                           */
/*  The second case is when ci1's start point is equal to or to the right    */
/*  of ci2's end point:                                                      */
/*                                                                           */
/*            [ ci1 ]                  [ ci1 ]                               */
/*      [ ci2 ]           or   [ ci2 ]                                       */
/*                                                                           */
/*  This case is symmetrical with the first case: the result is FALSE if     */
/*  ci1's start point is strictly to the right; if it is equal it is TRUE    */
/*  iff both intervals are closed, again assuming both intervals non-empty.  */
/*                                                                           */
/*  These two cases cover the only ways that these two intervals could be    */
/*  disjoint.  When neither case applies, the two intervals intersect.       */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN CaseIntervalsIntersect(CASE_INTERVAL ci1, CASE_INTERVAL ci2)
{
  int cmp1, cmp2;

  /* handle first case as described above */
  cmp1 = ExprConstCmp(ci1->end_point, ci2->start_point);
  if( cmp1 < 0 )
    return FALSE;
  else if( cmp1 == 0 )
    return ci1->end_closed && ci2->start_closed;

  /* handle second case as described above */
  cmp2 = ExprConstCmp(ci2->end_point, ci1->start_point);
  if( cmp2 < 0 )
    return FALSE;
  else if( cmp2 == 0 )
    return ci2->end_closed && ci1->start_closed;

  /* if neither case applies, they intersect */
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN CaseIntervalMake(FILE_POS file_pos, EXPR start_point,            */
/*    EXPR end_point, CASE_ENTRY action, CASE_INTERVAL *res)                 */
/*                                                                           */
/*  Make a new closed case interval from these attributes, unless it         */
/*  does not pan out.                                                        */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN CaseIntervalMake(FILE_POS file_pos, EXPR start_point,
  EXPR end_point, CASE_ENTRY action, CASE_INTERVAL *res)
{
  /* check for empty and make the interval */
  if( IntervalIsEmpty(start_point, TRUE, end_point, TRUE) )
  {
    *res = NULL;
    fprintf(stderr, "%s: empty case label range\n",
      FilePosShow(start_point->file_pos));
    return FALSE;
  }
  else
  {
    *res = CaseIntervalNew(file_pos, action, start_point, TRUE,end_point,TRUE);
    return TRUE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void DoGap(TYPE expr_type, CASE_INTERVAL prev_ci, CASE_INTERVAL succ_ci, */
/*    EXPR *start_point, BOOLEAN *start_closed,                              */
/*    EXPR *end_point, BOOLEAN *end_closed)                                  */
/*                                                                           */
/*  Get the start and end points of an interval that fills the gap between   */
/*  prev_ci's endpoint and succ_ci's start point.                            */
/*                                                                           */
/*  If prev_ci is NULL, there is no predecessor so the interval is to        */
/*  start at the smallest legal value.  If succ_ci is NULL, there is no      */
/*  successor so the interval is to end at the largest legal value.          */
/*                                                                           */
/*****************************************************************************/

static void DoGap(TYPE expr_type, CASE_INTERVAL prev_ci, CASE_INTERVAL succ_ci,
  EXPR *start_point, BOOLEAN *start_closed,
  EXPR *end_point, BOOLEAN *end_closed)
{
  /* get start point */
  if( prev_ci == NULL )
  {
    *start_point = TypeMin(expr_type);
    *start_closed = TRUE;
  }
  else
  {
    *start_point = prev_ci->end_point;
    *start_closed = !prev_ci->end_closed;
  }

  /* get end point */
  if( succ_ci == NULL )
  {
    *end_point = TypeMax(expr_type);
    *end_closed = TRUE;
  }
  else
  {
    *end_point = succ_ci->start_point;
    *end_closed = !succ_ci->start_closed;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN CaseIntervalGetGap(EXPR_CASE e, CASE_INTERVAL prev_ci,           */
/*    CASE_INTERVAL succ_ci, CASE_INTERVAL *res)                             */
/*                                                                           */
/*  Look into the question of whether a gap is required between prev_ci      */
/*  and succ_ci.                                                             */
/*                                                                           */
/*  Return TRUE if this operation did not detect any errors; that is, if     */
/*  it was not the case that a gap was required but e had no else case       */
/*  to supply its action.  Print an error message in the error case,         */
/*  unless this error message has already been printed for this e.           */
/*                                                                           */
/*  Set *res to a new case interval object if a gap is required, or to       */
/*  NULL if it is not.                                                       */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN CaseIntervalGetGap(EXPR_CASE e, CASE_INTERVAL prev_ci,
  CASE_INTERVAL succ_ci, CASE_INTERVAL *res)
{
  EXPR start_point, end_point;  CASE_ENTRY last_entry;
  BOOLEAN start_closed, end_closed;  CASE_ENTRY action;
  if( DEBUG4)
  {
    fprintf(stderr, "[ CaseIntervalGetGap(e: %s, ", TypeShow(e->type, NULL));
    CaseIntervalDebug(prev_ci, stderr);
    fprintf(stderr, ", ");
    CaseIntervalDebug(succ_ci, stderr);
    fprintf(stderr, ", *res)\n");
  }

  /* get the action */
  last_entry = ArrayLast(e->entries);
  action = (last_entry->labels == NULL ? last_entry : NULL);

  /* get the gap */
  DoGap(e->test_expr->type, prev_ci, succ_ci, &start_point, &start_closed,
    &end_point, &end_closed);

  /* make a case interval if the gap is non-empty */
  if( IntervalIsEmpty(start_point, start_closed, end_point, end_closed) )
    *res = NULL;
  else
  {
    /* it's an error if *res is non-empty but action is NULL */
    if( action == NULL )
    {
      if( !e->error_printed )
      {
	fprintf(stderr, "%s: case expression does not handle all values\n",
	  FilePosShow(e->file_pos));
	e->error_printed = TRUE;
      }
      db_return(DEBUG4, "CaseIntervalGetGap", FALSE);
    }
    *res = CaseIntervalNew(action->sub_expr->file_pos, action,
      start_point, start_closed, end_point, end_closed);
  }

  /* wrapup */
  if( DEBUG4 )
  {
    fprintf(stderr, "] CaseIntervalGetGap returning TRUE, *res = ");
    CaseIntervalDebug(*res, stderr);
    fprintf(stderr, "\n");
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void StringBEObjInit(ASTRING key, CODEGEN_OBJ *be_obj)                   */
/*                                                                           */
/*  Find string backend functions in the string class.                       */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_OBJ String_Lt = NULL;
static CODEGEN_OBJ String_Le = NULL;

static void StringBEObjInit(ASTRING key, CODEGEN_OBJ *be_obj)
{
  CLASS_VIEW cv;  FEFN_FEATURE_SET fvs;  BEFN_FEATURE f;
  cv = ClassOrigClassView(ClassString);
  if( !ClassViewRetrieveFEFnFeatureSet(cv, AStringToUString(key), &fvs) )
  {
    fprintf(stderr, "required feature \"<\" not present in class \"%s\"\n",
      NameShow(ClassViewName(cv)));
    exit(1);
  }
  f = FEFnFeatureBEFnFeature(FEFnFeatureSetSoleUncoercedMember(fvs));
  *be_obj = BEFnBEObj((BEFN) f);
}


/*****************************************************************************/
/*                                                                           */
/*  void LowerLimitTest(CASE_INTERVAL ci, CODEGEN_OBJ test_be_var,           */
/*    CODEGEN_TYPE test_be_type, CODEGEN be)                                 */
/*                                                                           */
/*  Print a C condition that will be true of every case point in or          */
/*  above ci, and false of every case point below ci.  The value to be       */
/*  tested is kept in test_var_name.                                         */
/*                                                                           */
/*****************************************************************************/

static void LowerLimitTest(CASE_INTERVAL ci, CODEGEN_OBJ test_be_var,
  CODEGEN_TYPE test_be_type, CODEGEN be)
{
  if( String_Lt == NULL )
  {
    StringBEObjInit("<", &String_Lt);
    StringBEObjInit("<=", &String_Le);
  }
  if( ExprIsString(ci->end_point) )
    Call2(ci->start_closed ? String_Le : String_Lt,
      ExprCodeGen(ci->start_point, NULL, test_be_type, be), Var(test_be_var));
  else
    Call2(ci->start_closed ? be->le : be->lt,
      ExprCodeGen(ci->start_point, NULL, test_be_type, be), Var(test_be_var));
}


/*****************************************************************************/
/*                                                                           */
/*  void UpperLimitTest(CASE_INTERVAL ci, CODEGEN_OBJ test_be_var,           */
/*    CODEGEN_TYPE test_be_type, CODEGEN be)                                 */
/*                                                                           */
/*  Print a C condition that will be true of every case point in or          */
/*  below ci, and false of every case point above ci.  The value to be       */
/*  tested is kept in test_var_name.                                         */
/*                                                                           */
/*****************************************************************************/

static void UpperLimitTest(CASE_INTERVAL ci, CODEGEN_OBJ test_be_var,
  CODEGEN_TYPE test_be_type, CODEGEN be)
{
  if( String_Lt == NULL )
  {
    StringBEObjInit("<", &String_Lt);
    StringBEObjInit("<=", &String_Le);
  }
  if( ExprIsString(ci->end_point) )
    Call2(ci->end_closed ? String_Le : String_Lt, Var(test_be_var),
      ExprCodeGen(ci->start_point, NULL, test_be_type, be));
  else
    Call2(ci->end_closed ? be->le : be->lt, Var(test_be_var),
      ExprCodeGen(ci->start_point, NULL, test_be_type, be));
}


/*****************************************************************************/
/*                                                                           */
/*  void InclusionTest(CASE_INTERVAL ci, CODEGEN_OBJ test_be_var,            */
/*    CODEGEN_TYPE test_be_type, CODEGEN be)                                 */
/*                                                                           */
/*  Print a test that will be true of every case point in ci.                */
/*                                                                           */
/*****************************************************************************/

void InclusionTest(CASE_INTERVAL ci, CODEGEN_OBJ test_be_var,
  CODEGEN_TYPE test_be_type, CODEGEN be)
{
  if( ExprConstCmp(ci->start_point, ci->end_point) == 0 )
  {
    /* end points equal, do an equality test */
    assert(ci->start_closed && ci->end_closed);
    Call2(be->eq, Var(test_be_var),
      ExprCodeGen(ci->start_point, NULL, test_be_type, be));
  }
  else
  {
    /* end points unequal, do an interval test */
    Call2(be->logical_and,
      LowerLimitTest(ci, test_be_var, test_be_type, be),
      UpperLimitTest(ci, test_be_var, test_be_type, be));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "code generation (case partitions)".                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void CasePartitionDebug(CASE_PARTITION cpart, int indent, FILE *fp)      */
/*                                                                           */
/*  Debug print of cpart onto fp with the given indent.                      */
/*                                                                           */
/*****************************************************************************/

static void CasePartitionDebug(CASE_PARTITION cpart, int indent, FILE *fp)
{
  CASE_INTERVAL ci;
  fprintf(fp, "%*s[ case partition\n", indent, "");
  ArrayForEach(cpart, ci)
  {
    fprintf(fp, "%*s", indent + 2, "");
    CaseIntervalDebug(ci, fp);
    fprintf(fp, "\n");
  }
  fprintf(fp, "%*s]\n", indent, "");
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN CasePartitionMake(EXPR_CASE e, CASE_PARTITION *case_partition)   */
/*                                                                           */
/*  Make the case partition corresponding to case expression e.  If there    */
/*  are any problems, print error messages and return FALSE.                 */
/*                                                                           */
/*  There are subtleties in this code.  It works for any test expression     */
/*  type that has a non-empty range of values.                               */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN CasePartitionMake(EXPR_CASE e, CASE_PARTITION *case_partition)
{
  static CASE_PARTITION case_intervals = NULL;
  int i;  BOOLEAN success;  CASE_INTERVAL ci, prev_ci, gci;
  CASE_ENTRY entry;  CASE_LABEL label;

  /* build the initial array of non-default case intervals and sort them */
  if( DEBUG2 )
    fprintf(stderr, "[ CasePartitionMake(e: %s)\n", TypeShow(e->type, NULL));
  ArrayFresh(case_intervals);
  success = TRUE;
  ArrayForEach(e->entries, entry)
    if( entry->labels != NULL )
    {
      ArrayForEach(entry->labels, label)
	if( CaseIntervalMake(label->first_label->file_pos, label->first_label,
	  label->second_label==NULL ? label->first_label : label->second_label,
	  entry, &ci) )
	  ArrayAddLast(case_intervals, ci);
	else
	  success = FALSE;
    }
  ArraySort(case_intervals, &CaseIntervalIncreasingCmp);

  /* check for non-disjoint case intervals; merge adjacent if same action */
  for( i = 1;  i < ArraySize(case_intervals);  i++ )
  {
    prev_ci = ArrayGet(case_intervals, i-1);
    ci = ArrayGet(case_intervals, i);
    if( CaseIntervalsIntersect(prev_ci, ci) )
    {
      fprintf(stderr, "%s: cases not disjoint (other is at %s)\n",
	FilePosShow(ci->file_pos), FilePosShow(prev_ci->file_pos));
      success = FALSE;
    }
    else if(prev_ci->action==ci->action && CaseIntervalsMergeable(prev_ci,ci))
    {
      prev_ci->end_point = ci->end_point;
      prev_ci->end_closed = ci->end_closed;
      ArrayRemove(case_intervals, i);
      i--;
    }
  }
  if( !success )
    db_return(DEBUG2, "CasePartitionMake", FALSE);

  /* register each case interval with its entry */
  ArrayForEach(case_intervals, ci)
    ArrayAddLast(ci->action->case_intervals, ci);

  /* for each element of case_intervals, insert preceding gap and element */
  ArrayInit(case_partition);
  prev_ci = NULL;
  ArrayForEach(case_intervals, ci)
  {
    /* get gci, the gap preceding ci, and insert it if found */
    if( !CaseIntervalGetGap(e, prev_ci, ci, &gci) )
      success = FALSE;
    if( gci != NULL )
    {
      ArrayAddLast(*case_partition, gci);
      ArrayAddLast(gci->action->case_intervals, gci);
    }
  
    /* insert ci itself */
    ArrayAddLast(*case_partition, ci);
    prev_ci = ci;
  }

  /* append the final gap, if any */
  if( !CaseIntervalGetGap(e, prev_ci, NULL, &gci) )
    success = FALSE;
  if( gci != NULL )
    ArrayAddLast(*case_partition, gci);

  if( DEBUG2 && success )
    CasePartitionDebug(*case_partition, 2, stderr);
  db_return(DEBUG2, "CasePartitionMake", success);
}


/*****************************************************************************/
/*                                                                           */
/*  float CasePartitionDensity(CASE_PARTITION cpart, int first, int last)    */
/*                                                                           */
/*  Return the density of that non-empty part of case partition cpart that   */
/*  stretches from first to last inclusive.                                  */
/*                                                                           */
/*  For strings this will be 0 unless there is a single case interval        */
/*  whose value is a single point.                                           */
/*                                                                           */
/*****************************************************************************/

static float CasePartitionDensity(CASE_PARTITION cpart, int first, int last)
{
  CASE_INTERVAL first_ci, last_ci;
  EXPR first_point, last_point;
  assert(0 <= first && first <= last && last < ArraySize(cpart));
  first_ci = ArrayGet(cpart, first);
  first_point = first_ci->start_point;
  last_ci = ArrayGet(cpart, last);
  last_point = last_ci->end_point;
  if( ExprIsString(first_point) )
    return 0.0;
  else
    return (float) (last - first + 1) / ((float) ExprConstIntValue(last_point) -
      (float) ExprConstIntValue(first_point) + 1);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "code generation (the actual C code printing)".                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void ExprCodeGenOrJump(CASE_ENTRY action, CODEGEN_OBJ res_be_var,        */
/*    CODEGEN_TYPE res_be_type, int count, CODEGEN be)                       */
/*                                                                           */
/*  Either generate the code for action, or else generate a goto             */
/*  statement directing flow to a point where the code for action            */
/*  was previously generated.                                                */
/*                                                                           */
/*  There are in fact three cases here.  First, count is the number          */
/*  of intervals that will be covered by this print.  If it is equal         */
/*  to the number of case intervals in the action, then this is clearly      */
/*  the only time that this code will be generated, so don't worry about     */
/*  any labels or gotos.                                                     */
/*                                                                           */
/*  Second, if action->jump_label == NULL, then this code has not been       */
/*  generated before.  Generate it now, but precede it with a label          */
/*  stored in action->jump_label.                                            */
/*                                                                           */
/*  Third, if action->jump_label != NULL, then this code must have been      */
/*  generated previously, so just print a jump to that label.  In this       */
/*  case, count must be strictly less than the number of case intervals      */
/*  known to the action.                                                     */
/*                                                                           */
/*****************************************************************************/

static void ExprCodeGenOrJump(CASE_ENTRY action, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, int count, CODEGEN be)
{
  assert(count <= ArraySize(action->case_intervals));
  if( count == ArraySize(action->case_intervals) )
  {
    /* this code will only be generated once, right now */
    assert(action->jump_label == NULL);
    ExprCodeGen(action->sub_expr, res_be_var, res_be_type, be);
  }
  else if( action->jump_label == NULL )
  {
    /* this code will be generated again later; right now is the first */
    action->jump_label = be->VarMake("CASE_BRANCH", NULL);
    be->GotoTarget(action->jump_label);
    ExprCodeGen(action->sub_expr, res_be_var, res_be_type, be);
  }
  else
  {
    /* this code has been generated before, so just jump to it */
    be->GotoStmt(action->jump_label);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCaseCodeGenSwitchCases(CODEGEN_TYPE be_type,                    */
/*    CASE_PARTITION cpart, int first, int last, CASE_ENTRY dft_action,      */
/*    CODEGEN_OBJ res_be_var, CODEGEN_TYPE res_be_type, CODEGEN be)          */
/*                                                                           */
/*  Generate the cases of a switch.                                          */
/*                                                                           */
/*****************************************************************************/

static void ExprCaseCodeGenSwitchCases(CODEGEN_TYPE be_type,
  CASE_PARTITION cpart, int first, int last, CASE_ENTRY dft_action,
  CODEGEN_OBJ res_be_var, CODEGEN_TYPE res_be_type, CODEGEN be)
{
  CASE_INTERVAL ci, ci2;  BOOLEAN dft_needed;  CASE_ENTRY action;
  int count, ind, ind2, i, first_included, last_included;

  /* print ordinary cases */
  dft_needed = FALSE;
  for( ind = first;  ind <= last;  ind++ )
  {
    ci = ArrayGet(cpart, ind);
    if( ci->action == NULL )
    {
      /* already taken care of - do nothing */
    }
    else if( ci->action == dft_action )
    {
      /* accumulate these for the default case */
      dft_needed = TRUE;
    }
    else
    {
      /* print all cases in switch that share ci->action */
      action = ci->action;
      count = 0;
      for( ind2 = ind;  ind2 <= last;  ind2++ )
      {
	ci2 = ArrayGet(cpart, ind2);
	if( ci2->action == action )
	{
	  if( ci2->start_point == ci2->end_point && ci2->start_closed &&
	      ClassIsEnum(TypeClass(ci2->start_point->type)) )
	  {
	    be->SwitchCaseBegin();
	    ExprCodeGen(ci2->start_point, NULL, be_type, be);
	    be->SwitchCaseEnd();
	  }
	  else
	  {
	    first_included = ci2->start_closed ?
	      ExprConstIntValue(ci2->start_point) :
	      ExprConstIntValue(ci2->start_point) + 1;
	    last_included = ci2->end_closed ?
	      ExprConstIntValue(ci2->end_point) :
	      ExprConstIntValue(ci2->end_point) - 1;
	    for( i = first_included;  i <= last_included;  i++ )
	      be->SwitchCaseInt(i);
	  }
	  count++;
	  ci2->action = NULL;
	}
      }

      /* print ci->action */
      be->SwitchActionBegin();
      ExprCodeGenOrJump(action, res_be_var, res_be_type, count, be);
      be->SwitchActionEnd(res_be_var != be->ret);
    }
  }

  if( dft_needed )
  {
    /* a genuine default case */
    assert(dft_action != NULL);
    be->SwitchCase(NULL);  /* i.e. default */
    be->SwitchActionBegin();
    ExprCodeGenOrJump(dft_action, res_be_var, res_be_type, 1, be);
    be->SwitchActionEnd(res_be_var != be->ret);
  }
  else
  {
    /* just to keep the C compiler happy */
    be->SwitchCase(NULL);  /* i.e. default */
    be->SwitchActionBegin();
    be->Fail();
    be->SwitchActionEnd(FALSE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCaseCodeGenSwitch(CODEGEN_OBJ test_be_var, CODEGEN_TYPE be_type,*/
/*    CASE_PARTITION cpart, int first, int last, CASE_ENTRY dft_action,      */
/*    CODEGEN_OBJ res_be_var, CODEGEN_TYPE res_be_type, CODEGEN be)          */
/*                                                                           */
/*  Generate C code for a case expression whose test expression has been     */
/*  evaluated and stored in test_be_var, and whose case partition is         */
/*  cpart; only that part between first and last is to be printed.  Utilize  */
/*  a C switch statement.  If dft_action != NULL, all cases that carry       */
/*  dft_action are to be grouped under a default case.                       */
/*                                                                           */
/*****************************************************************************/

static void ExprCaseCodeGenSwitch(CODEGEN_OBJ test_be_var, CODEGEN_TYPE be_type,
  CASE_PARTITION cpart, int first, int last, CASE_ENTRY dft_action,
  CODEGEN_OBJ res_be_var, CODEGEN_TYPE res_be_type, CODEGEN be)
{
  /* statement header */
  assert(0 <= first && first <= last && last < ArraySize(cpart));
  assert(res_be_var != NULL);
  be->SwitchBegin();
  Var(test_be_var);
  be->SwitchContinue();

  /* switch cases */
  ExprCaseCodeGenSwitchCases(be_type, cpart, first, last, dft_action,
    res_be_var, res_be_type, be);

  /* statement footer */
  be->SwitchEnd();
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCaseCodeGenDivideInTwo(CODEGEN_OBJ test_be_var,                 */
/*    CODEGEN_TYPE test_be_type, CASE_PARTITION cpart, int first, int last,  */
/*    int mid, CASE_ENTRY dft_action, CODEGEN_OBJ res_be_var,                */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Like ExprCaseCodeGenSubInterval just below, except that a decision has   */
/*  already been made to divide cpart[first..last] in two, with mid being    */
/*  the index of the last element of the first division.                     */
/*                                                                           */
/*****************************************************************************/
static void ExprCaseCodeGenSubInterval(CODEGEN_OBJ test_be_var,
  CODEGEN_TYPE test_be_type, CASE_PARTITION cpart, int first, int last,
  CASE_ENTRY dft_action, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be);

static void ExprCaseCodeGenDivideInTwo(CODEGEN_OBJ test_be_var,
  CODEGEN_TYPE test_be_type, CASE_PARTITION cpart, int first, int last,
  int mid, CASE_ENTRY dft_action, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  CASE_INTERVAL ci;
  ci = ArrayGet(cpart, mid);
  IfElse(
    UpperLimitTest(ci, test_be_var, test_be_type, be),
    Indent(ExprCaseCodeGenSubInterval(test_be_var, test_be_type, cpart,
      first, mid, dft_action, res_be_var, res_be_type, be)),
    Indent(ExprCaseCodeGenSubInterval(test_be_var, test_be_type, cpart,
      mid + 1, last, dft_action, res_be_var, res_be_type, be))
  );
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCaseCodeGenDivideInThree(CODEGEN_OBJ test_be_var,               */
/*    CODEGEN_TYPE test_be_type, CASE_PARTITION cpart, int first, int last,  */
/*    int mid1, int mid2, CASE_ENTRY dft_action, CODEGEN_OBJ res_be_var,     */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Like ExprCaseCodeGenSubInterval just below, except that a decision has   */
/*  already been made to divide cpart[first..last] in three, with mid1       */
/*  being the index of the last element of the first division, and mid2      */
/*  being the index of the last element of the second division.              */
/*                                                                           */
/*  If the first and last divisions have a single interval each, and         */
/*  those intervals have a common action, a special case is invoked in       */
/*  which the if condition is "above first *and* below last".                */
/*                                                                           */
/*****************************************************************************/

static void ExprCaseCodeGenDivideInThree(CODEGEN_OBJ test_be_var,
  CODEGEN_TYPE test_be_type, CASE_PARTITION cpart, int first, int last,
  int mid1, int mid2, CASE_ENTRY dft_action, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  CASE_INTERVAL c1, c2, d1, d2;
  c1 = ArrayGet(cpart, first);
  c2 = ArrayGet(cpart, last);
  if( mid1 == first && mid2 == last-1 && c1->action == c2->action )
  {
    /* the special case explained in the second paragraph above */
    d1 = ArrayGet(cpart, first + 1);
    d2 = ArrayGet(cpart, last - 1);
    IfElse( (d1 == d2 ? InclusionTest(d1, test_be_var, test_be_type, be) :
       Call2(be->logical_and,
	 LowerLimitTest(d1, test_be_var, test_be_type, be),
	 UpperLimitTest(d2, test_be_var, test_be_type, be))),
      Indent(ExprCaseCodeGenSubInterval(test_be_var, test_be_type, cpart,
	first + 1, last - 1, dft_action, res_be_var, res_be_type, be)),
      Indent(ExprCaseCodeGenSubInterval(test_be_var, test_be_type, cpart, last,
	last, dft_action, res_be_var, res_be_type, be))
    );
  }
  else
  {
    /* a more conventional split into three */
    d1 = ArrayGet(cpart, mid1);
    d2 = ArrayGet(cpart, mid2);
    IfElse2( UpperLimitTest(d1, test_be_var, test_be_type, be),
      Indent(ExprCaseCodeGenSubInterval(test_be_var, test_be_type, cpart,
	first, mid1, dft_action, res_be_var, res_be_type, be)),
      UpperLimitTest(d2, test_be_var, test_be_type, be),
      Indent(ExprCaseCodeGenSubInterval(test_be_var, test_be_type, cpart,
	mid1 + 1, mid2, dft_action, res_be_var, res_be_type, be)),
      Indent(ExprCaseCodeGenSubInterval(test_be_var, test_be_type, cpart,
	mid2 + 1, last, dft_action, res_be_var, res_be_type, be))
    );
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCaseCodeGenSubInterval(CODEGEN_OBJ test_be_var,                 */
/*    CODEGEN_TYPE test_be_type, CASE_PARTITION cpart, int first, int last,  */
/*    CASE_ENTRY dft_action, CODEGEN_OBJ res_be_var,                         */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Generate code for a case expression whose test expression has been       */
/*  evaluated and stored in test_var_name, and whose case partition is       */
/*  that part of cpart which stretches from first to last inclusive; this    */
/*  will be non-empty.  Utilize nested if statements, changing to a switch   */
/*  if the density is high enough; or just generate the code if there is     */
/*  only one case in the cpart.                                              */
/*                                                                           */
/*****************************************************************************/
#define width(first, last) ((last) - (first) + 1)

static void ExprCaseCodeGenSubInterval(CODEGEN_OBJ test_be_var,
  CODEGEN_TYPE test_be_type, CASE_PARTITION cpart, int first, int last,
  CASE_ENTRY dft_action, CODEGEN_OBJ res_be_var, CODEGEN_TYPE res_be_type,
  CODEGEN be)
{
  CASE_INTERVAL ci;  int mid;
  assert(0 <= first && first <= last && last < ArraySize(cpart));
  if( width(first, last) == 1 )
  {
    /* single interval, generate its code directly */
    ci = ArrayGet(cpart, first);
    ExprCodeGenOrJump(ci->action, res_be_var, res_be_type, 1, be);
  }
  else if( CasePartitionDensity(cpart, first, last) >= 0.2 )
  {
    /* high density, generate a C switch */
    ExprCaseCodeGenSwitch(test_be_var, test_be_type, cpart, first, last,
      dft_action, res_be_var, res_be_type, be);
  }
  else if( width(first, last) == 2 ||
    CasePartitionDensity(cpart, first + 1, last) >= 0.2 )
  {
    /* high density if first case interval is picked off */
    ExprCaseCodeGenDivideInTwo(test_be_var, test_be_type, cpart,
      first, last, first, dft_action, res_be_var, res_be_type, be);
  }
  else if( CasePartitionDensity(cpart, first, last - 1) >= 0.2 )
  {
    /* high density if last case interval is picked off */
    ExprCaseCodeGenDivideInTwo(test_be_var, test_be_type, cpart, first,
      last, last - 1, dft_action, res_be_var, res_be_type, be);
  }
  else if( width(first, last) == 3 ||
    CasePartitionDensity(cpart, first + 1, last - 1) >= 0.2 )
  {
    /* high density if first and last case intervals are picked off */
    ExprCaseCodeGenDivideInThree(test_be_var, test_be_type, cpart, first,
      last, first, last - 1, dft_action, res_be_var, res_be_type, be);
  }
  else if( width(first, last) > 16 && FALSE )
  {
    /* low density and only one range action, generate a hash table */
    /* still to do */
  }
  else
  {
    /* divide in two and use an if statement */
    mid = (first + last) / 2;
    ExprCaseCodeGenDivideInTwo(test_be_var, test_be_type, cpart, first,
      last, mid, dft_action, res_be_var, res_be_type, be);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprCaseCodeGen(EXPR_CASE expr_case, CODEGEN_OBJ res_be_var,        */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Implement the specification of ExprCodeGen for case expression e.        */
/*                                                                           */
/*****************************************************************************/

void ExprCaseCodeGen(EXPR_CASE expr_case, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  CODEGEN_OBJ test_be_var;  CODEGEN_TYPE test_be_type;
  CASE_ENTRY dft_action;  CASE_ENTRY entry, last_entry;
  CASE_PARTITION case_partition;

  /* make the case partition */
  if( !CasePartitionMake(expr_case, &case_partition) )
    exit(1);

  /* get the default action and test expression backend type */
  last_entry = ArrayLast(expr_case->entries);
  dft_action = (last_entry->labels == NULL ? last_entry : NULL);
  test_be_type = TypeBEType(expr_case->test_expr->type, be);

  if( !ExprLargeScale(expr_case->test_expr) &&
    CasePartitionDensity(case_partition, 0, ArraySize(case_partition)-1)>=0.2 )
  {
    /* small-scale test expr and switch, so avoid variable holding test expr */
    be->SwitchBegin();
    ExprCodeGen(expr_case->test_expr, NULL, test_be_type, be);
    be->SwitchContinue();
    ExprCaseCodeGenSwitchCases(test_be_type, case_partition, 0,
      ArraySize(case_partition)-1, dft_action, res_be_var, res_be_type, be);
    be->SwitchEnd();
  }
  else
  {
    /* evaluate the test expression and store in temporary value */
    assert(res_be_var != NULL);
    test_be_var = be->VarMakeAndDeclare("test", NULL, test_be_type);
    ExprCodeGen(expr_case->test_expr, test_be_var, test_be_type, be);

    /* do the rest */
    ExprCaseCodeGenSubInterval(test_be_var, test_be_type, case_partition, 0,
      ArraySize(case_partition) - 1, dft_action, res_be_var, res_be_type, be);
  }

  /* clear out the jump labels and case partition, in case done again */
  ArrayForEach(expr_case->entries, entry)
    entry->jump_label = NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprCaseInitOrder(EXPR_CASE expr_case, int visit_num,            */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of ExprInitOrder on case expression e.       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprCaseInitOrder(EXPR_CASE expr_case, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  CASE_ENTRY entry;  int i;
  for( i = 0;  i < ArraySize(expr_case->entries);  i++ )
  {
    entry = ArrayGet(expr_case->entries, i);
    if( !ExprInitOrder(entry->sub_expr, visit_num, report, fun) )
      return FALSE;
  }
  return TRUE;
}
