/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_default.c                                             */
/*  DESCRIPTION:  Nonpareil default parameter value expressions              */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"


/*****************************************************************************/
/*                                                                           */
/*  EXPR_DEFAULT (default parameter values)                                  */
/*                                                                           */
/*****************************************************************************/

struct expr_default_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* the identifying token             */
  USTRING		param_name;	/* param name token when := present  */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */
};


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprDefaultNew(FILE_POS file_pos, USTRING param_name, TYPE type)    */
/*                                                                           */
/*  Return an new EXPR_DEFAULT with these attributes.                        */
/*                                                                           */
/*****************************************************************************/

EXPR ExprDefaultNew(FILE_POS file_pos, USTRING param_name, TYPE type)
{
  EXPR_DEFAULT res;
  ExprNew(res, EXPR_DEFAULT, KIND_EXPR_DEFAULT, file_pos, param_name);
  res->type = type;
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprDefaultCopyUninstantiated(EXPR e, ARRAY_FEFN_PARAM orig_params, */
/*    ARRAY_FEFN_PARAM copy_params)                                          */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on default         */
/*  expression e.  Since default expressions are inserted only when          */
/*  manifesting, this function should never be called.                       */
/*                                                                           */
/*****************************************************************************/

EXPR ExprDefaultCopyUninstantiated(EXPR_DEFAULT expr_default,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  assert(FALSE);
  return NULL;  /* keep compiler happy */
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprDefaultManifest(EXPR_DEFAULT *expr_default, CONTEXT context, */
/*    TYPE self_type, BEFN encl_befn)                                        */
/*                                                                           */
/*  Carry out the specification of ExprManifest on a default expression.     */
/*                                                                           */
/*  Actually there is nothing to do, because default expressions are         */
/*  created not parsed, and they are created with fields already set.        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprDefaultManifest(EXPR_DEFAULT *expr_default, CONTEXT context,
  TYPE self_type, BEFN encl_befn)
{
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprDefaultCodeGen(EXPR_DEFAULT expr_default,                       */
/*    CODEGEN_OBJ res_be_var, CODEGEN_TYPE res_be_type, CODEGEN be)          */
/*                                                                           */
/*  Generate code for the value of the default value of a parameter.         */
/*                                                                           */
/*  Implementation note:  this call assumes that the value actually          */
/*  written has C type "int", and it will insert a cast if any other         */
/*  C type is required, in the usual way.                                    */
/*                                                                           */
/*****************************************************************************/

void ExprDefaultCodeGen(EXPR_DEFAULT expr_default, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  if( res_be_var != NULL )
    be->VarAsstBegin(res_be_var);
  Cast(res_be_type, ClassBEType(ClassInt), Var(NPBack_Dft_Param_Val));
  if( res_be_var != NULL )
    be->AsstEnd();
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprDefaultDebug(EXPR_DEFAULT expr_default, CONTEXT cxt,            */
/*    BOOLEAN show_types, FILE *fp, int print_style)                         */
/*                                                                           */
/*  Debug print of default expression expr_default on fp.                    */
/*                                                                           */
/*****************************************************************************/

void ExprDefaultDebug(EXPR_DEFAULT expr_default, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style)
{
  fprintf(fp, "<dft %s>", TypeShow(expr_default->type, cxt));
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprDefaultInitOrder(EXPR_DEFAULT expr_default, int visit_num,   */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of ExprInitOrder on default value            */
/*  expression expr_default.  Since the actual default values are searched   */
/*  while searching the body of the function, there is nothing to do here.   */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprDefaultInitOrder(EXPR_DEFAULT expr_default, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  return TRUE;
}
