/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_fnhead.c                                              */
/*  DESCRIPTION:  Expression with default parameter values.                  */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"
#define DEBUG1 0
#define DEBUG2 0


/*****************************************************************************/
/*                                                                           */
/*  EXPR_FNHEAD                                                              */
/*                                                                           */
/*****************************************************************************/

typedef struct body_set_rec {
  ARRAY_FEFN_PARAM	parameters;	/* parameters having this default    */
  EXPR			dft_val;	/* their common default value        */
} *BODY_SET;

typedef ARRAY(BODY_SET) ARRAY_BODY_SET;

struct expr_fnhead_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* the identifying token             */
  USTRING		param_name;	/* param name when := present        */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */

  ARRAY_BODY_SET	body_sets;	/* holds the bodies                  */
  EXPR			subexpression;	/* the actual function body          */
  BOOLEAN		manifested;	/* debug - TRUE when manifested      */
  BOOLEAN		body_sets_done;	/* when code gen for body sets done  */
};


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprFnHeadNew(FILE_POS file_pos)                                    */
/*                                                                           */
/*  Make a new function head expression with these attributes.               */
/*                                                                           */
/*****************************************************************************/

EXPR ExprFnHeadNew(FILE_POS file_pos)
{
  EXPR_FNHEAD res;
  ExprNew(res, EXPR_FNHEAD, KIND_EXPR_FNHEAD, file_pos, NULL);
  res->body_sets = NULL;
  res->subexpression = NULL;
  res->manifested = FALSE;
  res->body_sets_done = FALSE;
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  BODY_SET BodySetNew(EXPR body)                                           */
/*                                                                           */
/*  Return a new body set with no parameters and the given body.             */
/*                                                                           */
/*****************************************************************************/

static BODY_SET BodySetNew(EXPR body)
{
  BODY_SET res;
  GetMemory(res, BODY_SET);
  ArrayInit(&res->parameters);
  res->dft_val = body;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprFnHeadAddFEFnParamDefaultVal(EXPR e, FEFN_PARAM param,EXPR expr)*/
/*                                                                           */
/*  Add a new parameter with the given body to e.                            */
/*                                                                           */
/*****************************************************************************/

void ExprFnHeadAddFEFnParamDefaultVal(EXPR e, FEFN_PARAM param, EXPR expr)
{
  EXPR_FNHEAD expr_fnhead;  BODY_SET body_set;
  assert(e != NULL);
  ExprKindCheck(e, KIND_EXPR_FNHEAD, "ExprFnHeadAddParamDefaultVal");
  expr_fnhead = (EXPR_FNHEAD) e;

  /* make sure body sets array is initialized */
  if( expr_fnhead->body_sets == NULL )
    ArrayInit(&expr_fnhead->body_sets);

  /* get the body set to add param to (make one if necessary) */
  if( ArraySize(expr_fnhead->body_sets) > 0 &&
      ArrayLast(expr_fnhead->body_sets)->dft_val == expr )
    body_set = ArrayLast(expr_fnhead->body_sets);
  else
  {
    body_set = BodySetNew(expr);
    ArrayAddLast(expr_fnhead->body_sets, body_set);
  }

  /* add param to body_set */
  ArrayAddLast(body_set->parameters, param);
}


/*****************************************************************************/
/*                                                                           */
/*  BODY_SET BodySetCopyUninstantiated(BODY_SET body_set,                    */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Helper function for ExprFnHeadCopyUninstantiated, just below.  Make      */
/*  a copy of body_set, taking care to replace all references to             */
/*  orig_params by references to copy_params.                                */
/*                                                                           */
/*****************************************************************************/

static BODY_SET BodySetCopyUninstantiated(BODY_SET body_set,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  BODY_SET res;  FEFN_PARAM orig_param, copy_param;  int pos;
  GetMemory(res, BODY_SET);
  ArrayInit(&res->parameters);
  ArrayForEach(body_set->parameters, orig_param)
  {
    if( !ArrayContains(orig_params, orig_param, &pos) )
      assert(FALSE);
    copy_param = ArrayGet(copy_params, pos);
    ArrayAddLast(res->parameters, copy_param);
  }
  res->dft_val =
    ExprCopyUninstantiated(body_set->dft_val, orig_params, copy_params);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprFnHeadCopyUninstantiated(EXPR_FNHEAD expr_fnhead,               */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on fnhead          */
/*  expression e.                                                            */
/*                                                                           */
/*****************************************************************************/

EXPR ExprFnHeadCopyUninstantiated(EXPR_FNHEAD expr_fnhead,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_FNHEAD res;  BODY_SET body_set;
  ExprNew(res, EXPR_FNHEAD, KIND_EXPR_FNHEAD, expr_fnhead->file_pos,
    expr_fnhead->param_name);
  res->body_sets = NULL;
  if( expr_fnhead->body_sets != NULL )
  {
    ArrayInit(&res->body_sets);
    ArrayForEach(expr_fnhead->body_sets, body_set)
      ArrayAddLast(res->body_sets,
	BodySetCopyUninstantiated(body_set, orig_params, copy_params));
  }
  res->manifested = FALSE;
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprFnHeadAddSubExpression(EXPR e, EXPR subexpression)              */
/*                                                                           */
/*  Add subexpression (the body proper) to e.                                */
/*                                                                           */
/*****************************************************************************/

void ExprFnHeadAddSubExpression(EXPR e, EXPR subexpression)
{
  EXPR_FNHEAD expr_fnhead;
  ExprKindCheck(e, KIND_EXPR_FNHEAD, "ExprFnHeadAddSubExpression");
  expr_fnhead = (EXPR_FNHEAD) e;
  assert(expr_fnhead->subexpression == NULL);
  expr_fnhead->subexpression = subexpression;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprFnHeadInsertPrecondition(EXPR e, EXPR precondition)             */
/*                                                                           */
/*  Interpose precondition between fnhead expression e and its subexpr.      */
/*  This is used for letdefs where the full panoply of separate functions    */
/*  for preconditions is not wanted.                                         */
/*                                                                           */
/*****************************************************************************/

void ExprFnHeadInsertPrecondition(EXPR e, EXPR precondition)
{
  EXPR_FNHEAD expr_fnhead;
  ExprKindCheck(e, KIND_EXPR_FNHEAD, "ExprFnHeadAddSubExpression");
  expr_fnhead = (EXPR_FNHEAD) e;
  assert(expr_fnhead->subexpression != NULL);
  ExprPrecondAddSubExpression(precondition, expr_fnhead->subexpression);
  expr_fnhead->subexpression = precondition;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprFnHeadInsertCoercion(EXPR e, COERCION c, CONTEXT cxt,           */
/*    TYPE self_type)                                                        */
/*                                                                           */
/*  Coerce the subexpression of expr_fnhead e.                               */
/*                                                                           */
/*****************************************************************************/

void ExprFnHeadInsertCoercion(EXPR e, COERCION c, CONTEXT cxt,
  TYPE self_type)
{
  EXPR_FNHEAD expr_fnhead;
  ExprKindCheck(e, KIND_EXPR_FNHEAD, "ExprFnHeadAddInsertCoercion");
  expr_fnhead =  (EXPR_FNHEAD) e;
  assert(expr_fnhead->subexpression != NULL);
  CoercionInsert(c, &expr_fnhead->subexpression, cxt, self_type);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprFnHeadManifest(EXPR_FNHEAD *e, CONTEXT cxt, TYPE self_type,  */
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Manifest function head expression e.                                     */
/*                                                                           */
/*  Only the local preconditions are manifested, not the inherited ones;     */
/*  inherited preconditions are shared with other functions, and those       */
/*  other functions are responsible for manifesting them.                    */
/*                                                                           */
/*  The parameter default values are not manifested by this call, because    */
/*  they need a different cxt.  ExprFnHeadManifestParamDefaultVal does them. */
/*                                                                           */
/*  An expr_fnhead is large-scale if its subexpression is large-scale or     */
/*  it has at least one precondition or default parameter value.             */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprFnHeadManifest(EXPR_FNHEAD *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn)
{
  EXPR_FNHEAD expr_fnhead;
  expr_fnhead = *e;
  if( DEBUG2 )
  {
    fprintf(stderr, "[ ExprFnHeadManifest(e, ...), e =\n  ");
    ExprFnHeadDebug(expr_fnhead, cxt, TRUE, stderr, 2);
    fprintf(stderr, "\n");
  }
  assert(!expr_fnhead->manifested);
  expr_fnhead->manifested = TRUE;

  /* already large-scale if we have even one default parameter */
  expr_fnhead->large_scale = (expr_fnhead->body_sets != NULL);

  /* manifest subexpression, if any, and set type field */
  expr_fnhead->type = NULL;
  if( expr_fnhead->subexpression != NULL )
  {
    if( !ExprManifest(&expr_fnhead->subexpression, cxt, self_type, encl_befn) )
      db_return(DEBUG2, "ExprFnHeadManifest", FALSE);
    expr_fnhead->type = expr_fnhead->subexpression->type;
    expr_fnhead->large_scale |= expr_fnhead->subexpression->large_scale;
  }

  db_return(DEBUG2, "ExprFnHeadManifest", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprFnHeadManifestParamDefaultVal(EXPR *e, FEFN_PARAM param,     */
/*    CONTEXT cxt, TYPE self_type, BEFN encl_befn)                           */
/*                                                                           */
/*  Manifest a parameter default value.  If the value is shared among        */
/*  several parameters, manifest it only once, when given the first          */
/*  parameter.                                                               */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprFnHeadManifestParamDefaultVal(EXPR e, FEFN_PARAM param,
  CONTEXT cxt, TYPE self_type, BEFN encl_befn)
{
  BODY_SET body_set;  EXPR_FNHEAD expr_fnhead;  int i, j;  FEFN_PARAM param2;
  BOOLEAN res;
  ExprKindCheck(e, KIND_EXPR_FNHEAD, "ExprFnHeadManifestParamDefaultVal");
  expr_fnhead = (EXPR_FNHEAD) e;

  if( DEBUG1 )
  {
    fprintf(stderr, "[ ExprFnHeadManifestParamDefaultVal(e, %s: %s, ...)\n",
      FilePosShow(FEFnParamFilePos(param)), NameShow(FEFnParamName(param)));
    ExprFnHeadDebug(expr_fnhead, cxt, FALSE, stderr, 2);
  }

  /* find the body set containing param - it must be there */
  assert(expr_fnhead->body_sets != NULL);
  for( i = 0;  i < ArraySize(expr_fnhead->body_sets);  i++ )
  {
    body_set = ArrayGet(expr_fnhead->body_sets, i);
    for( j = 0;  j < ArraySize(body_set->parameters);  j++ )
    {
      param2 = ArrayGet(body_set->parameters, j);
      if( param2 == param )
	goto FOUND;
    }
  }
  assert(FALSE);

  /* instantiate the body once, for the first parameter it applies to */
  FOUND:
  if( j == 0 )
    res = ExprManifest(&body_set->dft_val, cxt, self_type, encl_befn);
  else
    res = TRUE;
  db_return(DEBUG1, "ExprFnHeadManifestParamDefaultVal", res);
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprFnHeadCodeGenDftParamValues(EXPR e, CODEGEN be)                 */
/*                                                                           */
/*  Generate code on be to set parameter default values.                     */
/*                                                                           */
/*****************************************************************************/

void ExprFnHeadCodeGenDftParamValues(EXPR e, CODEGEN be)
{
  BODY_SET body_set;  EXPR_FNHEAD expr_fnhead;
  FEFN_PARAM param, param2;  int i;
  CODEGEN_TYPE param_be_type;
  CODEGEN_OBJ param_be_var, param_be_var2;
  ExprKindCheck(e, KIND_EXPR_FNHEAD, "ExprFnHeadCodeGen");
  expr_fnhead = (EXPR_FNHEAD) e;
  assert(!expr_fnhead->body_sets_done);
  expr_fnhead->body_sets_done = TRUE;

  if( expr_fnhead->body_sets != NULL )
  ArrayForEach(expr_fnhead->body_sets, body_set)
  {
    /* first parameter */
    param = ArrayFirst(body_set->parameters);
    param_be_var = BEFnParamBEVar(FEFnParamBEFnParam(param));
    param_be_type = BEFnParamBEType(FEFnParamBEFnParam(param));
    If( Equal(Var(param_be_var),
	  ExprDefaultCodeGen(NULL, NULL, param_be_type, be)),
      body_set->dft_val->large_scale ?
       ExprCodeGen(body_set->dft_val, param_be_var, param_be_type, be) :
       Indent(ExprCodeGen(body_set->dft_val, param_be_var, param_be_type, be))
    );

    /* subsequent parameters with the same default value */
    for( i = 1;  i < ArraySize(body_set->parameters);  i++ )
    {
      param2 = ArrayGet(body_set->parameters, i);
      param_be_var2 = BEFnParamBEVar(FEFnParamBEFnParam(param2));
      If( Equal(Var(param_be_var2),
	  ExprDefaultCodeGen(NULL, NULL, param_be_type, be)),
	Indent(VarAssign(param_be_var2, Var(param_be_var)))
      );
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprFnHeadCodeGen(EXPR_FNHEAD expr_fnhead, CODEGEN_OBJ res_be_var,  */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Generate code for this function head expression.                         */
/*                                                                           */
/*  It is acceptable for the expression to have a null subexpression; this   */
/*  occurs with creation functions.  We simply generate nothing for the      */
/*  subexpression in that case.                                              */
/*                                                                           */
/*****************************************************************************/

void ExprFnHeadCodeGen(EXPR_FNHEAD expr_fnhead, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  /* default parameter values */
  if( !expr_fnhead->body_sets_done )
    ExprFnHeadCodeGenDftParamValues((EXPR) expr_fnhead, be);

  /* subexpression */
  if( expr_fnhead->subexpression != NULL )
    ExprCodeGen(expr_fnhead->subexpression, res_be_var, res_be_type, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprFnHeadDebug(EXPR_FNHEAD expr_fnhead, CONTEXT cxt,               */
/*    BOOLEAN show_types, FILE *fp, int print_style)                         */
/*                                                                           */
/*  Debug print of fn_head expression.                                       */
/*                                                                           */
/*****************************************************************************/

void ExprFnHeadDebug(EXPR_FNHEAD expr_fnhead, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style)
{
  BODY_SET body_set;  FEFN_PARAM param;

  fprintf(fp, "fn_head [");
  begin_indent;

  /* debug parameter default values */
  if( expr_fnhead->body_sets != NULL )
    ArrayForEach(expr_fnhead->body_sets, body_set)
    {
      next_line;
      ArrayForEach(body_set->parameters, param)
	fprintf(fp, "%s ", NameShow(FEFnParamName(param)));
      fprintf(fp, ":= ");
      ExprDebug(body_set->dft_val, cxt, show_types, fp, print_style);
    }

  /* debug subexpression, if any */
  if( expr_fnhead->subexpression != NULL )
  {
    next_line;
    ExprDebug(expr_fnhead->subexpression, cxt, show_types, fp, print_style);
  }
  end_indent;
  next_line;
  fprintf(fp, "]");
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprFnHeadInitOrder(EXPR_FNHEAD expr_fnhead, int visit_num,      */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of ExprInitOrder on function header          */
/*  expression expr_fnhead.  This involves searching through the default     */
/*  values of the parameters, even if they are not necessarily called in     */
/*  any given instance.                                                      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprFnHeadInitOrder(EXPR_FNHEAD expr_fnhead, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  BODY_SET body_set;  int i;

  /* explore body sets */
  if( expr_fnhead->body_sets != NULL )
    for( i = 0;  i < ArraySize(expr_fnhead->body_sets);  i++ )
    {
      body_set = ArrayGet(expr_fnhead->body_sets, i);
      if( !ExprInitOrder(body_set->dft_val, visit_num, report, fun) )
	return FALSE;
    }

  /* explore the subexpression */
  if( expr_fnhead->subexpression != NULL )
    if( !ExprInitOrder(expr_fnhead->subexpression, visit_num, report, fun) )
      return FALSE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprFnHeadWrapsBuiltin(EXPR e, ARRAY_BEFN_PARAM parameters,      */
/*    CODEGEN_OBJ *be_obj)                                                   */
/*                                                                           */
/*  Return TRUE if e is just a call on a bultin function with exactly        */
/*  the same parameters as the enclosing function.                           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprFnHeadWrapsBuiltin(EXPR e, ARRAY_BEFN_PARAM parameters,
  CODEGEN_OBJ *be_obj)
{
  EXPR_FNHEAD expr_fnhead;
  ExprKindCheck(e, KIND_EXPR_FNHEAD, "ExprFnHeadWrapsBuiltin");
  expr_fnhead = (EXPR_FNHEAD) e;

  /* must be no default parameters */
  if( expr_fnhead->body_sets != NULL )
    return FALSE;

  /* the subexpression must be a call expression with the right parameters */
  return expr_fnhead->subexpression != NULL &&
      expr_fnhead->subexpression->kind_tag == KIND_EXPR_CALL &&
      ExprCallWrapsBuiltin(expr_fnhead->subexpression, parameters, be_obj);
}
