/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_if.c                                                  */
/*  DESCRIPTION:  Nonpareil if expressions                                   */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"


/*****************************************************************************/
/*                                                                           */
/*  CONDITION (private)                                                      */
/*                                                                           */
/*  A condition, that is, a Boolean expression lying between if or elseif    */
/*  and then.  It may have one of these three types:                         */
/*                                                                           */
/*    CONDITION_ORDINARY      An ordinary condition, such as "x > y"         */
/*                                                                           */
/*    CONDITION_DOWN_NOVAR    A downcast condition without a variable,       */
/*                            such as "poly is square"                       */
/*                                                                           */
/*    CONDITION_DOWN_VAR      A downcast condition with a variable,          */
/*                            such as "poly is s: square".                   */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  CONDITION_ORDINARY,
  CONDITION_DOWN_NOVAR,
  CONDITION_DOWN_VAR
} CONDITION_TYPE;

typedef struct condition_rec {
  CONDITION_TYPE	cond_type;	/* type of condition                 */
  EXPR			subexpr;	/* subexpression, present always     */
  TYPE			test_type;	/* if downcast                       */
  FEFN_DOWNDEF		downdef;	/* if downcast with variable         */
} *CONDITION;

typedef ARRAY(CONDITION) ARRAY_CONDITION;


/*****************************************************************************/
/*                                                                           */
/*  EXPR_IF                                                                  */
/*                                                                           */
/*  This holds one if expression, with its conditions in e->conditions       */
/*  and its subexpressions in e->subexpressions, satisfying                  */
/*                                                                           */
/*    ArrayLength(e->subexpressions) = ArrayLength(e->conditions) + 1        */
/*                                                                           */
/*****************************************************************************/

struct expr_if_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* file position of expression       */
  USTRING		param_name;	/* param name token when := present  */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */
  ARRAY_CONDITION	conditions;	/* the conditions                    */
  ARRAY_EXPR		subexpressions;	/* the subexpressions                */
};


/*****************************************************************************/
/*                                                                           */
/*  CONDITION ConditionMake(CONDITION_TYPE cond_type, EXPR subexpr,          */
/*    FEFN_DOWNDEF downdef)                                                  */
/*                                                                           */
/*  Return a new CONDITION object with these attributes.                     */
/*                                                                           */
/*****************************************************************************/

static CONDITION ConditionMake(CONDITION_TYPE cond_type, EXPR subexpr,
  TYPE test_type, FEFN_DOWNDEF downdef)
{
  CONDITION res;
  GetMemory(res, CONDITION);
  res->cond_type = cond_type;
  res->subexpr = subexpr;
  res->test_type = test_type;
  res->downdef = downdef;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ParseCondition(TOKEN *t, SYSTEM_VIEW sv, CONDITION *res)         */
/*                                                                           */
/*  Parse a condition, including any downcasting part.                       */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN NameDefAndColon(TOKEN *t)
{
  /* return TRUE if input begins with a name definition and colon */
  if( LexType(curr_token) == TK_PREFIX || LexType(curr_token) == TK_POSTFIX ||
    LexType(curr_token) == TK_INFIX || LexType(curr_token) == TK_INFIXR )
    return TRUE;
  if( LexType(curr_token) == TK_IDENTIFIER &&
      LexType(LexNext(curr_token)) == TK_COLON )
    return TRUE;
  return FALSE;
}

static BOOLEAN ParseCondition(TOKEN *t, SYSTEM_VIEW sv, CONDITION *res)
{

  FILE_POS file_pos, name_file_pos;  NAME name;  TYPE test_type;
  EXPR expression;  CONDITION_TYPE cond_type;  FEFN_DOWNDEF downdef;

  /* skip initial if/elsif, known to be there */
  next_token;
  file_pos = LexFilePos(curr_token);

  /* parse test expression */
  if( !ExprParse(t, sv, &expression) )
    return FALSE;

  if( LexType(curr_token) != TK_IS )
  {
    /* an ordinary condition (no downcasting) */
    cond_type = CONDITION_ORDINARY;
    test_type = NULL;
    downdef = NULL;
  }
  else if( next_token, NameDefAndColon(t) )
  {
    /* a downcasting condition with a downcast variable */
    cond_type = CONDITION_DOWN_VAR;
    name_file_pos = LexFilePos(curr_token);
    if( !NameDefParse(t, SystemViewOpTable(sv), &name) )
    {
      fprintf(stderr, "internal error in ParseCondition\n");
      exit(1);
    }
    if( !TypeParse(t, &test_type) )
      return FALSE;
    downdef = FEFnDownDefMake(name_file_pos, name, test_type);
  }
  else
  {
    /* a downcasting condition but with no downcast variable */
    cond_type = CONDITION_DOWN_NOVAR;
    if( !TypeParse(t, &test_type) )
      return FALSE;
    downdef = NULL;
  }

  /* parse then and return */
  skip(TK_THEN, "\"then\" keyword in \"if\" expression");
  *res = ConditionMake(cond_type, expression, test_type, downdef);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  static BOOLEAN ExprIfParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)          */
/*                                                                           */
/*  Parse an if expression.                                                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprIfParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  EXPR_IF res_if;  EXPR sub_expr;  CONDITION cond;

  /* set up res_if and *res */
  ExprNew(res_if, EXPR_IF, KIND_EXPR_IF, LexFilePos(curr_token), NULL);
  ArrayInit(&res_if->conditions);
  ArrayInit(&res_if->subexpressions);
  *res = (EXPR) res_if;

  /* parse first condition, from "if" to "then" inclusive */
  if( !ParseCondition(t, sv, &cond) )
    return FALSE;
  ArrayAddLast(res_if->conditions, cond);

  /* parse expression following first "then" */
  if( !ExprParse(t, sv, &sub_expr) )
    return FALSE;
  ArrayAddLast(res_if->subexpressions, sub_expr);

  while( LexType(curr_token) == TK_ELSIF )
  {
    /* parse another condition, from "elsif" to "then" inclusive */
    if( !ParseCondition(t, sv, &cond) )
      return FALSE;
    ArrayAddLast(res_if->conditions, cond);

    /* parse expression following "then" */
    if( !ExprParse(t, sv, &sub_expr) )
      return FALSE;
    ArrayAddLast(res_if->subexpressions, sub_expr);
  }

  /* parse concluding else part */
  skip(TK_ELSE, "\"elsif\" or \"else\" of \"if\" expression");
  if( LexType(curr_token) == TK_IF )
  {
    fprintf(stderr, "%s: \"if\" not allowed after \"else\"; use \"elsif\"\n",
      LexPos(curr_token));
    return FALSE;
  }
  if( !ExprParse(t, sv, &sub_expr) )
    return FALSE;
  ArrayAddLast(res_if->subexpressions, sub_expr);
  skip(TK_END, "concluding \"end\" of \"if\" expression");
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  CONDITION ConditionCopyUninstantiated(CONDITION cond,                    */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Make a copy of uninstantiated condition cond.                            */
/*                                                                           */
/*****************************************************************************/

static CONDITION ConditionCopyUninstantiated(CONDITION cond,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  CONDITION res;
  GetMemory(res, CONDITION);
  res->cond_type = cond->cond_type;
  res->subexpr = ExprCopyUninstantiated(cond->subexpr,orig_params,copy_params);
  res->downdef = cond->downdef == NULL ? NULL :
    FEFnDownDefCopyUninstantiated(cond->downdef, orig_params, copy_params);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprIfCopyUninstantiated(EXPR_IF expr_if,                           */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on if              */
/*  expression expr_if.                                                      */
/*                                                                           */
/*****************************************************************************/

EXPR ExprIfCopyUninstantiated(EXPR_IF expr_if,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_IF res;  CONDITION cond;  EXPR e;
  ExprNew(res, EXPR_IF, KIND_EXPR_IF, expr_if->file_pos, expr_if->param_name);
  ArrayInit(&res->conditions);
  ArrayForEach(expr_if->conditions, cond)
    ArrayAddLast(res->conditions,
      ConditionCopyUninstantiated(cond, orig_params, copy_params));
  ArrayInit(&res->subexpressions);
  ArrayForEach(expr_if->subexpressions, e)
    ArrayAddLast(res->subexpressions,
      ExprCopyUninstantiated(e, orig_params, copy_params));
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprIfManifest(EXPR_IF *e, CONTEXT cxt, TYPE self_type,          */
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Carry out the specification of ExprManifest on an if expression.         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprIfManifest(EXPR_IF *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn)
{
  EXPR_IF expr_if;  EXPR *sub; int i;  TYPE joined_type;  CONDITION cond;
  COERCION c;
  expr_if = (EXPR_IF) *e;
  expr_if->large_scale = FALSE;

  joined_type = NULL;
  for( i = 0;  i < ArraySize(expr_if->subexpressions);  i++ )
  {
    /* manifest ith condition unless last time through */
    if( i < ArraySize(expr_if->conditions) )
    {
      cond = ArrayGet(expr_if->conditions, i);
      switch( cond->cond_type )
      {
	case CONDITION_ORDINARY:

	  /* ordinary condition, no downcasting */
	  if( !ExprManifest(&cond->subexpr, cxt, self_type, encl_befn) )
	    return FALSE;
	  if( !TypeIsSubType(cond->subexpr->type, ClassType(ClassBool),&c,cxt) )
	  {
	    fprintf(stderr, "%s: test expression does not have type %s\n",
	      FilePosShow(cond->subexpr->file_pos),
	      NameShow(ClassName(ClassBool, cxt)));
	    return FALSE;
	  }
	  CoercionInsert(c, &cond->subexpr, cxt, self_type);
	  if( cond->subexpr->large_scale )
	    expr_if->large_scale = TRUE;
	  break;


	case CONDITION_DOWN_NOVAR:
	case CONDITION_DOWN_VAR:

	  /* downcasting condition, with or without downcast variable */
	  expr_if->large_scale = TRUE;
	  if( !ExprManifest(&cond->subexpr, cxt, self_type, encl_befn) )
	    return FALSE;

	  /* manifest the downcast type, if any (shared with any downdef) */
	  if( !TypeManifest(cond->test_type, cxt, FALSE) )
	    return FALSE;

	  /* if target type not subtype of test expr type, never succeeds */
	  if( !TypeIsSubType(cond->test_type, cond->subexpr->type, &c, cxt) )
	  {
	    fprintf(stderr, "%s: downcast never succeeds\n",
	      FilePosShow(cond->subexpr->file_pos));
	    return FALSE;
	  }

	  /* if test expr type is subtype of target type, always succeeds */
	  if( TypeIsSubType(cond->subexpr->type, cond->test_type, &c, cxt) )
	  {
	    fprintf(stderr, "%s: downcast always succeeds\n",
	      FilePosShow(cond->subexpr->file_pos));
	    return FALSE;
	  }

	  /* add condition to cxt if variable present */
          if( cond->cond_type == CONDITION_DOWN_VAR )
	  {
	    assert(cond->downdef != NULL);
	    if( !ContextInsertFEFnDownDef(cxt, cond->downdef) )
	      return FALSE;
	  }
	  break;


	default:

	  assert(FALSE);
      }
    }

    /* manifest ith subexpression */
    sub = &ArrayGet(expr_if->subexpressions, i);
    if( !ExprManifest(sub, cxt, self_type, encl_befn) )
      return FALSE;
    if( (*sub)->large_scale )
      expr_if->large_scale = TRUE;

    /* join ith subexpression's type into joined_type */
    if( !TypeJoin(&joined_type, (*sub)->type, cxt) )
    {
      fprintf(stderr, "%s: incompatible type of \"if\" branch\n",
	FilePosShow(cond->subexpr->file_pos));
      return FALSE; 
    }

    /* remove downcast name if any */
    if( i < ArraySize(expr_if->conditions) &&
	cond->cond_type == CONDITION_DOWN_VAR )
      ContextDelete(cxt, (NAMED) cond->downdef);
  }

  /* success */
  (*e)->type = joined_type;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ConditionDebug(CONDITION c, CONTEXT cxt, BOOLEAN show_types,        */
/*    FILE *fp)                                                              */
/*                                                                           */
/*  Debug print of condition c on *fp.                                       */
/*                                                                           */
/*****************************************************************************/

static void ConditionDebug(CONDITION c, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp)
{
  ExprDebug(c->subexpr, cxt, show_types, fp, SINGLE_LINE);
  switch( c->cond_type )
  {
    case CONDITION_ORDINARY:
      
      break;

    case CONDITION_DOWN_NOVAR:
      
      fprintf(fp, " is %s", TypeShow(c->test_type, cxt));
      break;

    case CONDITION_DOWN_VAR:
      
      fprintf(fp, " is %s: %s", NameShow(FEFnName((FEFN) c->downdef)),
	TypeShow(c->test_type, cxt));
      break;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprIfDebug(EXPR_IF expr_if, CONTEXT cxt, BOOLEAN show_types,       */
/*    FILE *fp, int print_style)                                             */
/*                                                                           */
/*  Debug print of if expr e on *fp.                                         */
/*                                                                           */
/*****************************************************************************/

void ExprIfDebug(EXPR_IF expr_if, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style)
{
  int i;
  for( i = 0;  i < ArraySize(expr_if->conditions);  i++ )
  {
    CONDITION cond = ArrayGet(expr_if->conditions, i);
    EXPR sub_expr = ArrayGet(expr_if->subexpressions, i);
    fprintf(fp, i == 0 ? "if " : "elseif ");
    ConditionDebug(cond, cxt, show_types, fp);
    fprintf(fp, " then");
    begin_indent;
    next_line;
    ExprDebug(sub_expr, cxt, show_types, fp, print_style);
    end_indent;
    next_line;
  }
  fprintf(fp, "else");
  begin_indent;
  next_line;
  ExprDebug(ArrayLast(expr_if->subexpressions), cxt, show_types,fp,print_style);
  end_indent;
  next_line;
  fprintf(fp, "end");
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprIfCodeGenDowncastTest(CONDITION cond, CODEGEN_OBJ be_var,       */
/*    CODEGEN_TYPE be_type, CODEGEN be)                                      */
/*                                                                           */
/*  Generate a downcast test which first assigns cond->subexpr to be_var     */
/*  (of type be_type), then tests its type against cond->test_type.          */
/*                                                                           */
/*****************************************************************************/

static void ExprIfCodeGenDowncastTest(CONDITION cond, CODEGEN_OBJ be_var,
  CODEGEN_TYPE be_type, CODEGEN be)
{
  ARRAY_CLASS type_classes;  CLASS class;
  ExprCodeGen(cond->subexpr, be_var, be_type, be);
  type_classes = TypeClasses(cond->test_type);
  assert(ArraySize(type_classes) == 1); /* must be checked earlier! */
  class = ArrayFirst(type_classes);
  be->IfStmtBegin();
  Equal(Call1(NPBack_Type_Tag, Var(be_var)), Var(ClassTypeTag(class)));
  be->IfStmtContinue();
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprIfCodeGenFrom(EXPR_IF expr_if, CODEGEN_OBJ res_be_var,          */
/*    CODEGEN_TYPE res_be_type, CODEGEN be, int i)                           */
/*                                                                           */
/*  Like ExprIfCodeGen below, except that expr_if is known to be large       */
/*  scale, and we have to generate everything from condition(i) onwards,     */
/*  not the whole expression.  If condition(i) is small-scale, generate      */
/*                                                                           */
/*      if( condition(i) )                                                   */
/*        sub_expr(i)                                                        */
/*      else<ch><recurse>                                                    */
/*                                                                           */
/*  If condition(i) is large-scale, generate                                 */
/*                                                                           */
/*      bool tmpi;                                                           */
/*      ...                                                                  */
/*      tmpi = ...;                                                          */
/*      if( tmpi )                                                           */
/*        sub_expr(i)                                                        */
/*      else<ch><recurse>                                                    */
/*                                                                           */
/*  Here <ch> and <recurse> depend on condition(i+1):                        */
/*                                                                           */
/*      condition(i+1)  <ch>     <recurse>                                   */
/*      --------------------------------------------------------------       */
/*      Nonexistent     newline  CodeGen the last (else) subexpression       */
/*      Small-scale     space    ExprIfCodeGenFrom(..., i+1)                 */
/*      Large-scale     newline  ExprIfCodeGenFrom(..., i+1)                 */
/*      --------------------------------------------------------------       */
/*                                                                           */
/*  See the comment of ExprIfCodeGen below for what this makes altogether.   */
/*                                                                           */
/*****************************************************************************/

static void ExprIfCodeGenFrom(EXPR_IF expr_if, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be, int i)
{
  CONDITION cond;  EXPR sub_expr;  CODEGEN_TYPE cond_type;
  CODEGEN_OBJ be_var;
  assert(expr_if->large_scale);
  assert(i < ArraySize(expr_if->conditions));
  assert(res_be_var != NULL);

  /* "if( condition )" or "else if( condition )", possibly large-scale */
  cond = ArrayGet(expr_if->conditions, i);
  cond_type = TypeBEType(ExprType(cond->subexpr), be);
  if( cond->subexpr->large_scale )
  {
    if( i > 0 )
      be->IfStmtElse();
    be->BlockBegin();
    switch( cond->cond_type )
    {
      case CONDITION_ORDINARY:

	/* condition is large-scale but not downcast */
	be_var = be->VarMakeAndDeclare("tmp", NULL, cond_type);
	ExprCodeGen(cond->subexpr, be_var, cond_type, be);
	be->IfStmtBegin();
	Var(be_var);
	be->IfStmtContinue();
	break;


      case CONDITION_DOWN_NOVAR:

	/* condition is large-scale and downcast, with no downcast variable */
	be_var = be->VarMakeAndDeclare("tmp", NULL, cond_type);
        ExprIfCodeGenDowncastTest(cond, be_var, cond_type, be);
	break;


      case CONDITION_DOWN_VAR:

	/* condition is large-scale and downcast, with a downcast variable */
	be_var = be->VarMakeAndDeclare(NULL,
	  NameRep(FEFnName((FEFN) cond->downdef)), cond_type);
	FEFnDownDefMakeBEFn(cond->downdef, be_var);
        ExprIfCodeGenDowncastTest(cond, be_var, cond_type, be);
	break;

    }
  }
  else
  {
    /* condition is small-scale (and must not be a downcasting condition */
    assert(cond->cond_type == CONDITION_ORDINARY);
    if( i == 0 )
      be->IfStmtBegin();
    else
      be->IfStmtElseIf();
    ExprCodeGen(cond->subexpr, NULL, cond_type, be);
    be->IfStmtContinue();
  }

  /* "sub_expr(i)" */
  sub_expr = ArrayGet(expr_if->subexpressions, i);
  if( sub_expr->large_scale )
    ExprCodeGen(sub_expr, res_be_var, res_be_type, be);
  else
    Indent(ExprCodeGen(sub_expr, res_be_var, res_be_type, be));

  if( i == ArraySize(expr_if->conditions) - 1 )
  {
    /* "else sub_expr" */
    be->IfStmtElse();
    sub_expr = ArrayGet(expr_if->subexpressions, i+1);
    if( sub_expr->large_scale )
      ExprCodeGen(sub_expr, res_be_var, res_be_type, be);
    else
      Indent(ExprCodeGen(sub_expr, res_be_var, res_be_type, be));
  }
  else
  {
    /* recurse on next condition  */
    ExprIfCodeGenFrom(expr_if, res_be_var, res_be_type, be, i+1);
  }

  /* end if and close off declarations block if made one */
  be->IfStmtEnd();
  if( cond->subexpr->large_scale )
  {
    be->BlockEnd();
    be->ObjFree(be_var);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprIfCodeGen(EXPR_IF expr_if, CODEGEN_OBJ res_be_var,              */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Carry out the specification of ExprCodeGen on if expr expr_if.           */
/*                                                                           */
/*  In the simplest case, where the entire expression is small-scale, the    */
/*  generated code will look like this:                                      */
/*                                                                           */
/*    (condition1 ? subexpr1 : condition2 ? subexpr2 : ... : subexprk)       */
/*                                                                           */
/*  possibly spread over several lines.  When the expression is              */
/*  large-scale but the conditions are all small-scale (a likely case)       */
/*  the generated code will look like this:                                  */
/*                                                                           */
/*    if( condition1 )                                                       */
/*    {                                                                      */
/*      ...                                                                  */
/*    }                                                                      */
/*    else if( condition2 )                                                  */
/*    {                                                                      */
/*      ...                                                                  */
/*    }                                                                      */
/*    else                                                                   */
/*    {                                                                      */
/*      ...                                                                  */
/*    }                                                                      */
/*                                                                           */
/*  although any small-scale branches will omit the braces.  In the most     */
/*  elaborate case, when the conditions are large-scale too, the code        */
/*  generated will look like this:                                           */
/*                                                                           */
/*    bool tmp1;                                                             */
/*    ...                                                                    */
/*    tmp1 = ...;                                                            */
/*    if( tmp1 )                                                             */
/*    {                                                                      */
/*      ...                                                                  */
/*    }                                                                      */
/*    else                                                                   */
/*    {                                                                      */
/*      bool tmp2;                                                           */
/*      ...                                                                  */
/*      tmp2 = ...;                                                          */
/*      if( tmp2 )                                                           */
/*      {                                                                    */
/*        ...                                                                */
/*      }                                                                    */
/*      else                                                                 */
/*      {                                                                    */
/*        ...                                                                */
/*      }                                                                    */
/*    }                                                                      */
/*                                                                           */
/*  with the more elaborate form introduced only for those conditions that   */
/*  need it, i.e. that are large-scale.                                      */
/*                                                                           */
/*  When the condition is a downcast test, tmpk will be the downcast         */
/*  variable and the condition will become tmpk->type_tag == whatever.       */
/*                                                                           */
/*****************************************************************************/

void ExprIfCodeGen(EXPR_IF expr_if, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  int i;  CONDITION cond;  EXPR sub_expr;
  if( expr_if->large_scale )
    ExprIfCodeGenFrom(expr_if, res_be_var, res_be_type, be, 0);
  else
  {
    /* whole expression is small-scale: use ?: */
    if( res_be_var != NULL )
      be->VarAsstBegin(res_be_var);
    be->CastBegin(res_be_type, TypeBEType(expr_if->type, be));
    be->IfSmallBegin();
    for( i = 0;  i < ArraySize(expr_if->conditions);  i++ )
    {
      cond = ArrayGet(expr_if->conditions, i);
      sub_expr = ArrayGet(expr_if->subexpressions, i);
      if( i > 0 )
	be->IfSmallElseIf();
      ExprCodeGen(cond->subexpr, NULL,
	TypeBEType(ExprType(cond->subexpr), be), be);
      be->IfSmallContinue();
      ExprCodeGen(sub_expr, NULL, res_be_type, be);
    }
    sub_expr = ArrayGet(expr_if->subexpressions, i);
    be->IfSmallElse();
    ExprCodeGen(sub_expr, NULL, res_be_type, be);
    be->IfSmallEnd();
    be->CastEnd();
    if( res_be_var != NULL )
      be->AsstEnd();
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprIfInitOrder(EXPR_IF expr_if, int visit_num, BOOLEAN *report, */
/*     BEFN_SYSTEM_INIT fun)                                                 */
/*                                                                           */
/*  Carry out the specification of ExprInitOrder on if expression e.         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprIfInitOrder(EXPR_IF expr_if, int visit_num, BOOLEAN *report,
   BEFN_SYSTEM_INIT fun)
{
  CONDITION cond;  EXPR sub_expr;  int i;
  for( i = 0;  i < ArraySize(expr_if->conditions);  i++ )
  {
    cond = ArrayGet(expr_if->conditions, i);
    if( !ExprInitOrder(cond->subexpr, visit_num, report, fun) )
      return FALSE;
  }
  for( i = 0;  i < ArraySize(expr_if->subexpressions);  i++ )
  {
    sub_expr = ArrayGet(expr_if->subexpressions, i);
    if( !ExprInitOrder(sub_expr, visit_num, report, fun) )
      return FALSE;
  }
  return TRUE;
}
