/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_let.c                                                 */
/*  DESCRIPTION:  Nonpareil let expressions                                  */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"
#define	DEBUG1 0


/*****************************************************************************/
/*                                                                           */
/*  EXPR_LET                                                                 */
/*                                                                           */
/*  A let expression.  The definitions are in e->letdefs, and the            */
/*  expression following the "in" keyword is in e->subexpr.                  */
/*                                                                           */
/*****************************************************************************/

struct expr_let_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* file position of expression       */
  USTRING		param_name;	/* param name token when := present  */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */
  ARRAY_FEFN_LETDEF	letdefs;	/* the definitions, in order         */
  EXPR			subexpr;	/* the expression after "in"	     */
};


/*****************************************************************************/
/*                                                                           */
/*  EXPR_LET LetExprNew(FILE_POS file_pos, ARRAY_FEFN_LETDEF letdefs,        */
/*    EXPR subexpr)                                                          */
/*                                                                           */
/*  Make a new, empty let expression with these attributes.                  */
/*                                                                           */
/*****************************************************************************/

static EXPR_LET LetExprNew(FILE_POS file_pos, ARRAY_FEFN_LETDEF letdefs,
  EXPR subexpr)
{
  EXPR_LET res;
  ExprNew(res, EXPR_LET, KIND_EXPR_LET, file_pos, NULL);
  res->letdefs = letdefs;
  res->subexpr = subexpr;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprLetParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)                */
/*                                                                           */
/*  Parse a let expression.                                                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprLetParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  ARRAY_FEFN_LETDEF letdefs;  FEFN_LETDEF letdef;  int code1, code2;
  ARRAY_NAME names;  FILE_POS save_pos, file_pos;
  TYPE_VARS type_vars;  TYPE result_type;
  ARRAY_FEFN_PARAM params;
  EXPR expr_fnhead, expr_precond, subexpr;
  PARAM_BODIES_TYPE param_bodies_type;

  /* skip initial let, known to be there */
  save_pos = LexFilePos(curr_token);
  next_token;

  /* parse letdefs */
  ArrayInit(&letdefs);
  while( LexType(curr_token) != TK_IN )
  {
    /* parse letdef */
    params = NULL;
    param_bodies_type = PARAM_BODIES_FIXED;
    if( !FEFnParse(t, TRUE, FALSE, FEATURE_NONCREATION, FALSE, TRUE, FALSE,
	FALSE, NULL, FALSE, sv, &file_pos, &names, &type_vars, &params,
	&result_type, &expr_fnhead, &param_bodies_type, &expr_precond,
	&code1, &code2) )
      return FALSE;
    assert(expr_fnhead != NULL);

    /* insert precondition, if any */
    if( expr_precond != NULL )
      ExprFnHeadInsertPrecondition(expr_fnhead, expr_precond);

    /* only one name allowed here */
    if( ArraySize(names) > 1 )
    {
      fprintf(stderr, "%s: multiple names in letdef useless and not allowed\n",
	FilePosShow(save_pos));
      return FALSE;
    }

    /* make and store the letdef */
    letdef = FEFnLetDefNew(file_pos, ArrayFirst(names), type_vars, params,
      result_type, expr_fnhead);
    ArrayAddLast(letdefs, letdef);
  }

  /* parse concluding in part and make result */
  skip(TK_IN, "\"in\" keyword, or definition in \"let\" expression");
  if( !ExprParse(t, sv, &subexpr) )
    return FALSE;
  skip(TK_END, "concluding \"end\" keyword of \"let\" expression");
  *res = (EXPR) LetExprNew(save_pos, letdefs, subexpr);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprFunParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)                */
/*                                                                           */
/*  Parse one anonymous function; it's stored as a let expression:           */
/*                                                                           */
/*    fun(a, b: int): real := sqrt(a + b)                                    */
/*                                                                           */
/*  becomes                                                                  */
/*                                                                           */
/*    let fun1(a, b: int): real := sqrt(a + b) in fun1 end                   */
/*                                                                           */
/*  where fun1 is an invented name.  However, kind field remains             */
/*  EXPR_FUN as a record of where the let came from.                         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprFunParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  ARRAY_FEFN_LETDEF letdefs;  FEFN_LETDEF letdef;  ARRAY_FEFN_PARAM params;
  TYPE_VARS type_vars;  TYPE result_type;
  EXPR expr_fnhead, expr_precond, subexpr;  ARRAY_NAME names;  NAME name;
  FILE_POS file_pos;  int code1, code2;
  PARAM_BODIES_TYPE param_bodies_type;

  /* parse the function */
  param_bodies_type = PARAM_BODIES_FIXED;
  if( !FEFnParse(t, FALSE, TRUE, FEATURE_NONCREATION, FALSE, TRUE, FALSE,
	FALSE, NULL, FALSE, sv, &file_pos, &names, &type_vars, &params,
	&result_type, &expr_fnhead, &param_bodies_type, &expr_precond,
	&code1, &code2) )
    return FALSE;
  assert(expr_fnhead != NULL);

  /* insert precondition, if any */
  if( expr_precond != NULL )
    ExprFnHeadInsertPrecondition(expr_fnhead, expr_precond);

  /* make one letdef */
  ArrayInit(&letdefs);
  name = ArrayFirst(names);
  letdef = FEFnLetDefNew(file_pos, name, type_vars, params, result_type,
    expr_fnhead);
  ArrayAddLast(letdefs, letdef);

  /* make result object */
  subexpr = ExprMakeRawCall(file_pos, NameRep(name), NULL, NULL);
  *res = (EXPR) LetExprNew(file_pos, letdefs, subexpr);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprLetCopyUninstantiated(EXPR_LET expr_let,                        */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on let             */
/*  expression expr_let.                                                     */
/*                                                                           */
/*****************************************************************************/

EXPR ExprLetCopyUninstantiated(EXPR_LET expr_let,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_LET res;
  ExprNew(res, EXPR_LET, expr_let->kind_tag, expr_let->file_pos,
    expr_let->param_name);
  res->letdefs =
    FEFnLetDefsCopyUninstantiated(expr_let->letdefs, orig_params, copy_params);
  res->subexpr =
    ExprCopyUninstantiated(expr_let->subexpr, orig_params, copy_params);
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprLetManifest(EXPR_LET *e, CONTEXT cxt, TYPE self_type,        */
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Carry out the specification of ExprManifest on a let expression.         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprLetManifest(EXPR_LET *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn)
{
  EXPR_LET expr_let;  FEFN_LETDEF letdef;
  expr_let = *e;
  expr_let->large_scale = FALSE;

  /* add letdef names to context and manifest their functions */
  ArrayForEach(expr_let->letdefs, letdef)
    if( !FEFnLetDefManifest(letdef, cxt, self_type, encl_befn,
	&expr_let->large_scale) )
      return FALSE;
  
  /* manifest the part after "in" */
  if( !ExprManifest(&expr_let->subexpr, cxt, self_type, encl_befn) )
    return FALSE;
  expr_let->type = expr_let->subexpr->type;

  /* delete the letdef names from context in reverse order */
  ArrayForEachReverse(expr_let->letdefs, letdef)
    ContextDelete(cxt, (NAMED) letdef);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprLetDebug(EXPR_LET expr_let, CONTEXT cxt, BOOLEAN show_types,    */
/*    FILE *fp, int print_style)                                             */
/*                                                                           */
/*  Debug print of expr e on *fp.                                            */
/*                                                                           */
/*****************************************************************************/

void ExprLetDebug(EXPR_LET expr_let, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style)
{
  FEFN_LETDEF letdef;
  fprintf(fp, "let");
  begin_indent;
  ArrayForEach(expr_let->letdefs, letdef)
  {
    next_line;
    FEFnLetDefDebug(letdef, cxt, show_types, fp, print_style);
  }
  end_indent;
  next_line;
  fprintf(fp, "in");
  begin_indent;
  next_line;
  ExprDebug(expr_let->subexpr, cxt, show_types, fp, print_style);
  end_indent;
  next_line;
  fprintf(fp, "end");
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprLetCodeGen(EXPR_LET expr_let, CODEGEN_OBJ res_be_var,           */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Carry out the specification of ExprCodeGen on let expression expr_let.   */
/*                                                                           */
/*  Let expressions can be small-scale, if all their definitions are         */
/*  functions with parameters.  Only unparameterized ones are printed here.  */
/*                                                                           */
/*****************************************************************************/

void ExprLetCodeGen(EXPR_LET expr_let, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  if( expr_let->large_scale )
  {
    /* print non-function letdef declarations */
    be->BlockBegin();
    FEFnLetDefsVarEnterScope(expr_let->letdefs, be);

    /* print subexpression */
    ExprCodeGen(expr_let->subexpr, res_be_var, res_be_type, be);

    /* end declarations */
    be->BlockEnd();

    /* unregister the names, since they are going out of scope */
    FEFnLetDefsVarLeaveScope(expr_let->letdefs, be);
  }
  else
  {
    /* small-scale, so no definitions; just print the subexpression */
    ExprCodeGen(expr_let->subexpr, res_be_var, res_be_type, be);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprLetInitOrder(EXPR_LET expr_let, int visit_num,               */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of ExprInitOrder on let expression           */
/*  expr_let.  Note that there is no need to explore the bodies of the       */
/*  let definitions here, since that will be done when they are called;      */
/*  and if they aren't called, they don't matter anyway.                     */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprLetInitOrder(EXPR_LET expr_let, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  return ExprInitOrder(expr_let->subexpr, visit_num, report, fun);
}
