/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_list.c                                                */
/*  DESCRIPTION:  Nonpareil manifest list expressions                        */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"


/*****************************************************************************/
/*                                                                           */
/*  EXPR_LIST                                                                */
/*                                                                           */
/*  A list of zero or more expressions, held in e->subexpressions.           */
/*                                                                           */
/*****************************************************************************/

struct expr_list_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* file position of expression       */
  USTRING		param_name;	/* param name when := present        */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */
  ARRAY_EXPR		subexpressions;	/* the subexpressions                */
};


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprListParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)               */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprListParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  EXPR sub_expr;  EXPR_LIST res_list;

  /* set up res_list and *res */
  ExprNew(res_list, EXPR_LIST, KIND_EXPR_LIST, LexFilePos(curr_token), NULL);
  ArrayInit(&res_list->subexpressions);
  *res = (EXPR) res_list;

  /* skip initial left bracket, known to be there */
  next_token;

  /* parse optional expression sequence */
  if( LexType(curr_token) != TK_RIGHT_BRACKET )
  {
    /* nonempty list case */
    if( !ExprParse(t, sv, &sub_expr) )
      return FALSE;
    ArrayAddLast(res_list->subexpressions, sub_expr);
    while( LexType(curr_token) == TK_COMMA )
    {
      next_token;
      if( !ExprParse(t, sv, &sub_expr) )
        return FALSE;
      ArrayAddLast(res_list->subexpressions, sub_expr);
    }
    skip(TK_RIGHT_BRACKET, "\",\" or closing \"]\" of manifest list");
  }
  else
  {
    /* empty list case */
    next_token;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprListCopyUninstantiated(EXPR_LIST expr_list,                     */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on list            */
/*  expression e.                                                            */
/*                                                                           */
/*****************************************************************************/

EXPR ExprListCopyUninstantiated(EXPR_LIST expr_list,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_LIST res;  EXPR e;
  ExprNew(res, EXPR_LIST, KIND_EXPR_LIST, expr_list->file_pos,
    expr_list->param_name);
  res->subexpressions = NULL;
  if( expr_list->subexpressions != NULL )
  {
    ArrayInit(&res->subexpressions);
    ArrayForEach(expr_list->subexpressions, e)
      ArrayAddLast(res->subexpressions,
	ExprCopyUninstantiated(e, orig_params, copy_params));
  }
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprListManifest(EXPR_LIST *e, CONTEXT cxt, TYPE self_type,      */
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Carry out the specification of ExprManifest on a list expression.        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprListManifest(EXPR_LIST *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn)
{
  EXPR tail_expr, sub_expr;  EXPR_LIST expr_list;  ARRAY_EXPR params;
  expr_list = *e;

  /* we're working backwards and initially tail_expr is the empty list */
  tail_expr = ExprMakeRawCall((*e)->file_pos,
    NameRep(ClassName(ClassEList, cxt)), NULL, NULL);

  ArrayForEachReverse(expr_list->subexpressions, sub_expr)
  {
    /* make a parameter list consisting of this subexpression plus the tail */
    ArrayInit(&params);
    ArrayAddLast(params, sub_expr);
    ArrayAddLast(params, tail_expr);

    /* make the tail be an implicit call to the nlist creation function */
    tail_expr = ExprMakeRawCall(sub_expr->file_pos,
      NameRep(ClassName(ClassNList, cxt)), NULL, params);
  }

  /* now the current tail replaces *e and gets manifested */
  *e = (EXPR_LIST) tail_expr;
  return ExprManifest((EXPR *) e, cxt, self_type, encl_befn);
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprListDebug(EXPR_LIST e, CONTEXT cxt, BOOLEAN show_types,         */
/*    FILE *fp, int print_style)                                             */
/*                                                                           */
/*  Debug print of expr e on *fp.                                            */
/*                                                                           */
/*****************************************************************************/

void ExprListDebug(EXPR_LIST e, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style)
{
  EXPR_LIST expr_list;  EXPR child;
  ExprKindCheck(e, KIND_EXPR_LIST, "ExprListDebug");
  expr_list = (EXPR_LIST) e;
  fprintf(fp, "[");
  ArrayForEach(expr_list->subexpressions, child)
  {
    if( child != ArrayFirst(expr_list->subexpressions) )  fprintf(fp, ", ");
    ExprDebug(child, cxt, show_types, fp, SINGLE_LINE);
  }
  fprintf(fp, "]");
}
