/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_lit.c                                                 */
/*  DESCRIPTION:  Nonpareil literal expressions                              */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"
#define DEBUG1 0
#define DEBUG2 0

/*****************************************************************************/
/*                                                                           */
/*  EXPR_LIT                                                                 */
/*                                                                           */
/*****************************************************************************/

struct expr_lit_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* the identifying token             */
  USTRING		param_name;	/* param name when := present        */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */
  USTRING		value;		/* string value of literal           */
  SYSTEM_VIEW		system_view;	/* saved for string literals use     */
  BEFN_FEATURE		predef;		/* KIND_EXPR_LIT_STRING only         */
  int			int_val;	/* for integers and enums only       */
};


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprLitParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)                */
/*                                                                           */
/*  Parse one literal, returning the resulting EXPR in *res.                 */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprLitParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  EXPR_LIT e;  KIND_TAG kind_tag;
  switch( LexType(curr_token) )
  {

    case TK_FALSE:
    case TK_TRUE:

      /* boolean literal */
      kind_tag = KIND_EXPR_LIT_BOOLEAN;
      break;


    case TK_LIT_CHARACTER:

      /* character literal */
      kind_tag = KIND_EXPR_LIT_CHAR;
      break;


    case TK_LIT_STRING:

      /* string literal */
      kind_tag = KIND_EXPR_LIT_STRING;
      break;


    case TK_LIT_INTEGER:

      /* integer literal */
      kind_tag = KIND_EXPR_LIT_INTEGER;
      break;


    case TK_LIT_REAL:

      /* real literal */
      kind_tag = KIND_EXPR_LIT_REAL;
      break;


    default:

      /* error case */
      assert(FALSE);
  }
  ExprNew(e, EXPR_LIT, kind_tag, LexFilePos(curr_token), NULL);
  e->value = LexValue(curr_token);
  e->system_view = sv;
  e->predef = NULL;
  next_token;
  *res = (EXPR) e;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprLitCopyUninstantiated(EXPR_LIT expr_lit,                        */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on a literal       */
/*  expression.                                                              */
/*                                                                           */
/*****************************************************************************/

EXPR ExprLitCopyUninstantiated(EXPR_LIT expr_lit,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_LIT res;
  ExprNew(res, EXPR_LIT, expr_lit->kind_tag, expr_lit->file_pos,
    expr_lit->param_name);
  res->value = expr_lit->value;
  res->system_view = expr_lit->system_view;
  res->predef = NULL;
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprLitManifest(EXPR_LIT *e, CONTEXT context, TYPE self_type,    */
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Carry out the specification of ExprManifest on literal expressions.      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprLitManifest(EXPR_LIT *e, CONTEXT context, TYPE self_type,
  BEFN encl_befn)
{
  EXPR_LIT expr_lit = *e;
  switch( expr_lit->kind_tag )
  {
    case KIND_EXPR_LIT_BOOLEAN:

      expr_lit->type = ClassType(ClassBool);
      expr_lit->int_val =
	UStringEqual(expr_lit->value, AStringToUString("true")) ? 1 : 0;
      break;


    case KIND_EXPR_LIT_CHAR:

      expr_lit->type = ClassType(ClassChar);
      assert(UStringLength(expr_lit->value) == 1);
      expr_lit->int_val = UStringGet(expr_lit->value, 0);
      break;


    case KIND_EXPR_LIT_STRING:

      expr_lit->predef = SystemViewRegisterLiteralString(expr_lit->system_view,
	expr_lit->value, expr_lit->file_pos);
      expr_lit->type = ClassType(ClassString);
      break;


    case KIND_EXPR_LIT_INTEGER:

      expr_lit->int_val = UStringToInteger(expr_lit->value);
      if( expr_lit->int_val <= UCHAR_MAX )
        expr_lit->type = ClassType(ClassUByte);
      else if( expr_lit->int_val <= USHRT_MAX )
        expr_lit->type = ClassType(ClassUShort);
      else
	expr_lit->type = ClassType(ClassUInt);
      break;


    case KIND_EXPR_LIT_REAL:

      expr_lit->type = ClassType(ClassReal);
      break;


    default:

      assert(FALSE);
  }
  expr_lit->large_scale = FALSE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprLitDebug(EXPR_LIT expr_lit, CONTEXT cxt, BOOLEAN show_types,    */
/*    FILE *fp, int print_style)                                             */
/*                                                                           */
/*  Debug print of expr e on *fp.                                            */
/*                                                                           */
/*****************************************************************************/

void ExprLitDebug(EXPR_LIT expr_lit, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style)
{
  switch( expr_lit->kind_tag )
  {
    case KIND_EXPR_LIT_BOOLEAN:
    case KIND_EXPR_LIT_INTEGER:
    case KIND_EXPR_LIT_REAL:

      fprintf(fp, "%s", UStringToUTF8(expr_lit->value));
      break;


    case KIND_EXPR_LIT_CHAR:

      fprintf(fp, "'%s'", UStringToUTF8(expr_lit->value));
      break;


    case KIND_EXPR_LIT_STRING:

      fprintf(fp, "\"%s\"", UStringToUTF8(expr_lit->value));
      break;


    default:

      assert(FALSE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprLitFromChar(FILE_POS file_pos, UCHAR ch)                        */
/*                                                                           */
/*  Return a mainfested literal character with the given value.              */
/*                                                                           */
/*****************************************************************************/

EXPR ExprLitFromChar(FILE_POS file_pos, UCHAR ch)
{
  EXPR_LIT e;
  ExprNew(e, EXPR_LIT, KIND_EXPR_LIT_CHAR, file_pos, NULL);
  e->type = ClassType(ClassChar);
  UStringBeginWithSize(1);
  UStringAdd(ch);
  e->value = UStringEnd();
  e->predef = NULL;
  return (EXPR) e;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprLitFromInt(FILE_POS file_pos, int val)                          */
/*                                                                           */
/*  Return a manifested literal integer expression with the given value.     */
/*                                                                           */
/*****************************************************************************/

EXPR ExprLitFromInt(FILE_POS file_pos, int val)
{
  EXPR_LIT e;
  ExprNew(e, EXPR_LIT, KIND_EXPR_LIT_INTEGER, file_pos, NULL);
  e->type = ClassType(ClassInt);
  e->value = UStringFromInteger(val);
  e->predef = NULL;
  e->int_val = val;
  return (EXPR) e;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprLitFromBool(FILE_POS file_pos, BOOLEAN val)                     */
/*                                                                           */
/*  Return a manifested literal boolean expression with the given value.     */
/*                                                                           */
/*****************************************************************************/

EXPR ExprLitFromBool(FILE_POS file_pos, BOOLEAN val)
{
  EXPR_LIT e;
  static USTRING true_str = NULL;
  static USTRING false_str = NULL;
  if( true_str == NULL )
  {
    true_str = AStringToUString("true");
    false_str = AStringToUString("false");
  }
  ExprNew(e, EXPR_LIT, KIND_EXPR_LIT_BOOLEAN, file_pos, NULL);
  e->type = ClassType(ClassBool);
  e->value = val ? true_str : false_str;
  e->predef = NULL;
  return (EXPR) e;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprLitFromString(FILE_POS file_pos, USTRING str, SYSTEM_VIEW sv)   */
/*                                                                           */
/*  Return a manifested literal string expression with the given value.      */
/*                                                                           */
/*****************************************************************************/

EXPR ExprLitFromString(FILE_POS file_pos, USTRING str, SYSTEM_VIEW sv)
{
  EXPR_LIT e;
  ExprNew(e, EXPR_LIT, KIND_EXPR_LIT_STRING, file_pos, NULL);
  e->value = str;
  e->predef = SystemViewRegisterLiteralString(sv, str, file_pos);
  e->type = ClassType(ClassString);
  return (EXPR) e;
}


/*****************************************************************************/
/*                                                                           */
/*  int ExprLitConstIntValue(EXPR e)                                         */
/*                                                                           */
/*  Carry out the specification of ExprConstIntValue on literal expression   */
/*  e.                                                                       */
/*                                                                           */
/*****************************************************************************/

int ExprLitConstIntValue(EXPR e)
{
  switch( e->kind_tag )
  {
    case KIND_EXPR_LIT_CHAR:
    case KIND_EXPR_LIT_INTEGER:

      return ((EXPR_LIT) e)->int_val;


    default:

      fprintf(stderr, "%s: attempt to find int value of unsuitable expr\n",
	FilePosShow(e->file_pos));
      exit(1);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int ExprLitIntValue(EXPR e)                                              */
/*                                                                           */
/*  Return the integer value of e.  It must have such a value.               */
/*                                                                           */
/*****************************************************************************/

int ExprLitIntValue(EXPR e)
{
  /* get past any type casts in e */
  while( e->kind_tag == KIND_EXPR_CALL )
    e = ExprCallSoleParameter(e);
  return ExprLitConstIntValue(e);
}


/*****************************************************************************/
/*                                                                           */
/*  int ExprLitConstCmp(EXPR e1, EXPR e2)                                    */
/*                                                                           */
/*  Carry out the specification of ExprConstCmp on literal expressions       */
/*  e1 and e2.                                                               */
/*                                                                           */
/*****************************************************************************/

int ExprLitConstCmp(EXPR e1, EXPR e2)
{
  EXPR_LIT el1, el2;  int res;
  if( DEBUG1 )
  {
    fprintf(stderr, "[ ExprLitConstCmp(");
    ExprDebug(e1, NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ", ");
    ExprDebug(e2, NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, ")\n");
  }

  assert(e1->kind_tag == e2->kind_tag);
  el1 = (EXPR_LIT) e1;
  el2 = (EXPR_LIT) e2;
  switch( e1->kind_tag )
  {
    case KIND_EXPR_LIT_CHAR:
    case KIND_EXPR_LIT_INTEGER:

      res = el1->int_val - el2->int_val;
      if( DEBUG1 )
	fprintf(stderr, "] ExprLitConstCmp returning %d (char or int)\n", res);
      return res;


    case KIND_EXPR_LIT_STRING:

      res = UStringCmp(el1->value, el2->value);
      if( DEBUG1 )
	fprintf(stderr, "] ExprLitConstCmp returning %d (string)\n", res);
      return res;


    default:

      fprintf(stderr, "%s: attempt to compare incomparable expressions\n",
	FilePosShow(e1->file_pos));
      if( DEBUG2 )
	fprintf(stderr, "  (kind_tag is %s)\n", KindShow(e1->kind_tag));
      exit(1);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprLitCodeGen(EXPR_LIT expr_lit, CODEGEN_OBJ res_be_var,           */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Carry out the specification of ExprCodeGen on literal expression e.      */
/*                                                                           */
/*****************************************************************************/

void ExprLitCodeGen(EXPR_LIT expr_lit, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  if( res_be_var != NULL )
    be->VarAsstBegin(res_be_var);
  be->CastBegin(res_be_type, TypeBEType(expr_lit->type, be));
  switch( expr_lit->kind_tag )
  {
    case KIND_EXPR_LIT_BOOLEAN:

      be->LiteralConst((ASTRING) UStringToUTF8(expr_lit->value));
      break;


    case KIND_EXPR_LIT_INTEGER:
    case KIND_EXPR_LIT_REAL:

      be->LiteralConst(UStringNumberToAString(expr_lit->value));
      break;


    case KIND_EXPR_LIT_CHAR:

      be->LiteralChar(UStringGet(expr_lit->value, 0));
      break;


    case KIND_EXPR_LIT_STRING:

      BEFnCallCodeGen((BEFN) expr_lit->predef, NULL, res_be_type, be);
      break;


    default:

      assert(FALSE);
  }
  be->CastEnd();
  if( res_be_var != NULL )
    be->AsstEnd();
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprLitInitOrder(EXPR_LIT expr_lit, int visit_num,               */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Implement ExprInitOrder for literal expressions.                         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprLitInitOrder(EXPR_LIT expr_lit, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  return expr_lit->predef == NULL ? TRUE :
    BEFnFeatureInitOrder(expr_lit->predef, visit_num, report, fun);
}
