/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_paren.c                                               */
/*  DESCRIPTION:  Nonpareil parenthesized expressions and tuples             */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"


/*****************************************************************************/
/*                                                                           */
/*  EXPR_PAREN                                                               */
/*                                                                           */
/*  A parenthesized expression; e->subexpr is the subexpression.             */
/*                                                                           */
/*****************************************************************************/

struct expr_paren_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* file position of expression       */
  USTRING		param_name;	/* param name when := present        */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */
  EXPR			subexpr;	/* the subexpression                 */
};


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprParenNew(FILE_POS file_pos, EXPR sub_expr)                      */
/*                                                                           */
/*  Return a new paren expression with the given attributes.                 */
/*                                                                           */
/*****************************************************************************/

EXPR ExprParenNew(FILE_POS file_pos, EXPR sub_expr)
{
  EXPR_PAREN res_paren;
  ExprNew(res_paren, EXPR_PAREN, KIND_EXPR_PAREN, file_pos, NULL);
  res_paren->subexpr = sub_expr;
  return (EXPR) res_paren;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprParenCopyUninstantiated(EXPR_PAREN expr_paren,                  */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on parenthesized   */
/*  expression e.                                                            */
/*                                                                           */
/*****************************************************************************/

EXPR ExprParenCopyUninstantiated(EXPR_PAREN expr_paren,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_PAREN res;
  ExprNew(res, EXPR_PAREN, KIND_EXPR_PAREN, expr_paren->file_pos, NULL);
  res->subexpr =
    ExprCopyUninstantiated(expr_paren->subexpr, orig_params, copy_params);
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprParenManifest(EXPR_PAREN *expr_paren, CONTEXT cxt,           */
/*    TYPE self_type, BEFN encl_befn)                                        */
/*                                                                           */
/*  Carry out the specification of ExprManifest on a paren expression.       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprParenManifest(EXPR_PAREN *e, CONTEXT cxt,
  TYPE self_type, BEFN encl_befn)
{
  EXPR_PAREN expr_paren = *e;
  if( !ExprManifest(&expr_paren->subexpr, cxt, self_type, encl_befn) )
    return FALSE;
  expr_paren->type = expr_paren->subexpr->type;
  expr_paren->large_scale = expr_paren->subexpr->large_scale;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprParenDebug(EXPR_PAREN expr_paren, CONTEXT cxt,                  */
/*    BOOLEAN show_types, FILE *fp, int print_style)                         */
/*                                                                           */
/*  Debug print of paren expr e on *fp.                                      */
/*                                                                           */
/*****************************************************************************/

void ExprParenDebug(EXPR_PAREN expr_paren, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style)
{
  fprintf(fp, "(");
  ExprDebug(expr_paren->subexpr, cxt, show_types, fp, SINGLE_LINE);
  fprintf(fp, ")");
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprParenCodeGen(EXPR_PAREN expr_paren, CODEGEN_OBJ res_be_var,     */
/*    CODEGEN_TYPE res_be_type, CODEGEN be)                                  */
/*                                                                           */
/*  Carry out the specification of ExprCodeGen on paren expression e.        */
/*  There is no need to print these parentheses in the generated code,       */
/*  since the printed code is always fully parenthesized anyway.  Indeed     */
/*  it would be wrong, if the enclosed expression is large-scale.            */
/*                                                                           */
/*****************************************************************************/

void ExprParenCodeGen(EXPR_PAREN expr_paren, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  ExprCodeGen(expr_paren->subexpr, res_be_var, res_be_type, be);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprParenInitOrder(EXPR_PAREN expr_paren, int visit_num,         */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of ExprInitOrder on a parenthesized          */
/*  expression.                                                              */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprParenInitOrder(EXPR_PAREN expr_paren, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  return ExprInitOrder(expr_paren->subexpr, visit_num, report, fun);
}
