/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_precond.c                                             */
/*  DESCRIPTION:  Expression holding a sequence of preconditions.            */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"
#define DEBUG1 0
#define DEBUG2 0


/*****************************************************************************/
/*                                                                           */
/*  EXPR_PRECOND                                                             */
/*                                                                           */
/*****************************************************************************/

struct expr_precond_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* the identifying token             */
  USTRING		param_name;	/* param name when := present        */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */

  ARRAY_EXPR		preconditions;	/* the precondition expressions      */
  EXPR			subexpression;	/* optional subexpression            */
};


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprPrecondNew(FILE_POS file_pos)                                   */
/*                                                                           */
/*  Make a new function head expression with these attributes.               */
/*                                                                           */
/*****************************************************************************/

EXPR ExprPrecondNew(FILE_POS file_pos)
{
  EXPR_PRECOND res;
  ExprNew(res, EXPR_PRECOND, KIND_EXPR_PRECOND, file_pos, NULL);
  ArrayInit(&res->preconditions);
  res->subexpression = NULL;
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprPrecondAddPrecondition(EXPR e, EXPR precondition)               */
/*                                                                           */
/*  Add a precondition expression to e.                                      */
/*                                                                           */
/*****************************************************************************/

void ExprPrecondAddPrecondition(EXPR e, EXPR precondition)
{
  EXPR_PRECOND expr_precond;
  ExprKindCheck(e, KIND_EXPR_PRECOND, "ExprPrecondAddPrecondition");
  expr_precond = (EXPR_PRECOND) e;
  ArrayAddLast(expr_precond->preconditions, precondition);
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprPrecondAddSubExpression(EXPR e, EXPR subexpression)             */
/*                                                                           */
/*  Add subexpr to e.  Thus must be no subexpression already present.        */
/*                                                                           */
/*****************************************************************************/

void ExprPrecondAddSubExpression(EXPR e, EXPR subexpression)
{
  EXPR_PRECOND expr_precond;
  ExprKindCheck(e, KIND_EXPR_PRECOND, "ExprPrecondAddSubExpression");
  expr_precond = (EXPR_PRECOND) e;
  assert(expr_precond->subexpression == NULL);
  expr_precond->subexpression = subexpression;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprPrecondCopyUninstantiated(EXPR e, ARRAY_FEFN_PARAM orig_params, */
/*    ARRAY_FEFN_PARAM copy_params)                                          */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on fnhead          */
/*  expression e.                                                            */
/*                                                                           */
/*****************************************************************************/

EXPR ExprPrecondCopyUninstantiated(EXPR_PRECOND expr_precond,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_PRECOND res;  EXPR e1, e2;
  ExprNew(res, EXPR_PRECOND, KIND_EXPR_PRECOND, expr_precond->file_pos,
    expr_precond->param_name);
  ArrayInit(&res->preconditions);
  ArrayForEach(expr_precond->preconditions, e1)
  {
    e2 = ExprCopyUninstantiated(e1, orig_params, copy_params);
    ArrayAddLast(res->preconditions, e2);
  }
  res->subexpression = expr_precond->subexpression == NULL ? NULL :
    ExprCopyUninstantiated(expr_precond->subexpression, orig_params,
      copy_params);
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprPrecondManifest(EXPR_PRECOND *e, CONTEXT cxt, TYPE self_type,*/
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Manifest precondition expression e.                                      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprPrecondManifest(EXPR_PRECOND *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn)
{
  EXPR_PRECOND expr_precond;  int i;  EXPR ep;  COERCION c;
  expr_precond = *e;
  if( DEBUG2 )
  {
    fprintf(stderr, "[ ExprPrecondManifest(e, ...), e =\n  ");
    ExprPrecondDebug(expr_precond, cxt, TRUE, stderr, 2);
    fprintf(stderr, "\n");
  }

  /* always large-scale */
  expr_precond->large_scale = TRUE;

  /* manifest and organize preconditions */
  for( i = 0;  i < ArraySize(expr_precond->preconditions);  i++ )
  {
    /* manifest the precondition */
    ep = ArrayGet(expr_precond->preconditions, i);
    if( !ExprManifest(&ep, cxt, self_type, encl_befn) )
      db_return(DEBUG2, "ExprPrecondManifest", TRUE);

    /* make sure its type is boolean */
    if( !TypeIsSubType(ExprType(ep), ClassType(ClassBool), &c, cxt) )
    {
      fprintf(stderr, "%s: precondition has type %s\n",
	FilePosShow(ExprFilePos(ep)), TypeShow(ExprType(ep), cxt));
      db_return(DEBUG2, "FEFnPrecondManifest type", FALSE);
    }

    /* enclose in assert() */
    ep = ExprEncloseInInstantiatedAssert(ep);

    /* replace in array */
    ArrayPut(expr_precond->preconditions, i, ep);
  }

  /* manifest subexpression, if present */
  if( expr_precond->subexpression != NULL )
    ExprManifest(&expr_precond->subexpression, cxt, self_type, encl_befn);

  db_return(DEBUG2, "ExprPrecondManifest", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprPrecondInitOrder(EXPR_PRECOND expr_precond, int visit_num,   */
/*    BOOLEAN *report, BEFN_SYSTEM_INIT fun)                                 */
/*                                                                           */
/*  Carry out the specification of ExprInitOrder on precondition e.          */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprPrecondInitOrder(EXPR_PRECOND expr_precond, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun)
{
  EXPR e;
  ArrayForEach(expr_precond->preconditions, e)
    if( !ExprInitOrder(e, visit_num, report, fun) )
      return FALSE;
  if( expr_precond->subexpression != NULL )
    if( !ExprInitOrder(expr_precond->subexpression, visit_num, report, fun) )
      return FALSE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprPrecondCodeGen(EXPR_PRECOND expr_precond,                       */
/*    CODEGEN_OBJ res_be_var, CODEGEN_TYPE res_be_type, CODEGEN be)          */
/*                                                                           */
/*  Generate code for this precondition expression.  Its elements already    */
/*  include the enclosure in assert().                                       */
/*                                                                           */
/*****************************************************************************/

void ExprPrecondCodeGen(EXPR_PRECOND expr_precond, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be)
{
  EXPR e;
  ArrayForEach(expr_precond->preconditions, e)
    Stmt(ExprCodeGen(e, NULL, be->void_type, be));
  if( expr_precond->subexpression != NULL )
    ExprCodeGen(expr_precond->subexpression, res_be_var, res_be_type, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprPrecondDebug(EXPR_PRECOND expr_precond, CONTEXT cxt,            */
/*    BOOLEAN show_types, FILE *fp, int print_style)                         */
/*                                                                           */
/*  Debug print of preconditions expression expr_precond.                    */
/*                                                                           */
/*****************************************************************************/

void ExprPrecondDebug(EXPR_PRECOND expr_precond, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style)
{
  EXPR e;
  fprintf(fp, "precond ");
  ArrayForEach(expr_precond->preconditions, e)
  {
    next_line;
    ExprDebug(e, cxt, show_types, fp, print_style);
  }
  if( expr_precond->subexpression != NULL )
  {
    next_line;
    fprintf(fp, "  sub ");
    ExprDebug(expr_precond->subexpression, cxt, show_types, fp, print_style);
  }
}
