/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_tuple.c                                               */
/*  DESCRIPTION:  Nonpareil manifest tuples                                  */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include "expr.h"


/*****************************************************************************/
/*                                                                           */
/*  EXPR_TUPLE                                                               */
/*                                                                           */
/*  A tuple of two or more expressions, held in e->subexpressions.           */
/*                                                                           */
/*****************************************************************************/

struct expr_tuple_rec {
  KIND_TAG		kind_tag;	/* what kind of expr this is         */
  FILE_POS		file_pos;	/* file position of expression       */
  USTRING		param_name;	/* param name when := present        */
  TYPE			type;		/* actual type when manifested       */
  BOOLEAN		large_scale;	/* contains let or case              */
  CODEGEN_OBJ		be_var;		/* temp field used by code gen       */
  ARRAY_EXPR		subexpressions;	/* the subexpressions                */
};


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprParenOrTupleParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)       */
/*                                                                           */
/*  Parse a parenthesized expression or manifest tuple.                      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprParenOrTupleParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res)
{
  EXPR sub_expr;  FILE_POS file_pos;

  /* save and skip initial left parenthesis, known to be there */
  file_pos = LexFilePos(curr_token);
  next_token;

  /* parse first inner expression */
  if( !ExprParse(t, sv, &sub_expr) )
    return FALSE;

  if( LexType(curr_token) == TK_COMMA )
  {
    /* tuple case: initialize res_tuple, including first subexpression */
    EXPR_TUPLE res_tuple;
    ExprNew(res_tuple, EXPR_TUPLE, KIND_EXPR_TUPLE, file_pos, NULL);
    ArrayInit(&res_tuple->subexpressions);
    ArrayAddLast(res_tuple->subexpressions, sub_expr);

    /* get remaining subexpressions */
    while( LexType(curr_token) == TK_COMMA )
    {
      next_token;
      if( !ExprParse(t, sv, &sub_expr) )
        return FALSE;
      ArrayAddLast(res_tuple->subexpressions, sub_expr);
    }

    /* skip concluding right parenthesis */
    skip(TK_RIGHT_PAREN, "\",\" or closing \")\" of tuple");

    /* set *res to res_tuple */
    *res = (EXPR) res_tuple;
  }
  else
  {
    /* parenthesized expression case: skip right parenthesis and set *res */
    skip(TK_RIGHT_PAREN, "\")\" (or \",\" of tuple)");
    *res = ExprParenNew(file_pos, sub_expr);
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR ExprTupleCopyUninstantiated(EXPR_TUPLE expr_tuple,                  */
/*    ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)            */
/*                                                                           */
/*  Carry out the specification of ExprCopyUninstantiated on tuple           */
/*  expression e.                                                            */
/*                                                                           */
/*****************************************************************************/

EXPR ExprTupleCopyUninstantiated(EXPR_TUPLE expr_tuple,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  EXPR_TUPLE res;  EXPR e;
  ExprNew(res, EXPR_TUPLE, KIND_EXPR_TUPLE, expr_tuple->file_pos,
    expr_tuple->param_name);
  ArrayInit(&res->subexpressions);
  ArrayForEach(expr_tuple->subexpressions, e)
    ArrayAddLast(res->subexpressions,
      ExprCopyUninstantiated(e, orig_params, copy_params));
  return (EXPR) res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ExprTupleManifest(EXPR_TUPLE *e, CONTEXT cxt, TYPE self_type,    */
/*    BEFN encl_befn)                                                        */
/*                                                                           */
/*  Carry out the specification of ExprManifest on a tuple expression.       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ExprTupleManifest(EXPR_TUPLE *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn)
{
  CLASS c;  int len;
  EXPR_TUPLE expr_tuple = *e;

  /* determine mc, the relevant tuple class */
  len = ArraySize(expr_tuple->subexpressions);
  assert(len >= 2);
  if( len > MAX_TUPLE )
  {
    fprintf(stderr, "%s: too many components (%d) in tuple - max is %d\n",
      FilePosShow(expr_tuple->file_pos), len, MAX_TUPLE);
    return FALSE;
  }
  c = ClassTuple[len];

  /* simply add the class name as the node value, change type, and manifest */
  *e = (EXPR_TUPLE) ExprMakeRawCall(expr_tuple->file_pos,
	NameRep(ClassName(c, cxt)), NULL, expr_tuple->subexpressions);
  return ExprManifest((EXPR *) e, cxt, self_type, encl_befn);
}


/*****************************************************************************/
/*                                                                           */
/*  void ExprTupleDebug(EXPR_TUPLE expr_tuple, CONTEXT cxt,                  */
/*    BOOLEAN show_types, FILE *fp, int print_style)                         */
/*                                                                           */
/*  Debug print of tuple expr e on *fp.                                      */
/*                                                                           */
/*****************************************************************************/

void ExprTupleDebug(EXPR_TUPLE expr_tuple, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style)
{
  EXPR child;
  fprintf(fp, "(");
  ArrayForEach(expr_tuple->subexpressions, child)
  {
    if( child != ArrayFirst(expr_tuple->subexpressions) )
      fprintf(fp, ", ");
    ExprDebug(child, cxt, show_types, fp, SINGLE_LINE);
  }
  fprintf(fp, ")");
}
