/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         externs.h                                                  */
/*  DESCRIPTION:  Type definitions and external function headers             */
/*                                                                           */
/*****************************************************************************/
#include <assert.h>
#include <limits.h>
#include <stdlib.h>
#include <time.h>
/* more #includes below */


/*****************************************************************************/
/*                                                                           */
/*  MISCELLANEOUS CONSTANTS                                                  */
/*                                                                           */
/*  In addition to these symbols, three strings are expected to be passed    */
/*  to the compilation using the -D (preprocessor define) option of cc:      */
/*                                                                           */
/*    NPC_DIR_SEP     Directory separator, e.g. "/"                          */
/*    NPC_LIB_DIR     Library dir., e.g. "/home/jeff/nonpareil/compiler/lib" */
/*    NPC_C_COMPILER  C compiler, e.g. "gcc -g -ansi -pedantic -Wall"        */
/*                                                                           */
/*  MAX_TUPLE is the maximum number of elements allowed in a tuple;          */
/*  MAX_FUN_PARAMS is the maximum number of parameters allowed in            */
/*  a function.  Either or both may be increased, but if you do this you     */
/*  also need to delete files "tuples" and/or "functs" in the core module.   */
/*  The compiler will generate these files automatically when missing,       */
/*  using MAX_TUPLE and MAX_FUN_PARAMS to determine how many classes to      */
/*  declare in each file.  O(MAX_FUN_PARAMS^2) classes are created in the    */
/*  functs file, so increasing MAX_FUN_PARAMS a lot is a bad idea.           */
/*                                                                           */
/*  A similar automatic creation of file "char" in the core module will be   */
/*  carried out, based on files in NPC_LIB_DIR.  See the README file in      */
/*  that directory for more information.                                     */
/*                                                                           */
/*****************************************************************************/

#define	NPC_VERSION   "Nonpareil Compiler Version 1.02 (August 2005)"
#define MIN_INFIX_PRECEDENCE	   1	/* minimum precedence of infix ops   */
#define MAX_INFIX_PRECEDENCE	 100	/* maximum precedence of infix ops   */
#define	MAX_TUPLE	 	  16    /* max tuple params                  */
#define	MAX_FUN_PARAMS		  12    /* max function params               */


/*****************************************************************************/
/*                                                                           */
/*  KIND_TAG                                                                 */
/*                                                                           */
/*  Type tag distinguishing types of objects.                                */
/*                                                                           */
/*  Indenting indicates inheritance.  Not all object types appear here,      */
/*  only those needing a type tag because they are involved in inheritance   */
/*  relationships: type objects, expressions, and named entities.            */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  KIND_LOCATED,				/* just filepos                      */
    KIND_TYPEV,				/* a var invocation type expression  */
    KIND_TYPEC,				/* a class or meet type expression   */
    KIND_CTYPE,				/* ctype expr. (private to type.c)   */
    KIND_EXPR,				/* an expression (no name)           */
      KIND_EXPR_LIT_BOOLEAN,		/* boolean literals		     */
      KIND_EXPR_LIT_CHAR,		/* character literals		     */
      KIND_EXPR_LIT_STRING,		/* string literals		     */
      KIND_EXPR_LIT_INTEGER,		/* integer literals		     */
      KIND_EXPR_LIT_REAL,		/* real literals		     */
      KIND_EXPR_CALL,			/* call expressions		     */
      KIND_EXPR_PAREN,			/* parenthesized expressions	     */
      KIND_EXPR_TUPLE,			/* manifest tuple expressions	     */
      KIND_EXPR_LIST,			/* manifest list expressions	     */
      KIND_EXPR_ARRAY,			/* manifest array expressions	     */
      KIND_EXPR_LET,			/* let expressions		     */
      KIND_EXPR_FUN,			/* anonymous function expressions    */
      KIND_EXPR_IF,			/* if expressions		     */
      KIND_EXPR_CASE,			/* case expressions		     */
      KIND_EXPR_FNHEAD,			/* holds param defaults and preconds */
      KIND_EXPR_PRECOND,		/* holds list of preconditions       */
      KIND_EXPR_DEFAULT, 		/* default parameter value           */
    KIND_NAMED,				/* filepos and name                  */
      KIND_CLASS_VIEW,			/* a class view                      */
      KIND_FEFN_FEATURE_SET,		/* a feature view set                */
      KIND_TYPE_VAR,			/* a type variable                   */
      /* KIND_FEFN: abstract superclass of front-end functions */
        KIND_FEFN_CREATION,		/* front-end creation function       */
        KIND_FEFN_CREDFT,		/* front-end creation feature dft    */
	KIND_FEFN_FEATURE,		/* front-end feature                 */
	KIND_FEFN_PRECOND,		/* front-end precondition            */
	KIND_FEFN_BUILTIN,		/* front-end builtin function        */
        KIND_FEFN_LETDEF,		/* front-end let definition          */
	KIND_FEFN_DOWNDEF,		/* front-end downcast variable       */
        KIND_FEFN_PARAM,		/* front-end parameter               */
        KIND_FEFN_INVARIANT,		/* front-end invariant               */
      /* KIND_BEFN: abstract superclass of back-end functions */
        KIND_BEFN_CREATION,		/* back-end creation function        */
        KIND_BEFN_CREDFT,		/* back-end creation feature dft val */
        KIND_BEFN_FEATURE,		/* back-end feature                  */
	KIND_BEFN_PRECOND,		/* back-end precondition             */
	KIND_BEFN_BUILTIN,		/* back-end builtin function         */
	KIND_BEFN_LETDEF,		/* back-end let definition           */
	KIND_BEFN_DOWNDEF,		/* back-end downcast variable        */
        KIND_BEFN_PARAM,		/* back-end parameter                */
        KIND_BEFN_INVARIANT,		/* back-end invariant                */
        KIND_BEFN_CLASS_INIT,		/* back-end class init function      */
        KIND_BEFN_ENUM_INIT,		/* back-end all_enumerated init fn   */
        KIND_BEFN_SYSTEM_INIT		/* back-end system init function     */
} KIND_TAG;


/*****************************************************************************/
/*                                                                           */
/*  PARAM_KIND - a type tag distinguishing kinds of parameters.              */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  PARAM_HIDDEN = 0,
  PARAM_OPTIONAL = 1,
  PARAM_SELF = 2,
  PARAM_COMPULSORY = 3
} PARAM_KIND;


/*****************************************************************************/
/*                                                                           */
/*  PARAMS_BODIES - whether an array of parameters has bodies or not.        */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  PARAM_BODIES_OPEN,		/* state of param bodies not yet determined  */
  PARAM_BODIES_ABSTRACT,	/* optional param bodies must be abstract    */
  PARAM_BODIES_CONCRETE,	/* optional param bodies must be concrete    */
  PARAM_BODIES_FIXED		/* optional param bodies must be present     */
} PARAM_BODIES_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  FEATURE_TYPE - a type tag distinguishing types of features.              */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  FEATURE_CREATION,				/* creation feature          */
  FEATURE_NONCREATION,				/* ordinary noncreation      */
  FEATURE_NONCREATION_CLONE,			/* clone noncreation feature */
  FEATURE_PREDEF,				/* ordinary predef feature   */
  FEATURE_PREDEF_ALL_PREDEFINED,		/* all_predefined feature    */
  FEATURE_PREDEF_ALL_ENUMERATED,		/* all_enumerated feature    */
  FEATURE_PREDEF_ALL_ENUM_TRIE,			/* all_enum_trie feature     */
  FEATURE_PREDEF_LEGAL_CODE,			/* legal_code feature        */
  FEATURE_PREDEF_WITH_CODE			/* with_code feature         */
} FEATURE_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  TYPE_POINTER_STATUS                                                      */
/*                                                                           */
/*  Indicates whether a type is represented by a pointer to a record or      */
/*  not.  Type variables could be either, depending on how they are          */
/*  instantiated, so their status is VAR rather than YES or NO.              */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  TYPE_POINTER_YES,
  TYPE_POINTER_NO,
  TYPE_POINTER_VAR
} TYPE_POINTER_STATUS;


/*****************************************************************************/
/*                                                                           */
/*  CLASS_VIEW_TYPE                                                          */
/*                                                                           */
/*  Indicates whether a class view is original to a system view, or          */
/*  imported, or imported and extended (i.e. with a class extension).        */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  CLASS_VIEW_ORIGINAL,
  CLASS_VIEW_IMPORTED,
  CLASS_VIEW_EXTENDED
} CLASS_VIEW_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  CASE_TYPE                                                                */
/*                                                                           */
/*  Every type has one of these.  It indicates whether the type is           */
/*  enumerated, integral, string, or something else, and is used in          */
/*  case expressions to decide whether the test expression has a suitable    */
/*  type, and if so what particular category.                                */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  CASE_TYPE_ENUM,
  CASE_TYPE_INTEGRAL,
  CASE_TYPE_STRING,
  CASE_TYPE_OTHER
} CASE_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  MAJOR TYPES                                                              */
/*                                                                           */
/*  Here are typedefs for the major types used by this compiler, and for     */
/*  arrays and symbol tables of them as required.  These are all forward     */
/*  declared to make mutually recursive records easy to do later.            */
/*                                                                           */
/*  The types are divided into five sets:                                    */
/*                                                                           */
/*    Library modules                                                        */
/*    Expressions                                                            */
/*    Front-end functions                                                    */
/*    Back-end functions                                                     */
/*    Other types                                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  LIBRARY MODULES                                                          */
/*                                                                           */
/*  These are usable separately, hence they have their own header files,     */
/*  which are included here.                                                 */
/*                                                                           */
/*****************************************************************************/

#include "memory.h"
#include "boolean.h"
#include "utypes.h"
#include "uchar.h"
#include "ustring.h"
#include "array.h"
#include "iset.h"
#include "usymtab.h"
#include "asymtab.h"
#include "codegen.h"


/*****************************************************************************/
/*                                                                           */
/*  EXPRESSIONS                                                              */
/*                                                                           */
/*  These all inherit from type EXPR.  EXPR itself is the only expression    */
/*  type made visible by including this file.                                */
/*                                                                           */
/*****************************************************************************/

typedef struct expr_rec *			EXPR;
typedef ARRAY(EXPR)				ARRAY_EXPR;


/*****************************************************************************/
/*                                                                           */
/*  FRONT-END FUNCTIONS                                                      */
/*                                                                           */
/*  These all inherit from type FEFN, except FEFN_FEATURE_SET.               */
/*                                                                           */
/*****************************************************************************/

/* front-end functions generally (fefn.c) */
#define NO_CODE_NUM -1
typedef struct fefn_rec *			FEFN;
typedef ARRAY(FEFN)				ARRAY_FEFN;

/* front-end creation functions (fefn_creation.c) */
typedef struct fefn_creation_rec *		FEFN_CREATION;

/* front-end creation feature default values (fefn_credft.c) */
typedef struct fefn_credft_rec *		FEFN_CREDFT;

/* front-end feature views (fefn_feature.c) */
typedef struct fefn_feature_rec *		FEFN_FEATURE;
typedef ARRAY(FEFN_FEATURE)			ARRAY_FEFN_FEATURE;
typedef SYMTAB(ARRAY_FEFN_FEATURE)		SYMTAB_FEFN_FEATURE;

/* front-end feature view sets (fefn_feature.c) */
typedef struct fefn_feature_set_rec *		FEFN_FEATURE_SET;
typedef ARRAY(FEFN_FEATURE_SET)			ARRAY_FEFN_FEATURE_SET;
typedef SYMTAB(ARRAY_FEFN_FEATURE_SET)		SYMTAB_FEFN_FEATURE_SET;

/* front-end feature preconditions */
typedef struct fefn_precond_rec *		FEFN_PRECOND;
typedef ARRAY(FEFN_PRECOND)			ARRAY_FEFN_PRECOND;

/* front-end builtin functions (fefn_builtin.c) */
typedef struct fefn_builtin_rec *		FEFN_BUILTIN;

/* front-end letdefs (fefn_letdef.c) */
typedef struct fefn_letdef_rec *		FEFN_LETDEF;
typedef ARRAY(FEFN_LETDEF)			ARRAY_FEFN_LETDEF;

/* front-end downdefs (fefn_downdef.c) */
typedef struct fefn_downdef_rec *		FEFN_DOWNDEF;
typedef ARRAY(FEFN_DOWNDEF)			ARRAY_FEFN_DOWNDEF;

/* front-end parameters (fefn_param.c) */
typedef struct fefn_param_rec *			FEFN_PARAM;
typedef ARRAY(FEFN_PARAM)			ARRAY_FEFN_PARAM;

/* front-end invariants (fefn_invariant.c) */
typedef struct fefn_invariant_rec *		FEFN_INVARIANT;


/*****************************************************************************/
/*                                                                           */
/*  BACK-END FUNCTIONS                                                       */
/*                                                                           */
/*  These all inherit from type BEFN, except BEFN_FEATURE_INSTANCE.          */
/*                                                                           */
/*****************************************************************************/

/* back-end functions generally (befn.c) */
typedef struct befn_rec *			BEFN;
typedef ARRAY(BEFN)				ARRAY_BEFN;

/* back-end creation functions (befn_creation.c) */
typedef struct befn_creation_rec *		BEFN_CREATION;

/* back-end creation feature default values (befn_credft.c) */
typedef struct befn_credft_rec *		BEFN_CREDFT;

/* back-end features (befn_feature.c) */
typedef struct befn_feature_rec *		BEFN_FEATURE;
typedef ARRAY(BEFN_FEATURE)			ARRAY_BEFN_FEATURE;
typedef SYMTAB(ARRAY_BEFN_FEATURE)		SYMTAB_BEFN_FEATURE;

typedef struct befn_feature_instance_rec *	BEFN_FEATURE_INSTANCE;
typedef ARRAY(BEFN_FEATURE_INSTANCE)		ARRAY_BEFN_FEATURE_INSTANCE;

/* back-end feature preconditions */
typedef struct befn_precond_rec *		BEFN_PRECOND;
typedef ARRAY(BEFN_PRECOND)			ARRAY_BEFN_PRECOND;

/* back-end builtin functions (befn_builtin.c) */
typedef struct befn_builtin_rec *		BEFN_BUILTIN;

/* back-end letdefs (befn_letdef.c) */
typedef struct befn_letdef_rec *		BEFN_LETDEF;

/* back-end downdefs (befn_downdef.c) */
typedef struct befn_downdef_rec *		BEFN_DOWNDEF;

/* back-end parameters (befn_param.c) */
typedef struct befn_param_rec *			BEFN_PARAM;
typedef ARRAY(BEFN_PARAM)			ARRAY_BEFN_PARAM;

/* back-end invariants (befn_invariant.c) */
typedef struct befn_invariant_rec *		BEFN_INVARIANT;

/* class initialization function (befn_class_init.c) */
typedef struct befn_class_init_rec *		BEFN_CLASS_INIT;

/* class all_enumerated initialization function (befn_enum_init.c) */
typedef struct befn_enum_init_rec *		BEFN_ENUM_INIT;

/* system initialization function (befn_system_init.c) */
typedef struct befn_system_init_rec *		BEFN_SYSTEM_INIT;


/*****************************************************************************/
/*                                                                           */
/*  OTHER TYPES                                                              */
/*                                                                           */
/*****************************************************************************/

/* file positions (lexer.c) */
typedef struct file_pos_rec *			FILE_POS;
typedef ARRAY(FILE_POS)				ARRAY_FILE_POS;

/* tokens (lexer.c) */
typedef struct token_rec *			TOKEN;
typedef ARRAY(TOKEN)				ARRAY_TOKEN;
typedef SYMTAB(ARRAY_TOKEN)			SYMTAB_TOKEN;

/* names (name.c) */
typedef struct name_rec *			NAME;
typedef ARRAY(NAME) 				ARRAY_NAME;
typedef ARRAY(ARRAY_NAME) 			ARRAY_ARRAY_NAME;
typedef SYMTAB(ARRAY_NAME)			SYMTAB_NAME;
typedef ARRAY(SYMTAB_NAME)			ARRAY_SYMTAB_NAME;

/* named objects (name.c) */
typedef struct named_rec *			NAMED;
typedef ARRAY(NAMED)				ARRAY_NAMED;
typedef ARRAY(ARRAY_NAMED)			ARRAY_ARRAY_NAMED;
typedef SYMTAB(ARRAY_NAMED)			SYMTAB_NAMED;
typedef ARRAY(SYMTAB_NAMED)			ARRAY_SYMTAB_NAMED;

/* contexts (context.c) */
typedef struct context_rec *			CONTEXT;

/* types (type.c) */
typedef struct type_rec *	TYPE;
typedef ARRAY(TYPE)		ARRAY_TYPE;
typedef struct type_var_rec *	TYPE_VAR;
typedef ARRAY(TYPE_VAR)		TYPE_VARS;
typedef struct itype_rec *	ITYPE;
typedef ARRAY(ITYPE)		ITYPES;

/* coercions (coercion.c) */
typedef struct coercion_rec *			COERCION;
typedef ARRAY(COERCION)				ARRAY_COERCION;

/* classes (class.c) */
typedef struct class_rec *			CLASS;
typedef ARRAY(CLASS)				ARRAY_CLASS;

/* class views (class_view.c) */
typedef struct class_view_rec *			CLASS_VIEW;
typedef ARRAY(CLASS_VIEW)			ARRAY_CLASS_VIEW;
typedef SYMTAB(ARRAY_CLASS_VIEW)		SYMTAB_CLASS_VIEW;
typedef struct class_view_update_rec *		CLASS_VIEW_UPDATE;

/* systems (system.c) */
typedef struct system_rec *			SYSTEM;

/* system views (system_view.c) */
typedef struct system_view_rec *		SYSTEM_VIEW;
typedef ARRAY(SYSTEM_VIEW)			ARRAY_SYSTEM_VIEW;
typedef SYMTAB(ARRAY_SYSTEM_VIEW)		SYMTAB_SYSTEM_VIEW;

/* auto (auto.c) */
/* no typedefs associated with this module */


/*****************************************************************************/
/*                                                                           */
/*  TOKEN_TYPE - the type of a token                                         */
/*                                                                           */
/*****************************************************************************/

typedef enum {

  /* predefined punctuation symbols */
  TK_COLON, TK_COLON_EQUALS, TK_COMMA, TK_EXCLAM, TK_EXCLAM_EXCLAM,
  TK_DOT, TK_DOT_DOT, TK_LEFT_BRACE, TK_RIGHT_BRACE, TK_LEFT_BRACKET,
  TK_RIGHT_BRACKET, TK_LEFT_BAR_BRACKET, TK_RIGHT_BAR_BRACKET,
  TK_LEFT_PAREN, TK_RIGHT_PAREN,

  /* predefined words: keywords, self, false, true */
  TK_AS, TK_BUILTIN, TK_CASE, TK_CLASS, TK_COERCE, TK_CREATION, TK_ELSE,
  TK_ELSIF, TK_END, TK_ENUM, TK_EXTEND, TK_EXTENSION, TK_FALSE,
  TK_FILTER, TK_FUN, TK_GENESIS, TK_IF, TK_IN, TK_INFIX, TK_INFIXR, TK_INHERIT,
  TK_INTRODUCE, TK_INVARIANT, TK_IS, TK_LET, TK_LOCAL, TK_MEET,
  TK_MODULE, TK_NONCREATION, TK_NORENAME, TK_PREDEFINED, TK_PREFIX,
  TK_PRIVATE, TK_POSTFIX, TK_RENAME, TK_REQUIRE, TK_SELF, TK_SYSTEM,
  TK_THEN, TK_TRUE, TK_WHEN, TK_USE, TK_YIELD,

  /* miscellaneous */
  TK_IDENTIFIER,		/* an identifier, not a keyword              */
  TK_PUNCTSEQ,			/* a punctuation sequence, not predefinded   */
  TK_LIT_INTEGER,		/* an integer literal                        */
  TK_LIT_REAL,			/* a real literal                            */
  TK_LIT_CHARACTER,		/* a character literal                       */
  TK_LIT_STRING, 		/* a string literal                          */
  TK_END_FILE			/* end of input                              */

} TOKEN_TYPE;


/*****************************************************************************/
/*                                                                           */
/*  MACROS FOR PARSING                                                       */
/*                                                                           */
/*  curr_token - the current unprocessed token                               */
/*  next_token - move to next token                                          */
/*  check(ttype, exp) - check ttype, else print error and return FALSE.      */
/*  skip(ttype, exp) - skip ttype, else print error and return FALSE.        */
/*                                                                           */
/*  prohibit1(cond, fmt, p1)                  Complain and return FALSE      */
/*  prohibit2(cond, fmt, p1, p2)              if the condition holds         */
/*  prohibit3(cond, fmt, p1, p2, p3)                                         */
/*  prohibit4(cond, fmt, p1, p2, p3, p4)                                     */
/*                                                                           */
/*****************************************************************************/
#define is_keyword(t) ((t >= TK_AS) && (t <= TK_YIELD))

#define curr_token (*t)

#define next_token  curr_token = LexNext(curr_token)

#define check(ttype, exp)						\
if( LexType(curr_token) != ttype )					\
{									\
  if( is_keyword(LexType(curr_token)) && ttype == TK_IDENTIFIER )	\
    fprintf(stderr, "%s: expected %s here (\"%s\" is a keyword)\n",	\
      LexPos(curr_token), exp, LexShow(curr_token));			\
  else									\
    fprintf(stderr, "%s: expected %s here\n", LexPos(curr_token), exp);	\
  return FALSE;								\
}									\

#define skip(ttype, exp)  check(ttype, exp) else { next_token; }

#define prohibit1(cond, fmt, p1)					\
if( cond ) return fprintf(stderr, (fmt), (p1)) && FALSE

#define prohibit2(cond, fmt, p1, p2)					\
if( cond ) return fprintf(stderr, (fmt), (p1), (p2)) && FALSE

#define prohibit3(cond, fmt, p1, p2, p3)				\
if( cond ) return fprintf(stderr, (fmt), (p1), (p2), (p3)) && FALSE

#define prohibit4(cond, fmt, p1, p2, p3, p4)				\
if( cond ) return fprintf(stderr, (fmt), (p1), (p2), (p3), (p4)) && FALSE


/*****************************************************************************/
/*                                                                           */
/*  MACROS FOR DEBUG PRINTING                                                */
/*                                                                           */
/*  Debug printing is a simple form of prettyprinting which allows entities  */
/*  to be either printed on one line, or else on multiple lines with a       */
/*  given indent at the start of each line.                                  */
/*                                                                           */
/*  Whichever form is used, an entity never prints any initial space and     */
/*  never prints any following space.  It is the caller's responsibility     */
/*  to insert these spaces.  For example, if multi-line printing is          */
/*  in effect, the entity will print an indent at the start of the           */
/*  second and subsequent lines, not the first; and it will never print a    */
/*  newline at the end of itself.                                            */
/*                                                                           */
/*  It is not possible to move from single-line to multi-line printing,      */
/*  but it is possible to move from multi-line to single-line.  When         */
/*  doing that it is the caller's responsibility (as above) to ensure that   */
/*  the right spacing surrounds the called entity.                           */
/*                                                                           */
/*  This is controlled by a single parameter called print_style, an int.     */
/*  If print_style >= 0, multi-line printing is in effect and print_style    */
/*  is the amount of indent to start each line with.  If print_style < 0     */
/*  (i.e. has value SINGLE_LINE) then single-line printing is in effect.     */
/*                                                                           */
/*  The macros and their meanings are:                                       */
/*                                                                           */
/*  begin_indent;   Increase the indent level if multi-line.                 */
/*                                                                           */
/*  end_indent;     Decrease the indent level if multi-line.                 */
/*                                                                           */
/*  leave_line;     In single-line printing, do nothing.  In multi-line      */
/*                  printing, insert a blank line.                           */
/*                                                                           */
/*  next_line;      In single-line printing, just print a space.  In         */
/*                  multi-line printing, move to next line and print         */
/*                  an indent in preparation for new material on that line.  */
/*                                                                           */
/*****************************************************************************/

#define	SINGLE_LINE	-1
#define begin_indent	if( print_style >= 0 ) print_style += 2
#define end_indent	if( print_style >= 2 ) print_style -= 2
#define leave_line	if( print_style >= 0 ) fprintf(fp, "\n")
#define next_line	if( print_style >= 0 )				\
			  fprintf(fp, "\n%*s", print_style, "");	\
			else						\
			  fprintf(fp, " ")

#define db_return(db, str, val)						\
{									\
  BOOLEAN xxx_zzz = val;						\
  if( db )								\
    fprintf(stderr, "] %s returning %s\n", str, bool(xxx_zzz));		\
  return xxx_zzz;							\
}

#define db_return_val(db, str, show, val)				\
{									\
  if( db )								\
    fprintf(stderr, "] %s returning %s\n", str, show);			\
  return val;								\
}


/*****************************************************************************/
/*                                                                           */
/*  EXTERNAL FUNCTIONS - EXPRESSIONS                                         */
/*                                                                           */
/*  Headers for the various subexpression types are in expr.h.               */
/*                                                                           */
/*****************************************************************************/

/* simple queries */
/* extern BOOLEAN ExprIsSimple(EXPR e); */

/* parse */
extern FILE_POS ExprFilePos(EXPR e);
extern void ExprSetParamName(EXPR e, USTRING val);
extern USTRING ExprParamName(EXPR e);
extern USTRING UnderscorePrefix(TOKEN token);
extern BOOLEAN ExprDecoratedFactorParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
extern BOOLEAN ExprCallInfixSeqParse(TOKEN *t, SYSTEM_VIEW sv, EXPR *res);
#define ExprParse(t, op_table, res) ExprCallInfixSeqParse(t, op_table, res)

/* uninstantiated copy */
extern EXPR ExprCopyUninstantiated(EXPR e, ARRAY_FEFN_PARAM orig_params,
  ARRAY_FEFN_PARAM copy_params);

/* manifest */
extern BOOLEAN ExprManifest(EXPR *e, CONTEXT cxt, TYPE self_type,
  BEFN encl_cfunct);
extern TYPE ExprType(EXPR e);
extern BOOLEAN ExprLargeScale(EXPR e);

/* constant expressions */
extern BOOLEAN ExprIsConst(EXPR e, EXPR *root);
extern int ExprConstCmp(EXPR e1, EXPR e2);
extern int ExprConstIntValue(EXPR e);

/* code generation */
extern void ExprCodeGen(EXPR e, CODEGEN_OBJ res_be_var,
  CODEGEN_TYPE res_be_type, CODEGEN be);
extern BOOLEAN ExprInitOrder(EXPR e, int visit_num, BOOLEAN *report,
  BEFN_SYSTEM_INIT fun);

/* debug */
extern void ExprArrayDebug(ARRAY_EXPR ae, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp);
extern void ExprDebug(EXPR e, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style);


/*****************************************************************************/
/*                                                                           */
/*  External functions - other expression classes                            */
/*                                                                           */
/*****************************************************************************/

/* expr_lit.c */
extern EXPR ExprLitFromChar(FILE_POS file_pos, UCHAR ch);
extern EXPR ExprLitFromInt(FILE_POS file_pos, int val);
extern EXPR ExprLitFromBool(FILE_POS file_pos, BOOLEAN val);
extern EXPR ExprLitFromString(FILE_POS file_pos, USTRING str, SYSTEM_VIEW sv);

/* expr_array.c */
extern EXPR ExprArrayEmpty(FILE_POS file_pos);
extern EXPR ExprArrayFromString(FILE_POS file_pos, USTRING str);

/* expr_call.c */
extern EXPR ExprMakeRawCall(FILE_POS file_pos, USTRING value,
  ARRAY_TYPE generics, ARRAY_EXPR parameters);
extern EXPR ExprMakeDottedCall(USTRING value, EXPR param);
extern EXPR ExprMakeBuiltinCall(FILE_POS file_pos, USTRING value,
  ARRAY_EXPR parameters);
extern BOOLEAN ExprAbbreviatedCallParse(TOKEN *t, USTRING str, SYSTEM_VIEW sv,
  EXPR *res);
extern void ExprCallInsertCoercion(EXPR *e, FEFN_FEATURE fv, CONTEXT cxt,
  TYPE self_type);
extern EXPR ExprMakeInstantiatedFeatureCall(FEFN_FEATURE fv);
extern EXPR ExprEncloseInInstantiatedAssert(EXPR e);

/* expr_fnhead.c */
extern EXPR ExprFnHeadNew(FILE_POS file_pos);
extern void ExprFnHeadAddFEFnParamDefaultVal(EXPR e, FEFN_PARAM param, EXPR expr);
extern void ExprFnHeadAddSubExpression(EXPR e, EXPR subexpression);
extern void ExprFnHeadInsertPrecondition(EXPR e, EXPR precondition);
extern void ExprFnHeadInsertCoercion(EXPR e, COERCION c, CONTEXT cxt,
  TYPE self_type);
extern BOOLEAN ExprFnHeadManifestParamDefaultVal(EXPR e, FEFN_PARAM param,
  CONTEXT cxt, TYPE self_type, BEFN encl_cfunct);
extern BOOLEAN ExprFnHeadWrapsBuiltin(EXPR e, ARRAY_BEFN_PARAM parameters,
  CODEGEN_OBJ *be_obj);
extern void ExprFnHeadCodeGenDftParamValues(EXPR e, CODEGEN be);

/* expr_precond.c */
extern EXPR ExprPrecondNew(FILE_POS file_pos);
extern void ExprPrecondAddPrecondition(EXPR e, EXPR precondition);
extern void ExprPrecondAddSubExpression(EXPR e, EXPR subexpression);

/* expr_default.c */
extern void ExprDefaultHeaderDefinitions(FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*  EXTERNAL FUNCTIONS - FRONT-END FUNCTIONS                                 */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  External functions - fefn.c                                              */
/*                                                                           */
/*****************************************************************************/

extern NAME FEFnName(FEFN fun);
extern FILE_POS FEFnFilePos(FEFN fun);
extern TYPE FEFnResultType(FEFN fun);
extern BEFN FEFnBEFn(FEFN fun);
extern void FEFnAddParameter(FEFN fun, FEFN_PARAM param);
extern BOOLEAN FEFnParse(TOKEN *t, BOOLEAN is_letdef, BOOLEAN is_anon,
  FEATURE_TYPE ftype, BOOLEAN is_coerce, BOOLEAN encl_private,
  BOOLEAN builtin, BOOLEAN extension, USTRING predef_name,
  BOOLEAN is_enum, SYSTEM_VIEW sv, FILE_POS *file_pos,
  ARRAY_NAME *names, TYPE_VARS *type_vars, ARRAY_FEFN_PARAM *params,
  TYPE *result_type, EXPR *expr_fnhead, PARAM_BODIES_TYPE *param_bodies_type,
  EXPR *expr_precond, int *code1, int *code2);
/* ***
extern BOOLEAN FEFnManifest(FEFN fun, CONTEXT cxt, BOOLEAN is_public,
  BOOLEAN shadow_lim, BOOLEAN do_interface, EXPR body, TYPE self_type,
  BEFN encl_befn);
*** */
extern BOOLEAN FEFnAddHiddenParameter(FEFN fefn, NAMED *named);
extern void FEFnSignature(FEFN fun, TYPE_VARS *sig_vars,
  ARRAY_FEFN_PARAM *sig_params, TYPE *sig_result_type);
extern TYPE FEFnCallResultType(ARRAY_FEFN_PARAM sig_params,
  TYPE sig_result_type, int i, FILE_POS file_pos);
extern UTF8 FEFnSignatureShow(FEFN fun, CONTEXT cxt);
extern ARRAY_FEFN_PARAM FEFnParams(FEFN fun);
extern int FEFnParamsCount(FEFN fun);
extern int FEFnCompulsoryParamsCount(FEFN fun);
extern FEFN_PARAM FEFnSelfParam(FEFN fun);


/*****************************************************************************/
/*                                                                           */
/*  External functions - fefn_creation.c                                     */
/*                                                                           */
/*****************************************************************************/

extern FEFN_CREATION FEFnCreationNew(FILE_POS file_pos, NAME name,
  TYPE_VARS type_vars, ARRAY_FEFN_PARAM parameters, TYPE result_type,
  CLASS_VIEW orig_cv, BEFN_CREATION befn_creation);
extern void FEFnCreationSetResultTypeAndBEFnCreation(FEFN_CREATION cfv,
  TYPE result_type, BEFN_CREATION befn_creation);
extern BOOLEAN FEFnCreationManifest(FEFN_CREATION cfv, CONTEXT cxt,
  TYPE self_type);
/* ***
extern BEFN_CREATION FEFnCreationBEFnCreation(FEFN_CREATION cfv);
*** */


/*****************************************************************************/
/*                                                                           */
/*  External functions - fefn_credft.c                                       */
/*                                                                           */
/*****************************************************************************/

extern FEFN_CREDFT FEFnCreDftNew(FEFN_FEATURE orig_cfeature, EXPR body);
extern FEFN_FEATURE FEFnCreDftOrigCreationFeature(FEFN_CREDFT fefn_credft);
extern BOOLEAN FEFnCreDftManifest(FEFN_CREDFT fefn_credft, CONTEXT cxt,
  TYPE self_type);
extern void DEFnCreDftDebug(FEFN_CREDFT fefn_credft, CONTEXT cxt, FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*  External functions - fefn_feature.c                                      */
/*                                                                           */
/*****************************************************************************/

/* feature views basics */
extern FILE_POS FEFnFeatureFilePos(FEFN_FEATURE fv);
extern NAME FEFnFeatureName(FEFN_FEATURE fv);
extern CLASS_VIEW FEFnFeatureClassView(FEFN_FEATURE fv);
extern BOOLEAN FEFnFeatureIsPrivate(FEFN_FEATURE fv);
extern BOOLEAN FEFnFeatureIsCoerce(FEFN_FEATURE fv);
extern BOOLEAN FEFnFeatureIsImported(FEFN_FEATURE fv);
extern BOOLEAN FEFnFeatureIsNoRename(FEFN_FEATURE fv);
extern FEATURE_TYPE FEFnFeatureFType(FEFN_FEATURE fv);
extern int FEFnFeatureCode(FEFN_FEATURE fv);
extern BOOLEAN FEFnFeatureIsAbstract(FEFN_FEATURE fv);
extern COERCION FEFnFeatureCoercion(FEFN_FEATURE fv);
extern BEFN_FEATURE FEFnFeatureBEFnFeature(FEFN_FEATURE fv);
extern BOOLEAN FEFnFeatureSetParse(TOKEN *t, CLASS_VIEW cv,
  FEATURE_TYPE ftype, BOOLEAN builtin, BOOLEAN extension, SYSTEM_VIEW sv);
extern FEFN_FEATURE FEFnFeatureAllPredefinedMake(FILE_POS file_pos,
  CLASS_VIEW cv, SYSTEM_VIEW sv);
extern FEFN_FEATURE FEFnFeatureAllEnumeratedMake(FILE_POS file_pos,
  CLASS_VIEW cv, SYSTEM_VIEW sv);
extern void FEFnFeatureDebug(FEFN_FEATURE fv, CONTEXT cxt, FILE *fp);

/* the clone feature view */
extern FEFN_FEATURE FEFnFeatureCloneNew(CLASS_VIEW cv);

/* other auto-generated feature views */
extern void FEFnFeatureRegisterToInt(CLASS_VIEW cv, FILE_POS file_pos);
extern void FEFnFeatureRegisterWithCode(CLASS_VIEW cv, FILE_POS file_pos);
extern void FEFnFeatureRegisterLegalCode(CLASS_VIEW cv, FILE_POS file_pos);

/* feature view sets basics */
extern NAME FEFnFeatureSetName(FEFN_FEATURE_SET fvs);
extern FEFN_FEATURE FEFnFeatureSetSoleUncoercedMember(FEFN_FEATURE_SET fvs);
extern int FEFnFeatureSetSize(FEFN_FEATURE_SET fvs);
extern FEFN_FEATURE FEFnFeatureSetFEFnFeature(FEFN_FEATURE_SET fvs, int i);
extern void FEFnFeatureSetDebug(FEFN_FEATURE_SET fvs, CONTEXT cxt, FILE *fp,
  int print_style);

/* creation, copying, and checking */
extern FEFN_FEATURE_SET FEFnFeatureSetNew(FILE_POS file_pos, NAME name);
extern FEFN_FEATURE_SET FEFnFeatureSetNewFromFEFnFeature(FEFN_FEATURE fv);
extern FEFN_FEATURE FEFnFeatureCopyImported(FEFN_FEATURE fv, FILE_POS new_pos,
  NAME new_name, CLASS_VIEW new_cv);
extern BOOLEAN FEFnFeatureSetCheckInherited(FEFN_FEATURE_SET parent_fvs,
  FEFN_FEATURE_SET fvs, CLASS_VIEW parent_cv, CLASS_VIEW cv, CONTEXT cxt);
extern FEFN_FEATURE_SET FEFnFeatureSetCopyInherited(FEFN_FEATURE_SET fvs,
  CLASS_VIEW cv);
extern BOOLEAN FEFnFeatureSetAddCoercedCopy(FEFN_FEATURE_SET fvs,
  FEFN_FEATURE fv, COERCION coercion, CLASS_VIEW cv);
extern BOOLEAN FEFnFeatureSetMerge(FEFN_FEATURE_SET *target,
  FEFN_FEATURE_SET source, CONTEXT cxt, FILE_POS file_pos);

/* miscellaneous */
extern BOOLEAN FEFnFeatureManifestInterface(FEFN_FEATURE fv, CONTEXT cxt);
extern BOOLEAN FEFnFeatureManifestBody(FEFN_FEATURE fv, CONTEXT cxt,
  TYPE self_type);
extern void FEFnFeatureAddToClass(FEFN_FEATURE fv, CLASS c);
extern void FEFnFeatureAddToCreation(FEFN_FEATURE fv, FEFN_CREATION cfv);
extern FEFN_CREDFT FEFnCreationDefaultValue(FEFN_FEATURE fv);
extern void FEFnFeatureAddToClone(FEFN_FEATURE fv, FEFN_FEATURE clone_fv);


/*****************************************************************************/
/*                                                                           */
/*  External functions - fefn_precond.c                                      */
/*                                                                           */
/*****************************************************************************/

extern FEFN_PRECOND FEFnPrecondNew(FEFN_FEATURE fefn_feature, EXPR body);
extern FEFN_PRECOND FEFnPrecondCopyUninstantiated(FEFN_PRECOND fefn_precond,
  FEFN_FEATURE fv, ARRAY_FEFN_PARAM orig_params,
  ARRAY_FEFN_PARAM copy_params);
extern FEFN_FEATURE FEFnPrecondOrigFeature(FEFN_PRECOND fefn_precond);
extern BOOLEAN FEFnPrecondManifest(FEFN_PRECOND fefn_precond, CONTEXT cxt,
  TYPE self_type);
extern void FEFnPrecondDebug(FEFN_PRECOND fefn_precond, CONTEXT cxt, FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*  External functions - fefn_builtin.c                                      */
/*                                                                           */
/*****************************************************************************/

extern FEFN_BUILTIN BuiltinAssertFn;
extern void FEFnBuiltinInitialize(CODEGEN be);
extern void FEFnBuiltinInitFeatures(CODEGEN be);
/* ***
extern void BuiltinInsertEnumToContent(FILE_POS file_pos, USTRING str,
  CLASS_VIEW code_cv, CLASS_VIEW content_cv);
*** */
extern void FEFnBuiltinInsertEnumToInt(FILE_POS file_pos, USTRING str,
  CLASS_VIEW code_cv);
extern void FEFnBuiltinInsertIntToEnum(FILE_POS file_pos, USTRING str,
  CLASS_VIEW cv);
extern BOOLEAN FEFnBuiltinFind(USTRING str, FILE_POS file_pos, int param_count,
  FEFN_BUILTIN *bif);
extern void FEFnBuiltinFinalizeAll(CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - fefn_letdef.c                                       */
/*                                                                           */
/*****************************************************************************/

extern FEFN_LETDEF FEFnLetDefNew(FILE_POS file_pos, NAME name,
  TYPE_VARS type_vars, ARRAY_FEFN_PARAM parameters, TYPE result_type,
  EXPR body);
extern ARRAY_FEFN_LETDEF FEFnLetDefsCopyUninstantiated(ARRAY_FEFN_LETDEF letdefs,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);
/* ***
extern BOOLEAN FEFnLetDefAddHiddenParameter(FEFN_LETDEF letdef, NAMED *named);
*** */
extern BOOLEAN FEFnLetDefManifest(FEFN_LETDEF letdef, CONTEXT cxt,
  TYPE self_type, BEFN encl_cfunct, BOOLEAN *let_large_scale);
extern void FEFnLetDefsVarEnterScope(ARRAY_FEFN_LETDEF letdefs, CODEGEN be);
extern void FEFnLetDefsVarLeaveScope(ARRAY_FEFN_LETDEF letdefs, CODEGEN be);
extern void FEFnLetDefDebug(FEFN_LETDEF letdef, CONTEXT cxt,
  BOOLEAN show_types, FILE *fp, int print_style);


/*****************************************************************************/
/*                                                                           */
/*  External functions - fefn_downdef.c                                      */
/*                                                                           */
/*****************************************************************************/

extern FEFN_DOWNDEF FEFnDownDefMake(FILE_POS file_pos, NAME name,
  TYPE result_type);
extern void FEFnDownDefMakeBEFn(FEFN_DOWNDEF fefn_downdef, CODEGEN_OBJ be_var);
extern FEFN_DOWNDEF FEFnDownDefCopyUninstantiated(FEFN_DOWNDEF fefn_downdef,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params);


/*****************************************************************************/
/*                                                                           */
/*  External functions - fefn_param.c                                        */
/*                                                                           */
/*****************************************************************************/

extern FEFN_PARAM FEFnParamNew(FILE_POS file_pos, NAME name, TYPE result_type,
  BEFN_PARAM befn_param, PARAM_KIND param_kind, BOOLEAN is_private,
  FEFN hidden_value);
extern void FEFnParamSetBEFnParam(FEFN_PARAM param, BEFN_PARAM befn_param);
extern BOOLEAN FEFnParamIsPrivate(FEFN_PARAM param);
extern FEFN FEFnParamHiddenValue(FEFN_PARAM param);
extern FILE_POS FEFnParamFilePos(FEFN_PARAM param);
extern NAME FEFnParamName(FEFN_PARAM param);
extern TYPE FEFnParamType(FEFN_PARAM param);
extern PARAM_KIND FEFnParamKind(FEFN_PARAM param);
extern BEFN_PARAM FEFnParamBEFnParam(FEFN_PARAM param);
extern BOOLEAN FEFnParamIsOptional(FEFN_PARAM param);
extern FEFN_FEATURE FEFnParamCreationFEFnFeature(FEFN_PARAM param);
extern FEFN_PARAM FEFnParamCopy(FEFN_PARAM param, BOOLEAN is_private);
extern ARRAY_FEFN_PARAM FEFnParamListCopy(ARRAY_FEFN_PARAM parameters);
extern ARRAY_FEFN_PARAM FEFnParamListCopyImported(ARRAY_FEFN_PARAM parameters,
  BOOLEAN make_private);
extern ARRAY_FEFN_PARAM FEFnParamListCopyUninstantiated(
  ARRAY_FEFN_PARAM parameters);
extern BOOLEAN FEFnParamListParse(TOKEN *t, FEATURE_TYPE ftype,
  BOOLEAN is_coerce, BOOLEAN encl_private, SYSTEM_VIEW sv,
  ARRAY_FEFN_PARAM *parameters, EXPR *expr_fnhead,
  PARAM_BODIES_TYPE *param_bodies_type);
extern BOOLEAN FEFnParamIsRedefinition(FEFN_PARAM parent, FEFN_PARAM child);
extern BOOLEAN FEFnParamManifest(FEFN_PARAM param, CONTEXT cxt,
  BOOLEAN is_public);
extern void FEFnParamListRegisterBaseParameters(ARRAY_FEFN_PARAM parameters,
  BEFN_FEATURE base_feature);
extern UTF8 FEFnParamShow(FEFN_PARAM param, CONTEXT cxt);
extern UTF8 FEFnParamListShow(ARRAY_FEFN_PARAM parameters, CONTEXT cxt);


/*****************************************************************************/
/*                                                                           */
/*  External functions - fefn_invariant.c                                    */
/*                                                                           */
/*****************************************************************************/

extern FEFN_INVARIANT FEFnInvtNew(FILE_POS file_pos, CLASS_VIEW cv);
extern void FEFnInvtAddExpr(FEFN_INVARIANT iv, EXPR e);
extern BOOLEAN FEFnInvtManifest(FEFN_INVARIANT iv, CONTEXT cxt);


/*****************************************************************************/
/*                                                                           */
/*  EXTERNAL FUNCTIONS - BACK-END FUNCTIONS                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  External functions - befn.c                                              */
/*                                                                           */
/*****************************************************************************/

extern KIND_TAG BEFnKindTag(BEFN fun);
extern void BEFnAddParameter(BEFN fun, BEFN_PARAM param);
extern int BEFnParamsCount(BEFN fun);
extern ARRAY_BEFN_PARAM BEFnParameters(BEFN fun);
extern BEFN_PARAM BEFnFindParamFromCreationFeature(BEFN befn, BEFN_FEATURE f);
extern void BEFnAddInnerBEFn(BEFN fun, BEFN inner_fun);
extern void BEFnSetUtilized(BEFN fun);
extern BOOLEAN BEFnUtilized(BEFN fun);
extern void BEFnFinalize(BEFN fun, CODEGEN be);
extern CODEGEN_OBJ BEFnBEObj(BEFN fun);
extern CODEGEN_TYPE BEFnBEType(BEFN fun);
extern BOOLEAN BEFnInitOrder(BEFN befn, int visit_num, BOOLEAN *report,
  BEFN_SYSTEM_INIT fun);
extern void BEFnCodeGenParamHash(BEFN fun, CODEGEN_OBJ be_var_sig, CODEGEN be);
extern void BEFnCodeGen(BEFN fun, CODEGEN be);
extern void BEFnCallCodeGen(BEFN fun, ARRAY_EXPR sorted_actuals,
  CODEGEN_TYPE res_be_type, CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_creation.c                                     */
/*                                                                           */
/*****************************************************************************/

extern BEFN_CREATION BEFnCreationNew(CLASS class);
/* extern void BEFnCreationAddBody(BEFN_CREATION cfn, EXPR body); */
extern BOOLEAN BEFnCreationInitOrder(BEFN_CREATION cfn, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);
extern void BEFnCreationFinalize(BEFN_CREATION cfn, CODEGEN be);
extern CODEGEN_OBJ BEFnCreationBEObj(BEFN_CREATION cfn);
extern void BEFnCreationCodeGenBody(BEFN_CREATION cfn, CODEGEN_OBJ res_be_var,
  CODEGEN be);
extern void BEFnCreDftCodeGenCall(BEFN_CREDFT befn_credft,
  ARRAY_BEFN_PARAM params, CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_credft.c                                       */
/*                                                                           */
/*****************************************************************************/

extern BEFN_CREDFT BEFnCreDftNew(FEFN_CREDFT fefn_credft, EXPR body);
extern BOOLEAN BEFnCreDftInitOrder(BEFN_CREDFT befn_credft, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);
extern void BEFnCreDftFinalize(BEFN_CREDFT befn_credft, CODEGEN be);
extern void BEFnCreDftCodeGenBody(BEFN_CREDFT befn_credft,
  CODEGEN_OBJ res_be_var, CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_feature.c                                      */
/*                                                                           */
/*****************************************************************************/

extern ASTRING FeatureTypeShow(FEATURE_TYPE ftype);
extern FEATURE_TYPE BEFnFeatureFType(BEFN_FEATURE f);
extern void BEFnResetBaseFEFn(BEFN_FEATURE f, FEFN_FEATURE base_fv);
extern BEFN_FEATURE BEFnFeatureMake(CLASS base_class, FEFN_FEATURE base_fv,
  FEATURE_TYPE ftype, int code1, int code2);
extern BEFN_PARAM BEFnFeatureSelfParam(BEFN_FEATURE f);
extern FILE_POS BEFnFeatureOrigFilePos(BEFN_FEATURE f);
extern NAME BEFnFeatureOrigName(BEFN_FEATURE f);
extern BEFN_FEATURE BEFnFeatureNewAnon(FILE_POS file_pos, CLASS c, EXPR body);
extern BEFN_FEATURE BEFnFeatureNewEnumTrie(CLASS c);
extern void BEFnFeatureSetPredefIndex(BEFN_FEATURE f, int index);
extern BEFN_FEATURE BEFnFeatureRoot(BEFN_FEATURE f);
extern BEFN_FEATURE_INSTANCE BEFnFeatureRegisterNewInstance(
  BEFN_FEATURE feature, CLASS class, FILE_POS file_pos, EXPR body);
extern FILE_POS BEFnFeatureInstanceFilePos(BEFN_FEATURE_INSTANCE fi);
extern void BEFnFeatureRegisterExistingInstance(BEFN_FEATURE_INSTANCE fi,
  CLASS class);
extern void BEFnFeatureRegisterPrecondition(BEFN_FEATURE_INSTANCE fi,
  BEFN_PRECOND precond);
extern EXPR BEFnFeatureSoleBody(BEFN_FEATURE f);
extern CLASS BEFnFeatureSoleClass(BEFN_FEATURE f);
extern void BEFnFeatureMerge(BEFN_FEATURE target, BEFN_FEATURE source);
extern BOOLEAN BEFnFeatureHasCodeNumbers(BEFN_FEATURE f, int *code1,int *code2);
extern void BEFnFeatureSetCodeNumbers(BEFN_FEATURE f, int code1, int code2);
extern BOOLEAN BEFnFeatureListSortByCode(ARRAY_BEFN_FEATURE features,
  BOOLEAN *has_range);
extern void BEFnFeatureSetCreationDefaultValue(BEFN_FEATURE f,
  BEFN_CREDFT befn_credft);
extern BEFN_CREDFT BEFnFeatureCreationDefaultValue(BEFN_FEATURE f);

/* predefined features */
extern BOOLEAN BEFnFeatureInitOrder(BEFN_FEATURE f, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);

/* code generation */
extern TYPE_POINTER_STATUS BEFnFeatureSwizzleStatus(BEFN_FEATURE f,
  int *swizzle_index);
extern void BEFnFeatureCodeGenInitCreation(BEFN_FEATURE f, ISET layout_set,
  CODEGEN be);
extern CODEGEN_OBJ BEFnFeatureLayoutBEObj(BEFN_FEATURE f);
extern int BEFnFeatureOffset(BEFN_FEATURE f);
extern int BEFnFeatureFieldWidth(BEFN_FEATURE f, CODEGEN be);
extern int BEFnFeatureOffsetCmp(const void *t1, const void *t2);
extern void BEFnFeatureFinalize(BEFN_FEATURE f, CODEGEN be);
extern void BEFnFeatureCodeGenBody(BEFN_FEATURE f, CODEGEN_OBJ res_be_var,
  CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_precond.c                                      */
/*                                                                           */
/*****************************************************************************/

extern BEFN_PRECOND BEFnPrecondNew(FEFN_PRECOND fefn_precond, EXPR body);
extern void BEFnPrecondFinalize(BEFN_PRECOND befn_precond, CODEGEN be);
extern BOOLEAN BEFnPrecondInitOrder(BEFN_PRECOND befn_precond, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);
extern void BEFnPrecondCodeGenBody(BEFN_PRECOND befn_precond,
  CODEGEN_OBJ res_be_var, CODEGEN be);
extern void BEFnPrecondCodeGenCall(BEFN_PRECOND befn_precond,
  ARRAY_BEFN_PARAM parameters, CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_builtin.c                                      */
/*                                                                           */
/*****************************************************************************/

extern BEFN_BUILTIN BEFnBuiltinNew(FEFN_BUILTIN bf, CODEGEN_OBJ be_obj);
extern void BEFnBuiltinFinalize(BEFN_BUILTIN bcf, CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_letdef.c                                       */
/*                                                                           */
/*****************************************************************************/

extern BEFN_LETDEF BEFnLetDefNew(FEFN_LETDEF letdef, EXPR body);
extern void BEFnLetDefFinalize(BEFN_LETDEF fun, CODEGEN be);
extern void BEFnLetDefUnFinalize(BEFN_LETDEF fun, CODEGEN be);
extern BOOLEAN BEFnLetDefInitOrder(BEFN_LETDEF befn, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);
extern void BEFnLetDefCodeGenBody(BEFN_LETDEF letdef, CODEGEN_OBJ res_be_var,
  CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_downdef.c                                      */
/*                                                                           */
/*****************************************************************************/

extern BEFN_DOWNDEF BEFnDownDefMake(FEFN_DOWNDEF fefn_downdef,
  CODEGEN_OBJ be_obj);
extern void BEFnDownDefFinalize(BEFN_DOWNDEF fun, CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_param.c                                        */
/*                                                                           */
/*****************************************************************************/

extern BEFN_PARAM BEFnParamNew(PARAM_KIND param_kind, FEFN_PARAM np_param,
  BEFN_FEATURE creation_feature);
extern PARAM_KIND BEFnParamKind(BEFN_PARAM param);
extern BOOLEAN BEFnParamIsOptional(BEFN_PARAM param);
extern FEFN_PARAM BEFnParamOrigFEFnParam(BEFN_PARAM befn_param);
extern void BEFnParamResetOrigFEFnParam(BEFN_PARAM befn_param,
  FEFN_PARAM fefn_param);
extern void BEFnParamListInsert(ARRAY_BEFN_PARAM *befn_params,
  BEFN_PARAM befn_param);
extern CODEGEN_TYPE BEFnParamBEType(BEFN_PARAM param);
extern CODEGEN_OBJ BEFnParamBEVar(BEFN_PARAM param);
extern void BEFnParamListSetBETypes(ARRAY_BEFN_PARAM params, CODEGEN be);
extern void BEFnParamListPrototypeDeclare(ARRAY_BEFN_PARAM params,
  CODEGEN_OBJ be_obj, CODEGEN be);
extern void BEFnParamListSetBENamesAndDeclare(ARRAY_BEFN_PARAM params,
  CODEGEN_OBJ be_obj, CODEGEN be);
extern int BEFnParamIndex(BEFN_PARAM param);
extern BOOLEAN BEFnParamHasCreationFeature(BEFN_PARAM param);
extern BEFN_FEATURE BEFnParamCreationFeature(BEFN_PARAM param);
extern BEFN_PARAM BEFnParamListFind(ARRAY_BEFN_PARAM params,
  BEFN_FEATURE cfeature);
extern UTF8 BEFnParamShow(BEFN_PARAM param);
extern UTF8 BEFnParamListShow(ARRAY_BEFN_PARAM parameters);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_invariant.c                                    */
/*                                                                           */
/*****************************************************************************/

extern BEFN_INVARIANT BEFnInvtNew(CLASS c, FEFN_PARAM orig_self_param);
extern void BEFnInvtAddExpr(BEFN_INVARIANT invt, EXPR e);
extern BEFN_PARAM BEFnInvtSelfParam(BEFN_INVARIANT invt);
extern void BEFnInvtFinalize(BEFN_INVARIANT invt, CODEGEN be);
extern BOOLEAN BEFnInvtInitOrder(BEFN_INVARIANT befn, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun);
extern void BEFnInvtCodeGenBody(BEFN_INVARIANT invt, CODEGEN_OBJ res_be_var,
  CODEGEN be);
extern void BEFnInvtCallCodeGen(BEFN_INVARIANT invt, CODEGEN_TYPE self_be_type,
  CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_class_init.c                                   */
/*                                                                           */
/*****************************************************************************/

extern BEFN_CLASS_INIT BEFnClassInitMake(CLASS c);
extern void BEFnClassInitFinalize(BEFN_CLASS_INIT fun, CODEGEN be);
extern BOOLEAN BEFnClassInitInitOrder(BEFN_CLASS_INIT befn, int visit_num,
  BOOLEAN *report, BEFN_SYSTEM_INIT fun, int predef_index);
extern void BEFnClassInitCodeGenBody(BEFN_CLASS_INIT fun,
  CODEGEN_OBJ res_be_var, CODEGEN be);
extern void BEFnClassInitCodeGenCall(BEFN_CLASS_INIT fun, CODEGEN be);
extern CODEGEN_OBJ BEFnClassInitCodeGenLoadBEObj(BEFN_CLASS_INIT fun);
extern void BEFnClassInitCodeGenLoadFn(BEFN_CLASS_INIT fun, CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_enum_init.c                                    */
/*                                                                           */
/*****************************************************************************/

extern BEFN_ENUM_INIT BEFnEnumInitMake(CLASS c);
extern void BEFnEnumInitFinalize(BEFN_ENUM_INIT fun, CODEGEN be);
extern void BEFnEnumInitCodeGenBody(BEFN_ENUM_INIT fun, CODEGEN_OBJ res_be_var,
  CODEGEN be);
extern void BEFnEnumInitCodeGenCall(BEFN_ENUM_INIT fun, CODEGEN be);
extern void BEFnEnumInitCodeGenLoadFn(BEFN_ENUM_INIT fun, CODEGEN be);
extern CODEGEN_OBJ BEFnEnumInitCodeGenLoadBEObj(BEFN_ENUM_INIT fun);


/*****************************************************************************/
/*                                                                           */
/*  External functions - befn_system_init.c                                  */
/*                                                                           */
/*****************************************************************************/

extern BEFN_SYSTEM_INIT BEFnSystemInitMake(SYSTEM sys);
extern void BEFnSystemInitRegisterClass(BEFN_SYSTEM_INIT fun, CLASS c);
extern void BEFnSystemInitFinalize(BEFN_SYSTEM_INIT fun, CODEGEN be);
extern void BEFnSystemInitCodeGenBody(BEFN_SYSTEM_INIT fun,
  CODEGEN_OBJ res_be_var, CODEGEN be);
extern void BEFnSystemInitCodeGenLoadFn(BEFN_SYSTEM_INIT fun,
  CODEGEN_OBJ be_fn, CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  EXTERNAL FUNCTIONS - OTHER TYPES                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  External functions - lexer.c                                             */
/*                                                                           */
/*****************************************************************************/

extern void LexAddPos(FILE_POS pos, AFACTORY af);
extern FILE_POS LexFilePos(TOKEN t);
extern FILE_POS FilePosNew(USTRING file_name, short col_num, int line_num);
extern TOKEN LexTokenNew(TOKEN_TYPE type, FILE_POS file_pos, USTRING value);
extern void LexTokenListFree(TOKEN token);
extern UTF8 FilePosShow(FILE_POS pos);
extern UTF8 LexPos(TOKEN t);
extern TOKEN_TYPE LexType(TOKEN t);
extern TOKEN LexNext(TOKEN t);
extern USTRING LexValue(TOKEN t);
extern UTF8 LexShow(TOKEN t);
extern BOOLEAN LexFile(USTRING fdir, USTRING file, TOKEN *tokens,
  SYMTAB_NAMED op_table, FILE_POS ename_pos);
extern BOOLEAN LexLine(FILE *fp, TOKEN *tokens);
extern BOOLEAN LexIsName(TOKEN token);
extern void LexDebug(TOKEN tokens, FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*  External functions - name.c                                              */
/*                                                                           */
/*****************************************************************************/

/* construction and query */
extern NAME NameNew(USTRING rep);
extern NAME NameNewInfix(USTRING rep, int precedence);
extern NAME NameNewFromPrev(NAME value, NAMED prev);
extern USTRING NameRep(NAME name);
extern USTRING NameKey(NAME name);
extern BOOLEAN NameIsOperator(NAME name);
extern BOOLEAN NameIsPrefixOperator(NAME name);
extern BOOLEAN NameIsInfixOperator(NAME name);
extern BOOLEAN NameIsPostfixOperator(NAME name);
extern int NamePrecedence(NAME name);

/* parsing */
extern BOOLEAN NameParse(TOKEN *t, TOKEN *token);
extern BOOLEAN NameDefParse(TOKEN *t, SYMTAB_NAMED op_table, NAME *name);
extern BOOLEAN NameDefListParse(TOKEN *t, SYMTAB_NAMED op_table,
  ARRAY_NAME *names);

/* miscellaneous tests */
extern BOOLEAN NameEqual(NAME n1, NAME n2);
extern BOOLEAN NameKindEqual(NAME n1, NAME n2);
extern BOOLEAN NameCompatible(NAME n1, FILE_POS p1, NAME n2, FILE_POS p2,
  ASTRING str);
extern BOOLEAN NameConsistentWithParameters(NAME n, int param_count,
  FILE_POS err_pos);
extern BOOLEAN NameOperatorConsistent(NAME name, FILE_POS pos,
  SYMTAB_NAMED op_table, ASTRING str);
extern BOOLEAN NameIsInfix(USTRING str, SYMTAB_NAMED op_table, NAME *name);

/* display and debug */
extern void NameAdd(NAME name, AFACTORY af);
extern UTF8 NameShow(NAME name);
extern UTF8 NameFullShow(NAME name);
extern ASTRING NameKindShow(NAME name);
extern void NameDefListDebug(ARRAY_NAME names, FILE *fp);

/* named entities */
extern NAMED NamedNew(FILE_POS file_pos, NAME name);
extern KIND_TAG NamedKind(NAMED named);
extern FILE_POS NamedFilePos(NAMED named);
extern NAME NamedName(NAMED named);
extern ASTRING KindShow(KIND_TAG kind_tag);
/* extern BOOLEAN NamedIsFunction(NAMED named); */
extern BOOLEAN NamedIsLocal(NAMED named);


/*****************************************************************************/
/*                                                                           */
/*  External functions - context.c                                           */
/*                                                                           */
/*****************************************************************************/

extern CONTEXT ContextNew(SYMTAB_CLASS_VIEW class_views);
extern void ContextPushFEFnFeatureSets(CONTEXT cxt,
  SYMTAB_FEFN_FEATURE_SET fvs, BOOLEAN shadow_lim);
extern void ContextPushEmpty(CONTEXT cxt, FEFN fefn, BOOLEAN shadow_lim);
/* ***
extern BOOLEAN ContextPushTypeVars(CONTEXT cxt, TYPE_VARS tv,
  BOOLEAN shadow_lim);
*** */
extern void ContextPop(CONTEXT cxt, BOOLEAN free_it);
extern BOOLEAN ContextInsertGeneric(CONTEXT cxt, TYPE_VAR v);
extern BOOLEAN ContextInsertFEFnLetDef(CONTEXT cxt, FEFN_LETDEF ld);
extern BOOLEAN ContextInsertFEFnParam(CONTEXT cxt, FEFN_PARAM param);
extern BOOLEAN ContextInsertFEFnDownDef(CONTEXT cxt, FEFN_DOWNDEF c);
extern BOOLEAN ContextRetrieve(CONTEXT cxt, USTRING key, NAMED *value);
extern BOOLEAN ContextRetrieveClassView(CONTEXT cxt, USTRING key,
  CLASS_VIEW *class_view);
extern void ContextDelete(CONTEXT cxt, NAMED named);
extern void ContextClearTopLevel(CONTEXT cxt);
/* extern USTRING ContextNewName(CONTEXT cxt, USTRING str); */
extern void ContextDebug(CONTEXT cxt, ASTRING header, FILE *fp,
  int print_style);


/*****************************************************************************/
/*                                                                           */
/*  External functions - type.c                                              */
/*                                                                           */
/*****************************************************************************/

/* the Show submodule */
/* ***
extern void TypeDebugOn(void);
extern void TypeDebugOff(void);
*** */
extern UTF8 TypeVarsShow(TYPE_VARS tv, CONTEXT cxt);
extern UTF8 TypeShow(TYPE type, CONTEXT cxt);
extern UTF8 TypeArrayShow(ARRAY_TYPE agen, CONTEXT cxt);
extern UTF8 TypeITypesShow(ITYPES itypes, CONTEXT cxt);
extern FILE_POS TypeFilePos(TYPE type);

/* the Make submodule */
extern TYPE_VAR TypeVarMake(FILE_POS file_pos, NAME name);
extern NAME TypeVarName(TYPE_VAR tv);
extern TYPE TypeMakeFromVar(FILE_POS file_pos, TYPE_VAR v);
extern TYPE TypeMakeUninstantiatedClassType(FILE_POS file_pos,
  USTRING class_name, ARRAY_TYPE generics);
extern TYPE TypeMakeInstantiatedClassType(FILE_POS file_pos, CLASS c,
  TYPE_VARS c_vars);
extern TYPE TypeMakeInstantiatedClassType2(FILE_POS file_pos, CLASS c,
  ARRAY_TYPE generics);
extern TYPE TypeMakeFunctionType(ARRAY_TYPE generics, FILE_POS file_pos);
extern TYPE TypeMakeUnconstrainedRange(FILE_POS file_pos, NAME name);

/* the Parse submodule */
extern BOOLEAN TypeOptionalActualGenericsParse(TOKEN *t, ARRAY_TYPE *res);
extern BOOLEAN TypeParse(TOKEN *t, TYPE *type);
extern BOOLEAN TypeOptionalFormalGenericsParse(TOKEN *t, FEATURE_TYPE ftype,
  BOOLEAN is_coerce, TYPE_VARS *res);

/* the LevelOneValid submodule */
extern BOOLEAN TypeVarsBegin(TYPE_VARS agen, CONTEXT cxt);
extern void TypeVarsEnd(TYPE_VARS agen, CONTEXT cxt);
extern BOOLEAN TypeLevelOneValid(TYPE t, CONTEXT cxt, BOOLEAN is_public);
extern BOOLEAN TypeVarsBeginLevelOneValid(TYPE_VARS tv, CONTEXT cxt,
  BOOLEAN is_public);
extern BOOLEAN TypeWithVarsLevelOneValid(TYPE t, TYPE_VARS tv, CONTEXT cxt);

/* the Variable submodule */
extern void TypeVarsForwardBegin(TYPE_VARS tv1, TYPE_VARS tv2,
  FILE_POS file_pos);
extern void TypeVarsForwardEnd(TYPE_VARS tv1);

/* the Copy submodule */
extern TYPE TypeCopyUninstantiated(TYPE t);
extern TYPE_VARS TypeVarsCopyUninstantiated(TYPE_VARS type_vars);
extern TYPE TypeCopy(TYPE t);
extern TYPE_VARS TypeVarsCopy(TYPE_VARS type_vars, FILE_POS file_pos);

/* the Unify submodule */
extern void TypeRangeMarkBegin(void);
extern void TypeRangeMarkEnd(BOOLEAN confirm);
extern BOOLEAN TypeVarsInferBegin(TYPE_VARS tv, ARRAY_TYPE *actuals,
  FILE_POS file_pos, BOOLEAN confirm, CONTEXT cxt);
extern void TypeVarsInferEnd(TYPE_VARS tv, ARRAY_TYPE *actuals,
  BOOLEAN confirm);
extern TYPE TypeFixRange(TYPE type);

/* the Equality submodule */
extern BOOLEAN TypeEqual(TYPE t1, TYPE t2, CONTEXT cxt);
extern BOOLEAN TypeVarsEquivalent(TYPE_VARS tv1, TYPE_VARS tv2, CONTEXT cxt,
  FILE_POS file_pos);

/* the Subtype submodule */
extern BOOLEAN TypeAncestorsContainIType(ITYPES ancestors, CLASS c,
  ITYPE *res);
extern BOOLEAN TypeIsSubType(TYPE lower, TYPE higher, COERCION *coercion,
  CONTEXT cxt);

/* the Join and Meet submodule */
extern void TypeITypeSort(ITYPES itypes);
extern BOOLEAN TypeJoin(TYPE *t1, TYPE t2, CONTEXT cxt);
extern BOOLEAN TypeMeet(TYPE *t1, TYPE t2, CONTEXT cxt);

/* the LevelTwoValid submodule */
extern BOOLEAN TypeCheckInheritance(TYPE type, CLASS c, BOOLEAN coerce,
  CONTEXT cxt);
extern CLASS TypeITypeClass(ITYPE itype);
extern COERCION TypeITypeCoercion(ITYPE itype);
extern ARRAY_TYPE TypeITypeGenerics(ITYPE itype);
extern void TypeAddITypes(ITYPES itypes, TYPE type, int weight, COERCION co);
extern BOOLEAN TypeBuildAncestorSet(CLASS c, ITYPES *ancestor_set,
  CONTEXT cxt);

/* the LevelThreeValid submodule */
extern BOOLEAN TypeLevelThreeValid(TYPE type, CONTEXT cxt);
extern BOOLEAN TypeVarsLevelThreeValid(TYPE_VARS type_vars, CONTEXT cxt);
extern BOOLEAN TypeManifest(TYPE t, CONTEXT cxt, BOOLEAN is_public);
extern BOOLEAN TypeVarsBeginManifest(TYPE_VARS tv, CONTEXT cxt,
  BOOLEAN is_public);

/* the Features submodule */
extern BOOLEAN TypeInheritFEFnFeatureSets(CLASS_VIEW cv, ITYPES ancestor_set,
  CONTEXT cxt);
extern BOOLEAN TypeRetrieveFEFnFeatureSet(TYPE t, CONTEXT cxt,
  FILE_POS file_pos, USTRING key, FEFN_FEATURE_SET *fvs);

/* the Swizzle submodule */
extern void TypeSwizzleRegister(TYPE type);
extern int TypeSwizzleSet(TYPE_VARS type_vars);
extern TYPE_POINTER_STATUS TypeIsPointer(TYPE type, int *swizzle_index);

/* the CodeGen submodule */
extern ARRAY_CLASS TypeClasses(TYPE t);
extern BOOLEAN TypeIsClassType(TYPE t, CLASS *class);
extern CLASS TypeClass(TYPE t);
extern CASE_TYPE TypeCaseType(TYPE t);
extern int TypeStructWidth(TYPE t);
extern CODEGEN_TYPE TypeBEType(TYPE t, CODEGEN be);
extern EXPR TypeMin(TYPE t);
extern EXPR TypeMax(TYPE t);
/* ***
extern void ITypeInvtCallCodeGen(ITYPE itype, CODEGEN_TYPE self_be_type,
  CODEGEN be);
*** */


/*****************************************************************************/
/*                                                                           */
/*  External functions - coercion.c                                          */
/*                                                                           */
/*****************************************************************************/

extern COERCION CoercionMake(FEFN_FEATURE fv);
extern COERCION CoercionCopyAndExtend(COERCION co, FEFN_FEATURE fv);
extern COERCION CoercionCopyAndAppend(COERCION co1, COERCION co2);
extern int CoercionLength(COERCION co);
extern BOOLEAN CoercionEqual(COERCION co1, COERCION co2);
extern void CoercionInsert(COERCION co, EXPR *e, CONTEXT cxt, TYPE self_type);
extern UTF8 CoercionShow(COERCION co);


/*****************************************************************************/
/*                                                                           */
/*  External functions - class.c                                             */
/*                                                                           */
/*****************************************************************************/

/* establish */
extern CLASS ClassNew(FILE_POS file_pos, CLASS_VIEW orig_cv,
  TYPE_VARS type_vars, TYPE inherit_type, BOOLEAN is_enum, BOOLEAN is_builtin);
extern FILE_POS ClassFilePos(CLASS c);
extern CLASS_VIEW ClassOrigClassView(CLASS c);
extern CLASS_VIEW ClassToView(CLASS c, CONTEXT cxt);
extern void ClassRegisterName(CLASS class, NAME name);
extern NAME ClassName(CLASS c, CONTEXT cxt);
extern BOOLEAN ClassIsBuiltin(CLASS c);
extern CASE_TYPE ClassCaseType(CLASS c);

/* establish predefined classes */
extern CLASS ClassVoid;
extern CLASS ClassObject;
extern CLASS ClassBool;
extern CLASS ClassByte;
extern CLASS ClassShort;
extern CLASS ClassInt;
extern CLASS ClassUByte;
extern CLASS ClassUShort;
extern CLASS ClassUInt;
extern CLASS ClassReal;
extern CLASS ClassChar;
extern CLASS ClassList;
extern CLASS ClassNList;
extern CLASS ClassEList;
extern CLASS ClassArray;
extern CLASS ClassString;
extern CLASS ClassTuple[MAX_TUPLE + 1];
extern CLASS ClassFun[MAX_FUN_PARAMS + 1];
extern CLASS ClassFunRep[MAX_FUN_PARAMS+1];
extern CLASS ClassFunNM[MAX_FUN_PARAMS+1][MAX_FUN_PARAMS+1];
extern BOOLEAN ClassInitPredefined(SYSTEM_VIEW sv);

/* validate interface types */
extern TYPE_VARS ClassVars(CLASS c);
extern void ClassRegisterCoerceResultType(CLASS c, TYPE type, FEFN_FEATURE cfv);
extern BOOLEAN ClassCheckBuiltinInheritance(CLASS c, CLASS parent_c);
extern BOOLEAN ClassValidateInterfaceTypes1(CLASS c, CONTEXT cxt);
extern BOOLEAN ClassValidateInterfaceTypes2(CLASS c, CONTEXT cxt);
extern BOOLEAN ClassValidateInterfaceTypes3(CLASS c, CONTEXT cxt);
extern BOOLEAN ClassValidateInterfaceTypes4(CLASS c, CONTEXT cxt);
extern int ClassSortKey(CLASS c);
extern ITYPES ClassParentSet(CLASS c);
extern ITYPES ClassAncestorSet(CLASS c);
extern int ClassEquivalenceId(CLASS c);
extern BOOLEAN ClassIsEnum(CLASS c);
extern BOOLEAN ClassIsAbstract(CLASS c);
extern TYPE ClassType(CLASS c);
extern void ClassDebugTypes(CLASS c, CONTEXT cxt, FILE *fp, int print_style);

/* creation, initialization, and invariants */
extern BEFN_CREATION ClassBEFnCreation(CLASS c);
extern BEFN_CLASS_INIT ClassBEFnClassInit(CLASS c);
extern BEFN_ENUM_INIT ClassBEFnEnumInit(CLASS c);
extern BOOLEAN ClassFindInitializationOrder(CLASS c, BEFN_SYSTEM_INIT fun);
extern void ClassMakeInvariant(CLASS c, FEFN_PARAM orig_self_param);
extern BEFN_INVARIANT ClassInvariant(CLASS c);

/* features */
extern void ClassAddBEFnFeature(CLASS c, BEFN_FEATURE f, BOOLEAN is_abstract,
  BOOLEAN original);
extern BEFN_FEATURE ClassAllPredefinedBEFnFeature(CLASS c);
extern BEFN_FEATURE ClassAllEnumeratedBEFnFeature(CLASS c);
extern BEFN_FEATURE ClassAllEnumTrieBEFnFeature(CLASS c);
extern void ClassRegisterPredefinedObjects(CLASS c,
  ARRAY_BEFN_FEATURE features);
extern void ClassPromotePredefinedObjects(CLASS c);
extern ARRAY_BEFN_FEATURE ClassPredefs(CLASS c);
extern BEFN_FEATURE ClassLegalCode(CLASS c);

/* enumerated classes */
extern ISET ClassEnumCodeSetInit(CLASS c);
extern void ClassAugmentCodeSet(CLASS c, ISET enum_code_set);
extern void ClassNotifyCodeSet(CLASS c, ISET enum_code_set);
extern BOOLEAN ClassEnumAssignCodes(CLASS c);
extern int ClassEnumMaxCode(CLASS c);
extern int ClassEnumTrieHeight(CLASS c);
extern int ClassEnumTrieWidth(CLASS c);
extern BOOLEAN ClassHasNoCreationFeatures(CLASS c);
extern ARRAY_BEFN_FEATURE ClassCreationFeatures(CLASS c);
extern BOOLEAN ClassEnumHasCodeRange(CLASS c);

/* disjoint sets */
extern void ClassDSMerge(CLASS c1, CLASS c2);
extern BEFN_FEATURE ClassDSCloneBEFnFeature(CLASS c, FEFN_FEATURE fv,
  FEFN_PARAM fv_self);
/* extern void ClassDSSetCloneBEFnFeature(CLASS c, BEFN_FEATURE f); */
extern void ClassDSFinalizeComponent(CLASS c);
extern ARRAY_CLASS ClassDSComponent(CLASS c);
extern void ClassDSDebug(CLASS c, FILE *fp);

/* code generation, phase 1 (initialization) */
extern void ClassCodeGenInitAll(ARRAY_CLASS classes, CODEGEN be);
extern void ClassCodeGenInitCreationFeatures(CLASS class, CODEGEN be);
extern void ClassCodeGenInitFunctions(CLASS c, CODEGEN be);

/* code generation, phase 2 (actual code generation) */
extern CODEGEN_TYPE ClassBEType(CLASS c);
extern CODEGEN_TYPE ClassBEContentType(CLASS c);
extern void ClassEnumCodeGenTrieAccess(CLASS c, CODEGEN_OBJ be_obj, CODEGEN be);
extern BOOLEAN ClassIsObjRef(CLASS c);
extern BOOLEAN ClassIsSwizzle(CLASS c);
extern int ClassFieldWidth(CLASS c, CODEGEN be);
extern EXPR ClassMin(CLASS c);
extern EXPR ClassMax(CLASS c);
extern CODEGEN_OBJ ClassSwizzleFn(CLASS c);
extern CODEGEN_OBJ ClassUnSwizzleFn(CLASS c);

/* typedefs and structs */
extern void ClassTypeDefPrint(CLASS class, CODEGEN be);
extern BOOLEAN ClassHasTypeTag(CLASS c);
extern void ClassTypeTagPrint(CLASS class, CODEGEN be);
extern CODEGEN_OBJ ClassTypeTag(CLASS class);
extern void ClassTypeStructPrint(CLASS class, CODEGEN be);
extern int ClassStructWidth(CLASS class);

/* executable code */
extern void ClassInvtCallCodeGen(CLASS c, CODEGEN_TYPE self_be_type,
  CODEGEN be);
extern void ClassCodeGenFunctionsInitFile(CLASS c, CODEGEN be);
extern void ClassCodeGenFunctionsLoadFile(CLASS c, CODEGEN be);
extern void ClassCodeGenFunctionsMainFile(CLASS c, CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - class_view.c                                        */
/*                                                                           */
/*****************************************************************************/

/* simple queries */
extern FEFN_CREATION ClassViewFEFnCreation(CLASS_VIEW cv);
extern FILE_POS ClassViewFilePos(CLASS_VIEW cv);
extern NAME ClassViewName(CLASS_VIEW cv);
extern CLASS_VIEW_TYPE ClassViewType(CLASS_VIEW cv);
extern CLASS ClassViewClass(CLASS_VIEW cv);
extern SYSTEM_VIEW ClassViewSystemView(CLASS_VIEW cv);
extern BOOLEAN ClassViewIsPrivate(CLASS_VIEW cv);
extern BOOLEAN ClassViewIsNoRename(CLASS_VIEW cv);
extern CLASS_VIEW ClassViewContentView(CLASS_VIEW cv);

/* establish */
extern CLASS_VIEW ClassViewImportCopy(CLASS_VIEW cv, FILE_POS file_pos,
  NAME name, BOOLEAN is_private, SYSTEM_VIEW sv);
extern BOOLEAN ClassViewImportUninheritedFEFnFeatures(CLASS_VIEW target,
  CLASS_VIEW source, CLASS_VIEW_UPDATE class_view_update,
  SYMTAB_NAMED op_table, FILE_POS dft_pos);
extern BOOLEAN ClassViewRegisterOriginalFEFnFeature(CLASS_VIEW cv,
  FEFN_FEATURE fv);
extern BOOLEAN ClassViewRetrieveFEFnFeatureSet(CLASS_VIEW cv, USTRING key,
  FEFN_FEATURE_SET *fvs);
extern BOOLEAN ParseOptionalClassViewUpdate(TOKEN *t, SYMTAB_NAMED op_table,
  CLASS_VIEW_UPDATE *res);
extern BOOLEAN ClassViewListParse(TOKEN *t, SYSTEM_VIEW sv,
  FILE_POS file_pos, BOOLEAN extension);
extern BOOLEAN ClassViewVoidClassMake(SYSTEM_VIEW sv);

/* validate interface types */
extern BOOLEAN ClassViewValidateInterfaceTypes0(CLASS_VIEW cv);
extern void ClassViewAddToAllPredefined(CLASS_VIEW cv, FEFN_FEATURE fv);
extern BOOLEAN ClassViewValidateInterfaceTypes1(CLASS_VIEW cv, CONTEXT cxt);
extern BOOLEAN ClassViewValidateInterfaceTypes5(CLASS_VIEW cv, CONTEXT cxt);

/* feature inheritance */
extern BOOLEAN ClassViewInheritParentFEFnFeatureSets(CLASS_VIEW cv,
  CLASS_VIEW ancestor_cv, CONTEXT cxt);
extern BOOLEAN ClassViewInheritFeatures1(CLASS_VIEW cv, CONTEXT cxt);
extern BOOLEAN ClassViewInheritFeatures2(CLASS_VIEW cv, CONTEXT cxt);

/* validate expressions (and other things done during that major stage) */
extern BOOLEAN ClassViewValidateExpressions(CLASS_VIEW cv, CONTEXT cxt);

/* debug */
extern void ClassViewDebug(CLASS_VIEW cv, CONTEXT cxt, FILE *fp,
  int print_style);


/*****************************************************************************/
/*                                                                           */
/*  External functions - system.c                                            */
/*                                                                           */
/*****************************************************************************/

extern void SystemRegisterClass(SYSTEM sys, CLASS c);
extern ARRAY_CLASS SystemClasses(SYSTEM sys);
extern BOOLEAN SystemRetrieveView(SYSTEM sys, USTRING str, SYSTEM_VIEW *sv);
extern void SystemModuleInitFileHeaderIncludes(SYSTEM sys, CODEGEN be);
extern void SystemModuleLoadFileHeaderIncludes(SYSTEM sys, CODEGEN be);
extern void SystemModuleMainFileHeaderIncludes(SYSTEM sys, CODEGEN be);
extern void SystemModuleInitFileCodeIncludes(SYSTEM sys, CODEGEN be);
extern void SystemModuleLoadFileCodeIncludes(SYSTEM sys, CODEGEN be);
extern void SystemModuleMainFileCodeIncludes(SYSTEM sys, CODEGEN be);
extern void SystemCodeFileIncludes(SYSTEM sys, CODEGEN be);
extern CODEGEN_EXEC SystemBEInitExec(SYSTEM sys);
extern CODEGEN_EXEC SystemBELoadExec(SYSTEM sys);
extern BOOLEAN SystemCompile(CODEGEN be, ASTRING data_dir, SYSTEM *res);
extern SYSTEM_VIEW SystemSystemView(SYSTEM sys);
extern void SystemDebug(SYSTEM s, CONTEXT cxt, FILE *fp, int print_style);
extern void SystemDebugSystemViewNames(SYSTEM s, SYSTEM_VIEW curr_sv, FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*  External functions - system_view.c                                       */
/*                                                                           */
/*****************************************************************************/

/* establish */
extern SYSTEM_VIEW SystemViewNew(FILE_POS file_pos, NAME name, SYSTEM system);
extern FILE_POS SystemViewFilePos(SYSTEM_VIEW sv);
extern NAME SystemViewName(SYSTEM_VIEW sv);
extern SYSTEM SystemViewSystem(SYSTEM_VIEW sv);
extern SYMTAB_NAMED SystemViewOpTable(SYSTEM_VIEW sv);
extern BOOLEAN SystemViewInsertClassView(SYSTEM_VIEW sv, CLASS_VIEW cv,
  CLASS_VIEW *cv2);
extern BOOLEAN SystemViewRetrieveClassView(SYSTEM_VIEW sv, USTRING key,
  CLASS_VIEW *cv);
extern SYMTAB_CLASS_VIEW SystemViewClassViews(SYSTEM_VIEW sv);
extern BOOLEAN SystemViewRetrieveSystemView(SYSTEM_VIEW sv, USTRING key,
  SYSTEM_VIEW *sv2);
extern ARRAY_SYSTEM_VIEW SystemViewSystemViews(SYSTEM_VIEW sv);
extern BOOLEAN SystemViewImport(SYSTEM_VIEW target, SYSTEM_VIEW source,
  BOOLEAN is_private, TOKEN *t);
extern BOOLEAN SystemViewEstablish(TOKEN module_name_token, SYSTEM system,
  SYSTEM_VIEW *res);

/* validate interface types */
extern BOOLEAN SystemViewValidateInterfaceTypes(SYSTEM_VIEW sv);

/* inherit features */
extern BOOLEAN SystemViewInheritFeatures(SYSTEM_VIEW sv);

/* validate expressions */
extern BOOLEAN SystemViewValidateExpressions(SYSTEM_VIEW sv);

/* code generation */
extern CODEGEN_FILE SystemViewBEFile(SYSTEM_VIEW sv);
extern CODEGEN_FILE SystemViewBEInitFile(SYSTEM_VIEW sv);
extern CODEGEN_FILE SystemViewBELoadFile(SYSTEM_VIEW sv);
/* extern MODULE_INIT_FN SystemViewInitBEFn(SYSTEM_VIEW sv); */
extern CODEGEN_OBJ SystemViewSwizzleFn(SYSTEM_VIEW sv);
extern CODEGEN_OBJ SystemViewUnSwizzleFn(SYSTEM_VIEW sv);
extern void SystemViewRegisterClass(SYSTEM_VIEW sv, CLASS class);
extern BOOLEAN SystemViewLastClassWithTypeTag(SYSTEM_VIEW sv, CLASS *c);
extern BEFN_FEATURE SystemViewRegisterLiteralString(SYSTEM_VIEW sv,
  USTRING str, FILE_POS file_pos);
extern ASTRING TimingTimeStamp(void);
extern void SystemViewCodeGenInit(SYSTEM_VIEW sv, CODEGEN be);
extern void SystemViewCodeGenTypeDefs(SYSTEM_VIEW sv, CODEGEN be);
extern void SystemViewCodeGenTypeTags(SYSTEM_VIEW sv, CODEGEN be);
extern void SystemViewCodeGenTypeStructs(SYSTEM_VIEW sv, CODEGEN be);
extern BOOLEAN SystemViewCodeGenInitFile(SYSTEM_VIEW sv, CODEGEN be);
extern BOOLEAN SystemViewCodeGenLoadFile(SYSTEM_VIEW sv, CODEGEN be);
extern BOOLEAN SystemViewCodeGenMainFile(SYSTEM_VIEW sv, CODEGEN be);

/* debug */
extern void SystemViewDebug(SYSTEM_VIEW sv, CONTEXT cxt,
  FILE *fp, int print_style);
extern void SystemViewDebugSystemViewNames(SYSTEM_VIEW sv, SYSTEM_VIEW curr_sv,
  FILE *fp);


/*****************************************************************************/
/*                                                                           */
/*  External functions - auto.c                                              */
/*                                                                           */
/*****************************************************************************/

extern BOOLEAN AutoTupleClasses(USTRING module_dir_name);
extern BOOLEAN AutoFunctionClasses(USTRING module_dir_name);
extern BOOLEAN AutoCharClass(USTRING module_dir_name);


/*****************************************************************************/
/*                                                                           */
/*  External functions - np_back.c                                           */
/*                                                                           */
/*****************************************************************************/

/* files */
extern CODEGEN_FILE NPBack_File_Stdlib;		/* "stdlib" file	     */
extern CODEGEN_FILE NPBack_File_Stdio;		/* "stdio" file		     */
extern CODEGEN_FILE NPBack_File_String;		/* "string" file	     */
extern CODEGEN_FILE NPBack_File_Limits;		/* "limits" file	     */
extern CODEGEN_FILE NPBack_File_Float;		/* "float" file		     */
extern CODEGEN_FILE NPBack_File_Math;		/* "math" file	     	     */
extern CODEGEN_FILE NPBack_File_Assert;		/* "assert" file	     */

/* types */
extern CODEGEN_TYPE NPBack_Cache_Obj_Type;	/* CACHE_OBJ		     */
extern CODEGEN_TYPE NPBack_Cache_Fun_Type;	/* CACHE_FUN		     */

/* constants */
extern CODEGEN_OBJ NPBack_False;		/* false		     */
extern CODEGEN_OBJ NPBack_True;			/* true			     */
extern CODEGEN_OBJ NPBack_Dft_Param_Val;	/* default param value       */

/* variables and functions (miscellaneous) */
extern CODEGEN_OBJ NPBack_Self;			/* self			     */
extern CODEGEN_OBJ NPBack_Other;		/* other		     */
extern CODEGEN_OBJ NPBack_Res;			/* res			     */
extern CODEGEN_OBJ NPBack_NPSysObj1;		/* npsys_obj1		     */
extern CODEGEN_OBJ NPBack_NPSysObj2;		/* npsys_obj2		     */
extern CODEGEN_OBJ NPBack_I;			/* i			     */
extern CODEGEN_OBJ NPBack_Code;			/* code			     */
extern CODEGEN_OBJ NPBack_Sig;			/* sig			     */
extern CODEGEN_OBJ NPBack_Mem;			/* mem			     */
extern CODEGEN_OBJ NPBack_Obj_Cache;		/* npsys_obj_cache	     */
extern CODEGEN_OBJ NPBack_Obj_Cache_Insert;	/* npsys_obj_cache_insert()  */
extern CODEGEN_OBJ NPBack_Obj_New;		/* npsys_obj_new(..)	     */
extern CODEGEN_OBJ NPBack_Obj_Cache_Len;	/* npsys_obj_cache_len	     */
extern CODEGEN_OBJ NPBack_Fun_Cache;		/* npsys_fun_cache	     */
extern CODEGEN_OBJ NPBack_Fun_Cache_Insert;	/* npsys_fun_cache_insert()  */
extern CODEGEN_OBJ NPBack_Fun_New;		/* npsys_fun_new(..)	     */
extern CODEGEN_OBJ NPBack_Fun_Cache_Len;	/* npsys_fun_cache_len	     */
extern CODEGEN_OBJ NPBack_Trie[4];		/* trie1, ... , trie4 macros */
extern CODEGEN_OBJ NPBack_SubTrie[4];		/* val, subtrie1 .. subtrie3 */
extern CODEGEN_OBJ NPBack_NullTrie[4];		/* null_value .. null_trie3  */
extern CODEGEN_OBJ NPBack_Signature;		/* $1->signature	     */
extern CODEGEN_OBJ NPBack_Type_Tag;		/* $1->type_tag		     */
extern CODEGEN_OBJ NPBack_Swizzle_Bits;		/* $1->swizzle_bits	     */
extern CODEGEN_OBJ NPBack_Array_Length;		/* $1->length		     */
extern CODEGEN_OBJ NPBack_Array_Get;		/* $1->elems[$2]	     */
extern CODEGEN_OBJ NPBack_EnumSet;		/* enum_set($1, ... , $n)    */
extern CODEGEN_OBJ NPBack_EnumSetSeq;		/* enum_set_seq($1 .. $n)    */
extern CODEGEN_OBJ NPBack_EnumInvtSeq;		/* enum_invt_seq($1 .. $n)   */
extern CODEGEN_OBJ NPBack_Swizzle_Field;  	/* swizzle_field($1, $2, $3) */
extern CODEGEN_OBJ NPBack_Unswizzle_Field;	/* unswizzle_field($1, $2)   */

/* array and string creation functions */
extern CODEGEN_OBJ NPBack_Array_Create_Fn1;
extern CODEGEN_OBJ NPBack_Array_Create_Fn2;
extern CODEGEN_OBJ NPBack_String_Create_Fn1;
extern CODEGEN_OBJ NPBack_String_Create_Fn2;
extern CODEGEN_OBJ NPBack_String_Make0;
extern CODEGEN_OBJ NPBack_String_Make1;
extern CODEGEN_OBJ NPBack_String_Make2;
extern CODEGEN_OBJ NPBack_String_Make3;
extern CODEGEN_OBJ NPBack_String_Make4;
extern CODEGEN_OBJ NPBack_String_Make5;

/* initialization */
extern void NPBackInit(CODEGEN be);


/*****************************************************************************/
/*                                                                           */
/*  External functions - codegen_c.c                                         */
/*                                                                           */
/*****************************************************************************/

extern CODEGEN CodeGenC(ASTRING code_dir);


/*****************************************************************************/
/*                                                                           */
/*  External functions - main.c                                              */
/*                                                                           */
/*****************************************************************************/

void MainCopyrightNotice(FILE *fp);
