/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         fefn.c                                                     */
/*  DESCRIPTION:  Front-end functions                                        */
/*                                                                           */
/*****************************************************************************/
#include <string.h>
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0
#define DEBUG6 0
#define DEBUG7 0


/*****************************************************************************/
/*                                                                           */
/*  Front-end functions                                                      */
/*  -------------------                                                      */
/*                                                                           */
/*  There are several kinds of front-end functions, which all inherit        */
/*  from FEFN:                                                               */
/*                                                                           */
/*     FEFN_CREATION       A view of a Nonpareil creation function           */
/*     FEFN_CREDFT         A creation feature default value                  */
/*     FEFN_FEATURE        A view of a Nonpareil feature                     */
/*     FEFN_PRECOND        A precondition of a Nonpareil feature             */
/*     FEFN_BUILTIN        A Nonpareil builtin function                      */
/*     FEFN_LETDEF         A let definition                                  */
/*     FEFN_CONDITION      A downcast variable                               */
/*     FEFN_PARAM          A Nonpareil parameter                             */
/*     FEFN_INVARIANT      A class invariant                                 */
/*                                                                           */
/*  NB the befn field may have its type redefined in these classes,          */
/*  but always to a type which is a subtype of BEFN.                         */
/*                                                                           */
/*****************************************************************************/

struct fefn_rec {
  KIND_TAG		kind_tag;	/* kind of entity                    */
  FILE_POS		file_pos;	/* position of function name in file */
  NAME			name;		/* name of function                  */
  TYPE_VARS		type_vars;	/* generic parameters (NULL if none) */
  ARRAY_FEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  TYPE			result_type;	/* result type                       */
  BEFN			befn;		/* the corresponding back-end fn,    */
 					/* not for direct use: feature case! */
};


/*****************************************************************************/
/*                                                                           */
/*  NAME FEFnName(FEFN fun)                                                  */
/*                                                                           */
/*  Return the name of fun.                                                  */
/*                                                                           */
/*****************************************************************************/

NAME FEFnName(FEFN fun)
{
  return fun->name;
}


/*****************************************************************************/
/*                                                                           */
/*  FILE_POS FEFnFilePos(FEFN fun)                                           */
/*                                                                           */
/*  Return the file position of fun.                                         */
/*                                                                           */
/*****************************************************************************/

FILE_POS FEFnFilePos(FEFN fun)
{
  return fun->file_pos;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE FEFnResultType(FEFN fun)                                            */
/*                                                                           */
/*  Return the result type of fun.                                           */
/*                                                                           */
/*****************************************************************************/

TYPE FEFnResultType(FEFN fun)
{
  return fun->result_type;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN FEFnBEFn(FEFN fun)                                                  */
/*                                                                           */
/*  Return the C function that implements fun.                               */
/*                                                                           */
/*****************************************************************************/

BEFN FEFnBEFn(FEFN fun)
{
  if( fun->kind_tag == KIND_FEFN_FEATURE )
    return (BEFN) FEFnFeatureBEFnFeature((FEFN_FEATURE) fun);
  else
    return fun->befn;
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnAddParameter(FEFN fun, FEFN_PARAM param)                        */
/*                                                                           */
/*  Add param to fun.                                                        */
/*                                                                           */
/*****************************************************************************/

void FEFnAddParameter(FEFN fun, FEFN_PARAM param)
{
  if( fun->parameters == NULL )
    ArrayInit(&fun->parameters);
  ArrayAddLast(fun->parameters, param);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnParsePreconditions(TOKEN *t, FEATURE_TYPE ftype,             */
/*    SYSTEM_VIEW sv, EXPR *expr_precond)                                    */
/*                                                                           */
/*  Parse some optional preconditions and add them to *expr_precond.  The    */
/*  grammar is                                                               */
/*                                                                           */
/*      [ "require" { expr } "end" ]                                         */
/*                                                                           */
/*  If ftype is FEATURE_PREDEF, preconditions are not permitted.             */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FEFnParsePreconditions(TOKEN *t, FEATURE_TYPE ftype,
  SYSTEM_VIEW sv, EXPR *expr_precond)
{
  EXPR e;
  *expr_precond = NULL;
  if( LexType(curr_token) == TK_REQUIRE )
  {
    *expr_precond = ExprPrecondNew(LexFilePos(curr_token));
    if( ftype == FEATURE_PREDEF )
    {
      fprintf(stderr,
	"%s: precondition not permitted in predefined object feature\n",
	LexPos(curr_token));
      return FALSE;
    }
    next_token;
    while( LexType(curr_token) != TK_END )
    {
      if( !ExprParse(t, sv, &e) )
        return FALSE;
      ExprPrecondAddPrecondition(*expr_precond, e);
    }
    next_token;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnParse(TOKEN *t, BOOLEAN is_letdef, BOOLEAN is_anon,          */
/*    FEATURE_TYPE ftype, BOOLEAN is_coerce, BOOLEAN encl_private,           */
/*    BOOLEAN builtin, BOOLEAN extension, USTRING predef_name,               */
/*    BOOLEAN is_enum, SYSTEM_VIEW sv, FILE_POS *file_pos,                   */
/*    ARRAY_NAME *names, TYPE_VARS *type_vars, ARRAY_FEFN_PARAM *params,     */
/*    TYPE *result_type, EXPR *expr_fnhead, PARAM_BODIES_TYPE *param_b_type, */
/*    EXPR *expr_precond, int *code1, int *code2)                            */
/*                                                                           */
/*  Parse one function, including the body.  Functions occur in three        */
/*  places: let definitions, anonymous functions, and feature views.         */
/*  Each type has a slightly different grammar, and feature views vary       */
/*  depending on whether they are creation, noncreation, predefined, etc.    */
/*  This code handles all the types together, to make clear both the         */
/*  similarities and the differences.                                        */
/*                                                                           */
/*  If encl_private is TRUE, this function is private, so its parameters     */
/*  are automatically private and should not be marked so explicitly.        */
/*                                                                           */
/*  If is_letdef is TRUE, we are parsing a let definition, so a result       */
/*  type is not compulsory but a body is compulsory.                         */
/*                                                                           */
/*  If is_anon is TRUE we are parsing an anonymous function, so the          */
/*  keyword "fun" appears where a list of function names otherwise would.    */
/*  Also, the compulsory body takes the syntactic form of a decorated        */
/*  factor rather than an arbitrary expression.                              */
/*                                                                           */
/*  If neither is TRUE we are parsing a feature view.  In this case, the     */
/*  parameters ftype, is_coerce, builtin, and extension are                  */
/*  relevant, and they indicate whether the feature view is of a creation    */
/*  feature, is a coercion feature, lies in a builtin class, lies in a       */
/*  class extension, or is of a predefined object.  These conditions         */
/*  determine variations in the basic grammar.                               */
/*                                                                           */
/*  If ftype is FEATURE_PREDEF, then predef_name is non-NULL and contains    */
/*  the class name to insert when the syntax of the function body takes the  */
/*  abbreviated form allowed to predefined objects.  Otherwise predef_name   */
/*  is not used and will be NULL.  Also in this case, is_enum may be TRUE    */
/*  and indicates whether the enclosing class is enumerated.  If it is,      */
/*  code integers are allowed following the optional name.                   */
/*                                                                           */
/*  Parameters *file_pos, *names, *type_vars, *params, *result_type,         */
/*  *expr_fnhead, *expr_precond, *code1, and *code2 return the various       */
/*  attributes found during the parse.  It is left to the caller to          */
/*  assemble these into whatever kinds of functions are required.            */
/*  In the case of an anonymous function, *names will contain a              */
/*  single element, "fun".  Parameter *params must be defined on entry,      */
/*  either to NULL or to an array of parameters (which in the case of        */
/*  non-predefined feature views will already contain a "self" parameter).   */
/*                                                                           */
/*  Parameter *expr_fnhead is non-NULL if and only if the function has a     */
/*  body; it contains default values for parameters as well as the body.     */
/*                                                                           */
/*  Parameter *param_bodies_type is both an input and an output parameter.   */
/*  On entry its value may be either PARAM_BODIES_OPEN, meaning that         */
/*  optional parameters are free to either all have default values or        */
/*  all have none; or PARAM_BODIES_FIXED, meaning that optional parameters   */
/*  must all have default values.  On exit, it may hold any of the four      */
/*  values.                                                                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnParse(TOKEN *t, BOOLEAN is_letdef, BOOLEAN is_anon,
  FEATURE_TYPE ftype, BOOLEAN is_coerce, BOOLEAN encl_private,
  BOOLEAN builtin, BOOLEAN extension, USTRING predef_name,
  BOOLEAN is_enum, SYSTEM_VIEW sv, FILE_POS *file_pos,
  ARRAY_NAME *names, TYPE_VARS *type_vars, ARRAY_FEFN_PARAM *params,
  TYPE *result_type, EXPR *expr_fnhead, PARAM_BODIES_TYPE *param_bodies_type,
  EXPR *expr_precond, int *code1, int *code2)
{
  EXPR body;  FILE_POS body_pos;

  if( DEBUG2 )
    fprintf(stderr, "[ FEFnParse(%s: %s)\n", LexPos(curr_token),
      LexShow(curr_token));

  /* no sign of a body yet */
  *expr_fnhead = NULL;

  /* names */
  *file_pos = LexFilePos(curr_token);
  if( is_anon )
  {
    /* anonymous function, current token known to be "fun" */
    ArrayInit(names);
    ArrayAddLast(*names, NameNew(LexValue(curr_token)));
    next_token;
  }
  else if( ftype == FEATURE_PREDEF && LexType(curr_token) != TK_IDENTIFIER )
  {
    /* anonymous predefined; no *names */
    *names = NULL;
  }
  else
  {
    /* ordinary case of names list */
    if( !NameDefListParse(t, SystemViewOpTable(sv), names) )
      db_return(DEBUG2, "FEFnParse", FALSE);
  }

  /* generics */
  if( !TypeOptionalFormalGenericsParse(t, ftype, is_coerce, type_vars) )
    db_return(DEBUG2, "FEFnParse", FALSE);

  /* parameters */
  if( ftype != FEATURE_PREDEF &&
      !FEFnParamListParse(t, ftype, is_coerce, encl_private, sv, params,
	expr_fnhead, param_bodies_type) )
    db_return(DEBUG2, "FEFnParse", FALSE);

  /* result type */
  if( ftype == FEATURE_PREDEF )
  {
    /* predefined object feature, result type is prohibited */
    *result_type = NULL;
    if( LexType(curr_token) == TK_COLON )
    {
      fprintf(stderr,
	"%s: result type prohibited in predefined object feature\n",
	LexPos(curr_token));
      db_return(DEBUG2, "FEFnParse", FALSE);
    }
  }
  else if( is_letdef || is_anon )
  {
    /* let def or anonymous function, result type is optional */
    *result_type = NULL;
    if( LexType(curr_token) == TK_COLON )
    {
      next_token;
      if( !TypeParse(t, result_type) )
	db_return(DEBUG2, "FEFnParse", FALSE);
    }
  }
  else
  {
    /* feature view, result type is compulsory */
    skip(TK_COLON, "colon preceding compulsory result type");
    if( !TypeParse(t, result_type) )
      db_return(DEBUG2, "FEFnParse", FALSE);
  }

  /* if enumerated class, optional code or codes */
  *code1 = *code2 = NO_CODE_NUM;
  if( LexType(curr_token) == TK_LIT_INTEGER )
  {
    if( !is_enum )
    {
      fprintf(stderr, "%s: code number in non-enum class\n",
	LexPos(curr_token));
      db_return(DEBUG2, "FEFnParse", FALSE);
    }
    *code1 = *code2 = UStringToInteger(LexValue(curr_token));
    if( *code1 < 0 )
    {
      fprintf(stderr, "%s: code number is negative\n", LexPos(curr_token));
      db_return(DEBUG2, "FEFnParse", FALSE);
    }
    next_token;
    if( LexType(curr_token) == TK_DOT_DOT )
    {
      next_token;
      check(TK_LIT_INTEGER, "integer code after ..");
      *code2 = UStringToInteger(LexValue(curr_token));
      next_token;
      if( *code2 < 0 )
      {
	fprintf(stderr, "%s: integer code after .. is negative\n",
	  LexPos(curr_token));
	db_return(DEBUG2, "FEFnParse", FALSE);
      }
      else if( *code2 < *code1 )
      {
	fprintf(stderr, "%s: empty integer code range\n", LexPos(curr_token));
	db_return(DEBUG2, "FEFnParse", FALSE);
      }
      else if( *code1 < *code2 && *names != NULL )
      {
	fprintf(stderr, "%s: code range not permitted when name present\n",
	  LexPos(curr_token));
	db_return(DEBUG2, "FEFnParse", FALSE);
      }
    }
  }

  /* optional require clause */
  if( !FEFnParsePreconditions(t, ftype, sv, expr_precond) )
    db_return(DEBUG2, "FEFnParse", FALSE);

  /* body, NULL if not found */
  body = NULL;
  body_pos = LexFilePos(curr_token);
  if( is_letdef )
  {
    /* let def, body is a compulsory expression */
    skip(TK_COLON_EQUALS, ":= (not =)");
    if( !ExprParse(t, sv, &body) )
      db_return(DEBUG2, "FEFnParse", FALSE);
  }
  else if( is_anon )
  {
    /* anonymous function, body is a compulsory decorated factor */
    skip(TK_COLON_EQUALS, ":= (not =)");
    if( !ExprDecoratedFactorParse(t, sv, &body) )
      db_return(DEBUG2, "FEFnParse", FALSE);
  }
  else
  {
    /* feature view, body is usually an optional expression */
    if( LexType(curr_token) == TK_COLON_EQUALS ||
	(LexType(curr_token) == TK_PUNCTSEQ &&
	 UStringEqual(LexValue(curr_token), AStringToUString("="))) )
    {
      skip(TK_COLON_EQUALS, ":= (not =)");
      if( !ExprParse(t, sv, &body) )
	db_return(DEBUG2, "FEFnParse", FALSE);
    }
    else if( ftype == FEATURE_PREDEF )
    {
      if( !ExprAbbreviatedCallParse(t, predef_name, sv, &body) )
	db_return(DEBUG2, "FEFnParse", FALSE);
    }
    else if( builtin )
    {
      fprintf(stderr, "%s: expected := here (builtin class)\n",
	LexPos(curr_token));
      db_return(DEBUG2, "FEFnParse", FALSE);
    }
    else if( extension )
    {
      fprintf(stderr, "%s: expected := here (class extension)\n",
	LexPos(curr_token));
      db_return(DEBUG2, "FEFnParse", FALSE);
    }
  }

  /* sort out body wrt param_bodies_type, and add it to *expr_fnhead */
  if( body == NULL )
  {
    /* no body, so parameter bodies must not be concrete */
    if( *param_bodies_type == PARAM_BODIES_CONCRETE )
    {
      fprintf(stderr,
	"%s: function without body has optional parameter with default value\n",
	FilePosShow(*file_pos));
      return FALSE;
    }
  }
  else
  {
    /* body, so parameter bodies must not be abstract */
    if( *param_bodies_type == PARAM_BODIES_ABSTRACT )
    {
      fprintf(stderr,
	"%s: function with body has optional parameter without default value\n",
	FilePosShow(*file_pos));
      return FALSE;
    }
    if( *expr_fnhead == NULL )
      *expr_fnhead = ExprFnHeadNew(body_pos);
    ExprFnHeadAddSubExpression(*expr_fnhead, body);
  }
  assert(*code1 == NO_CODE_NUM || *code1 <= *code2);
  db_return(DEBUG2, "FEFnParse", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnManifest(FEFN fun, CONTEXT cxt, BOOLEAN is_public,           */
/*    BOOLEAN shadow_lim, BOOLEAN do_interface, EXPR body, TYPE self_type,   */
/*    FEFN_PRECOND fefn_precond, BEFN encl_befn)                             */
/*                                                                           */
/*  Manifest fun: type check it, and prepare it for code generation.         */
/*                                                                           */
/*  The current context is cxt.  When pushing a new level onto the           */
/*  context, to hold the generic and actual parameters of the function,      */
/*  parameter shadow_lim determines whether this level is the limit of       */
/*  shadow checking within the function or not.  It will be the limit        */
/*  for functions which are features, but not for functions which are        */
/*  let definitions.  As always with contexts, by function exit cxt must     */
/*  be restored to its state on entry, whether or not this function          */
/*  succeeds or not.                                                         */
/*                                                                           */
/*  Feature views have to be done in two steps:  first the interface, then   */
/*  later on the implementation.  This is achieved by first calling          */
/*  FEFnManifest with do_interface TRUE and body == NULL, then calling       */
/*  it later with do_interface FALSE and body != NULL.  On the other hand,   */
/*  let definitions (including anonymous functions) can be done all at once, */
/*  which is most efficiently achieved by a single call with both of these   */
/*  parameters set.                                                          */
/*                                                                           */
/*  If fefn_precond != NULL, it has to be manifested in the same context     */
/*  as the body.                                                             */
/*                                                                           */
/*****************************************************************************/

/* *** demoted to fefn_feature.c and fefn_letdef.c now; too many options here
BOOLEAN FEFnManifest(FEFN fun, CONTEXT cxt, BOOLEAN is_public,
  BOOLEAN shadow_lim, BOOLEAN do_interface, EXPR body, TYPE self_type,
  FEFN_PRECOND fefn_precond, BEFN encl_befn)
{
  FEFN_PARAM param;  COERCION c;  int i;  EXPR copy_body;

  if( DEBUG1 )
    fprintf(stderr, "[ FEFnManifest(%s%s, cxt, %s, %s, body %s NULL, ...)\n",
      NameShow(fun->name), FEFnSignatureShow(fun, cxt), bool(shadow_lim),
      bool(do_interface), body != NULL ? "!=" : "==");
      
  ** formal generics **
  if( do_interface )
  {
    if( !TypeVarsManifest(fun->type_vars, cxt, is_public) )
      db_return(DEBUG1, "FEFnManifest typevars-m", FALSE);
  }
  if( !ContextPushTypeVars(cxt, fun->type_vars, shadow_lim) )
    db_return(DEBUG1, "FEFnManifest typevars-b", ContextPop(cxt, TRUE))

  ** parameters **
  if( fun->parameters != NULL )
    for( i = 0;  i < ArraySize(fun->parameters);  i++ )
    {
      param = ArrayGet(fun->parameters, i);
      if( do_interface )
      {
	if( !FEFnParamManifest(param, cxt, is_public) )
	  db_return(DEBUG1, "FEFnManifest params-m", ContextPop(cxt, TRUE))
      }
      if( body != NULL )
      {
	if( FEFnParamKind(param) == PARAM_OPTIONAL )
	{
	  if( !ExprFnHeadManifestParamDefaultVal(body, param, cxt,
	      self_type, encl_befn) )
	    db_return(DEBUG1, "FEFnManifest params-d",ContextPop(cxt,TRUE))
	}
	if( !ContextInsertFEFnParam(cxt, param) )
	  db_return(DEBUG1, "FEFnManifest params-c", ContextPop(cxt, TRUE))
      }
    }

  ** optional result type **
  if( do_interface && fun->result_type != NULL )
  {
    if( !TypeManifest(fun->result_type, cxt, is_public) )
      db_return(DEBUG1, "FEFnManifest type-m", ContextPop(cxt, TRUE))
  }

  if( DEBUG1 )
    fprintf(stderr, "  FEFnManifest completed interface %s%s\n",
      NameShow(fun->name), FEFnSignatureShow(fun, cxt));

  ** optional body **
  if( body != NULL )
  {
    ** manifest body **
    copy_body = body;
    if( !ExprManifest(&copy_body, cxt, self_type, encl_befn) )
      db_return(DEBUG1, "FEFnManifest body", ContextPop(cxt, TRUE));
    assert(copy_body == body);

    ** utilize body type if have no function result type, else just check **
    TypeRangeMarkBegin();
    if( ExprType(body) != NULL )
    {
      if( fun->result_type == NULL )
	fun->result_type = ExprType(body);
      else
      {
	if( !TypeIsSubType(ExprType(body), fun->result_type, &c, cxt) )
	{
	  fprintf(stderr,
	    "%s: type of function body (%s) does not match result type (%s)\n",
	    FilePosShow(ExprFilePos(body)), TypeShow(ExprType(body), cxt),
	    TypeShow(fun->result_type, cxt));
	  TypeRangeMarkEnd(FALSE);
	  if( DEBUG1 )
	  {
	    fprintf(stderr, "  ExprType(body): %s\n",
	      TypeShow(ExprType(body), cxt)); 
	    fprintf(stderr, "  funct->result_type: %s\n",
	      TypeShow(fun->result_type, cxt)); 
	  }
	  db_return(DEBUG1, "FEFnManifest body-s", ContextPop(cxt, TRUE))
	}
	if( c != NULL )
	  ExprFnHeadInsertCoercion(body, c, cxt, self_type);
	  ** CoercionInsert(c, body, cxt, self_type); **
      }
    }
    TypeRangeMarkEnd(TRUE);
  }

  ** all done successfully **
  ContextPop(cxt, TRUE);
  db_return(DEBUG1, "FEFnManifest", TRUE)
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnAddHiddenParameter(FEFN fefn, NAMED *named)                  */
/*                                                                           */
/*  Add a hidden parameter to fefn, corresponding to the given named        */
/*  object, and change *named to the new hidden parameter.  Return TRUE      */
/*  if successful - FALSE can only mean an attempt to insert a hidden        */
/*  parameter after a call on the function has already been manifested.      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnAddHiddenParameter(FEFN fefn, NAMED *named)
{
  TYPE type;  FEFN_PARAM np_param;  BEFN_PARAM befn_param;
  BEFN_FEATURE cfeature;
  if( DEBUG3 )
    fprintf(stderr, "[ FEFnAddHiddenParameter(%s%s, \"%s\" %s)\n",
      NameShow(FEFnName(fefn)), FEFnSignatureShow(fefn, NULL),
      KindShow(NamedKind(*named)), NameShow(NamedName(*named)));

  /* make sure that a call on fefn has not already occurred */
  if( BEFnUtilized(fefn->befn) )
  {
    fprintf(stderr, "%s: recursive letdef uses outer variables: simplify!\n",
      FilePosShow(fefn->file_pos));
    return FALSE;
  }

  /* find out which creation feature this is related to, if any */
  cfeature = NULL;
  if( NamedKind(*named) == KIND_FEFN_PARAM )
  {
    befn_param = FEFnParamBEFnParam((FEFN_PARAM) *named);
    if( BEFnParamHasCreationFeature(befn_param) )
      cfeature = BEFnParamCreationFeature(befn_param);
  }

  /* make and add a Nonpareil hidden parameter */
  type = FEFnResultType((FEFN) *named);
  np_param = FEFnParamNew(NamedFilePos(*named), NamedName(*named), type,
    NULL, PARAM_HIDDEN, FALSE, (FEFN) *named);
  FEFnAddParameter((FEFN) fefn, np_param);

  /* make and add a C hidden parameter, and point np_param to befn_param */
  befn_param = BEFnParamNew(PARAM_HIDDEN, np_param, cfeature);
  BEFnAddParameter(fefn->befn, befn_param);
  FEFnParamSetBEFnParam(np_param, befn_param);
  if( DEBUG3 )
    fprintf(stderr, "] FEFnAddHiddenParameter returning %s%s\n",
      NameShow(FEFnName((FEFN) np_param)),
      FEFnSignatureShow((FEFN) np_param, NULL));
  *named = (NAMED) np_param;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnSignature(FEFN fun, TYPE_VARS *sig_vars,                        */
/*    ARRAY_FEFN_PARAM *sig_params, TYPE *sig_result_type)                   */
/*                                                                           */
/*  Set the last three parameters to the signature of fun.  Copy it,         */
/*  don't share it.                                                          */
/*                                                                           */
/*****************************************************************************/

void FEFnSignature(FEFN fun, TYPE_VARS *sig_vars,
  ARRAY_FEFN_PARAM *sig_params, TYPE *sig_result_type)
{
  *sig_vars = TypeVarsCopy(fun->type_vars, fun->file_pos);
  TypeVarsForwardBegin(fun->type_vars, *sig_vars, fun->file_pos);
  *sig_params = FEFnParamListCopy(fun->parameters);
  *sig_result_type = fun->result_type==NULL ? NULL : TypeCopy(fun->result_type);
  TypeVarsForwardEnd(fun->type_vars);
  if( DEBUG5 )
    fprintf(stderr, "FEFnSignature returning %s\n",
      FEFnSignatureShow(fun, NULL));
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE FEFnCallResultType(ARRAY_FEFN_PARAM sig_params,                     */
/*    TYPE sig_result_type, int i, FILE_POS file_pos)                        */
/*                                                                           */
/*  Return the result type of a call with these formal parameters and        */
/*  result type, for which i actual compulsory parameters were given.        */
/*  This will be a function from the remaining compulsory parameters         */
/*  to the result type, or just the result type if no remaining parameters.  */
/*                                                                           */
/*****************************************************************************/

TYPE FEFnCallResultType(ARRAY_FEFN_PARAM sig_params,
  TYPE sig_result_type, int i, FILE_POS file_pos)
{
  int j, ccount;  ARRAY_TYPE ptypes;  TYPE res;  FEFN_PARAM param;

  /* set ptypes to the types of any leftover compulsory parameters */
  ccount = 0;
  ptypes = NULL;
  if( sig_params != NULL )
    for( j = 0;  j < ArraySize(sig_params);  j++ )
    {
      param = ArrayGet(sig_params, j);
      if( !FEFnParamIsOptional(param) )
      {
	if( ccount < i )
	  ccount++;
	else
	{
	  if( ptypes == NULL )
	    ArrayInit(&ptypes);
	  ArrayAddLast(ptypes, FEFnParamType(param));
	}
      }
    }

  /* find the result type depending on whether there were leftovers or not */
  if( ptypes == NULL )
    res = sig_result_type;
  else
  {
    ArrayAddLast(ptypes, sig_result_type);
    res = TypeMakeFunctionType(ptypes, file_pos);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  UTF8 FEFnSignatureShow(FEFN fun, CONTEXT cxt)                            */
/*                                                                           */
/*  Show the signature of fun in cxt.                                        */
/*                                                                           */
/*****************************************************************************/

UTF8 FEFnSignatureShow(FEFN fun, CONTEXT cxt)
{
  static unsigned char buff[800];
  sprintf((char *) buff, "%s%s: %s", TypeVarsShow(fun->type_vars, cxt),
    FEFnParamListShow(fun->parameters, cxt),
    fun->result_type==NULL ? (UTF8) "(null)" : TypeShow(fun->result_type,cxt));
  return buff;
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_FEFN_PARAM FEFnParams(FEFN fun)                                    */
/*                                                                           */
/*  Return the parameters of fun.                                            */
/*                                                                           */
/*****************************************************************************/

ARRAY_FEFN_PARAM FEFnParams(FEFN fun)
{
  return fun->parameters;
}


/*****************************************************************************/
/*                                                                           */
/*  int FEFnParamsCount(FEFN fun)                                            */
/*                                                                           */
/*  Return the number of parameters of fun.                                  */
/*                                                                           */
/*****************************************************************************/

int FEFnParamsCount(FEFN fun)
{
  return fun->parameters == NULL ? 0 : ArraySize(fun->parameters);
}


/*****************************************************************************/
/*                                                                           */
/*  int FEFnCompulsoryParamsCount(FEFN fun)                                  */
/*                                                                           */
/*  Return the number of compulsory parameters in function fun.              */
/*                                                                           */
/*****************************************************************************/

int FEFnCompulsoryParamsCount(FEFN fun)
{
  int i, res;  FEFN_PARAM param;
  res = 0;
  if( fun->parameters != NULL )
    for( i = 0;  i < ArraySize(fun->parameters);  i++ )
    {
      param = ArrayGet(fun->parameters, i);
      if( !FEFnParamIsOptional(param) )
	res++;
    }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_PARAM FEFnSelfParam(FEFN fun)                                       */
/*                                                                           */
/*  Return the "self" parameter of fun.  It is an error if there isn't one.  */
/*                                                                           */
/*****************************************************************************/

FEFN_PARAM FEFnSelfParam(FEFN fun)
{
  FEFN_PARAM param;  int i;
  if( fun->parameters != NULL )
    for( i = 0;  i < ArraySize(fun->parameters);  i++ )
    {
      param = ArrayGet(fun->parameters, i);
      if( FEFnParamKind(param) == PARAM_SELF )
	return param;
    }
  assert(FALSE);
  return NULL;  /* keep compiler happy */
}
