/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         builtin.c                                                  */
/*  DESCRIPTION:  Nonpareil builtin expressions                              */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0


/*****************************************************************************/
/*                                                                           */
/*  FEFN_BUILTIN - a builtin function                                        */
/*                                                                           */
/*****************************************************************************/

struct fefn_builtin_rec {		/* inherits from FEFN                */
  KIND_TAG		kind_tag;	/* kind of entity                    */
  FILE_POS		file_pos;	/* not very relevant to builtins     */
  NAME			name;		/* name of function                  */
  TYPE_VARS		type_vars;	/* generic parameters                */
  ARRAY_FEFN_PARAM	parameters;	/* ordinary parameters               */
  TYPE			result_type;	/* result type                       */
  BEFN_BUILTIN		befn_builtin;	/* corresponding back-end function   */

  CLASS_VIEW		class_view;	/* non-NULL only in enum functions   */
};

typedef	ARRAY(FEFN_BUILTIN)		ARRAY_FEFN_BUILTIN;
typedef SYMTAB(ARRAY_FEFN_BUILTIN)	SYMTAB_FEFN_BUILTIN;


/*****************************************************************************/
/*                                                                           */
/*  Exported variables                                                       */
/*                                                                           */
/*****************************************************************************/

FEFN_BUILTIN BuiltinAssertFn = NULL;


/*****************************************************************************/
/*                                                                           */
/*  Module variables                                                         */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN		  initialized;	/* TRUE when module initialized      */
static SYMTAB_FEFN_BUILTIN  info_tab;	/* info about builtin operations     */
static ARRAY_FEFN_BUILTIN   info_array;	/* info_tab in order inserted        */


/*****************************************************************************/
/*                                                                           */
/*  FEFN_BUILTIN FEFnBuiltinNew(FILE_POS file_pos, NAME name,                */
/*    TYPE result_type, CODEGEN_OBJ be_obj)                                  */
/*                                                                           */
/*  Return a new FEFN_BUILTIN object with these attributes.                  */
/*                                                                           */
/*****************************************************************************/

static FEFN_BUILTIN FEFnBuiltinNew(FILE_POS file_pos, NAME name,
  TYPE result_type, CODEGEN_OBJ be_obj)
{
  FEFN_BUILTIN res;
  GetMemory(res, FEFN_BUILTIN);
  res->kind_tag = KIND_FEFN_BUILTIN;
  res->file_pos = file_pos;
  res->name = name;
  res->type_vars = NULL;
  res->parameters = NULL;
  res->result_type = result_type;
  res->befn_builtin = BEFnBuiltinNew(res, be_obj);
  res->class_view = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void AddParam(FEFN_BUILTIN bf, FILE_POS file_pos, ASTRING name,          */
/*    TYPE type)                                                             */
/*                                                                           */
/*  Add a compulsory parameter with this name and type to bf, as well as a   */
/*  corresponding C parameter to bf->befn_builtin.                           */
/*                                                                           */
/*****************************************************************************/

static void AddParam(FEFN_BUILTIN bf, FILE_POS file_pos, ASTRING name,
  TYPE type)
{
  FEFN_PARAM np_param;  BEFN_PARAM c_param;

  /* make the Nonpareil parameter and add it to bf */
  np_param = FEFnParamNew(file_pos, NameNew(AStringToUString(name)),
    type, NULL, PARAM_COMPULSORY, FALSE, NULL);
  FEFnAddParameter((FEFN) bf, np_param);

  /* make the C parameter and add it to the befn_builtin */
  c_param = BEFnParamNew(PARAM_COMPULSORY, np_param, NULL);
  BEFnAddParameter((BEFN) bf->befn_builtin, c_param);

  /* make np_param refer to c_param */
  FEFnParamSetBEFnParam(np_param, c_param);
}


/*****************************************************************************/
/*                                                                           */
/* void FEFnBuiltinInsertIntToEnum(FILE_POS file_pos, USTRING str,           */
/*  CLASS_VIEW cv)                                                           */
/*                                                                           */
/*  Insert a builtin function with signature "str(p1: int): cv" whose        */
/*  backend function is just a type cast, at least for now.                  */
/*                                                                           */
/*****************************************************************************/

void FEFnBuiltinInsertIntToEnum(FILE_POS file_pos, USTRING str, CLASS_VIEW cv)
{
  FEFN_BUILTIN bf, bf2;
  assert(initialized);
  if( DEBUG3 )
    fprintf(stderr, "[ FEFnBuiltinInsertIntToEnum(%s, %s, %s)\n",
      FilePosShow(file_pos), UStringToUTF8(str), NameShow(ClassViewName(cv)));
  bf = FEFnBuiltinNew(file_pos, NameNew(str), ClassType(ClassViewClass(cv)),
    NULL);
  bf->class_view = cv;
  AddParam(bf, file_pos, "p1", ClassType(ClassInt));
  if( !SymInsert(info_tab, str, bf, &bf2) )
    assert(FALSE);
  ArrayAddLast(info_array, bf);
  if( DEBUG3 )
    fprintf(stderr, "] FEFnBuiltinInsertIntToEnum returning\n");
}


/*****************************************************************************/
/*                                                                           */
/* void FEFnBuiltinInsertEnumToInt(FILE_POS file_pos, USTRING str,           */
/*    CLASS_VIEW cv)                                                         */
/*                                                                           */
/*  Insert a builtin function with signature "str(p1: cv): int" whose        */
/*  backend function is just a type cast, and will be inserted later.        */
/*                                                                           */
/*****************************************************************************/

void FEFnBuiltinInsertEnumToInt(FILE_POS file_pos, USTRING str, CLASS_VIEW cv)
{
  FEFN_BUILTIN bf, bf2;
  assert(initialized);
  if( DEBUG3 )
    fprintf(stderr, "[ FEFnBuiltinInsertEnumToInt(%s, %s, %s)\n",
      FilePosShow(file_pos), UStringToUTF8(str), NameShow(ClassViewName(cv)));
  bf = FEFnBuiltinNew(file_pos, NameNew(str), ClassType(ClassInt), NULL);
  bf->class_view = cv;
  AddParam(bf, file_pos, "p1", ClassType(ClassViewClass(cv)));
  if( !SymInsert(info_tab, str, bf, &bf2) )
    assert(FALSE);
  ArrayAddLast(info_array, bf);
  if( DEBUG3 )
    fprintf(stderr, "] FEFnBuiltinInsertEnumToInt returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void set0(ASTRING key, CLASS res, FILE_POS file_pos, CODEGEN_OBJ be_obj) */
/*                                                                           */
/*  Create a new builtin function with no parameters, and result type res.   */
/*                                                                           */
/*****************************************************************************/

static void set0(ASTRING key, CLASS res, FILE_POS file_pos, CODEGEN_OBJ be_obj)
{
  FEFN_BUILTIN bf, bf2;  USTRING str;
  str = AStringToUString(key);
  bf = FEFnBuiltinNew(file_pos, NameNew(str), ClassType(res), be_obj);
  SymInsert(info_tab, str, bf, &bf2);
  ArrayAddLast(info_array, bf);
}


/*****************************************************************************/
/*                                                                           */
/*  void set1(ASTRING key, CLASS p1, CLASS res, FILE_POS file_pos,           */
/*    CODEGEN_OBJ be_obj)                                                    */
/*                                                                           */
/*  Create a new builtin function with one paramater of type p1, and         */
/*  result type res.                                                         */
/*                                                                           */
/*****************************************************************************/

static void set1(ASTRING key, CLASS p1, CLASS res, FILE_POS file_pos,
  CODEGEN_OBJ be_obj)
{
  FEFN_BUILTIN bf, bf2;  USTRING str;
  str = AStringToUString(key);
  bf = FEFnBuiltinNew(file_pos, NameNew(str), ClassType(res), be_obj);
  AddParam(bf, file_pos, "p1", ClassType(p1));
  SymInsert(info_tab, str, bf, &bf2);
  ArrayAddLast(info_array, bf);
}


/*****************************************************************************/
/*                                                                           */
/*  void set2(ASTRING key, CLASS p1, CLASS p2, CLASS res, FILE_POS file_pos, */
/*    CODEGEN_OBJ be_obj)                                                    */
/*                                                                           */
/*  Create a new builtin function with these two parameter types             */
/*  and this result type.                                                    */
/*                                                                           */
/*****************************************************************************/

static void set2(ASTRING key, CLASS p1, CLASS p2, CLASS res, FILE_POS file_pos,
  CODEGEN_OBJ be_obj)
{
  FEFN_BUILTIN bf, bf2;  USTRING str;
  str = AStringToUString(key);
  bf = FEFnBuiltinNew(file_pos, NameNew(str), ClassType(res), be_obj);
  AddParam(bf, file_pos, "p1", ClassType(p1));
  AddParam(bf, file_pos, "p2", ClassType(p2));
  SymInsert(info_tab, str, bf, &bf2);
  ArrayAddLast(info_array, bf);
}


/*****************************************************************************/
/*                                                                           */
/*  void set2gen(ASTRING key, CLASS res, FILE_POS file_pos,                  */
/*    CODEGEN_OBJ be_obj)                                                    */
/*                                                                           */
/*  Create a new builtin function                                            */
/*                                                                           */
/*     builtin "key"(p1, p2: X): res                                         */
/*                                                                           */
/*  which calls be_obj.                                                      */
/*                                                                           */
/*****************************************************************************/

static void set2gen(ASTRING key, CLASS res, FILE_POS file_pos,
  CODEGEN_OBJ be_obj)
{
  FEFN_BUILTIN bf, bf2;  USTRING str;
  TYPE_VAR tv;  TYPE t;

  /* builtin function object */
  str = AStringToUString(key);
  bf = FEFnBuiltinNew(file_pos, NameNew(str), ClassType(res), be_obj);

  /* type vars: {X} */
  tv = TypeVarMake(file_pos, NameNew(AStringToUString("X")));
  ArrayInit(&bf->type_vars);
  ArrayAddLast(bf->type_vars, tv);

  /* p1 and p2: X */
  t = TypeMakeFromVar(file_pos, tv);
  AddParam(bf, file_pos, "p1", t);
  AddParam(bf, file_pos, "p2", t);

  /* finalize C function and insert into info_tab */
  SymInsert(info_tab, str, bf, &bf2);
  ArrayAddLast(info_array, bf);
}


/*****************************************************************************/
/*                                                                           */
/*  void set_array_length(ASTRING key, FILE_POS file_pos,                    */
/*    CODEGEN_OBJ be_obj)                                                    */
/*                                                                           */
/*  Create a new builtin function for the length call in the array class:    */
/*                                                                           */
/*    length: int := builtin "array_length"(self)                            */
/*                                                                           */
/*****************************************************************************/

static void set_array_length(ASTRING key, FILE_POS file_pos,
  CODEGEN_OBJ be_obj)
{
  FEFN_BUILTIN bf, bf2;  USTRING str;
  TYPE_VAR tv;  TYPE_VARS tvs;  TYPE t;

  /* builtin function object */
  str = AStringToUString(key);
  bf = FEFnBuiltinNew(file_pos, NameNew(str), ClassType(ClassInt), be_obj);

  /* type vars: {X} */
  ArrayInit(&bf->type_vars);
  tv = TypeVarMake(file_pos, NameNew(AStringToUString("X")));
  ArrayAddLast(bf->type_vars, tv);

  /* t: array{X} */
  ArrayInit(&tvs);
  ArrayAddLast(tvs, tv);
  t = TypeMakeInstantiatedClassType(file_pos, ClassArray, tvs);

  /* parameter */
  AddParam(bf, file_pos, "p1", t);

  /* finalize C function and insert into info_tab */
  SymInsert(info_tab, str, bf, &bf2);
  ArrayAddLast(info_array, bf);
}


/*****************************************************************************/
/*                                                                           */
/*  void set_array_get(ASTRING key, FILE_POS file_pos, CODEGEN_OBJ be_obj)   */
/*                                                                           */
/*  Create a new builtin function for the apply1 call in the array class:    */
/*                                                                           */
/*    apply1(i:int): X := builtin "array_get"(self, i)                       */
/*                                                                           */
/*****************************************************************************/

static void set_array_get(ASTRING key, FILE_POS file_pos, CODEGEN_OBJ be_obj)
{
  FEFN_BUILTIN bf, bf2;  USTRING str;
  TYPE_VAR tv;  TYPE_VARS tvs;  TYPE t1;

  /* builtin function object with type vars {X} and result type X */
  tv = TypeVarMake(file_pos, NameNew(AStringToUString("X")));
  str = AStringToUString(key);
  bf = FEFnBuiltinNew(file_pos, NameNew(str), TypeMakeFromVar(file_pos, tv),
    be_obj);
  ArrayInit(&bf->type_vars);
  ArrayAddLast(bf->type_vars, tv);

  /* p1: array_rep{X} */
  ArrayInit(&tvs);
  ArrayAddLast(tvs, tv);
  t1 = TypeMakeInstantiatedClassType(file_pos, ClassArray, tvs);
  AddParam(bf, file_pos, "p1", t1);

  /* p2: int */
  AddParam(bf, file_pos, "p2", ClassType(ClassInt));

  /* finalize C function and insert into info_tab */
  SymInsert(info_tab, str, bf, &bf2);
  ArrayAddLast(info_array, bf);
}


/*****************************************************************************/
/*                                                                           */
/*  void set_funrep_call(int n, FILE_POS file_pos, CODEGEN be)               */
/*                                                                           */
/*  Create a new builtin function for the call in the funrep-n class; e.g.   */
/*                                                                           */
/*    call(x1: X1, x2: X2): T := builtin "fun2_rep_call"(self, x1, x2)       */
/*                                                                           */
/*****************************************************************************/

static void set_funrep_call(int n, FILE_POS file_pos, CODEGEN be)
{
  FEFN_BUILTIN bf, bf2;  TYPE_VAR tv, tt;  int i;
  char key[20], buff[20];  USTRING str;

  /* funnrep_call: T */
  sprintf(key, "fun%d_rep_call", n);
  str = AStringToUString(key);
  tt = TypeVarMake(file_pos, NameNew(AStringToUString("T")));
  bf = FEFnBuiltinNew(file_pos, NameNew(str), TypeMakeFromVar(file_pos, tt),
    be->FunctionMakeFunRef(AStringCopy(key)));

  /* add type vars (X1, X2, ... Xn, T) to call */
  ArrayInit(&bf->type_vars);
  for( i = 0;  i < n;  i++ )
  {
    sprintf(buff, "X%d", i+1);
    tv = TypeVarMake(file_pos, NameNew(AStringToUString(buff)));
    ArrayAddLast(bf->type_vars, tv);
  }
  ArrayAddLast(bf->type_vars, tt);

  /* add self parameter to call */
  AddParam(bf, file_pos, "p0", ClassType(ClassFunRep[n]));

  /* add other parameters to cf */
  for( i = 0;  i < n;  i++ )
  {
    tv = ArrayGet(bf->type_vars, i);
    sprintf(buff, "p%d", i+1);
    AddParam(bf, file_pos, AStringCopy(buff), TypeMakeFromVar(file_pos, tv));
  }

  /* finalize C function and insert into info_tab */
  SymInsert(info_tab, str, bf, &bf2);
  ArrayAddLast(info_array, bf);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnBuiltinInitialize(CODEGEN be)                                   */
/*                                                                           */
/*  Initialize this module - to be done before anything else basically.      */
/*                                                                           */
/*  It is not our usual policy to pass the back end during initialization,   */
/*  but here it is used only to access backend builtin functions, such as    */
/*  be->add, be->array_get, etc.  It's convenient and harmless.              */
/*                                                                           */
/*****************************************************************************/

void FEFnBuiltinInitialize(CODEGEN be)
{
  SymInit(&info_tab);
  ArrayInit(&info_array);
  initialized = TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnBuiltinInitReal(FILE_POS pos, CODEGEN be)                       */
/*                                                                           */
/*  Initialize the builtin functions required by the real class.             */
/*                                                                           */
/*****************************************************************************/

static void FEFnBuiltinInitReal(FILE_POS pos, CODEGEN be)
{
  /* arithmetic operations */
  set2("real_add", ClassReal, ClassReal, ClassReal, pos, be->add);
  set2("real_sub", ClassReal, ClassReal, ClassReal, pos, be->subtract);
  set1("real_neg", ClassReal, ClassReal, pos, be->unary_minus);
  set2("real_mul", ClassReal, ClassReal, ClassReal, pos, be->multiply);
  set2("real_div", ClassReal, ClassReal, ClassReal, pos, be->divide);
  set1("real_trunc", ClassReal, ClassInt, pos, be->cast_to_int);

  /* comparisons */
  set2("real_eq", ClassReal, ClassReal, ClassBool, pos, be->eq);
  set2("real_ne", ClassReal, ClassReal, ClassBool, pos, be->ne);
  set2("real_lt", ClassReal, ClassReal, ClassBool, pos, be->lt);
  set2("real_le", ClassReal, ClassReal, ClassBool, pos, be->le);
  set2("real_gt", ClassReal, ClassReal, ClassBool, pos, be->gt);
  set2("real_ge", ClassReal, ClassReal, ClassBool, pos, be->ge);

  /* mathematical functions */
  set1("real_sin", ClassReal, ClassReal, pos,
    be->FunctionMake("sin", TRUE, TRUE));
  set1("real_cos", ClassReal, ClassReal, pos,
    be->FunctionMake("cos", TRUE, TRUE));
  set1("real_tan", ClassReal, ClassReal, pos,
    be->FunctionMake("tan", TRUE, TRUE));
  set1("real_asin", ClassReal, ClassReal, pos, 
    be->FunctionMake("asin", TRUE, TRUE));
  set1("real_acos", ClassReal, ClassReal, pos,
    be->FunctionMake("acos", TRUE, TRUE));
  set1("real_atan", ClassReal, ClassReal, pos,
    be->FunctionMake("atan", TRUE, TRUE));
  set2("real_atan2", ClassReal, ClassReal, ClassReal, pos,
    be->FunctionMake("atan2", TRUE, TRUE));
  set1("real_sinh", ClassReal, ClassReal, pos,
    be->FunctionMake("sinh", TRUE, TRUE));
  set1("real_cosh", ClassReal, ClassReal, pos,
    be->FunctionMake("cosh", TRUE, TRUE));
  set1("real_tanh", ClassReal, ClassReal, pos,
    be->FunctionMake("tanh", TRUE, TRUE));
  set1("real_exp", ClassReal, ClassReal, pos,
    be->FunctionMake("exp", TRUE, TRUE));
  set1("real_log", ClassReal, ClassReal, pos,
    be->FunctionMake("log", TRUE, TRUE));
  set1("real_log10", ClassReal, ClassReal, pos,
    be->FunctionMake("log10", TRUE, TRUE));
  set2("real_pow", ClassReal, ClassReal, ClassReal, pos,
    be->FunctionMake("pow", TRUE, TRUE));
  set1("real_sqrt", ClassReal, ClassReal, pos,
    be->FunctionMake("sqrt", TRUE, TRUE));
  set1("real_ceil", ClassReal, ClassReal, pos,
    be->FunctionMake("ceil", TRUE, TRUE));
  set1("real_floor", ClassReal, ClassReal, pos,
    be->FunctionMake("floor", TRUE, TRUE));
  set1("real_fabs", ClassReal, ClassReal, pos,
    be->FunctionMake("fabs", TRUE, TRUE));
  set2("real_ldexp", ClassReal, ClassInt, ClassReal, pos,
    be->FunctionMake("ldexp", TRUE, TRUE));
  set2("real_fmod", ClassReal, ClassReal, ClassReal, pos,
    be->FunctionMake("fmod", TRUE, TRUE));

  /* predefined object features */
  set0("real_max", ClassReal, pos, be->float_max);
  set0("real_min", ClassReal, pos, be->float_min);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnBuiltinInitInt(FILE_POS pos, CODEGEN be)                        */
/*                                                                           */
/*  Initialize the builtin functions required by the int class.              */
/*                                                                           */
/*****************************************************************************/

static void FEFnBuiltinInitInt(FILE_POS pos, CODEGEN be)
{
  /* arithmetic operations */
  set2("int_add", ClassInt, ClassInt, ClassInt, pos, be->add);
  set2("int_sub", ClassInt, ClassInt, ClassInt, pos, be->subtract);
  set1("int_neg", ClassInt, ClassInt, pos, be->unary_minus);
  set2("int_mul", ClassInt, ClassInt, ClassInt, pos, be->multiply);
  set2("int_div", ClassInt, ClassInt, ClassInt, pos, be->divide);
  set2("int_mod", ClassInt, ClassInt, ClassInt, pos, be->mod);

  /* comparisons */
  set2("int_eq", ClassInt, ClassInt, ClassBool, pos, be->eq);
  set2("int_ne", ClassInt, ClassInt, ClassBool, pos, be->ne);
  set2("int_lt", ClassInt, ClassInt, ClassBool, pos, be->lt);
  set2("int_le", ClassInt, ClassInt, ClassBool, pos, be->le);
  set2("int_gt", ClassInt, ClassInt, ClassBool, pos, be->gt);
  set2("int_ge", ClassInt, ClassInt, ClassBool, pos, be->ge);

  /* coercions and conversions */
  set1("int_to_real", ClassInt, ClassReal, pos, be->cast_to_float);
  set1("int_to_uint", ClassInt, ClassUInt, pos, be->cast_to_uint);
  set1("int_to_ushort", ClassInt, ClassUShort, pos, be->cast_to_ushort);
  set1("int_to_ubyte", ClassInt, ClassUByte, pos, be->cast_to_uchar);
  set1("int_to_short", ClassInt, ClassShort, pos, be->cast_to_short);
  set1("int_to_byte", ClassInt, ClassByte, pos, be->cast_to_char);

  /* predefined object features */
  set0("int_min", ClassInt, pos, be->int_min);
  set0("int_max", ClassInt, pos, be->int_max);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnBuiltinInitShort(FILE_POS pos, CODEGEN be)                      */
/*                                                                           */
/*  Initialize the builtin functions required by the short class.            */
/*                                                                           */
/*****************************************************************************/

static void FEFnBuiltinInitShort(FILE_POS pos, CODEGEN be)
{
  /* arithmetic operations */
  set2("short_add", ClassShort, ClassShort, ClassShort, pos, be->add);
  set2("short_sub", ClassShort, ClassShort, ClassShort, pos, be->subtract);
  set1("short_neg", ClassShort, ClassShort, pos, be->unary_minus);
  set2("short_mul", ClassShort, ClassShort, ClassShort, pos, be->multiply);
  set2("short_div", ClassShort, ClassShort, ClassShort, pos, be->divide);
  set2("short_mod", ClassShort, ClassShort, ClassShort, pos, be->mod);

  /* comparisons */
  set2("short_eq", ClassShort, ClassShort, ClassBool, pos, be->eq);
  set2("short_ne", ClassShort, ClassShort, ClassBool, pos, be->ne);
  set2("short_lt", ClassShort, ClassShort, ClassBool, pos, be->lt);
  set2("short_le", ClassShort, ClassShort, ClassBool, pos, be->le);
  set2("short_gt", ClassShort, ClassShort, ClassBool, pos, be->gt);
  set2("short_ge", ClassShort, ClassShort, ClassBool, pos, be->ge);

  /* coercions and conversions */
  set1("short_to_int", ClassShort, ClassInt, pos, be->cast_to_int);
  set1("short_to_uint", ClassShort, ClassUInt, pos, be->cast_to_uint);
  set1("short_to_ushort", ClassShort, ClassUShort, pos, be->cast_to_ushort);
  set1("short_to_ubyte", ClassShort, ClassUByte, pos,  be->cast_to_uchar);
  set1("short_to_byte", ClassShort, ClassByte, pos,  be->cast_to_char);

  /* predefined object features */
  set0("short_min", ClassShort, pos, be->short_min);
  set0("short_max", ClassShort, pos, be->short_max);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnBuiltinInitByte(FILE_POS pos, CODEGEN be)                       */
/*                                                                           */
/*  Initialize the builtin functions required by the byte class.             */
/*                                                                           */
/*****************************************************************************/

static void FEFnBuiltinInitByte(FILE_POS pos, CODEGEN be)
{
  /* arithmetic operations */
  set2("byte_add", ClassByte, ClassByte, ClassByte, pos, be->add);
  set2("byte_sub", ClassByte, ClassByte, ClassByte, pos, be->subtract);
  set1("byte_neg", ClassByte, ClassByte, pos, be->unary_minus);
  set2("byte_mul", ClassByte, ClassByte, ClassByte, pos, be->multiply);
  set2("byte_div", ClassByte, ClassByte, ClassByte, pos, be->divide);
  set2("byte_mod", ClassByte, ClassByte, ClassByte, pos, be->mod);

  /* comparisons */
  set2("byte_eq", ClassByte, ClassByte, ClassBool, pos, be->eq);
  set2("byte_ne", ClassByte, ClassByte, ClassBool, pos, be->ne);
  set2("byte_lt", ClassByte, ClassByte, ClassBool, pos, be->lt);
  set2("byte_le", ClassByte, ClassByte, ClassBool, pos, be->le);
  set2("byte_gt", ClassByte, ClassByte, ClassBool, pos, be->gt);
  set2("byte_ge", ClassByte, ClassByte, ClassBool, pos, be->ge);

  /* coercions and conversions */
  set1("byte_to_short", ClassByte, ClassShort, pos, be->cast_to_short);
  set1("byte_to_uint", ClassByte, ClassUInt, pos, be->cast_to_uint);
  set1("byte_to_ushort", ClassByte, ClassUShort, pos, be->cast_to_ushort);
  set1("byte_to_ubyte", ClassByte, ClassUByte, pos, be->cast_to_uchar);
  set1("byte_to_int", ClassByte, ClassInt, pos, be->cast_to_int);

  /* predefined object features */
  set0("byte_min", ClassByte, pos, be->char_min);
  set0("byte_max", ClassByte, pos, be->char_max);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnBuiltinInitUInt(FILE_POS pos, CODEGEN be)                       */
/*                                                                           */
/*  Initialize the builtin functions required by the uint class.             */
/*                                                                           */
/*****************************************************************************/

static void FEFnBuiltinInitUInt(FILE_POS pos, CODEGEN be)
{
  /* arithmetic operations */
  set2("uint_add", ClassUInt, ClassUInt, ClassUInt, pos, be->add);
  set2("uint_sub", ClassUInt, ClassUInt, ClassUInt, pos, be->subtract);
  set2("uint_mul", ClassUInt, ClassUInt, ClassUInt, pos, be->multiply);
  set2("uint_div", ClassUInt, ClassUInt, ClassUInt, pos, be->divide);
  set2("uint_mod", ClassUInt, ClassUInt, ClassUInt, pos, be->mod);

  /* comparisons */
  set2("uint_eq", ClassUInt, ClassUInt, ClassBool, pos, be->eq);
  set2("uint_ne", ClassUInt, ClassUInt, ClassBool, pos, be->ne);
  set2("uint_lt", ClassUInt, ClassUInt, ClassBool, pos, be->lt);
  set2("uint_le", ClassUInt, ClassUInt, ClassBool, pos, be->le);
  set2("uint_gt", ClassUInt, ClassUInt, ClassBool, pos, be->gt);
  set2("uint_ge", ClassUInt, ClassUInt, ClassBool, pos, be->ge);

  /* coercions and conversions */
  set1("uint_to_real", ClassUInt, ClassReal, pos, be->cast_to_float);
  set1("uint_to_int", ClassUInt, ClassInt, pos, be->cast_to_int);
  set1("uint_to_ushort", ClassUInt, ClassUShort, pos, be->cast_to_ushort);
  set1("uint_to_ubyte", ClassUInt, ClassUByte, pos, be->cast_to_uchar);
  set1("uint_to_short", ClassUInt, ClassShort, pos, be->cast_to_short);
  set1("uint_to_byte", ClassUInt, ClassByte, pos, be->cast_to_char);

  /* predefined object features */
  set0("uint_max", ClassUInt, pos, be->uint_max);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnBuiltinInitUShort(FILE_POS pos, CODEGEN be)                     */
/*                                                                           */
/*  Initialize the builtin functions required by the ushort class.           */
/*                                                                           */
/*****************************************************************************/

static void FEFnBuiltinInitUShort(FILE_POS pos, CODEGEN be)
{
  /* arithmetic operations */
  set2("ushort_add", ClassUShort, ClassUShort, ClassUShort, pos, be->add);
  set2("ushort_sub", ClassUShort, ClassUShort, ClassUShort, pos, be->subtract);
  set2("ushort_mul", ClassUShort, ClassUShort, ClassUShort, pos, be->multiply);
  set2("ushort_div", ClassUShort, ClassUShort, ClassUShort, pos, be->divide);
  set2("ushort_mod", ClassUShort, ClassUShort, ClassUShort, pos, be->mod);

  /* comparisons */
  set2("ushort_eq", ClassUShort, ClassUShort, ClassBool, pos, be->eq);
  set2("ushort_ne", ClassUShort, ClassUShort, ClassBool, pos, be->ne);
  set2("ushort_lt", ClassUShort, ClassUShort, ClassBool, pos, be->lt);
  set2("ushort_le", ClassUShort, ClassUShort, ClassBool, pos, be->le);
  set2("ushort_gt", ClassUShort, ClassUShort, ClassBool, pos, be->gt);
  set2("ushort_ge", ClassUShort, ClassUShort, ClassBool, pos, be->ge);

  /* coercions and conversions */
  set1("ushort_to_uint", ClassUShort, ClassUInt, pos, be->cast_to_uint);
  set1("ushort_to_ubyte", ClassUShort, ClassUByte, pos, be->cast_to_uchar);
  set1("ushort_to_int", ClassUShort, ClassInt, pos, be->cast_to_int);
  set1("ushort_to_short", ClassUShort, ClassShort, pos, be->cast_to_short);
  set1("ushort_to_byte", ClassUShort, ClassByte, pos, be->cast_to_char);

  /* predefined object features */
  set0("ushort_max", ClassUShort, pos, be->ushort_max);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnBuiltinInitUByte(FILE_POS pos, CODEGEN be)                      */
/*                                                                           */
/*  Initialize the builtin functions required by the ubyte class.            */
/*                                                                           */
/*****************************************************************************/

static void FEFnBuiltinInitUByte(FILE_POS pos, CODEGEN be)
{
  /* arithmetic operations */
  set2("ubyte_add", ClassUByte, ClassUByte, ClassUByte, pos, be->add);
  set2("ubyte_sub", ClassUByte, ClassUByte, ClassUByte, pos, be->subtract);
  set2("ubyte_mul", ClassUByte, ClassUByte, ClassUByte, pos, be->multiply);
  set2("ubyte_div", ClassUByte, ClassUByte, ClassUByte, pos, be->divide);
  set2("ubyte_mod", ClassUByte, ClassUByte, ClassUByte, pos, be->mod);

  /* comparisons */
  set2("ubyte_eq", ClassUByte, ClassUByte, ClassBool, pos, be->eq);
  set2("ubyte_ne", ClassUByte, ClassUByte, ClassBool, pos, be->ne);
  set2("ubyte_lt", ClassUByte, ClassUByte, ClassBool, pos, be->lt);
  set2("ubyte_le", ClassUByte, ClassUByte, ClassBool, pos, be->le);
  set2("ubyte_gt", ClassUByte, ClassUByte, ClassBool, pos, be->gt);
  set2("ubyte_ge", ClassUByte, ClassUByte, ClassBool, pos, be->ge);

  /* coercions and conversions */
  set1("ubyte_to_ushort", ClassUByte, ClassUShort, pos, be->cast_to_ushort);
  set1("ubyte_to_uint", ClassUByte, ClassUInt, pos, be->cast_to_uint);
  set1("ubyte_to_int", ClassUByte, ClassInt, pos, be->cast_to_int);
  set1("ubyte_to_short", ClassUByte, ClassShort, pos, be->cast_to_short);
  set1("ubyte_to_byte", ClassUByte, ClassByte, pos, be->cast_to_char);

  /* predefined object features */
  set0("ubyte_max", ClassUByte, pos, be->uchar_max);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnBuiltinInitFeatures(CODEGEN be)                                 */
/*                                                                           */
/*  Initialize the builtin features of the standard classes.  This call      */
/*  occurs after the standard classes have been established, when ClassReal  */
/*  etc. are all defined.                                                    */
/*                                                                           */
/*****************************************************************************/

void FEFnBuiltinInitFeatures(CODEGEN be)
{
  int n;  FILE_POS pos;

  assert(initialized);
  pos = FilePosNew(AStringToUString("builtin"), 0, 0);

  /* the builtin assert function, not in any list or table */
  BuiltinAssertFn = FEFnBuiltinNew(pos,
    NameNew(AStringToUString("builtin_assert")), ClassType(ClassVoid),
    be->assert_fn);
  AddParam(BuiltinAssertFn, pos, "p1", ClassType(ClassBool));

  /* generic */
  set2gen("eq", ClassBool, pos, be->eq);
  set2gen("ne", ClassBool, pos, be->ne);

  /* object */
  set2("object_eq", ClassObject, ClassObject, ClassBool, pos, be->eq);
  set2("object_ne", ClassObject, ClassObject, ClassBool, pos, be->ne);

  /* array */
  set_array_length("array_length", pos, NPBack_Array_Length);
  set_array_get("array_get", pos, NPBack_Array_Get);

  /* bool */
  set2("bool_and", ClassBool, ClassBool, ClassBool, pos, be->logical_and);
  set2("bool_or", ClassBool, ClassBool, ClassBool, pos, be->logical_or);
  set1("bool_not", ClassBool, ClassBool, pos, be->logical_not);

  /* funn_rep */
  for( n = 1;  n <= MAX_FUN_PARAMS;  n++ )
    set_funrep_call(n, pos, be);

  /* real and integral types */
  FEFnBuiltinInitReal(pos, be);
  FEFnBuiltinInitInt(pos, be);
  FEFnBuiltinInitShort(pos, be);
  FEFnBuiltinInitByte(pos, be);
  FEFnBuiltinInitUInt(pos, be);
  FEFnBuiltinInitUShort(pos, be);
  FEFnBuiltinInitUByte(pos, be);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnBuiltinFind(USTRING str, FILE_POS file_pos, int param_count, */
/*    FEFN_BUILTIN *bif)                                                     */
/*                                                                           */
/*  Find builtin *bif and return TRUE if found and param_count agrees with   */
/*  the number of formal parameters, or else return FALSE.                   */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnBuiltinFind(USTRING str, FILE_POS file_pos, int param_count,
  FEFN_BUILTIN *bif)
{
  int count;

  /* ensure this module is initialized */
  assert(initialized);

  /* look up str in info table */
  if( !SymRetrieve(info_tab, str, bif) )
  {
    fprintf(stderr, "%s: builtin %s not known\n", FilePosShow(file_pos),
      UStringToUTF8(str));
    return FALSE;
  }

  /* check that the number of parameters agrees with signature */
  count = (*bif)->parameters == NULL ? 0 : ArraySize((*bif)->parameters);
  if( param_count != count )
  {
    fprintf(stderr, "%s: expected %d parameters for builtin %s\n",
      FilePosShow(file_pos), ArraySize((*bif)->parameters), UStringToUTF8(str));
    return FALSE;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnBuiltinFinalizeAll(CODEGEN be)                                  */
/*                                                                           */
/*  Finalize all the C functions in the back end, including BuiltinAssertFn  */
/*  which is not in info_array since it's not visible to the end user.       */
/*                                                                           */
/*****************************************************************************/

void FEFnBuiltinFinalizeAll(CODEGEN be)
{
  FEFN_BUILTIN bf;
  ArrayForEach(info_array, bf)
    BEFnFinalize((BEFN) bf->befn_builtin, be);
  BEFnFinalize((BEFN) BuiltinAssertFn->befn_builtin, be);
}
