/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         creation_view.c                                            */
/*  DESCRIPTION:  A view of a creation function                              */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"


struct fefn_creation_rec {		/* inherits from FEFN                */
  KIND_TAG		kind_tag;	/* kind (KIND_CLASS_VIEW)            */
  FILE_POS		file_pos;	/* position of class view in input   */
  NAME			name;		/* name of class view                */
  TYPE_VARS		type_vars;	/* type variables                    */
  ARRAY_FEFN_PARAM	parameters;	/* one for each creation feat. view  */
  TYPE			result_type;	/* equivalent to class type          */
  BEFN_CREATION		creation_fn;	/* the corresponding C function      */

  CLASS_VIEW		orig_cv;	/* the class view holding this cfv   */
};


/*****************************************************************************/
/*                                                                           */
/*  FEFN_CREATION CreationViewNew(FILE_POS file_pos, NAME name,              */
/*    TYPE_VARS type_vars, ARRAY_FEFN_PARAM parameters, TYPE result_type,    */
/*    BEFN_CREATION creation_fn)                                             */
/*                                                                           */
/*  Create a new creation function view with these attributes.               */
/*                                                                           */
/*  Because of circularity, this call might receive NULL values for          */
/*  result_type and creation_fn (these depend on the class that this         */
/*  creation function view is a view of the creation function of, and        */
/*  that class might be not quite in existence at the time of this call).    */
/*  In that case, a call to the following function will quickly follow.      */
/*                                                                           */
/*****************************************************************************/

FEFN_CREATION FEFnCreationNew(FILE_POS file_pos, NAME name,
  TYPE_VARS type_vars, ARRAY_FEFN_PARAM parameters, TYPE result_type,
  CLASS_VIEW orig_cv, BEFN_CREATION creation_fn)
{
  FEFN_CREATION res;
  GetMemory(res, FEFN_CREATION);
  res->kind_tag = KIND_FEFN_CREATION;
  res->file_pos = file_pos;
  res->name = name;
  res->type_vars = type_vars;
  res->parameters = parameters;
  res->result_type = result_type;
  res->creation_fn = creation_fn;
  res->orig_cv = orig_cv;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnCreationSetResultTypeAndBEFnCreation(FEFN_CREATION cfv,         */
/*    TYPE result_type, BEFN_CREATION creation_fn)                           */
/*                                                                           */
/*  Set the result type and creation function fields of cfv.  See the        */
/*  comment to FEFnCreationNew for why this function is needed.              */
/*                                                                           */
/*****************************************************************************/

void FEFnCreationSetResultTypeAndBEFnCreation(FEFN_CREATION cfv,
  TYPE result_type, BEFN_CREATION creation_fn)
{
  cfv->result_type = result_type;
  cfv->creation_fn = creation_fn;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnCreationManifest(FEFN_CREATION cfv, CONTEXT cxt,             */
/*    TYPE self_type)                                                        */
/*                                                                           */
/*  Manifest creation function view cfv, including unimported default        */
/*  values of any parameters.  The context at the moment is just the         */
/*  usual external context, plus the type variables of the enclosing         */
/*  class view, nothing else.  This must be the context upon return.         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnCreationManifest(FEFN_CREATION cfv, CONTEXT cxt,
  TYPE self_type)
{
  FEFN_PARAM param;  FEFN_FEATURE cfeature;  FEFN_CREDFT fefn_credft;
  BOOLEAN success;  FEFN fefn;
  success = TRUE;
  if( cfv->parameters != NULL )
  {
    ContextPushEmpty(cxt, NULL, FALSE);
    ArrayForEach(cfv->parameters, param)
    {
      /* manifest the default value associated with param, if any */
      fefn = FEFnParamHiddenValue(param);
      assert(fefn != NULL);
      assert(NamedKind((NAMED) fefn) == KIND_FEFN_FEATURE);
      cfeature = (FEFN_FEATURE) fefn;
      if( !FEFnFeatureIsImported(cfeature) )
      {
	fefn_credft = FEFnCreationDefaultValue(cfeature);
	if( fefn_credft != NULL )
	{
	  if( !FEFnCreDftManifest(fefn_credft, cxt, self_type) )
	    success = FALSE;
	}
      }

      /* add param to context */
      if( !ContextInsertFEFnParam(cxt, param) )
	success = FALSE;
    }
    ContextPop(cxt, TRUE);
  }
  return success;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_CREATION FEFnCreationBEFnCreation(FEFN_CREATION cfv)                */
/*                                                                           */
/*  Return the creation function corresponding to this creation fn view.     */
/*                                                                           */
/*****************************************************************************/

/* *** apparently not used
BEFN_CREATION FEFnCreationBEFnCreation(FEFN_CREATION cfv)
{
  assert(cfv->creation_fn != NULL);
  return cfv->creation_fn;
}
*** */
