/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         fefn_credft.c                                              */
/*  DESCRIPTION:  A creation feature default value                           */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#define DEBUG1 0


struct fefn_credft_rec {		/* inherits from FEFN                */
  KIND_TAG		kind_tag;	/* kind (KIND_CLASS_VIEW)            */
  FILE_POS		file_pos;	/* position of class view in input   */
  NAME			name;		/* name of default value             */
  TYPE_VARS		type_vars;	/* type variables                    */
  ARRAY_FEFN_PARAM	parameters;	/* one for hidden parameter          */
  TYPE			result_type;	/* equivalent to class type          */
  BEFN_CREDFT		befn_credft;	/* the corresponding backend fn.     */

  FEFN_FEATURE		orig_cfeature;	/* the originating creation feature  */
  EXPR			body;		/* the body of the default value     */
};


/*****************************************************************************/
/*                                                                           */
/*  FEFN_CREDFT FEFnCreDftNew(FEFN_FEATURE orig_cfeature, EXPR body)         */
/*                                                                           */
/*  Create a new creation feature default value function with these          */
/*  attributes.                                                              */
/*                                                                           */
/*****************************************************************************/

FEFN_CREDFT FEFnCreDftNew(FEFN_FEATURE orig_cfeature, EXPR body)
{
  FEFN_CREDFT res;
  GetMemory(res, FEFN_CREDFT);
  res->kind_tag = KIND_FEFN_CREDFT;
  res->file_pos = FEFnFeatureFilePos(orig_cfeature);
  res->name = FEFnFeatureName(orig_cfeature);
  res->type_vars = NULL;
  res->parameters = NULL;
  res->result_type = FEFnResultType((FEFN) orig_cfeature);
  res->befn_credft = BEFnCreDftNew(res, body);
  res->orig_cfeature = orig_cfeature;
  res->body = body;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnCreDftOrigCreationFeature(FEFN_CREDFT fefn_credft)      */
/*                                                                           */
/*  Return the creation feature from which this creation feature default     */
/*  value originated.                                                        */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE FEFnCreDftOrigCreationFeature(FEFN_CREDFT fefn_credft)
{
  return fefn_credft->orig_cfeature;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnCreDftManifest(FEFN_CREDFT fefn_credft, CONTEXT cxt,         */
/*    TYPE self_type)                                                        */
/*                                                                           */
/*  Manifest this creation function default value.  It is an outer level     */
/*  function in the back end.  It also gathers hidden parameters.            */
/*                                                                           */
/*  Also pass the back end version of this function to the back end          */
/*  version of the creation feature.  This can't be done at creation         */
/*  time since the back end feature is not known then.                       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnCreDftManifest(FEFN_CREDFT fefn_credft, CONTEXT cxt,
  TYPE self_type)
{
  BOOLEAN success;  BEFN_FEATURE befn_feature;
  if( DEBUG1 )
  {
    fprintf(stderr, "[ FEFnCreDftManifest([");
    DEFnCreDftDebug(fefn_credft, cxt, stderr);
    fprintf(stderr, "], cxt, %s)\n", TypeShow(self_type, cxt));
  }

  /* manifest the body */
  success = TRUE;
  ContextPushEmpty(cxt, (FEFN) fefn_credft, FALSE);
  success = ExprManifest(&fefn_credft->body, cxt, self_type,
    (BEFN) fefn_credft->befn_credft);
  ContextPop(cxt, TRUE);

  /* pass the back end version of this function to the creation feature */
  befn_feature = FEFnFeatureBEFnFeature(fefn_credft->orig_cfeature);
  assert(befn_feature != NULL);
  BEFnFeatureSetCreationDefaultValue(befn_feature, fefn_credft->befn_credft);

  if( DEBUG1 )
  {
    fprintf(stderr, "] FEFnCreDftManifest returning %s, fefn_credft now: ",
      bool(success));
    DEFnCreDftDebug(fefn_credft, cxt, stderr);
    fprintf(stderr, "\n");
  }
  return success;
}


/*****************************************************************************/
/*                                                                           */
/*  void DEFnCreDftDebug(FEFN_CREDFT fefn_credft, FILE *fp)                  */
/*                                                                           */
/*  Debug print of fefn_credft.                                              */
/*                                                                           */
/*****************************************************************************/

void DEFnCreDftDebug(FEFN_CREDFT fefn_credft, CONTEXT cxt, FILE *fp)
{
  fprintf(fp, "%s%s := ", NameShow(fefn_credft->name),
    FEFnSignatureShow((FEFN) fefn_credft, cxt));
  ExprDebug(fefn_credft->body, cxt, FALSE, fp, SINGLE_LINE);
}
