/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         fefn_feature.c                                             */
/*  DESCRIPTION:  The view of one feature from one class                     */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0
#define DEBUG6 0
#define DEBUG7 0
#define DEBUG8 0
#define DEBUG9 0
#define DEBUG10 0
#define DEBUG11 0
#define DEBUG12 0
#define DEBUG13 0

/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE                                                             */
/*                                                                           */
/*  A FEFN_FEATURE is the view of one feature from one class.  It includes   */
/*  a view of the header of the feature, and an optional body.               */
/*                                                                           */
/*  The body of a fefn feature is non-NULL only when this front-end          */
/*  feature is responsible for manifesting it.  Thus, it is NULL in          */
/*  clone features (whose bodies are built-in and need no manifesting),      */
/*  and in imported and inherited features (whose bodies are manifested      */
/*  elsewhere).  Whether a noncreation feature is abstract or not does       */
/*  not affect this, since abstract features do have a body - an             */
/*  EXPR_FNHEAD holding any preconditions and default parameter values.      */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  FEFN_FEATURE_UNINHERITED,		/* based in this class/extension     */
  FEFN_FEATURE_INHERITED,		/* inherited, not redefined          */
  FEFN_FEATURE_REDEFINED		/* inherited and redefined           */
} FEFN_FEATURE_STATUS;

struct fefn_feature_rec {               /* inherits from FEFN                */
  KIND_TAG		kind_tag;	/* kind of entity                    */
  FILE_POS		file_pos;	/* position of function name in file */
  NAME			name;		/* name of function (NULL if anon)   */
  TYPE_VARS		type_vars;	/* generic parameters (NULL if none) */
  ARRAY_FEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  TYPE			result_type;	/* result type                       */
  BEFN_FEATURE		base_feature;	/* base feature; not for direct use, */
                                        /* call FEFnFeatureBEFnFeature(fv)!  */

  CLASS_VIEW		class_view;	/* the class view holding this fv    */
  FEFN_FEATURE_STATUS	status;		/* feature view status (inherited..) */
  FEATURE_TYPE		ftype;		/* type (creation, predef, etc.)     */
  BOOLEAN		is_imported;	/* TRUE if imported                  */
  BOOLEAN		is_private;	/* TRUE if feature is private here   */
  BOOLEAN		is_coerce;	/* TRUE if feature is coerce here    */
  BOOLEAN		is_norename;	/* TRUE if feature is norename       */
  BOOLEAN		is_abstract;	/* TRUE if feature has no body       */
  FEFN_CREDFT		fefn_credft;	/* creation features with body only  */
  int			code1;		/* only if is_predef                 */
  int			code2;		/* only if is_predef                 */
  BEFN_FEATURE_INSTANCE	effective_instance;  /* effective instance           */
  FEFN_PRECOND		precondition;	/* local precondition, if any        */
  ARRAY_FEFN_PRECOND	inherited_preconditions; /* inherited preconditions  */
  EXPR			body;		/* body of feature view, if any      */
  COERCION		coercion;	/* the coercion required to get here */
};


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE_SET                                                         */
/*                                                                           */
/*  A FEFN_FEATURE_SET is a set of feature views lying in the same class     */
/*  view and having the same name (i.e. they overload each other).           */
/*                                                                           */
/*  Although a NAME field is necessary here so that feature view sets        */
/*  can go into contexts, and perfectly well defined as far as the           */
/*  raw name goes, the file position is not so well defined.  It will be     */
/*  the file position of whatever element gets into fvs first.               */
/*                                                                           */
/*****************************************************************************/

struct fefn_feature_set_rec {
  KIND_TAG		kind_tag;	/* kind of entity                    */
  FILE_POS		file_pos;	/* position of function name in file */
  NAME			name;		/* name of function                  */
  ARRAY_FEFN_FEATURE	fefn_features;	/* the views, least coerced first    */
};



/*****************************************************************************/
/*                                                                           */
/*  Submodule "feature views basics".                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  FILE_POS FEFnFeatureFilePos(FEFN_FEATURE fv)                             */
/*                                                                           */
/*  Return the file position of fv.                                          */
/*                                                                           */
/*****************************************************************************/

FILE_POS FEFnFeatureFilePos(FEFN_FEATURE fv)
{
  return fv->file_pos;
}


/*****************************************************************************/
/*                                                                           */
/*  NAME FEFnFeatureName(FEFN_FEATURE fv)                                    */
/*                                                                           */
/*  The name of fv.                                                          */
/*                                                                           */
/*****************************************************************************/

NAME FEFnFeatureName(FEFN_FEATURE fv)
{
  return fv->name;
}


/*****************************************************************************/
/*                                                                           */
/*  CLASS_VIEW FEFnFeatureClassView(FEFN_FEATURE fv)                         */
/*                                                                           */
/*  Return the class view that fv lies in.                                   */
/*                                                                           */
/*****************************************************************************/

CLASS_VIEW FEFnFeatureClassView(FEFN_FEATURE fv)
{
  return fv->class_view;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureIsPrivate(FEFN_FEATURE fv)                            */
/*                                                                           */
/*  Return TRUE if fv is private.                                            */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureIsPrivate(FEFN_FEATURE fv)
{
  return fv->is_private;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureIsCoerce(FEFN_FEATURE fv)                             */
/*                                                                           */
/*  Return TRUE if fv is a coercion feature.                                 */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureIsCoerce(FEFN_FEATURE fv)
{
  return fv->is_coerce;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureIsImported(FEFN_FEATURE fv)                           */
/*                                                                           */
/*  Return TRUE if fv is imported.                                           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureIsImported(FEFN_FEATURE fv)
{
  return fv->is_imported;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureIsNoRename(FEFN_FEATURE fv)                           */
/*                                                                           */
/*  Return TRUE if fv is a norename feature, either because it is marked     */
/*  as such individually, or because it comes from a norename class view.    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureIsNoRename(FEFN_FEATURE fv)
{
  return fv->is_norename;
}


/*****************************************************************************/
/*                                                                           */
/*  FEATURE_TYPE FEFnFeatureFType(FEFN_FEATURE fv)                           */
/*                                                                           */
/*  Return the ftype (creation, predef, etc.) of fv.                         */
/*                                                                           */
/*****************************************************************************/

FEATURE_TYPE FEFnFeatureFType(FEFN_FEATURE fv)
{
  return fv->ftype;
}


/*****************************************************************************/
/*                                                                           */
/*  int FEFnFeatureCode(FEFN_FEATURE fv)                                     */
/*                                                                           */
/*  Return the code number of predefined object feature fv.  It must have    */
/*  a unique code number for this to be allowed.                             */
/*                                                                           */
/*****************************************************************************/

int FEFnFeatureCode(FEFN_FEATURE fv)
{
  assert(fv->ftype == FEATURE_PREDEF);
  assert(fv->code1 != NO_CODE_NUM);
  assert(fv->code1 == fv->code2);
  return fv->code1;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureIsAbstract(FEFN_FEATURE fv)                           */
/*                                                                           */
/*  Return TRUE if fv is abstract (has no body).                             */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureIsAbstract(FEFN_FEATURE fv)
{
  return fv->is_abstract;
}


/*****************************************************************************/
/*                                                                           */
/*  COERCION FEFnFeatureCoercion(FEFN_FEATURE fv)                            */
/*                                                                           */
/*  Return fv's coercion.                                                    */
/*                                                                           */
/*****************************************************************************/

COERCION FEFnFeatureCoercion(FEFN_FEATURE fv)
{
  return fv->coercion;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE FEFnFeatureBEFnFeature(FEFN_FEATURE fv)                     */
/*                                                                           */
/*  Return the feature that fv is a view of, or NULL if none yet.            */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE FEFnFeatureBEFnFeature(FEFN_FEATURE fv)
{
  return fv->base_feature == NULL ? NULL : BEFnFeatureRoot(fv->base_feature);
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnFeatureNew(FILE_POS file_pos, NAME name,                */
/*    TYPE_VARS type_vars, ARRAY_FEFN_PARAM parameters, TYPE result_type,    */
/*    CLASS_VIEW cv, FEFN_FEATURE_STATUS type, FEATURE_TYPE ftype,           */
/*    BOOLEAN is_imported, BOOLEAN is_private, BOOLEAN is_coerce,            */
/*    BOOLEAN is_norename, BOOLEAN is_abstract, int code1, int code2,        */
/*    FEFN_PRECOND precond, EXPR body)                                       */
/*                                                                           */
/*  Return a new feature view with these attributes, and null values for     */
/*  the others.                                                              */
/*                                                                           */
/*****************************************************************************/

static FEFN_FEATURE FEFnFeatureNew(FILE_POS file_pos, NAME name,
  TYPE_VARS type_vars, ARRAY_FEFN_PARAM parameters, TYPE result_type,
  CLASS_VIEW cv, FEFN_FEATURE_STATUS status, FEATURE_TYPE ftype,
  BOOLEAN is_imported, BOOLEAN is_private, BOOLEAN is_coerce,
  BOOLEAN is_norename, BOOLEAN is_abstract, int code1, int code2,
  FEFN_PRECOND precond, EXPR body)
{
  FEFN_FEATURE res;
  GetMemory(res, FEFN_FEATURE);
  res->kind_tag = KIND_FEFN_FEATURE;
  res->file_pos = file_pos;
  res->name = name;
  res->type_vars = type_vars;
  res->parameters = parameters;
  res->result_type = result_type;
  res->base_feature = NULL;

  res->class_view = cv;
  res->status = status;
  res->ftype = ftype;
  res->is_imported = is_imported;
  res->is_private = is_private;
  res->is_coerce = is_coerce;
  res->is_norename = is_norename;
  res->is_abstract = is_abstract;
  res->fefn_credft = ftype == FEATURE_CREATION && !is_abstract ?
    FEFnCreDftNew(res, body) : NULL;
  if( code1 != NO_CODE_NUM )
    assert(code1 <= code2);
  res->code1 = code1;
  res->code2 = code2;
  res->effective_instance = NULL;
  res->precondition = precond;
  res->inherited_preconditions = NULL;
  res->body = body;
  res->coercion = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnFeatureCopyUninstantiated(FEFN_FEATURE fv, NAME name)   */
/*                                                                           */
/*  Make a deep copy of uninstantiated feature view fv, changing only the    */
/*  name.  This is to disentangle multiple features declared together from   */
/*  each other.  Nothing has been manifested at the time of this call, and   */
/*  in fact fv is freshly made by a call to FEFnFeatureNew.                  */
/*                                                                           */
/*****************************************************************************/

static FEFN_FEATURE FEFnFeatureCopyUninstantiated(FEFN_FEATURE fv, NAME name)
{
  FEFN_FEATURE res;
  if( DEBUG11 )
  {
    fprintf(stderr, "[ FEFnFeatureCopyUninstantiated(");
    FEFnFeatureDebug(fv, NULL, stderr);
    fprintf(stderr, ", %s)\n", NameShow(name));
  }
  GetMemory(res, FEFN_FEATURE);
  res->kind_tag = fv->kind_tag;
  res->file_pos = fv->file_pos;
  res->name = name;
  res->type_vars = TypeVarsCopyUninstantiated(fv->type_vars);
  res->parameters = FEFnParamListCopyUninstantiated(fv->parameters);
  res->result_type = TypeCopyUninstantiated(fv->result_type);
  res->base_feature = NULL;

  res->class_view = fv->class_view;
  res->status = fv->status;
  res->ftype = fv->ftype;
  res->is_imported = fv->is_imported;
  res->is_private = fv->is_private;
  res->is_coerce = fv->is_coerce;
  res->is_norename = fv->is_norename;
  res->is_abstract = fv->is_abstract;
  res->code1 = fv->code1;
  res->code2 = fv->code2;
  res->effective_instance = fv->effective_instance;
  res->precondition = fv->precondition == NULL ? NULL :
    FEFnPrecondCopyUninstantiated(fv->precondition, res, fv->parameters,
      res->parameters);
  assert(fv->inherited_preconditions == NULL);
  res->inherited_preconditions = NULL;
  res->body = fv->body == NULL ? NULL :
    ExprCopyUninstantiated(fv->body, fv->parameters, res->parameters);
  res->fefn_credft = fv->fefn_credft == NULL ? NULL :
    FEFnCreDftNew(res, res->body);
  res->coercion = NULL;
  if( DEBUG11 )
    fprintf(stderr, "] FEFnFeatureCopyUninstantiated returning\n");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnFeatureCopyImported(FEFN_FEATURE fv, FILE_POS new_pos,  */
/*    NAME new_name, CLASS_VIEW new_cv)                                      */
/*                                                                           */
/*  Return a fresh copy of fv, for importing into a different module, with   */
/*  the name changed to new_name.                                            */
/*                                                                           */
/*  The bodies and preconditions of imported features are not imported,      */
/*  because having access to them would serve no useful purpose.             */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE FEFnFeatureCopyImported(FEFN_FEATURE fv, FILE_POS new_pos,
  NAME new_name, CLASS_VIEW new_cv)
{
  FEFN_FEATURE res;
  GetMemory(res, FEFN_FEATURE);
  res->kind_tag = KIND_FEFN_FEATURE;
  res->file_pos = new_pos;
  res->name = new_name;
  res->type_vars = fv->type_vars;
  res->parameters =
    FEFnParamListCopyImported(fv->parameters, ClassViewIsPrivate(new_cv));
  res->result_type = fv->result_type;
  res->base_feature = fv->base_feature;

  res->class_view = new_cv;
  res->status = fv->status;
  res->ftype = fv->ftype;
  res->is_imported = TRUE;
  assert(!fv->is_private);
  res->is_private = ClassViewIsPrivate(new_cv);
  res->is_coerce = fv->is_coerce;
  res->is_norename = fv->is_norename;
  res->is_abstract = fv->is_abstract;
  res->fefn_credft = NULL;  /* always null in imported features */
  res->code1 = fv->code1;
  res->code2 = fv->code2;
  res->effective_instance = fv->effective_instance;
  res->precondition = NULL;
  res->inherited_preconditions = NULL;
  res->body = NULL;
  res->coercion = fv->coercion;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnFeatureCopy(FEFN_FEATURE fv, COERCION co, CLASS_VIEW cv)*/
/*                                                                           */
/*  Copy fv with coercion co (possibly NULL) and class view cv.  This is     */
/*  used when inheriting a feature view, coerced or not.                     */
/*                                                                           */
/*****************************************************************************/

static FEFN_FEATURE FEFnFeatureCopy(FEFN_FEATURE fv, COERCION co, CLASS_VIEW cv)
{
  FEFN_FEATURE res;
  GetMemory(res, FEFN_FEATURE);
  res->kind_tag = fv->kind_tag;
  res->file_pos = fv->file_pos;
  res->name = fv->name;
  FEFnSignature((FEFN) fv, &res->type_vars, &res->parameters,
    &res->result_type);
  res->base_feature = fv->base_feature;

  res->class_view = cv;
  res->status = FEFN_FEATURE_INHERITED;
  res->ftype = fv->ftype;
  res->is_imported = fv->is_imported;
  res->is_private = fv->is_private;
  res->is_coerce = fv->is_coerce;
  res->is_norename = fv->is_norename;
  res->is_abstract = fv->is_abstract;
  res->fefn_credft = NULL;  /* always null in imported features */
  res->code1 = fv->code1;
  res->code2 = fv->code2;
  res->effective_instance = fv->effective_instance;
  res->body = NULL;
  res->precondition = NULL;
  res->inherited_preconditions = NULL;
  res->coercion = co;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureModifiersParse(TOKEN *t, FEATURE_TYPE ftype,          */
/*    CLASS_VIEW cv, BOOLEAN *is_private, BOOLEAN *is_coerce)                */
/*                                                                           */
/*  Parse the modifiers of a feature destined for cv according to grammer    */
/*                                                                           */
/*    [ "private"  |  "coerce" ]                                             */
/*                                                                           */
/*  and set *is_private and *is_coerce appropriately.  This code actually    */
/*  parses according to a more lenient grammar in which {} replaces [],      */
/*  so as to offer more informative error messages.                          */
/*                                                                           */
/*  If ftype is FEATURE_PREDEF, the "coerce" modifier is prohibited.         */
/*  If cv is a private class, the "private" keyword is prohibited but        */
/*  the feature is automatically private.  If cv is a public class it        */
/*  may not have a private coercion feature.                                 */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FEFnFeatureModifiersParse(TOKEN *t, FEATURE_TYPE ftype,
  CLASS_VIEW cv, BOOLEAN *is_private, BOOLEAN *is_coerce)
{
  *is_coerce = FALSE;
  *is_private = ClassViewIsPrivate(cv);
  while( LexType(curr_token) == TK_PRIVATE || LexType(curr_token) == TK_COERCE )
  {
    if( LexType(curr_token) == TK_PRIVATE )
    {
      if( ClassViewIsPrivate(cv) )
      {
	fprintf(stderr, "%s: \"private\" keyword in private class\n",
	  LexPos(curr_token));
	return FALSE;
      }
      else if( *is_private )
      {
	fprintf(stderr, "%s: \"private\" keyword given twice\n",
	  LexPos(curr_token));
	return FALSE;
      }
      else
        *is_private = TRUE;
    }
    else
    {
      if( *is_coerce )
      {
	fprintf(stderr, "%s: \"coerce\" keyword given twice\n",
	  LexPos(curr_token));
	return FALSE;
      }
      else if( ftype == FEATURE_PREDEF )
      {
	fprintf(stderr, "%s: \"coerce\" not allowed with predefined object\n",
	  LexPos(curr_token));
	return FALSE;
      }
      else
        *is_coerce = TRUE;
    }
    next_token;
  }
  if( *is_coerce && *is_private && !ClassViewIsPrivate(cv) )
  {
    fprintf(stderr, "%s: private coercion feature in public class\n",
      LexPos(curr_token));
    return FALSE;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureSetParse(TOKEN *t, CLASS_VIEW cv, FEATURE_TYPE ftype, */
/*    BOOLEAN builtin, BOOLEAN extension, SYSTEM_VIEW sv)                    */
/*                                                                           */
/*  Parse a feature view set according to the grammar                        */
/*                                                                           */
/*     [mods] [names] [gens] [parameters] : type [preconditions] [:= expr]   */
/*                                                                           */
/*  and register them with cv, the class view that they are going into.      */
/*                                                                           */
/*  If ftype is FEATURE_CREATION, these are creation features, whose grammar */
/*  is a subset of what is given above: there may be no gens, parameters, or */
/*  preconditions, although these things are parsed if present in the        */
/*  interest of giving informative error messages.                           */
/*                                                                           */
/*  If builtin or extension is TRUE, these are features of a builtin class   */
/*  or of a class extension, so if they are noncreation features they must   */
/*  have a body.                                                             */
/*                                                                           */
/*  If ftype is FEATURE_PREDEF, these are predefined features, so their      */
/*  grammar is rather different:                                             */
/*                                                                           */
/*     [names] [ := expr | "(" actual_parameters ")" ]                       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureSetParse(TOKEN *t, CLASS_VIEW cv, FEATURE_TYPE ftype,
  BOOLEAN builtin, BOOLEAN extension, SYSTEM_VIEW sv)
{
  BOOLEAN is_private, is_coerce;  ARRAY_NAME names;  NAME name;
  ARRAY_FEFN_PARAM params;  TYPE_VARS type_vars;  FEFN_FEATURE fv;
  EXPR expr_fnhead, expr_precond;  TYPE result_type;
  FILE_POS file_pos; FEFN_PARAM param;  int code1, code2;
  PARAM_BODIES_TYPE param_bodies_type;
  if( DEBUG6 )
    fprintf(stderr, "[ FEFnFeatureSetParse(-, -, %s, %s, %s, -)\n",
      FeatureTypeShow(ftype), bool(builtin), bool(extension));

  /* modifiers */
  if( !FEFnFeatureModifiersParse(t, ftype, cv, &is_private, &is_coerce) )
    db_return(DEBUG6, "FEFnFeatureSetParse modifiers", FALSE);

  /* self parameter, if not is_predef */
  params = NULL;
  if( ftype < FEATURE_PREDEF )
  {
    ArrayInit(&params);
    param = FEFnParamNew(LexFilePos(curr_token),
      NameNew(AStringToUString("self")),
      ClassType(ClassViewClass(cv)), NULL, PARAM_SELF, FALSE, NULL);
    ArrayAddLast(params, param);
  }

  /* the function */
  param_bodies_type = PARAM_BODIES_OPEN;
  if( !FEFnParse(t, FALSE, FALSE, ftype, is_coerce, is_private, builtin,
	extension, ftype == FEATURE_PREDEF ? NameRep(ClassViewName(cv)) : NULL,
	ClassIsEnum(ClassViewClass(cv)), sv, &file_pos, &names, &type_vars,
	&params, &result_type, &expr_fnhead, &param_bodies_type,
	&expr_precond, &code1, &code2) )
    db_return(DEBUG6, "FEFnFeatureSetParse function", FALSE);
  if( ftype == FEATURE_PREDEF )
  {
    assert(result_type == NULL);
    result_type = ClassType(ClassViewClass(cv));
  }

  /* a private creation feature of a public class must have a default value */
  if( ftype == FEATURE_CREATION && is_private &&
      !ClassViewIsPrivate(cv) && expr_fnhead == NULL )
  {
    fprintf(stderr,
      "%s: private creation feature %s of public class has no body\n",
      FilePosShow(file_pos), NameShow(ArrayFirst(names)));
    return FALSE;
  }

  if( names != NULL )
  {
    /* add one feature view for each name def */
    fv = NULL;
    ArrayForEach(names, name)
    {
      /* make first feature view, uninstantiated-copy the others */
      if( fv == NULL )
      {
	fv = FEFnFeatureNew(file_pos, name, type_vars, params, result_type,
	  cv, FEFN_FEATURE_UNINHERITED, ftype, FALSE, is_private, is_coerce,
	  ClassViewIsNoRename(cv), expr_fnhead == NULL, code1, code2,
	  NULL, expr_fnhead);
	if( expr_precond != NULL )
	  fv->precondition = FEFnPrecondNew(fv, expr_precond);
      }
      else
	fv = FEFnFeatureCopyUninstantiated(fv, name);

      /* check fv and register it */
      if( !NameConsistentWithParameters(fv->name,
	    FEFnCompulsoryParamsCount((FEFN) fv), fv->file_pos) )
	db_return(DEBUG6, "FEFnFeatureSetParse paramcount", FALSE);
      if( !ClassViewRegisterOriginalFEFnFeature(cv, fv) )
	db_return(DEBUG6, "FEFnFeatureSetParse register", FALSE);
    }
  }
  else
  {
    /* must be anonymous predefined object feature; make one feature view */
    assert(ftype == FEATURE_PREDEF && type_vars == NULL && params == NULL);
    fv = FEFnFeatureNew(file_pos, NULL, type_vars, params, result_type,
      cv, FEFN_FEATURE_UNINHERITED, ftype, FALSE, is_private, is_coerce,
      ClassViewIsNoRename(cv), expr_fnhead == NULL, code1, code2, NULL,
      expr_fnhead);
    if( !ClassViewRegisterOriginalFEFnFeature(cv, fv) )
      db_return(DEBUG6, "FEFnFeatureSetParse register predef", FALSE);
  }

  /* register coercion result type with class; last fv is the feature */
  if( is_coerce )
    ClassRegisterCoerceResultType(ClassViewClass(cv), result_type, fv);
  db_return(DEBUG6, "FEFnFeatureSetParse", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnFeatureAllPredefinedMake(FILE_POS file_pos,             */
/*    CLASS_VIEW cv, SYSTEM_VIEW sv)                                         */
/*                                                                           */
/*  Pretend that a predefined feature                                        */
/*                                                                           */
/*    all_predefined: array{cv}                                              */
/*                                                                           */
/*  appears in the input at file_pos.  Create the appropriate feature view.  */
/*  Don't worry about a body; the body is built from the predefined objects  */
/*  by back-end code.                                                        */
/*                                                                           */
/*  Calls on this function are only available after the first module has     */
/*  been read, because it relies on ClassArray being set.                    */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE FEFnFeatureAllPredefinedMake(FILE_POS file_pos, CLASS_VIEW cv,
  SYSTEM_VIEW sv)
{
  TYPE t, result_type;  ARRAY_TYPE generics;
  static USTRING all_predef_str = NULL;

  /* name is "all_predefined" */
  assert(!ClassIsEnum(ClassViewClass(cv)));
  if( all_predef_str == NULL )
    all_predef_str = AStringToUString("all_predefined");

  /* result type (instantiated) is array{cv} */
  assert(ClassArray != NULL);
  t = TypeMakeInstantiatedClassType2(file_pos, ClassViewClass(cv), NULL);
  ArrayInit(&generics);
  ArrayAddLast(generics, t);
  result_type = TypeMakeInstantiatedClassType2(file_pos, ClassArray, generics);

  /* make and return the feature view (body is NULL) */
  return FEFnFeatureNew(file_pos, NameNew(all_predef_str), NULL, NULL,
    result_type, cv, FEFN_FEATURE_UNINHERITED, FEATURE_PREDEF_ALL_PREDEFINED,
    FALSE, FALSE, FALSE, TRUE, FALSE, NO_CODE_NUM, NO_CODE_NUM, NULL, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnFeatureAllEnumeratedMake(FILE_POS file_pos,             */
/*    CLASS_VIEW cv, SYSTEM_VIEW sv)                                         */
/*                                                                           */
/*  Pretend that a predefined feature                                        */
/*                                                                           */
/*    all_enumerated: array{pair{cv, cv}} := [||]                            */
/*                                                                           */
/*  appears in the input at file_pos.  Create the appropriate feature view.  */
/*  Don't worry about a body; the body is built from the predefined objects  */
/*  by back-end code.                                                        */
/*                                                                           */
/*  Calls on this function are only available after the first module has     */
/*  been read, because it relies on ClassArray and ClassTuple[2] being set.  */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE FEFnFeatureAllEnumeratedMake(FILE_POS file_pos,
  CLASS_VIEW cv, SYSTEM_VIEW sv)
{
  TYPE t, result_type;  ARRAY_TYPE generics;
  static USTRING all_enum_str = NULL;

  /* name is "all_enumerated" */
  assert(ClassIsEnum(ClassViewClass(cv)));
  if( all_enum_str == NULL )
    all_enum_str = AStringToUString("all_enumerated");

  /* result type (instantiated) is array{pair{cv, cv}} */
  assert(ClassArray != NULL);
  ArrayInit(&generics);
  t = TypeMakeInstantiatedClassType2(file_pos, ClassViewClass(cv), NULL);
  ArrayAddLast(generics, t);
  t = TypeMakeInstantiatedClassType2(file_pos, ClassViewClass(cv), NULL);
  ArrayAddLast(generics, t);
  t = TypeMakeInstantiatedClassType2(file_pos, ClassTuple[2], generics);
  ArrayInit(&generics);
  ArrayAddLast(generics, t);
  result_type = TypeMakeInstantiatedClassType2(file_pos, ClassArray, generics);

  /* make and return the feature view */
  return FEFnFeatureNew(file_pos, NameNew(all_enum_str), NULL, NULL,
    result_type, cv, FEFN_FEATURE_UNINHERITED, FEATURE_PREDEF_ALL_ENUMERATED,
    FALSE, FALSE, FALSE, TRUE, FALSE, NO_CODE_NUM, NO_CODE_NUM, NULL, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnFeatureDebug(FEFN_FEATURE fv, CONTEXT cxt, FILE *fp)            */
/*                                                                           */
/*  Debug print of fv onto fp.                                               */
/*                                                                           */
/*****************************************************************************/

void FEFnFeatureDebug(FEFN_FEATURE fv, CONTEXT cxt, FILE *fp)
{
  /* print coercion if any */
  if( fv->coercion != NULL )
    fprintf(fp, "[%s] ", CoercionShow(fv->coercion));

  /* print name and function header */
  fprintf(fp, "%s%s%s: %s", NameShow(fv->name),
    TypeVarsShow(fv->type_vars, cxt),
    FEFnParamListShow(fv->parameters, cxt),
    TypeShow(fv->result_type, cxt));

  /* print state */
  fprintf(fp, " (%s%s)", fv->is_imported ? "imported, " : "",
    fv->status == FEFN_FEATURE_UNINHERITED ? "uninherited" :
    fv->status == FEFN_FEATURE_INHERITED ? "inherited" :
    fv->status == FEFN_FEATURE_REDEFINED ? "redefined" : "??");
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "the clone feature view"                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnFeatureCloneNew(CLASS_VIEW cv)                          */
/*                                                                           */
/*  Make a new clone feature view for this class view.                       */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE FEFnFeatureCloneNew(CLASS_VIEW cv)
{
  FEFN_PARAM param;  ARRAY_FEFN_PARAM params;  FEFN_FEATURE res;

  /* get dopey strings initialized */
  static USTRING self_str = NULL;
  static USTRING clone_str = NULL;
  if( self_str == NULL )
  {
    self_str = AStringToUString("self");
    clone_str = AStringToUString("clone");
  }

  if( DEBUG13 )
    fprintf(stderr, "[ FEFnFeatureCloneNew(%s)\n", NameShow(ClassViewName(cv)));

  /* make the sole parameter (initially): self */
  ArrayInit(&params);
  param = FEFnParamNew(ClassViewFilePos(cv), NameNew(self_str),
    ClassType(ClassViewClass(cv)), NULL, PARAM_SELF, FALSE, NULL);
  ArrayAddLast(params, param);

  /* make the new feature view; clone has a NULL body and non-NULL base f. */
  res = FEFnFeatureNew(ClassViewFilePos(cv), NameNew(clone_str), NULL,
    params, ClassType(ClassViewClass(cv)), cv, FEFN_FEATURE_UNINHERITED,
    FEATURE_NONCREATION_CLONE, FALSE, FALSE, FALSE, TRUE, FALSE, NO_CODE_NUM,
    NO_CODE_NUM, NULL, NULL);
  res->base_feature = ClassDSCloneBEFnFeature(ClassViewClass(cv), res, param);

  /* link self parameter of res to self parameter of res->base_feature */
  FEFnParamSetBEFnParam(param, BEFnFeatureSelfParam(res->base_feature));

  if( DEBUG13 )
    fprintf(stderr, "] FEFnFeatureCloneNew returning\n");
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "other auto-generated feature views"                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void AddParam(ARRAY_FEFN_PARAM np_params, FILE_POS file_pos,             */
/*    USTRING name_str, TYPE type)                                           */
/*                                                                           */
/*  Helper function for the following functions.  Add to params a new        */
/*  parameter with the given file pos, name, and type given by cv.           */
/*                                                                           */
/*****************************************************************************/

static void AddParam(ARRAY_FEFN_PARAM np_params, FILE_POS file_pos,
  USTRING name_str, TYPE type)
{
  FEFN_PARAM param;
  param = FEFnParamNew(file_pos, NameNew(name_str), type, NULL,
    PARAM_COMPULSORY, FALSE, NULL);
  ArrayAddLast(np_params, param);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnFeatureRegisterToInt(CLASS_VIEW cv, FILE_POS file_pos)          */
/*                                                                           */
/*  Register with cv a feature                                               */
/*                                                                           */
/*    to_int: int := builtin "C_to_int"(self)                                */
/*                                                                           */
/*  where C comes from cv.  The feature is to be uninstantiated,             */
/*  indistinguishable from a feature that has just been parsed.              */
/*                                                                           */
/*****************************************************************************/

void FEFnFeatureRegisterToInt(CLASS_VIEW cv, FILE_POS file_pos)
{
  ARRAY_EXPR params;  EXPR e, expr_fnhead;
  NAME name;  TYPE result_type;  FEFN_FEATURE fv;
  ARRAY_FEFN_PARAM np_params;
  static USTRING to_int_str = NULL;
  static USTRING int_str = NULL;
  static USTRING self_str = NULL;
  static USTRING suffix = NULL;
  USTRING builtin_str = NULL;
  if( to_int_str == NULL )
  {
    to_int_str = AStringToUString("to_int");
    int_str = AStringToUString("int");
    self_str = AStringToUString("self");
    suffix = AStringToUString("_to_int");
  }

  /* register the builtin function "C_to_int" */
  builtin_str = UStringCat(NameRep(ClassViewName(cv)), suffix);
  FEFnBuiltinInsertEnumToInt(file_pos, builtin_str, cv);

  /* make the body */
  ArrayInit(&params);
  ArrayAddLast(params, ExprMakeRawCall(file_pos, self_str, NULL, NULL));
  e = ExprMakeBuiltinCall(file_pos, builtin_str, params);
  expr_fnhead = ExprFnHeadNew(file_pos);
  ExprFnHeadAddSubExpression(expr_fnhead, e);

  /* make the name, formal parameters, and result type */
  name = NameNew(to_int_str);
  ArrayInit(&np_params);
  AddParam(np_params, file_pos, self_str, ClassType(ClassViewClass(cv)));
  result_type = ClassType(ClassInt);

  /* make the feature view */
  fv = FEFnFeatureNew(file_pos, name, NULL, np_params, result_type,
    cv, FEFN_FEATURE_UNINHERITED, FEATURE_NONCREATION, FALSE, FALSE,
    FALSE, TRUE, FALSE, NO_CODE_NUM, NO_CODE_NUM, NULL, expr_fnhead);

  /* check fv and register it */
  if( !NameConsistentWithParameters(fv->name,
	FEFnCompulsoryParamsCount((FEFN) fv), fv->file_pos) )
    assert(FALSE);
  if( !ClassViewRegisterOriginalFEFnFeature(cv, fv) )
    assert(FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnFeatureRegisterLegalCode(CLASS_VIEW cv, FILE_POS file_pos)      */
/*                                                                           */
/*  Register with cv a predefined feature                                    */
/*                                                                           */
/*    legal_code(i: int): bool                                               */
/*                                                                           */
/*  The feature is to be uninstantiated, indistinguishable from a feature    */
/*  that has just been parsed.  It has no body.                              */
/*                                                                           */
/*****************************************************************************/

void FEFnFeatureRegisterLegalCode(CLASS_VIEW cv, FILE_POS file_pos)
{
  EXPR expr_fnhead;
  NAME name;  TYPE result_type;  FEFN_FEATURE fv;
  ARRAY_FEFN_PARAM np_params;
  static USTRING legal_code_str = NULL;
  static USTRING i_str = NULL;
  if( legal_code_str == NULL )
  {
    legal_code_str = AStringToUString("legal_code");
    i_str = AStringToUString("i");
  }

  /* make the body */
  expr_fnhead = ExprFnHeadNew(file_pos);

  /* make the name, formal parameters, and result type */
  name = NameNew(legal_code_str);
  ArrayInit(&np_params);
  AddParam(np_params, file_pos, i_str, ClassType(ClassInt));
  result_type = ClassType(ClassBool);

  /* make the feature view */
  fv = FEFnFeatureNew(file_pos, name, NULL, np_params, result_type,
    cv, FEFN_FEATURE_UNINHERITED, FEATURE_PREDEF_LEGAL_CODE, FALSE, FALSE,
    FALSE, TRUE, FALSE, NO_CODE_NUM, NO_CODE_NUM, NULL, expr_fnhead);

  /* check fv and register it */
  if( !NameConsistentWithParameters(fv->name,
	FEFnCompulsoryParamsCount((FEFN) fv), fv->file_pos) )
    assert(FALSE);
  if( !ClassViewRegisterOriginalFEFnFeature(cv, fv) )
    assert(FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnFeatureRegisterWithCode(CLASS_VIEW cv, FILE_POS file_pos)       */
/*                                                                           */
/*  Register with cv a predefined feature                                    */
/*                                                                           */
/*    with_code(i: int): C                                                   */
/*                                                                           */
/*  where C comes from cv.  The feature is to be uninstantiated,             */
/*  indistinguishable from a feature that has just been parsed.              */
/*  It has no body.                                                          */
/*                                                                           */
/*****************************************************************************/

void FEFnFeatureRegisterWithCode(CLASS_VIEW cv, FILE_POS file_pos)
{
  EXPR expr_fnhead;
  NAME name;  TYPE result_type;  FEFN_FEATURE fv;
  ARRAY_FEFN_PARAM np_params;
  static USTRING with_code_str = NULL;
  static USTRING i_str = NULL;
  if( with_code_str == NULL )
  {
    with_code_str = AStringToUString("with_code");
    i_str = AStringToUString("i");
  }

  /* make the body */
  expr_fnhead = ExprFnHeadNew(file_pos);

  /* make the name, formal parameters, and result type */
  name = NameNew(with_code_str);
  ArrayInit(&np_params);
  AddParam(np_params, file_pos, i_str, ClassType(ClassInt));
  result_type = ClassType(ClassViewClass(cv));

  /* make the feature view */
  fv = FEFnFeatureNew(file_pos, name, NULL, np_params, result_type,
    cv, FEFN_FEATURE_UNINHERITED, FEATURE_PREDEF_WITH_CODE, FALSE, FALSE,
    FALSE, TRUE, FALSE, NO_CODE_NUM, NO_CODE_NUM, NULL, expr_fnhead);

  /* check fv and register it */
  if( !NameConsistentWithParameters(fv->name,
	FEFnCompulsoryParamsCount((FEFN) fv), fv->file_pos) )
    assert(FALSE);
  if( !ClassViewRegisterOriginalFEFnFeature(cv, fv) )
    assert(FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "feature view sets basics".                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  NAME FEFnFeatureSetName(FEFN_FEATURE_SET fvs)                            */
/*                                                                           */
/*  Return the name of this feature view set.                                */
/*                                                                           */
/*****************************************************************************/

NAME FEFnFeatureSetName(FEFN_FEATURE_SET fvs)
{
  return fvs->name;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnFeatureSetSoleUncoercedMember(FEFN_FEATURE_SET fvs)     */
/*                                                                           */
/*  Return the sole uncoerced member of fvs, or NULL if none.                */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE FEFnFeatureSetSoleUncoercedMember(FEFN_FEATURE_SET fvs)
{
  FEFN_FEATURE fv;
  assert(fvs->fefn_features != NULL);
  if( ArraySize(fvs->fefn_features) >= 1 )
  {
    fv = ArrayFirst(fvs->fefn_features);
    return CoercionLength(fv->coercion) == 0 ? fv : NULL;
  }
  else
    return NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  int FEFnFeatureSetSize(FEFN_FEATURE_SET fvs)                             */
/*                                                                           */
/*  Return the number of feature views in fvs.                               */
/*                                                                           */
/*****************************************************************************/

int FEFnFeatureSetSize(FEFN_FEATURE_SET fvs)
{
  return ArraySize(fvs->fefn_features);
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnFeatureSetFEFnFeature(FEFN_FEATURE_SET fvs, int i)      */
/*                                                                           */
/*  Return the ith feature view of fvs.                                      */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE FEFnFeatureSetFEFnFeature(FEFN_FEATURE_SET fvs, int i)
{
  assert(i < ArraySize(fvs->fefn_features));
  return ArrayGet(fvs->fefn_features, i);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnFeatureSetDebug(FEFN_FEATURE_SET fvs, CONTEXT cxt, FILE *fp,    */
/*    int print_style)                                                       */
/*                                                                           */
/*  Debug print of fvs onto fp with the given print style.                   */
/*                                                                           */
/*****************************************************************************/

void FEFnFeatureSetDebug(FEFN_FEATURE_SET fvs, CONTEXT cxt, FILE *fp,
  int print_style)
{
  FEFN_FEATURE fv;
  if( fvs == NULL )
    fprintf(fp, "%s", "NULL");
  else
  {
    ArrayForEach(fvs->fefn_features, fv)
    {
      if( fv != ArrayFirst(fvs->fefn_features) )
      {
	next_line;
      }
      FEFnFeatureDebug(fv, cxt, fp);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "creation, copying, and checking".                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE_SET FEFnFeatureSetNew(FILE_POS file_pos, NAME name)         */
/*                                                                           */
/*  Make a new, empty feature view set with these attributes.                */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE_SET FEFnFeatureSetNew(FILE_POS file_pos, NAME name)
{
  FEFN_FEATURE_SET res;
  GetMemory(res, FEFN_FEATURE_SET);
  res->kind_tag = KIND_FEFN_FEATURE_SET;
  res->file_pos = file_pos;
  res->name = name;
  ArrayInit(&res->fefn_features);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE_SET FEFnFeatureSetNewFromFEFnFeature(FEFN_FEATURE fv)       */
/*                                                                           */
/*  Make a new feature view set containing just fv.  Unlike other calls      */
/*  that add and meld feature views into feature view sets, this one does    */
/*  not copy fv at all.                                                      */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE_SET FEFnFeatureSetNewFromFEFnFeature(FEFN_FEATURE fv)
{
  FEFN_FEATURE_SET res;
  res = FEFnFeatureSetNew(fv->file_pos, fv->name);
  ArrayAddLast(res->fefn_features, fv);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN CheckConstantParts(FEFN_FEATURE pfv, FEFN_FEATURE cfv)           */
/*                                                                           */
/*  Check that the constant parts (feature type, type variables, and         */
/*  compulsory parameter views) of parent signature pfv are equal to those   */
/*  of child signature cfv.  Print an error message and return FALSE if not. */
/*  Used by FEFnFeatureSetCheckInherited.                                    */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN CheckConstantParts(FEFN_FEATURE pfv, FEFN_FEATURE cfv,
  CONTEXT cxt)
{
  int ip, ic, pcompulsory, ccompulsory;
  FEFN_PARAM pp, cp;  BOOLEAN success;

  /* check for incompatible ftype */
  if( cfv->ftype != pfv->ftype )
  {
    fprintf(stderr, "%s: redefinition into a different kind of feature\n",
      FilePosShow(cfv->file_pos));
    fprintf(stderr, "  %s: %s (inherited %s)\n", FilePosShow(pfv->file_pos),
      NameFullShow(pfv->name), FeatureTypeShow(pfv->ftype));
    fprintf(stderr, "  %s: %s (redefined as %s)\n", FilePosShow(cfv->file_pos),
      NameFullShow(cfv->name), FeatureTypeShow(cfv->ftype));
    db_return(DEBUG10, "FEFnFeatureSetCheckInherited", FALSE);
  }

  /* type variables must be equivalent */
  if( !TypeVarsEquivalent(pfv->type_vars, cfv->type_vars, cxt, pfv->file_pos) )
  {
    fprintf(stderr, "%s: redefinition of %s has incompatible generics\n",
      FilePosShow(cfv->file_pos), NameShow(cfv->name));
    fprintf(stderr, "  (incompatible parent definition is at %s)\n",
      FilePosShow(pfv->file_pos));
    return FALSE;
  }

  /* compulsory parameters must be equal in number */
  pcompulsory = 0;
  ArrayForEach(pfv->parameters, pp)
    if( !FEFnParamIsOptional(pp) )
      pcompulsory++;
  ccompulsory = 0;
  ArrayForEach(cfv->parameters, cp)
    if( !FEFnParamIsOptional(cp) )
      ccompulsory++;
  if( pcompulsory != ccompulsory )
  {
    fprintf(stderr, "%s: redefinition of %s has %s compulsory parameters\n",
      FilePosShow(cfv->file_pos), NameShow(cfv->name),
      ccompulsory < pcompulsory ? "too few" : "too many");
    success = FALSE;
  }

  /* compulsory parameters (known equal in number) must have same types */
  /* and also set the childrens' base parameters to the parents' */
  else
  {
    TypeVarsForwardBegin(pfv->type_vars, cfv->type_vars, pfv->file_pos);
    ip = 0;  ic = 0;  success = TRUE;
    while( ip < ArraySize(pfv->parameters) && ic < ArraySize(cfv->parameters) )
    {
      pp = ArrayGet(pfv->parameters, ip);
      cp = ArrayGet(cfv->parameters, ic);
      if( FEFnParamIsOptional(pp) )
	ip++;
      else if( FEFnParamIsOptional(cp) )
	ic++;
      else
      {
	if( FEFnParamKind(pp) == PARAM_SELF )
	{
	  /* self parameters match but don't need a type check */
	  assert(FEFnParamKind(cp) == PARAM_SELF);
	}
	else if( !TypeEqual(FEFnParamType(pp), FEFnParamType(cp), cxt) )
	{
	  fprintf(stderr, "%s: redefinition of %s differs in type of %s %s\n",
	    FilePosShow(cfv->file_pos), NameShow(cfv->name),
	    "compulsory parameter", NameShow(FEFnParamName(cp)));
	  fprintf(stderr, "  (incompatible parent definition is at %s)\n",
	    FilePosShow(pfv->file_pos));
	  success = FALSE;
	}
	if( !FEFnParamIsRedefinition(pp, cp) )
	  return FALSE;
	ip++;
	ic++;
      }
    }
    TypeVarsForwardEnd(pfv->type_vars);
  }
  return success;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN CheckVaryingParts(FEFN_FEATURE pfv, FEFN_FEATURE cfv,CONTEXT cxt)*/
/*                                                                           */
/*  Check that the varying parts (optional parameters, result type, and      */
/*  privacy) of parent feature view pfv are compatible with child feature    */
/*  view cfv.  Print an error message and return FALSE if not.  Used by      */
/*  FEFnFeatureSetCheckInherited.                                            */
/*                                                                           */
/*  Missing optional parameters of clone features are ignored.  They are     */
/*  added later on, elsewhere in the compiler.                               */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN CheckVaryingParts(FEFN_FEATURE pfv, FEFN_FEATURE cfv,CONTEXT cxt)
{
  FEFN_PARAM pp, cp;  BOOLEAN found, success;  COERCION c;

  /* every optional parameter in the parent must also be in the child */
  success = TRUE;
  TypeVarsForwardBegin(pfv->type_vars, cfv->type_vars, pfv->file_pos);
  ArrayForEach(pfv->parameters, pp)
    if( FEFnParamIsOptional(pp) )
    {
      found = FALSE;
      ArrayForEach(cfv->parameters, cp)
      {
	if( FEFnParamIsOptional(cp) )
	{
	  if( NameEqual(FEFnParamName(pp), FEFnParamName(cp)) )
	  {
	    found = TRUE;
	    if( !TypeEqual(FEFnParamType(pp), FEFnParamType(cp), cxt) )
	    {
	      fprintf(stderr,
		"%s: redefinition of %s differs in type of %s %s\n",
		FilePosShow(cfv->file_pos), NameShow(cfv->name),
		"optional parameter", NameShow(FEFnParamName(cp)));
	      fprintf(stderr, "  (incompatible parent definition is at %s)\n",
		FilePosShow(pfv->file_pos));
	      success = FALSE;
	    }
	    if( !FEFnParamIsRedefinition(pp, cp) )
	      return FALSE;
	    break;
	  }
	}
      }
      if( !found && pfv->ftype != FEATURE_NONCREATION_CLONE )
      {
	/* clone is a special case, its params arrive late and never wrong */
	fprintf(stderr,"%s: redefinition of %s lacks optional parameter %s\n",
	  FilePosShow(cfv->file_pos), NameShow(cfv->name),
	  NameShow(FEFnParamName(pp)));
	fprintf(stderr, "  (incompatible parent definition is at %s)\n",
	  FilePosShow(pfv->file_pos));
	success = FALSE;
      }
    }

  /* the child's result type must be a subtype of the parent's */
  if( !TypeIsSubType(cfv->result_type, pfv->result_type, &c, cxt) )
  {
    fprintf(stderr, "%s: redefinition of %s has incompatible result type\n",
      FilePosShow(cfv->file_pos), NameShow(cfv->name));
    fprintf(stderr, "  (incompatible parent definition is at %s)\n",
      FilePosShow(pfv->file_pos));
    if( DEBUG2 )
    {
      fprintf(stderr, "  cfv res type: %s\n", TypeShow(cfv->result_type, cxt));
      fprintf(stderr, "  pfv res type: %s\n", TypeShow(pfv->result_type, cxt));
    }
    success = FALSE;
  }
  TypeVarsForwardEnd(pfv->type_vars);

  /* privacy cannot increase */
  if( cfv->is_private && !pfv->is_private )
  {
    fprintf(stderr, "%s: private version of %s inherits public version\n",
      FilePosShow(cfv->file_pos), NameShow(cfv->name));
    fprintf(stderr, "  (the inherited version is at %s)\n",
      FilePosShow(BEFnFeatureInstanceFilePos(pfv->effective_instance)));
  }

  return success;
}


/*****************************************************************************/
/*                                                                           */
/*  void InheritPrecondition(FEFN_FEATURE cfv, FEFN_PRECOND fefn_precond)    */
/*                                                                           */
/*  Inherit fefn_precond into cfv, unless it is already present.             */
/*                                                                           */
/*****************************************************************************/

static void InheritPrecondition(FEFN_FEATURE cfv, FEFN_PRECOND fefn_precond)
{
  int pos;
  if( cfv->inherited_preconditions == NULL )
    ArrayInit(&cfv->inherited_preconditions);
  if( !ArrayContains(cfv->inherited_preconditions, fefn_precond, &pos) )
    ArrayAddLast(cfv->inherited_preconditions, fefn_precond);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureSetCheckInherited(FEFN_FEATURE_SET parent_fvs,        */
/*    FEFN_FEATURE_SET fvs, CLASS_VIEW cv, CONTEXT cxt)                      */
/*                                                                           */
/*  This function is called during the first part of feature inheritance.    */
/*  At that stage, no coercions are present so all feature view sets         */
/*  consist of a single, uncoerced element.                                  */
/*                                                                           */
/*  Parameter parent_fvs is being inherited into cv, where fvs with          */
/*  the same name is already present.  So make sure that parent_fvs is       */
/*  compatible with fvs, then ignore parent_fvs except that its base         */
/*  feature is copied into fvs's view if needed.                             */
/*                                                                           */
/*  If cfv->status == FEFN_FEATURE_INHERITED, then the child is not a child  */
/*  feature at all but a previously inherited parent in the case where       */
/*  there is no child.  In this case, both signatures must have the same     */
/*  effective instance, and no other checks are necessary.                   */
/*                                                                           */
/*  If cfv->status == FEFN_FEATURE_UNINHERITED, then the child is an         */
/*  original child feature, not previously checked against a parent.         */
/*  A full check of all items must be carried out.  The child's base         */
/*  feature is set to the parent's, and its type to FEFN_FEATURE_REDEFINED.  */
/*                                                                           */
/*  If cfv->status == FEFN_FEATURE_REDEFINED, then the child has previously  */
/*  been checked against a parent.  It therefore has the final value of      */
/*  its base feature, and this must be the same as the base feature of       */
/*  the parent, else it is an error.  If they are the same, then we can      */
/*  skip checking the creation status, type variables, and compulsory        */
/*  parameters, since they consequently must be the same.                    */
/*                                                                           */
/*  In case of a redefinition this code also implements inheritance          */
/*  of preconditions.                                                        */
/*                                                                           */
/*  Paramenter parent_cv is the class view containing parent_fvs.  It is     */
/*  used only in the special case of the clone feature, which is handled     */
/*  differently when inheriting from ClassObject.                            */
/*                                                                           */
/*  Parameter cv is the class view in which these feature views are being    */
/*  assembled, and is for error messages only.  In fact, it is used only     */
/*  in the FEFN_FEATURE_INHERITED case, since in the other two cases the     */
/*  (re-)definition provides the best location of the error point.  In the   */
/*  FEFN_FEATURE_INHERITED case there is no definition or redefinition, so   */
/*  the best we can do is the class view.                                    */
/*                                                                           */
/*  A test made at the point where this function is called ensures that      */
/*  predefined object feature views are never passed to this function        */
/*  in parent_fvs, but there could be some in fvs, which would be an error.  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureSetCheckInherited(FEFN_FEATURE_SET parent_fvs,
  FEFN_FEATURE_SET fvs, CLASS_VIEW parent_cv, CLASS_VIEW cv, CONTEXT cxt)
{
  FEFN_FEATURE pfv, cfv;  FEFN_PRECOND fefn_precond;

  if( DEBUG10 )
  {
    fprintf(stderr, "[ FEFnFeatureSetCheckInherited(");
    FEFnFeatureSetDebug(parent_fvs, cxt, stderr, SINGLE_LINE);
    fprintf(stderr, ", ");
    FEFnFeatureSetDebug(fvs, cxt, stderr, SINGLE_LINE);
    fprintf(stderr, ", %s, cxt)\n", NameShow(ClassViewName(cv)));
  }

  /* the feature views must be uncoerced and sole occupants of their sets */
  assert(ArraySize(parent_fvs->fefn_features) == 1);
  pfv = ArrayFirst(parent_fvs->fefn_features);
  assert(pfv->coercion == NULL);
  assert(ArraySize(fvs->fefn_features) == 1);
  cfv = ArrayFirst(fvs->fefn_features);
  assert(cfv->coercion == NULL);

  /* inheriting clone from object is ignored */
  if( cfv->ftype == FEATURE_NONCREATION_CLONE &&
      ClassViewClass(parent_cv) == ClassObject )
    db_return(DEBUG10, "FEFnFeatureSetCheckInherited (clone from object)",TRUE);

  /* inherit preconditions */
  if( pfv->inherited_preconditions != NULL )
    ArrayForEach(pfv->inherited_preconditions, fefn_precond)
      InheritPrecondition(cfv, fefn_precond);
  if( pfv->precondition != NULL )
    InheritPrecondition(cfv, pfv->precondition);

  /* rest depends on inherit state */
  assert(pfv->base_feature != NULL);
  switch( cfv->status )
  {

    case FEFN_FEATURE_UNINHERITED:

      /* redefinition of creation feature not permitted */
      if( cfv->ftype == FEATURE_CREATION )
      {
	fprintf(stderr, "%s: redefinition of creation feature %s\n",
	  FilePosShow(cfv->file_pos), NameFullShow(cfv->name));
	fprintf(stderr, "  (inherited feature is at %s)\n",
	  FilePosShow(pfv->file_pos));
	db_return(DEBUG10, "FEFnFeatureSetCheckInherited", FALSE);
      }

      /* feature type, type variables, compulsory parameters must agree */
      if( !CheckConstantParts(pfv, cfv, cxt) )
	db_return(DEBUG10, "FEFnFeatureSetCheckInherited constant", FALSE);

      /* optional parameters, result type, privacy must be compatible */
      if( !CheckVaryingParts(pfv, cfv, cxt) )
	db_return(DEBUG10, "FEFnFeatureSetCheckInherited varying", FALSE);

      /* base feature is inherited from first parent */
      if( cfv->ftype == FEATURE_NONCREATION_CLONE )
      {
	assert(cfv->base_feature != NULL);
	assert(cfv->base_feature == pfv->base_feature);
      }
      else
      {
	assert(cfv->base_feature == NULL);
	cfv->base_feature = pfv->base_feature;
      }
      cfv->status = FEFN_FEATURE_REDEFINED;
      break;


    case FEFN_FEATURE_INHERITED:

      /* effective instances must agree */
      assert(cfv->base_feature != NULL);
      if( cfv->effective_instance != pfv->effective_instance )
      {
	fprintf(stderr, "%s: in class %s, inherited versions of %s differ\n",
	  FilePosShow(ClassViewFilePos(cv)), NameShow(ClassViewName(cv)),
	  NameShow(cfv->name));
	fprintf(stderr, "  (the incompatible versions are at %s and %s)\n",
	  FilePosShow(BEFnFeatureInstanceFilePos(cfv->effective_instance)),
	  FilePosShow(BEFnFeatureInstanceFilePos(pfv->effective_instance)));
	db_return(DEBUG10, "FEFnFeatureSetCheckInherited inherited", FALSE);
      }

      /* privacy is the minimum of all the inherited versions */
      if( !pfv->is_private )
	cfv->is_private = FALSE;
      break;


    case FEFN_FEATURE_REDEFINED:

      /* base features must agree (except when inheriting clone from object) */
      assert(cfv->base_feature != NULL);
      if( FEFnFeatureBEFnFeature(cfv) != FEFnFeatureBEFnFeature(pfv) )
      {
	fprintf(stderr, "%s: redefinition of %s inherits incompatible %s\n",
	  FilePosShow(cfv->file_pos), NameShow(cfv->name), "base features");
	fprintf(stderr, "  (incompatible versions are at %s and %s)\n",
	  FilePosShow(BEFnFeatureOrigFilePos(cfv->base_feature)),
	  FilePosShow(BEFnFeatureOrigFilePos(pfv->base_feature)));
	db_return(DEBUG10, "FEFnFeatureSetCheckInherited redef", FALSE);
      }

      /* optional parameters, result type, privacy must be compatible */
      if( !CheckVaryingParts(pfv, cfv, cxt) )
	db_return(DEBUG10, "FEFnFeatureSetCheckInherited redef varying", FALSE);
      break;


    default:

      assert(FALSE);
  }

  db_return(DEBUG10, "FEFnFeatureSetCheckInherited", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE_SET FEFnFeatureSetCopyInherited(FEFN_FEATURE_SET parent_fvs,*/
/*    CLASS_VIEW cv)                                                         */
/*                                                                           */
/*  This function is called during Stage 6 of the front end.  At that        */
/*  stage, no coercions are present so all feature view sets consist of      */
/*  a single, uncoerced element.                                             */
/*                                                                           */
/*  In this case, parent_fvs is being inherited into class view cv, where    */
/*  no existing feature view set has the same name.  So copy parent_fvs,     */
/*  ready for insertion into that class view.  The copying ensures that      */
/*  any required type substitutions are made.                                */
/*                                                                           */
/*  A test made at the point where this function is called ensures that      */
/*  predefined object features views are never passed to this function.      */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE_SET FEFnFeatureSetCopyInherited(FEFN_FEATURE_SET parent_fvs,
  CLASS_VIEW cv)
{
  FEFN_FEATURE pfv, cfv;  FEFN_FEATURE_SET res;

  /* get the feature view and check all correct for Stage 6 */
  assert(ArraySize(parent_fvs->fefn_features) == 1);
  pfv = ArrayFirst(parent_fvs->fefn_features);
  assert(pfv->coercion == NULL);

  /* copy the feature view and package it into a new feature view set */
  assert(pfv->ftype != FEATURE_NONCREATION_CLONE);
  cfv = FEFnFeatureCopy(pfv, pfv->coercion, cv);
  res = FEFnFeatureSetNew(cfv->file_pos, cfv->name);
  ArrayAddLast(res->fefn_features, cfv);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int FEFnFeatureCoercionLengthCmp(const void *t1, const void *t2)         */
/*                                                                           */
/*  Comparison function for sorting an array of feature views into           */
/*  increasing coercion length.                                              */
/*                                                                           */
/*****************************************************************************/

static int FEFnFeatureCoercionLengthCmp(const void *t1, const void *t2)
{
  FEFN_FEATURE fv1 = * (FEFN_FEATURE *) t1;
  FEFN_FEATURE fv2 = * (FEFN_FEATURE *) t2;
  return CoercionLength(fv1->coercion) - CoercionLength(fv2->coercion);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureSetAddCoercedCopy(FEFN_FEATURE_SET fvs,               */
/*    FEFN_FEATURE fv, COERCION coercion, CLASS_VIEW cv)                     */
/*                                                                           */
/*  Add a new copy of fv to fvs (which lies in cv), coerced by coercion.     */
/*                                                                           */
/*  A check made where this function is called ensures that fv is never      */
/*  a predefined feature; but fvs could contain one, in which case           */
/*  it is an error.                                                          */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureSetAddCoercedCopy(FEFN_FEATURE_SET fvs,
  FEFN_FEATURE fv, COERCION coercion, CLASS_VIEW cv)
{
  FEFN_FEATURE cfv, new_fv;

  cfv = FEFnFeatureSetSoleUncoercedMember(fvs);
  if( cfv != NULL && cfv->ftype >= FEATURE_PREDEF )
  {
    /* inheriting onto predefined is not allowed */
    fprintf(stderr, "%s: inherited and predefined features name clash\n",
      FilePosShow(cfv->file_pos));
    fprintf(stderr, "  %s: %s (predefined)\n", FilePosShow(cfv->file_pos),
      NameFullShow(cfv->name));
    fprintf(stderr, "  %s: %s (inherited)\n", FilePosShow(fv->file_pos),
      NameFullShow(fv->name));
    return FALSE;
  }
  else
  {
    /* if a feature view with this coercion is already present, we */
    /* prefer it and ignore this one (but it's not an error)       */
    ArrayForEach(fvs->fefn_features, cfv)
      if( CoercionEqual(cfv->coercion, coercion) )
	return TRUE;

    /* all OK, add fv to fvs */
    new_fv = FEFnFeatureCopy(fv, coercion, cv);
    ArrayAddLast(fvs->fefn_features, new_fv);
    ArraySort(fvs->fefn_features, &FEFnFeatureCoercionLengthCmp);
    return TRUE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureMerge(FEFN_FEATURE tfv, FEFN_FEATURE sfv,             */
/*    CONTEXT cxt, FILE_POS file_pos)                                        */
/*                                                                           */
/*  Merge sfv into tsf.  Their coercions are the same, so they must be       */
/*  compatible.                                                              */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FEFnFeatureMerge(FEFN_FEATURE tfv, FEFN_FEATURE sfv,
  CONTEXT cxt, FILE_POS file_pos)
{
  int tcompulsory, scompulsory, ti, si;  FEFN_PARAM tp, sp;  BOOLEAN found;

  /* base feature instances must be equal */
  assert(tfv->ftype < FEATURE_PREDEF);
  assert(sfv->ftype < FEATURE_PREDEF);
  if( FEFnFeatureBEFnFeature(tfv) != FEFnFeatureBEFnFeature(sfv) )
  {
    fprintf(stderr, "%s: incompatible base features for %s in meet type\n",
      FilePosShow(file_pos), NameShow(tfv->name));
    fprintf(stderr, "  (incompatible base features declared at %s and %s)\n",
      FilePosShow(BEFnFeatureOrigFilePos(sfv->base_feature)),
      FilePosShow(BEFnFeatureOrigFilePos(tfv->base_feature)));
    return FALSE;
  }

  /* coercions will necessarily be equal, otherwise would not have called */
  assert(CoercionEqual(tfv->coercion, sfv->coercion));

  /* creation status does not need to be checked separately       */
  /* type variables do not need to be checked separately          */

  /* compulsory parameters must necessarily be equal in number */
  tcompulsory = 0;
  ArrayForEach(tfv->parameters, tp)
    if( !FEFnParamIsOptional(tp) )
      tcompulsory++;
  scompulsory = 0;
  ArrayForEach(sfv->parameters, sp)
    if( !FEFnParamIsOptional(sp) )
      scompulsory++;
  assert(tcompulsory == scompulsory);

  /* but compulsory parameters must also have the same names */
  ti = si = 0;
  while( ti < ArraySize(tfv->parameters) && si < ArraySize(sfv->parameters) )
  {
    tp = ArrayGet(tfv->parameters, ti);
    sp = ArrayGet(sfv->parameters, ti);
    if( FEFnParamIsOptional(tp) )
      ti++;
    else if( FEFnParamIsOptional(sp) )
      si++;
    else
    {
      if( !NameEqual(FEFnParamName(tp), FEFnParamName(sp)) )
      {
	fprintf(stderr, "%s: incompatible names for parameter in meet\n",
	  FilePosShow(file_pos));
	fprintf(stderr, "  (incompatible parameters are at %s and %s)\n",
	  FilePosShow(FEFnParamFilePos(tp)), FilePosShow(FEFnParamFilePos(sp)));
	return FALSE;
      }
      ti++;
      si++;
    }
  }

  /* optional parameters must be compatible, and new ones merged in */
  TypeVarsForwardBegin(sfv->type_vars, tfv->type_vars, file_pos);
  ArrayForEach(sfv->parameters, sp)  if( FEFnParamIsOptional(sp) )
  {
    found = FALSE;
    ArrayForEach(tfv->parameters, tp)  if( FEFnParamIsOptional(tp) )
    {
      if( NameEqual(FEFnParamName(tp), FEFnParamName(sp)) )
      {
	found = TRUE;
	if( !TypeEqual(FEFnParamType(tp), FEFnParamType(sp), cxt) )
	{
	  fprintf(stderr, "%s: incompatible types for parameter %s in meet\n",
	    FilePosShow(file_pos), NameShow(FEFnParamName(tp)));
	  fprintf(stderr, "  (incompatible parameters are at %s and %s)\n",
	    FilePosShow(FEFnParamFilePos(tp)), FilePosShow(FEFnParamFilePos(sp)));
	  TypeVarsForwardEnd(sfv->type_vars);
	  return FALSE;
	}
      }
    }
    if( !found )
    {
      /* sp not found, so merge it in, copying to get type right */
      sp = FEFnParamCopy(sp, FEFnParamIsPrivate(sp));
      ArrayAddLast(tfv->parameters, sp);
    }
  }

  /* result types must be met */
  if( !TypeMeet(&tfv->result_type, sfv->result_type, cxt) )
  {
    fprintf(stderr, "%s: incompatible result types for feature %s in meet\n",
      FilePosShow(file_pos), NameShow(tfv->name));
    fprintf(stderr, "  (incompatible features are at %s and %s)\n",
      FilePosShow(tfv->file_pos), FilePosShow(sfv->file_pos));
    TypeVarsForwardEnd(sfv->type_vars);
    return FALSE;
  }
  TypeVarsForwardEnd(sfv->type_vars);
  tfv->effective_instance = NULL;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureSetMerge(FEFN_FEATURE_SET *target,                    */
/*    FEFN_FEATURE_SET source, CONTEXT cxt, FILE_POS file_pos)               */
/*                                                                           */
/*  Merge copies of the feature views of source into target.  The copying    */
/*  may cause substitutions in the types of the feature view's function      */
/*  header.  Parameter *target may be NULL, in which case it must be         */
/*  initialized to an empty feature view set before we do anything else.     */
/*                                                                           */
/*  This function is called by TypeRetrieveFEFnFeatureSet where it is        */
/*  needed mainly when there is a meet type to retrieve a feature from,      */
/*  although it does utilize the type copying aspect of this function        */
/*  even when there is no meet type.                                         */
/*                                                                           */
/*  There may be any number of feature views in the sets.                    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureSetMerge(FEFN_FEATURE_SET *target,
  FEFN_FEATURE_SET source, CONTEXT cxt, FILE_POS file_pos)
{
  FEFN_FEATURE sfv, tfv;  BOOLEAN found;
  if( DEBUG8 )
  {
    fprintf(stderr, "  [ FEFnFeatureSetMerge(target, source)\n");
    fprintf(stderr, "      *target: ");
    FEFnFeatureSetDebug(*target, cxt, stderr, 6);
    fprintf(stderr, "\n");
    fprintf(stderr, "      source:  ");
    FEFnFeatureSetDebug(source, cxt, stderr, 6);
    fprintf(stderr, "\n");
  }

  /* make *target if currently null */
  if( *target == NULL )
    *target = FEFnFeatureSetNew(source->file_pos, source->name);

  /* merge in each source feature view */
  ArrayForEach(source->fefn_features, sfv)
  {
    found = FALSE;
    ArrayForEach((*target)->fefn_features, tfv)
    {
      if( CoercionEqual(tfv->coercion, sfv->coercion) )
      {
	/* same coercions, so these should merge unless predefined */
	if( tfv->ftype >= FEATURE_PREDEF || sfv->ftype >= FEATURE_PREDEF ||
	    !FEFnFeatureMerge(tfv, sfv, cxt, file_pos) )
	  return FALSE;
	found = TRUE;
      }
    }
    if( !found )
    {
      /* no element of target has same coercion as sfv; copy sfv into target */
      tfv = FEFnFeatureCopy(sfv, sfv->coercion, sfv->class_view);
      ArrayAddLast((*target)->fefn_features, tfv);
    }
  }

  /* sort *target into increasing coercion length */
  if( ArraySize((*target)->fefn_features) > 1 )
    ArraySort((*target)->fefn_features, &FEFnFeatureCoercionLengthCmp);
  if( DEBUG8 )
  {
    fprintf(stderr, "  ] FEFnFeatureSetMerge returning\n");
    fprintf(stderr, "      *target: ");
    FEFnFeatureSetDebug(*target, cxt, stderr, 6);
    fprintf(stderr, "\n");
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "miscellaneous".                                               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureManifestInterface(FEFN_FEATURE fv, CONTEXT cxt)       */
/*                                                                           */
/*  Instantiate the interface of fv in cxt.  Imported features already       */
/*  have instantiated interfaces, so we skip this step for them.             */
/*                                                                           */
/*  On entry, cxt has three layers:  a class views layer, a class view       */
/*  generic parameters layer, and a top empty layer that is used by this     */
/*  function.  On exit, these three layers are still present and the top     */
/*  layer may be dirty.  The caller is required to clean it up.  It's done   */
/*  this way because there are many error exit points in this code and       */
/*  it would be painful to clean up at each of those.                        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureManifestInterface(FEFN_FEATURE fv, CONTEXT cxt)
{
  FEFN_PARAM param;
  if( DEBUG4 )
    fprintf(stderr, "[ FEFnFeatureManifestInterface(%s)\n",
      NameShow(FEFnFeatureName(fv)));

  if( !fv->is_imported )
  {
    /* in current module, there has been no inheritance yet */
    assert(fv->status == FEFN_FEATURE_UNINHERITED);

    /* manifest formal generics and add them to context */
    if( !TypeVarsBeginManifest(fv->type_vars, cxt, !fv->is_private) )
      db_return(DEBUG4, "FEFnFeatureManifestInterface typevars 1", FALSE);

    /* manifest parameters (not their bodies) and add them to context */
    if( fv->parameters != NULL )
      ArrayForEach(fv->parameters, param)
	if( !FEFnParamManifest(param, cxt, !fv->is_private) )
	  db_return(DEBUG4, "FEFnFeatureManifestInterface params", FALSE);

    /* manifest the result type */
    if( !TypeManifest(fv->result_type, cxt, !fv->is_private) )
      db_return(DEBUG4, "FEFnFeatureManifestInterface result", FALSE);
  }
  
  /* success */
  db_return(DEBUG4, "FEFnFeatureManifestInterface", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnFeatureManifest(FEFN_FEATURE fv, CONTEXT cxt, TYPE self_type)*/
/*                                                                           */
/*  Instantiate the body of fv, including working out what each name         */
/*  inside it refers to, and the type of each subexpression.                 */
/*                                                                           */
/*  Any inner letdefs have to be told which C function they are used by,     */
/*  determining where in the C code they will be code generated.  For        */
/*  predefined object features this is the class initialization function;    */
/*  for creation features it is the class creation function; and for         */
/*  noncreation features it is the backend feature corresponding to fv.      */
/*                                                                           */
/*  On entry, cxt has four layers:  a class views layer, a class view        */
/*  generic parameters layer, a class view features layer, and a top empty   */
/*  layer that is used by this function.  On exit, these four layers are     */
/*  still present and the top layer may be dirty.  The caller is required    */
/*  to clean it up.  It's done this way because there are many error exit    */
/*  points in this code and it would be painful to clean up at each of them. */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnFeatureManifestBody(FEFN_FEATURE fv, CONTEXT cxt, TYPE self_type)
{
  BEFN encl_befn;  COERCION c;  FEFN_PARAM param;
  if( DEBUG4 )
    fprintf(stderr, "[ FEFnFeatureManifestBody(%s)\n",
      NameShow(FEFnFeatureName(fv)));

  /* creation, inherited, and clone features don't have to be done */
  assert(fv->ftype != FEATURE_CREATION);
  if( fv->status == FEFN_FEATURE_INHERITED ||
      fv->ftype == FEATURE_NONCREATION_CLONE )
    db_return(DEBUG4, "FEFnFeatureManifestBody", TRUE);

  /* find the back-end function appropriate to this feature */
  if( fv->ftype == FEATURE_PREDEF )
    encl_befn = (BEFN) ClassBEFnClassInit(ClassViewClass(fv->class_view));
  else
    encl_befn = (BEFN) FEFnFeatureBEFnFeature(fv);

  /* formal generics (types already manifested) */
  if( !TypeVarsBegin(fv->type_vars, cxt) )
    db_return(DEBUG4, "FEFnFeatureManifestBody typevars", FALSE);

  /* parameters (types already manifested) */
  if( fv->parameters != NULL && fv->body != NULL )
    ArrayForEach(fv->parameters, param)
    {
      /* manifest the default values of optional parameters */
      if( FEFnParamKind(param) == PARAM_OPTIONAL )
      {
	if( !ExprFnHeadManifestParamDefaultVal(fv->body, param, cxt,
	    self_type, encl_befn) )
	  db_return(DEBUG4, "FEFnFeatureManifestBody params 1", FALSE);
      }

      /* add parameter to context */
      if( !ContextInsertFEFnParam(cxt, param) )
	db_return(DEBUG4, "FEFnFeatureManifestBody params 2", FALSE);
    }

  /* the precondition, if any */
  if( fv->precondition != NULL )
    if( !FEFnPrecondManifest(fv->precondition, cxt, self_type) )
      db_return(DEBUG4, "FEFnFeatureManifestBody precond", FALSE);

  /* the body, if any */
  if( fv->body != NULL )
  {
    /* manifest the body */
    if( !ExprManifest(&fv->body, cxt, self_type, encl_befn) )
      db_return(DEBUG4, "FEFnFeatureManifestBody body", FALSE);

    /* check that body type matches the declared result type */
    if( ExprType(fv->body) != NULL )
    {
      TypeRangeMarkBegin();
      if( !TypeIsSubType(ExprType(fv->body), fv->result_type, &c, cxt) )
      {
	fprintf(stderr,
	  "%s: type of feature body (%s) does not match result type (%s)\n",
	  FilePosShow(ExprFilePos(fv->body)), TypeShow(ExprType(fv->body), cxt),
	  TypeShow(fv->result_type, cxt));
	TypeRangeMarkEnd(FALSE);
	db_return(DEBUG4, "FEFnFeatureManifestBody type", FALSE);
      }
      if( c != NULL )
	ExprFnHeadInsertCoercion(fv->body, c, cxt, self_type);
      TypeRangeMarkEnd(TRUE);
    }
  }

  /* *** old code
    res = FEFnManifest((FEFN) fv, cxt, is_public := FALSE, shadow_lim := TRUE,
      do_interface := FALSE, fv->body, self_type, encl_befn);
  *** */
  db_return(DEBUG4, "FEFnFeatureManifestBody", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnFeatureAddToClass(FEFN_FEATURE fv, CLASS c)                     */
/*                                                                           */
/*  We've added fv to some class view, so now we have to add the feature it  */
/*  refers to to the class corresponding to that class view.                 */
/*                                                                           */
/*  If fv is clone we still have to do this, but the feature we add is       */
/*  the clone feature from the disjoint set of classes containing c.         */
/*  It will already be linked to fv.  If fv is uninherited, we make this     */
/*  base feature point back to fv.  This ensures that whatever class         */
/*  originally stimulated the creation of the back-end clone feature, in     */
/*  the end it will appear to have been created in a class that does not     */
/*  inherit from any other class than object.  It is not essential to do     */
/*  this, but it makes the generated code for clone closer to that for       */
/*  other features.                                                          */
/*                                                                           */
/*****************************************************************************/

void FEFnFeatureAddToClass(FEFN_FEATURE fv, CLASS c)
{
  BEFN_FEATURE base_feature;  FEFN_PRECOND precond;
  if( DEBUG9 )
  {
    fprintf(stderr, "    [ FEFnFeatureAddToClass(");
    FEFnFeatureDebug(fv, NULL, stderr);
    fprintf(stderr, ", %s)\n", NameShow(ClassViewName(ClassOrigClassView(c))));
  }
  if( !fv->is_imported ) switch( fv->status )
  {
    case FEFN_FEATURE_UNINHERITED:

      /* original to class, so make a new feature and a new instance */
      if( fv->ftype == FEATURE_NONCREATION_CLONE )
      {
	/* clone already has base feature; see above for resets */
	assert(fv->base_feature != NULL);
	BEFnResetBaseFEFn(fv->base_feature, fv);
	BEFnParamResetOrigFEFnParam(BEFnFeatureSelfParam(fv->base_feature),
	  FEFnSelfParam((FEFN) fv));
	/* fv->base_feature = ClassDSCloneBEFnFeature(c, fv); */
      }
      else
      {
	/* ordinary feature view, just make a new feature */
	assert(fv->base_feature == NULL);
	fv->base_feature =
	  BEFnFeatureMake(c, fv, fv->ftype, fv->code1, fv->code2);
      }
      fv->effective_instance = BEFnFeatureRegisterNewInstance(
	fv->base_feature, c, fv->file_pos, fv->body);
      ClassAddBEFnFeature(c, fv->base_feature, fv->is_abstract, TRUE);
      FEFnParamListRegisterBaseParameters(fv->parameters, fv->base_feature);
      if( fv->precondition != NULL )
	BEFnFeatureRegisterPrecondition(fv->effective_instance,
	  (BEFN_PRECOND) FEFnBEFn((FEFN) fv->precondition));
      break;


    case FEFN_FEATURE_INHERITED:

      /* inherited, just needs adding, not creating */
      assert(fv->ftype != FEATURE_NONCREATION_CLONE);
      assert(fv->base_feature != NULL);
      base_feature = FEFnFeatureBEFnFeature(fv);
      assert(fv->effective_instance != NULL);
      BEFnFeatureRegisterExistingInstance(fv->effective_instance, c);
      ClassAddBEFnFeature(c, base_feature, fv->is_abstract, FALSE);
      FEFnParamListRegisterBaseParameters(fv->parameters, base_feature);
      break;


    case FEFN_FEATURE_REDEFINED:

      /* redefined, needs a new instance */
      assert(fv->base_feature != NULL);
      base_feature = FEFnFeatureBEFnFeature(fv);
      fv->effective_instance = BEFnFeatureRegisterNewInstance(base_feature,
	c, fv->file_pos, fv->body);
      ClassAddBEFnFeature(c, base_feature, fv->is_abstract, FALSE);
      FEFnParamListRegisterBaseParameters(fv->parameters, base_feature);

      /* add precondition back-end functions to new instance */
      if( fv->inherited_preconditions != NULL )
	ArrayForEach(fv->inherited_preconditions, precond)
	  BEFnFeatureRegisterPrecondition(fv->effective_instance,
	    (BEFN_PRECOND) FEFnBEFn((FEFN) precond));
      if( fv->precondition != NULL )
	BEFnFeatureRegisterPrecondition(fv->effective_instance,
	  (BEFN_PRECOND) FEFnBEFn((FEFN) fv->precondition));
      break;


    default:

      assert(FALSE);
  }
  if( DEBUG9 )
    fprintf(stderr, "    ] FEFnFeatureAddToClass returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnFeatureAddToCreation(FEFN_FEATURE fv, FEFN_CREATION cfv)        */
/*                                                                           */
/*  Feature view fv is a creation feature, and we need to update cfv, the    */
/*  creation function of its class view and possibly that of the class       */
/*  view's class to reflect the acceptance of fv into its class view.        */
/*                                                                           */
/*  This is done by adding a parameter to fv's class view's creation         */
/*  function view, and, if fv is not imported, adding a parameter to         */
/*  fv's class view's class's creation function.  If fv is imported,         */
/*  we need to link the new parameter to the back-end parameter of fv's      */
/*  original version.                                                        */
/*                                                                           */
/*****************************************************************************/

void FEFnFeatureAddToCreation(FEFN_FEATURE fv, FEFN_CREATION cfv)
{
  BEFN befn;  PARAM_KIND param_kind;  BEFN_PARAM befn_param;
  FEFN_PARAM param;
  assert(fv->ftype == FEATURE_CREATION);

  /* add a new front-end parameter to cfv, based on fv */
  param_kind = fv->is_abstract ? PARAM_COMPULSORY : PARAM_OPTIONAL;
  param = FEFnParamNew(fv->file_pos, fv->name, fv->result_type, NULL,
    param_kind, fv->is_private, (FEFN) fv);
  FEFnAddParameter((FEFN) cfv, param);

  /* register this parameter's type as one potentially requiring swizzling */
  TypeSwizzleRegister(FEFnParamType(param));

  /* find or make the corresponding back-end parameter */
  befn = FEFnBEFn((FEFN) cfv);
  if( fv->is_imported )
  {
    /* imported, so there must be a suitable parameter already; find it */
    befn_param = BEFnFindParamFromCreationFeature(befn, fv->base_feature);
  }
  else
  {
    /* not imported, so we need a new parameter of befn */
    assert(fv->base_feature != NULL);
    befn_param = BEFnParamNew(param_kind, param, fv->base_feature);
    BEFnAddParameter(befn, befn_param);
  }

  /* link the front-end parameter to befn_param */
  FEFnParamSetBEFnParam(param, befn_param);
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_CREDFT FEFnCreationDefaultValue(FEFN_FEATURE fv)                    */
/*                                                                           */
/*  Return the creation default value of fv, which must be a creation        */
/*  feature.  The returned value will be NULL if there is no default         */
/*  value.                                                                   */
/*                                                                           */
/*****************************************************************************/

FEFN_CREDFT FEFnCreationDefaultValue(FEFN_FEATURE fv)
{
  assert(fv->ftype == FEATURE_CREATION);
  return fv->fefn_credft;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_PARAM FEFnFeatureAddToClone(FEFN_FEATURE fv, FEFN_FEATURE clone_fv) */
/*                                                                           */
/*  Add a parameter to clone_fv corresponding to creation feature view fv.   */
/*                                                                           */
/*  Since clone feature views have the same parameters as creation           */
/*  functions, this is very similar to FEFnFeatureAddToCreation.  The        */
/*  main difference is that creation functions are never inherited, so       */
/*  each creation function has a distinct backend function, whereas          */
/*  clone features share backend functions and backend parameters with       */
/*  their ancestors.                                                         */
/*                                                                           */
/*****************************************************************************/

void FEFnFeatureAddToClone(FEFN_FEATURE fv, FEFN_FEATURE clone_fv)
{
  FEFN_PARAM param;  PARAM_KIND param_kind;
  BEFN befn;  BEFN_PARAM befn_param;

  /* add a new front-end parameter to clone_fv, based on fv */
  param_kind = PARAM_OPTIONAL;
  param = FEFnParamNew(clone_fv->file_pos, fv->name,
    fv->result_type, NULL, param_kind, fv->is_private, (FEFN) fv);
  FEFnAddParameter((FEFN) clone_fv, param);

  /* find or make the corresponding back-end parameter */
  befn = FEFnBEFn((FEFN) clone_fv);
  if( fv->is_imported || fv->status != FEFN_FEATURE_UNINHERITED )
  {
    /* imported or inherited, so befn_param must exist already; find it */
    befn_param = BEFnFindParamFromCreationFeature(befn, fv->base_feature);
  }
  else
  {
    /* not imported or inherited, so we need a new parameter of befn */
    befn_param = BEFnParamNew(PARAM_OPTIONAL, param, fv->base_feature);
    BEFnAddParameter(befn, befn_param);
  }

  /* link the front-end parameter to befn_param */
  FEFnParamSetBEFnParam(param, befn_param);
}
