/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         fefn_invariant.c                                           */
/*  DESCRIPTION:  Invariants                                                 */
/*                                                                           */
/*****************************************************************************/
#include <string.h>
#include "externs.h"
#define DEBUG1 0


/*****************************************************************************/
/*                                                                           */
/*  FEFN_INVARIANT                                                           */
/*                                                                           */
/*****************************************************************************/

struct fefn_invariant_rec {		/* inherits from FEFN                */
  KIND_TAG		kind_tag;	/* kind of entity                    */
  FILE_POS		file_pos;	/* position in file                  */
  NAME			name;		/* "invariant"                       */
  TYPE_VARS		type_vars;	/* generic parameters (always NULL)  */
  ARRAY_FEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  TYPE			result_type;	/* always boolean                    */
  BEFN_INVARIANT	befn_invariant;	/* corresponding C function          */
  CLASS_VIEW		class_view;	/* that this invariant view lies in  */
  ARRAY_EXPR		exprs;		/* the actual expressions            */
};


/*****************************************************************************/
/*                                                                           */
/*  FEFN_INVARIANT FEFnInvtNew(FILE_POS file_pos, CLASS_VIEW cv)             */
/*                                                                           */
/*  Return a new, empty invariant function with these attributes.            */
/*                                                                           */
/*****************************************************************************/

FEFN_INVARIANT FEFnInvtNew(FILE_POS file_pos, CLASS_VIEW cv)
{
  FEFN_INVARIANT res;  FEFN_PARAM param;
  GetMemory(res, FEFN_INVARIANT);

  /* make new fefn_invariant */
  res->kind_tag = KIND_FEFN_INVARIANT;
  res->file_pos = file_pos;
  res->name = NameNew(AStringToUString("invariant"));
  res->type_vars = NULL;
  ArrayInit(&res->parameters);
  param = FEFnParamNew(file_pos, NameNew(AStringToUString("self")),
    ClassType(ClassViewClass(cv)), NULL, PARAM_COMPULSORY, FALSE, NULL);
  ArrayAddLast(res->parameters, param);
  res->result_type = NULL;
  res->class_view = cv;
  ArrayInit(&res->exprs);

  /* make an invariant for this to point to, if not made already */
  if( ClassInvariant(ClassViewClass(cv)) == NULL )
    ClassMakeInvariant(ClassViewClass(cv), param);

  /* point res to the invariant, and point its self parameter to invt's */
  res->befn_invariant = ClassInvariant(ClassViewClass(cv));
  FEFnParamSetBEFnParam(param, BEFnInvtSelfParam(res->befn_invariant));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnInvtAddExpr(FEFN_INVARIANT iv, EXPR e)                          */
/*                                                                           */
/*  Add e as an invariant to iv.                                             */
/*                                                                           */
/*****************************************************************************/

void FEFnInvtAddExpr(FEFN_INVARIANT iv, EXPR e)
{
  ArrayAddLast(iv->exprs, e);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnInvtManifest(FEFN_INVARIANT iv, CONTEXT cxt)                 */
/*                                                                           */
/*  Manifest the expressions of iv in cxt, after first enclosing each        */
/*  of them in assert().  This can't be done earlier, because the            */
/*  assert function is not defined until after the first module has          */
/*  been read.                                                               */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnInvtManifest(FEFN_INVARIANT iv, CONTEXT cxt)
{
  TYPE self_type;  int i;  COERCION c;  EXPR inv;
  iv->result_type = ClassType(ClassVoid);
  ContextPushEmpty(cxt, NULL, TRUE);
  if( !ContextInsertFEFnParam(cxt, ArrayFirst(iv->parameters)) )
    db_return(DEBUG1, "FEFnInvtManifest param", FALSE);
  self_type = ClassType(ClassViewClass(iv->class_view));
  for( i = 0;  i < ArraySize(iv->exprs);  i++ )
  {
    /* manifest the invariant */
    inv = ArrayGet(iv->exprs, i);
    if( !ExprManifest(&inv, cxt, self_type, (BEFN) iv->befn_invariant) )
      db_return(DEBUG1, "FEFnInvtManifest body", FALSE);

    /* make sure its type is boolean */
    if( !TypeIsSubType(ExprType(inv), ClassType(ClassBool), &c, cxt) )
    {
      fprintf(stderr, "%s: invariant has type %s\n",
	FilePosShow(ExprFilePos(inv)), TypeShow(ExprType(inv), cxt));
      db_return(DEBUG1, "FEFnInvtManifest type", FALSE);
    }

    /* enclose in assert() */
    inv = ExprEncloseInInstantiatedAssert(inv);

    /* replace in array and report to backend function */
    ArrayPut(iv->exprs, i, inv);
    BEFnInvtAddExpr(iv->befn_invariant, inv);
  }
  ContextPop(cxt, TRUE);
  return TRUE;
}
