/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         expr_let.c                                                 */
/*  DESCRIPTION:  Nonpareil let expressions                                  */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#define	DEBUG1 0
#define	DEBUG2 0
#define	DEBUG3 0


/*****************************************************************************/
/*                                                                           */
/*  FEFN_LETDEF                                                              */
/*                                                                           */
/*  A let definition.                                                        */
/*                                                                           */
/*****************************************************************************/

struct fefn_letdef_rec {		/* inherits from FEFN                */
  KIND_TAG		kind_tag;	/* kind of entity                    */
  FILE_POS		file_pos;	/* position of function name in file */
  NAME			name;		/* name of function                  */
  TYPE_VARS		type_vars;	/* generic parameters (NULL if none) */
  ARRAY_FEFN_PARAM	parameters;	/* ordinary params (NULL if none)    */
  TYPE			result_type;	/* result type                       */
  BEFN_LETDEF		befn_letdef;	/* corresponding C function          */

  EXPR			body;		/* body of letdef                    */
};


/*****************************************************************************/
/*                                                                           */
/*  FEFN_LETDEF FEFnLetDefNew(FILE_POS file_pos, NAME name,                  */
/*    TYPE_VARS type_vars, ARRAY_FEFN_PARAM parameters, TYPE result_type,    */
/*    EXPR body)                                                             */
/*                                                                           */
/*  Return a new FEFN_LETDEF with these attributes.  Also initialize the     */
/*  corresponding C function.  It has parallel parameters, and it            */
/*  shares the body.                                                         */
/*                                                                           */
/*****************************************************************************/

FEFN_LETDEF FEFnLetDefNew(FILE_POS file_pos, NAME name, TYPE_VARS type_vars,
  ARRAY_FEFN_PARAM parameters, TYPE result_type, EXPR body)
{
  FEFN_LETDEF res;  FEFN_PARAM param;  BEFN_PARAM befn_param;
  GetMemory(res, FEFN_LETDEF);
  res->kind_tag = KIND_FEFN_LETDEF;
  res->file_pos = file_pos;
  res->name = name;
  res->type_vars = type_vars;
  res->parameters = parameters;
  res->result_type = result_type;
  res->befn_letdef = BEFnLetDefNew(res, body);
  if( parameters != NULL )
    ArrayForEach(parameters, param)
    {
      befn_param = BEFnParamNew(FEFnParamKind(param), param, NULL);
      BEFnAddParameter((BEFN) res->befn_letdef, befn_param);
      FEFnParamSetBEFnParam(param, befn_param);
    }
  res->body = body;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_FEFN_LETDEF FEFnLetDefsCopyUninstantiated(                         */
/*    ARRAY_FEFN_LETDEF letdefs, ARRAY_FEFN_PARAM orig_params,               */
/*    ARRAY_FEFN_PARAM copy_params)                                          */
/*                                                                           */
/*  Make a copy of uninstantiated letdefs array letdefs.                     */
/*                                                                           */
/*****************************************************************************/

ARRAY_FEFN_LETDEF FEFnLetDefsCopyUninstantiated(ARRAY_FEFN_LETDEF letdefs,
  ARRAY_FEFN_PARAM orig_params, ARRAY_FEFN_PARAM copy_params)
{
  ARRAY_FEFN_LETDEF res_array;  FEFN_LETDEF letdef, res;
  FEFN_PARAM param;  BEFN_PARAM befn_param;
  if( letdefs == NULL )
    return NULL;
  ArrayInit(&res_array);
  ArrayForEach(letdefs, letdef)
  {
    GetMemory(res, FEFN_LETDEF);
    res->kind_tag = KIND_FEFN_LETDEF;
    res->file_pos = letdef->file_pos;
    res->name = letdef->name;
    res->type_vars = TypeVarsCopyUninstantiated(letdef->type_vars);
    res->befn_letdef = BEFnLetDefNew(res, letdef->body);
    res->parameters = FEFnParamListCopyUninstantiated(letdef->parameters);
    if( res->parameters != NULL )
      ArrayForEach(res->parameters, param)
      {
	befn_param = BEFnParamNew(FEFnParamKind(param), param, NULL);
	BEFnAddParameter((BEFN) res->befn_letdef, befn_param);
	FEFnParamSetBEFnParam(param, befn_param);
      }
    res->result_type = letdef->result_type == NULL ? NULL :
      TypeCopyUninstantiated(letdef->result_type);
    res->body = ExprCopyUninstantiated(letdef->body, orig_params, copy_params);
    ArrayAddLast(res_array, res);
  }
  return res_array;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnLetDefAddHiddenParameter(FEFN_LETDEF letdef, NAMED *named)   */
/*                                                                           */
/*  Add a hidden parameter to letdef, corresponding to the given named       */
/*  object, and change *named to the new hidden parameter.  Return TRUE      */
/*  if successful - FALSE can only mean an attempt to insert a hidden        */
/*  parameter after a call on the function has already been manifested.      */
/*                                                                           */
/*****************************************************************************/

/* moved to fefn.c now
BOOLEAN FEFnLetDefAddHiddenParameter(FEFN_LETDEF letdef, NAMED *named)
{
  TYPE type;  FEFN_PARAM np_param;  BEFN_PARAM befn_param;
  if( DEBUG3 )
    fprintf(stderr, "[ FEFnLetDefAddHiddenParameter(%s%s, \"%s\" %s)\n",
      NameShow(FEFnName((FEFN) letdef)),
      FEFnSignatureShow((FEFN) letdef, NULL),
      KindShow(NamedKind(*named)), NameShow(NamedName(*named)));

  ** make sure that a call on letdef has not already occurred **
  if( BEFnUtilized( (BEFN) letdef->befn_letdef) )
  {
    fprintf(stderr, "%s: recursive letdef uses outer variables: simplify!\n",
      FilePosShow(letdef->file_pos));
    return FALSE;
  }

  ** make and add a Nonpareil hidden parameter **
  type = FEFnResultType((FEFN) *named);
  np_param = FEFnParamNew(NamedFilePos(*named), NamedName(*named), type,
    NULL, PARAM_HIDDEN, FALSE, NULL, (FEFN) *named);
  FEFnAddParameter((FEFN) letdef, np_param);

  ** make and add a C hidden parameter, and point np_param to befn_param **
  befn_param = BEFnParamNew(PARAM_HIDDEN, np_param, NULL);
  BEFnAddParameter((BEFN) letdef->befn_letdef, befn_param);
  FEFnParamSetBEFnParam(np_param, befn_param);
  if( DEBUG3 )
    fprintf(stderr, "] FEFnLetDefAddHiddenParameter returning %s%s\n",
      NameShow(FEFnName((FEFN) np_param)),
      FEFnSignatureShow((FEFN) np_param, NULL));
  *named = (NAMED) np_param;
  return TRUE;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnLetDefDoManifest(FEFN_LETDEF letdef, CONTEXT cxt,            */
/*    TYPE self_type, BEFN encl_befn)                                        */
/*                                                                           */
/*  Carry out the actual manifesting of the letdef, assuming that the        */
/*  context is set up correctly.                                             */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN FEFnLetDefDoManifest(FEFN_LETDEF letdef, CONTEXT cxt,
  TYPE self_type, BEFN encl_befn)
{
  FEFN_PARAM param;  COERCION c;

  /* add formal generics to context */
  if( !TypeVarsBeginManifest(letdef->type_vars, cxt, FALSE) )
    return FALSE;

  /* manifest parameters and their default values, and add them to context */
  if( letdef->parameters != NULL )
    ArrayForEach(letdef->parameters, param)
    {
      if( !FEFnParamManifest(param, cxt, FALSE) )
	return FALSE;
      if( FEFnParamKind(param) == PARAM_OPTIONAL )
      {
	if( !ExprFnHeadManifestParamDefaultVal(letdef->body, param, cxt,
	    self_type, encl_befn) )
	  return FALSE;
      }
      if( !ContextInsertFEFnParam(cxt, param) )
	return FALSE;
    }

  /* result type is optional; manifest it if present */
  if( letdef->result_type != NULL )
  {
    if( !TypeManifest(letdef->result_type, cxt, FALSE) )
      return FALSE;
  }

  /* manifest the body */
  if( !ExprManifest(&letdef->body, cxt, self_type, encl_befn) )
    return FALSE;

  /* utilize body type if have no function result type, else check */
  if( letdef->result_type == NULL )
    letdef->result_type = ExprType(letdef->body);
  else
  {
    TypeRangeMarkBegin();
    if( !TypeIsSubType(ExprType(letdef->body), letdef->result_type, &c, cxt) )
    {
      fprintf(stderr,
	"%s: type of function body (%s) does not match result type (%s)\n",
	FilePosShow(ExprFilePos(letdef->body)),
	TypeShow(ExprType(letdef->body), cxt),
	TypeShow(letdef->result_type, cxt));
      TypeRangeMarkEnd(FALSE);
      return FALSE;
    }
    if( c != NULL )
      ExprFnHeadInsertCoercion(letdef->body, c, cxt, self_type);
    TypeRangeMarkEnd(TRUE);
  }

  /* success */
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnLetDefManifest(FEFN_LETDEF letdef, CONTEXT cxt,              */
/*    TYPE self_type, BEFN encl_befn, BOOLEAN *let_large_scale)              */
/*                                                                           */
/*  Manifest letdef in the given context. If any of them have no             */
/*  parameters, they will be left inline; *let_large_scale has to be         */
/*  set to TRUE in this case, to tell the enclosing let expression that      */
/*  it is a large-scale expression.                                          */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnLetDefManifest(FEFN_LETDEF letdef, CONTEXT cxt, TYPE self_type,
  BEFN encl_befn, BOOLEAN *let_large_scale)
{
  BOOLEAN success;

  if( DEBUG1 )
    fprintf(stderr, "[ FEFnLetDefManifest(%p: %s%s)\n", (void *) letdef,
      NameShow(letdef->name), FEFnSignatureShow((FEFN) letdef, cxt));

  /* insert the letdef into the current context */
  if( !ContextInsertFEFnLetDef(cxt, letdef) )
    db_return(DEBUG1, "FEFnLetDefManifest", FALSE);

  /* much simpler if no parameters */
  if( letdef->parameters == NULL )
  {
    /* manifest the definition */
    success = FEFnLetDefDoManifest(letdef, cxt, self_type, encl_befn);
    if( !success )
      db_return(DEBUG1, "FEFnLetDefManifest", FALSE);

    /* definition remains inline so must be large-scale */
    *let_large_scale = TRUE;
  }
  else
  {
    /* inform encl_befn about this letdef */
    BEFnAddInnerBEFn(encl_befn, (BEFN) letdef->befn_letdef);

    /* manifest the function at a new context level */
    ContextPushEmpty(cxt, (FEFN) letdef, FALSE);
    success = FEFnLetDefDoManifest(letdef, cxt, self_type, encl_befn);
    ContextPop(cxt, TRUE);
    if( !success )
      db_return(DEBUG1, "FEFnLetDefManifest", FALSE);
  }
  assert(letdef->result_type != NULL);
  if( DEBUG1 )
    fprintf(stderr, "] FEFnLetDefManifest returning %s%s\n",
      NameShow(letdef->name), FEFnSignatureShow((FEFN) letdef, cxt));
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnLetDefsEnterScope(ARRAY_FEFN_LETDEF letdefs, CODEGEN be)        */
/*                                                                           */
/*  Called during code generation when entering a scope region with          */
/*  these letdefs defined in it.  Finalize the unparameterized ones and      */
/*  code gen their declarations and initializations.                         */
/*                                                                           */
/*****************************************************************************/

void FEFnLetDefsVarEnterScope(ARRAY_FEFN_LETDEF letdefs, CODEGEN be)
{
  FEFN_LETDEF letdef;
  if( DEBUG2 )
    fprintf(stderr, "[ FEFnLetDefsVarEnterScope(%d letdefs, be)\n",
      ArraySize(letdefs));

  /* finalize and declare letdef vars, using names registered with scope */
  ArrayForEach(letdefs, letdef)
    if( letdef->parameters == NULL )
      BEFnFinalize((BEFN) letdef->befn_letdef, be);

  /* print expressions assigned to decls */
  ArrayForEach(letdefs, letdef)
    if( letdef->parameters == NULL )
    {
      if( DEBUG2 )
	fprintf(stderr, "  FEFnLetDefsVarEnterScope printing expr for %s\n",
	  NameShow(letdef->name));
      ExprCodeGen(letdef->body, BEFnBEObj((BEFN) letdef->befn_letdef),
	BEFnBEType((BEFN) letdef->befn_letdef), be);
    }

  if( DEBUG2 )
    fprintf(stderr, "] FEFnLetDefsVarEnterScope\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnLetDefsVarLeaveScope(ARRAY_FEFN_LETDEF letdefs, CODEGEN be)     */
/*                                                                           */
/*  Matches FEFnLetDefsVarEnterScope; called when they leave scope.          */
/*                                                                           */
/*****************************************************************************/

void FEFnLetDefsVarLeaveScope(ARRAY_FEFN_LETDEF letdefs, CODEGEN be)
{
  FEFN_LETDEF letdef;
  ArrayForEach(letdefs, letdef)
    if( letdef->parameters == NULL )
      BEFnLetDefUnFinalize(letdef->befn_letdef, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnLetDefDebug(FEFN_LETDEF letdef, CONTEXT cxt,                    */
/*    BOOLEAN show_types, FILE *fp, int print_style)                         */
/*                                                                           */
/*  Debug print of letdef onto fp.  Needs fleshing out!                      */
/*                                                                           */
/*****************************************************************************/

void FEFnLetDefDebug(FEFN_LETDEF letdef, CONTEXT cxt, BOOLEAN show_types,
  FILE *fp, int print_style)
{
  fprintf(fp, "%s...", NameShow(letdef->name));
  /* FunctionDebug(letdef->function, cxt, show_types, fp, print_style); */
}
