/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         fefn_param.c                                               */
/*  DESCRIPTION:  A parameter of a front-end function                        */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0

struct fefn_param_rec {			/* inherits from FEFN                */
  KIND_TAG		kind_tag;	/* kind of entity (KIND_FEFN_PARAM)  */
  FILE_POS		file_pos;	/* position of param name in file    */
  NAME			name;		/* name of function                  */
  TYPE_VARS		type_vars;	/* always NULL in parameters         */
  ARRAY_FEFN_PARAM	parameters;	/* always NULL in parameters         */
  TYPE			result_type;	/* result type                       */
  BEFN_PARAM		befn_param;	/* corr parameter of corr C function */

  PARAM_KIND		param_kind;	/* self, compulsory, etc.            */
  BOOLEAN		is_private;	/* omit when importing encl sign.    */
  /* FEFN_FEATURE	feature_view;*/	/* in creation fn and clone views    */
  FEFN    		hidden_value;	/* if PARAM_HIDDEN                   */
};


/*****************************************************************************/
/*                                                                           */
/*  FEFN_PARAM FEFnParamNew(FILE_POS file_pos, NAME name, TYPE result_type,  */
/*    BEFN_PARAM befn_param, PARAM_KIND param_kind, BOOLEAN is_private,      */
/*    FEFN hidden_value)                                                     */
/*                                                                           */
/*  Make and return a new parameter with these attributes and default        */
/*  values for the others.                                                   */
/*                                                                           */
/*****************************************************************************/

FEFN_PARAM FEFnParamNew(FILE_POS file_pos, NAME name, TYPE result_type,
  BEFN_PARAM befn_param, PARAM_KIND param_kind, BOOLEAN is_private,
  FEFN hidden_value)
{
  FEFN_PARAM res;
  GetMemory(res, FEFN_PARAM);
  res->kind_tag = KIND_FEFN_PARAM;
  res->file_pos = file_pos;
  res->name = name;
  res->type_vars = NULL;
  res->parameters = NULL;
  res->result_type = result_type;
  res->param_kind = param_kind;
  res->is_private = is_private;
  res->befn_param = befn_param;
  res->hidden_value = hidden_value;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnParamSetBEFnParam(FEFN_PARAM param, BEFN_PARAM befn_param)      */
/*                                                                           */
/*  Set the befn_param attribute of param.                                   */
/*                                                                           */
/*****************************************************************************/

void FEFnParamSetBEFnParam(FEFN_PARAM param, BEFN_PARAM befn_param)
{
  assert(param->befn_param == NULL);
  param->befn_param = befn_param;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnParamIsPrivate(FEFN_PARAM param)                             */
/*                                                                           */
/*  Return TRUE if param is private.                                         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnParamIsPrivate(FEFN_PARAM param)
{
  return param->is_private;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN FEFnParamHiddenValue(FEFN_PARAM param)                              */
/*                                                                           */
/*  Return the fefn that is associated with param.  For a hidden parameter   */
/*  this will be the parameter, downdef, or letdef the parameter stands      */
/*  for.  It may also be the creation feature that gives rise to this        */
/*  parameter, in creation functions and clone features.  In a creation      */
/*  default value function it will be another parameter which itself has a   */
/*  hidden value which is a creation feature.                                */
/*                                                                           */
/*****************************************************************************/

FEFN FEFnParamHiddenValue(FEFN_PARAM param)
{
  /* assert(param->param_kind == PARAM_HIDDEN); */
  return param->hidden_value;
}


/*****************************************************************************/
/*                                                                           */
/*  FILE_POS FEFnParamFilePos(FEFN_PARAM param)                              */
/*                                                                           */
/*  Return the file position of param.                                       */
/*                                                                           */
/*****************************************************************************/

FILE_POS FEFnParamFilePos(FEFN_PARAM param)
{
  return param->file_pos;
}


/*****************************************************************************/
/*                                                                           */
/*  NAME FEFnParamName(FEFN_PARAM param)                                     */
/*                                                                           */
/*  Return the name of param.                                                */
/*                                                                           */
/*****************************************************************************/

NAME FEFnParamName(FEFN_PARAM param)
{
  return param->name;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE FEFnParamType(FEFN_PARAM param)                                     */
/*                                                                           */
/*  Return the type (that is, the result type) of param.                     */
/*                                                                           */
/*****************************************************************************/

TYPE FEFnParamType(FEFN_PARAM param)
{
  return param->result_type;
}


/*****************************************************************************/
/*                                                                           */
/*  PARAM_KIND FEFnParamKind(FEFN_PARAM param)                               */
/*                                                                           */
/*  Return the kind of parameter this is.                                    */
/*                                                                           */
/*****************************************************************************/

PARAM_KIND FEFnParamKind(FEFN_PARAM param)
{
  return param->param_kind;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_PARAM FEFnParamBEFnParam(FEFN_PARAM param)                          */
/*                                                                           */
/*  Return the C parameter corresponding to this Nonpareil parameter.        */
/*                                                                           */
/*****************************************************************************/

BEFN_PARAM FEFnParamBEFnParam(FEFN_PARAM param)
{
  assert(param->befn_param != NULL);
  return param->befn_param;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnParamIsOptional(FEFN_PARAM param)                            */
/*                                                                           */
/*  Return TRUE if param is optional (has a default value).                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnParamIsOptional(FEFN_PARAM param)
{
  return param->param_kind == PARAM_OPTIONAL;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_PARAM FEFnParamCopy(FEFN_PARAM param, BOOLEAN is_private)           */
/*                                                                           */
/*  Return a fresh copy of parameter param, setting its privacy value to     */
/*  is_private.  A copy is needed even if the privacy value does not change, */
/*  since substitutions might occur within the param type during the copy.   */
/*                                                                           */
/*****************************************************************************/

FEFN_PARAM FEFnParamCopy(FEFN_PARAM param, BOOLEAN is_private)
{
  return FEFnParamNew(param->file_pos, param->name,
    TypeCopy(param->result_type), param->befn_param, param->param_kind,
    is_private, param->hidden_value);
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_FEFN_PARAM FEFnParamListCopy(ARRAY_FEFN_PARAM parameters)          */
/*                                                                           */
/*  Return a fresh copy of this list of parameters.  The point of this       */
/*  is that substitutions might occur within their types during the copy.    */
/*                                                                           */
/*****************************************************************************/

ARRAY_FEFN_PARAM FEFnParamListCopy(ARRAY_FEFN_PARAM parameters)
{
  ARRAY_FEFN_PARAM res;  FEFN_PARAM param;
  if( parameters == NULL )
    return NULL;
  else
  {
    ArrayInit(&res);
    ArrayForEach(parameters, param)
      ArrayAddLast(res, FEFnParamCopy(param, param->is_private));
    return res;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_FEFN_PARAM FEFnParamListCopyImported(ARRAY_FEFN_PARAM parameters,  */
/*    BOOLEAN make_private)                                                  */
/*                                                                           */
/*  Make the kind of copy of a list of parameters that is needed when        */
/*  importing.  The parameters are copied in case their privacy value        */
/*  changes:  only non-private parameters are copied, but if make_private    */
/*  is TRUE these are themselves then marked private.                        */
/*                                                                           */
/*****************************************************************************/

ARRAY_FEFN_PARAM FEFnParamListCopyImported(ARRAY_FEFN_PARAM parameters,
  BOOLEAN make_private)
{
  ARRAY_FEFN_PARAM res;  FEFN_PARAM param;
  res = NULL;
  if( parameters != NULL )
  {
    ArrayInit(&res);
    ArrayForEach(parameters, param)
      if( !param->is_private )
	ArrayAddLast(res, FEFnParamCopy(param, make_private));
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_PARAM FEFnParamCopyUninstantiated(FEFN_PARAM param)                 */
/*                                                                           */
/*  Copy an uninstantiated parameter.                                        */
/*                                                                           */
/*****************************************************************************/

static FEFN_PARAM FEFnParamCopyUninstantiated(FEFN_PARAM param)
{
  return FEFnParamNew(param->file_pos, param->name,
    TypeCopyUninstantiated(param->result_type), param->befn_param,
    param->param_kind, param->is_private, param->hidden_value);
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_FEFN_PARAM FEFnParamListCopyUninstantiated(                        */
/*    ARRAY_FEFN_PARAM parameters)                                           */
/*                                                                           */
/*  Make the kind of copy of a list of parameters needed when the list       */
/*  is still uninstantiated, as used when disentangling two features         */
/*  declared together from each other.                                       */
/*                                                                           */
/*****************************************************************************/

ARRAY_FEFN_PARAM FEFnParamListCopyUninstantiated(ARRAY_FEFN_PARAM parameters)
{
  ARRAY_FEFN_PARAM res;  FEFN_PARAM param;
  res = NULL;
  if( parameters != NULL )
  {
    ArrayInit(&res);
    ArrayForEach(parameters, param)
      ArrayAddLast(res, FEFnParamCopyUninstantiated(param));
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ParameterSetParse(TOKEN *t, BOOLEAN is_coerce,                   */
/*    BOOLEAN encl_private, SYSTEM_VIEW sv, ARRAY_FEFN_PARAM parameters,     */
/*    EXPR *expr_fnhead, PARAM_BODIES_TYPE *param_bodies_type)               */
/*                                                                           */
/*  Parse a set of parameters and add them to parameters.  Add any default   */
/*  parameter values to *expr_fnhead, and update *param_bodies_type          */
/*  according to the definition given in FEFnParamListParse below.           */
/*                                                                           */
/*  If is_coerce is TRUE, the parameters belong to a coerce feature, which   */
/*  means that they must be optional.  If encl_private is TRUE, the          */
/*  enclosing function is private, which means these parameters are          */
/*  automatically private and may not be declared so.                        */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ParameterSetParse(TOKEN *t, BOOLEAN is_coerce,
  BOOLEAN encl_private, SYSTEM_VIEW sv, ARRAY_FEFN_PARAM parameters,
  EXPR *expr_fnhead, PARAM_BODIES_TYPE *param_bodies_type)
{
  ARRAY_NAME name_defs;  NAME name;  TYPE type;  EXPR body;
  FEFN_PARAM param;  FILE_POS file_pos, body_pos;  BOOLEAN is_private;
  PARAM_KIND param_kind;
 
  /* optional privacy flag */
  is_private = encl_private;
  if( LexType(curr_token) == TK_PRIVATE )
  {
    if( encl_private )
    {
      fprintf(stderr, "%s: \"private\" keyword redundant so prohibited here\n",
	LexPos(curr_token));
      return FALSE;
    }
    is_private = TRUE;
    next_token;
  }

  /* name definitions */
  if( !NameDefListParse(t, SystemViewOpTable(sv), &name_defs) )
    return FALSE;

  /* type (compulsory) */
  skip(TK_COLON, "colon preceding compulsory type");
  file_pos = LexFilePos(curr_token);
  if( !TypeParse(t, &type) )
    return FALSE;

  /* default value */
  body = NULL;
  param_kind = PARAM_COMPULSORY;
  if( LexType(curr_token) == TK_COLON_EQUALS ||
      (LexType(curr_token) == TK_PUNCTSEQ &&
       UStringEqual(LexValue(curr_token), AStringToUString("="))) )
  {
    skip(TK_COLON_EQUALS, ":= (not =)");
    param_kind = PARAM_OPTIONAL;
    if( LexType(curr_token)!=TK_COMMA && LexType(curr_token)!=TK_RIGHT_PAREN )
    {
      /* body is present; parse it */
      body_pos = LexFilePos(curr_token);
      if( !ExprParse(t, sv, &body) )
	return FALSE;

      /* must not contradict param_bodies_type */
      if( *param_bodies_type == PARAM_BODIES_ABSTRACT )
      {
	fprintf(stderr, "%s: optional parameter has default value, but a %s\n",
	  FilePosShow(body_pos), "previous optional parameter does not");
	return FALSE;
      }
      *param_bodies_type = PARAM_BODIES_CONCRETE;
    }
    else
    {
      /* body is not present; must not contradict param_bodies_type */
      if( *param_bodies_type == PARAM_BODIES_CONCRETE )
      {
	fprintf(stderr, "%s: optional parameter has no default value, but %s\n",
	  FilePosShow(file_pos), "a previous optional parameter does");
	return FALSE;
      }
      else if( *param_bodies_type == PARAM_BODIES_FIXED )
      {
	fprintf(stderr, "%s: missing optional parameter default value\n",
	  FilePosShow(file_pos));
	return FALSE;
      }
      *param_bodies_type = PARAM_BODIES_ABSTRACT;
    }
  }
  else if( is_coerce )
  {
    fprintf(stderr, "%s: \"coerce\" feature has non-optional parameter\n",
      LexPos(curr_token));
    return FALSE;
  }
  else if( is_private && !encl_private )
  {
    fprintf(stderr, "%s: private parameter %s of public feature not optional\n",
      LexPos(curr_token), NameShow(ArrayFirst(name_defs)));
    return FALSE;
  }

  /* add one parameter for each name def, updating expr_fnhead as reqd */
  ArrayForEach(name_defs, name)
  {
    param = FEFnParamNew(file_pos, name, type,NULL,param_kind,is_private,NULL);
    ArrayAddLast(parameters, param);
    if( body != NULL )
    {
      if( *expr_fnhead == NULL )
	*expr_fnhead = ExprFnHeadNew(file_pos);
      ExprFnHeadAddFEFnParamDefaultVal(*expr_fnhead, param, body);
    }
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnParamListParse(TOKEN *t,FEATURE_TYPE ftype,BOOLEAN is_coerce,*/
/*    BOOLEAN encl_private, SYSTEM_VIEW sv, ARRAY_FEFN_PARAM *parameters,    */
/*    EXPR *expr_fnhead, PARAM_BODIES_TYPE *param_bodies_type)               */
/*                                                                           */
/*  Parse an optional parameter list according to the grammar                */
/*                                                                           */
/*     [ "(" parameter_set { "," parameter_set } ")" ]                       */
/*                                                                           */
/*  Return the parameters in *parameters (which may be null initially,       */
/*  in which case it has to be initialized), and any default parameter       */
/*  values in expr_fnhead.                                                   */
/*                                                                           */
/*  If ftype is FEATURE_CREATION, these parameters belong to a creation      */
/*  feature, which means that there may not be any.  If is_coerce is TRUE,   */
/*  the parameters belong to a coercion feature, which means they should all */
/*  be optional.  If encl_private is TRUE, the enclosing function is         */
/*  known to be private, so a "private" keyword would be redundant here.     */
/*                                                                           */
/*  The return value *param_bodies_type reports whether the parameters were  */
/*  found to have bodies or not:                                             */
/*                                                                           */
/*    PARAM_BODIES_ABSTRACT    There are optional parameters and none of     */
/*                             them have bodies                              */
/*                                                                           */
/*    PARAM_BODIES_CONCRETE    There are optional parameters and all of      */
/*                             them have bodies                              */
/*                                                                           */
/*    PARAM_BODIES_OPEN        There are no optional parameters, and so      */
/*                             nothing is fixed either way                   */
/*                                                                           */
/*    PARAM_BODIES_FIXED       In this list, optional parameters must have   */
/*                             bodies                                        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnParamListParse(TOKEN *t, FEATURE_TYPE ftype, BOOLEAN is_coerce,
  BOOLEAN encl_private, SYSTEM_VIEW sv, ARRAY_FEFN_PARAM *parameters,
  EXPR *expr_fnhead, PARAM_BODIES_TYPE *param_bodies_type)
{
  *param_bodies_type = PARAM_BODIES_OPEN;
  if( LexType(curr_token) == TK_LEFT_PAREN )
  {
    if( ftype == FEATURE_CREATION )
    {
      fprintf(stderr, "%s: creation feature has parameter(s)\n",
	LexPos(curr_token));
      return FALSE;
    }
    if( ftype == FEATURE_PREDEF )
    {
      fprintf(stderr, "%s: predefined object feature has parameter(s)\n",
	LexPos(curr_token));
      return FALSE;
    }

    /* skip opening left parenthesis */
    next_token;

    /* make sure *parameters is initialized */
    if( *parameters == NULL )
      ArrayInit(parameters);

    /* parse first parameter set */
    if( !ParameterSetParse(t, is_coerce, encl_private, sv, *parameters,
	  expr_fnhead, param_bodies_type) )
      return FALSE;

    /* parse subsequent sets */
    while( LexType(curr_token) == TK_COMMA )
    {
      next_token;
      if( !ParameterSetParse(t, is_coerce, encl_private, sv, *parameters,
	  expr_fnhead, param_bodies_type) )
	return FALSE;
    }

    /* skip closing right parenthesis */
    skip(TK_RIGHT_PAREN, "comma or closing parenthesis of parameter list");
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnParamIsRedefinition(FEFN_PARAM parent, FEFN_PARAM child)     */
/*                                                                           */
/*  It has been established that child is a redefinition of parent, so       */
/*  make sure that the child has the same C parameter as the parent.         */
/*  (It may already have a C parameter, in which case it must be the same    */
/*  C parameter.)                                                            */
/*                                                                           */
/*  Also check that the privacy does not increase; return FALSE if it does.  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnParamIsRedefinition(FEFN_PARAM parent, FEFN_PARAM child)
{
  if( DEBUG3 )
    fprintf(stderr, "[ FEFnParamIsRedefinition(prnt %s %s, child %s %s)\n",
      FilePosShow(parent->file_pos), NameShow(parent->name),
      FilePosShow(child->file_pos), NameShow(child->name));

  /* check privacy */
  if( !parent->is_private && child->is_private )
  {
    fprintf(stderr, "%s: private parameter %s inherits public parameter\n",
      FilePosShow(child->file_pos), NameShow(child->name));
    fprintf(stderr, "  (the inherited parameter is at %s)\n",
      FilePosShow(parent->file_pos));
    return FALSE;
  }

  /* sort out back-end parameter */
  assert(parent->befn_param != NULL);
  if( child->befn_param == NULL )
    child->befn_param = parent->befn_param;
  else
    assert(child->befn_param == parent->befn_param);
  db_return(DEBUG3, "FEFnParamIsRedefinition", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnParamManifest(FEFN_PARAM param, CONTEXT cxt,                 */
/*    BOOLEAN is_public)                                                     */
/*                                                                           */
/*  Manifest param in context.  This does not include manifesting any        */
/*  default value; those are stored separately.  Self parameters are         */
/*  skipped because their types are manifested all along.                    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnParamManifest(FEFN_PARAM param, CONTEXT cxt, BOOLEAN is_public)
{
  BOOLEAN res;
  if( DEBUG1 )
    fprintf(stderr, "[ FEFnParamManifest(%s, cxt, is_public: %s)\n",
      NameShow(param->name), bool(is_public));
  assert(param->result_type != NULL);
  if( param->param_kind != PARAM_SELF )
    res = TypeManifest(param->result_type, cxt, is_public);
  else
    res = TRUE;
  if( DEBUG1 )
    fprintf(stderr, "] FEFnParamManifest returning %s, type is %s\n",
      bool(res), TypeShow(param->result_type, cxt));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnParamListRegisterBaseParameters(ARRAY_FEFN_PARAM parameters,    */
/*    BEFN_FEATURE base_feature)                                             */
/*                                                                           */
/*  These parameters come from a feature view.  Those whose "befn_param"     */
/*  fields are null are hereby declared to refer to parameters of            */
/*  base_feature, so make parameters for them there and set their            */
/*  "befn_param" fields to those parameters.                                 */
/*                                                                           */
/*****************************************************************************/

void FEFnParamListRegisterBaseParameters(ARRAY_FEFN_PARAM parameters,
  BEFN_FEATURE base_feature)
{
  FEFN_PARAM param;
  if( DEBUG5 )
    fprintf(stderr, "[ FEFnParamListRegisterBaseParameters(%s, %s)\n",
      FEFnParamListShow(parameters, NULL),
      NameShow(BEFnFeatureOrigName(base_feature)));
  assert(base_feature != NULL);
  if( parameters != NULL )
    ArrayForEach(parameters, param)
      if( param->befn_param == NULL )
      {
	if( DEBUG5 )
	  fprintf(stderr, "  befn_param for %s\n", FEFnParamShow(param, NULL));
	param->befn_param = BEFnParamNew(param->param_kind, param, NULL);
	BEFnAddParameter((BEFN) base_feature, param->befn_param);
      }
  if( DEBUG5 )
    fprintf(stderr, "] FEFnParamListRegisterBaseParameters returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnParamAddToAString(FEFN_PARAM param, AFACTORY af, CONTEXT cxt)   */
/*                                                                           */
/*  Add rendering of param to af.                                            */
/*                                                                           */
/*****************************************************************************/

static void FEFnParamAddToAString(FEFN_PARAM param, AFACTORY af, CONTEXT cxt)
{
  AStringAddFmt5(af, "%s[%s%p]: %s%s", (ASTRING) NameShow(param->name),
    param->befn_param != NULL ? "C" : "", (void *) param,
    (ASTRING) TypeShow(param->result_type, cxt),
    param->param_kind == PARAM_OPTIONAL ? " := _" : "");
}


/*****************************************************************************/
/*                                                                           */
/*  UTF8 FEFnParamShow(FEFN_PARAM param, CONTEXT cxt)                        */
/*                                                                           */
/*  Show param in cxt.                                                       */
/*                                                                           */
/*****************************************************************************/

UTF8 FEFnParamShow(FEFN_PARAM param, CONTEXT cxt)
{
  AFACTORY af = AStringBegin();
  FEFnParamAddToAString(param, af, cxt);
  return (UTF8) AStringEnd(af);
}


/*****************************************************************************/
/*                                                                           */
/*  UTF8 FEFnParamListShow(ARRAY_FEFN_PARAM parameters, CONTEXT cxt)         */
/*                                                                           */
/*  Return a static string displaying these parameters.                      */
/*                                                                           */
/*****************************************************************************/

UTF8 FEFnParamListShow(ARRAY_FEFN_PARAM parameters, CONTEXT cxt)
{
  FEFN_PARAM param;
  if( parameters == NULL || ArraySize(parameters) == 0 )
    return (UTF8) "";
  else
  {
    AFACTORY af = AStringBegin();
    AStringAddChar(af, '(');
    ArrayForEach(parameters, param)
    {
      if( param != ArrayFirst(parameters) )
	AStringAddAString(af, ", ");
      FEFnParamAddToAString(param, af, cxt);
    }
    AStringAddChar(af, ')');
    return (UTF8) AStringEnd(af);
  }
}
