/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         fefn_precond.c                                             */
/*  DESCRIPTION:  A creation feature default value                           */
/*                                                                           */
/*****************************************************************************/
#include "externs.h"
#define DEBUG1 0


struct fefn_precond_rec {		/* inherits from FEFN                */
  KIND_TAG		kind_tag;	/* kind (KIND_CLASS_VIEW)            */
  FILE_POS		file_pos;	/* position of class view in input   */
  NAME			name;		/* name of default value             */
  TYPE_VARS		type_vars;	/* type variables                    */
  ARRAY_FEFN_PARAM	parameters;	/* one for hidden parameter          */
  TYPE			result_type;	/* equivalent to class type          */
  BEFN_PRECOND		befn_precond;	/* the corresponding backend fn.     */

  FEFN_FEATURE		fefn_feature;	/* fefn feature holding this precond */
  EXPR			body;		/* body of precondition              */
};


/*****************************************************************************/
/*                                                                           */
/*  FEFN_PRECOND FEFnPrecondNew(FEFN_FEATURE fefn_feature, EXPR body)        */
/*                                                                           */
/*  Create a new front-end precondition with these attributes, and           */
/*  initially no parameters.                                                 */
/*                                                                           */
/*****************************************************************************/

FEFN_PRECOND FEFnPrecondNew(FEFN_FEATURE fefn_feature, EXPR body)
{
  FEFN_PRECOND res;
  GetMemory(res, FEFN_PRECOND);
  res->kind_tag = KIND_FEFN_PRECOND;
  res->file_pos = FEFnFeatureFilePos(fefn_feature);
  res->name = FEFnFeatureName(fefn_feature);
  res->type_vars = NULL;
  res->parameters = NULL;
  res->result_type = FEFnResultType((FEFN) fefn_feature);
  res->befn_precond = BEFnPrecondNew(res, body);
  res->fefn_feature = fefn_feature;
  res->body = body;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_PRECOND FEFnPrecondCopyUninstantiated(FEFN_PRECOND fefn_precond,    */
/*    FEFN_FEATURE fv, ARRAY_FEFN_PARAM orig_params,                         */
/*    ARRAY_FEFN_PARAM copy_params)                                          */
/*                                                                           */
/*  Return a copy of uninstantiated precondition function fefn_precond,      */
/*  but linked now to fv.                                                    */
/*                                                                           */
/*****************************************************************************/

FEFN_PRECOND FEFnPrecondCopyUninstantiated(FEFN_PRECOND fefn_precond,
  FEFN_FEATURE fv, ARRAY_FEFN_PARAM orig_params,
  ARRAY_FEFN_PARAM copy_params)
{
  EXPR e = ExprCopyUninstantiated(fefn_precond->body, orig_params, copy_params);
  return FEFnPrecondNew(fv, e);
}


/*****************************************************************************/
/*                                                                           */
/*  FEFN_FEATURE FEFnPrecondOrigFeature(FEFN_PRECOND fefn_precond)           */
/*                                                                           */
/*  Return the noncreation feature holding this precondition.                */
/*                                                                           */
/*****************************************************************************/

FEFN_FEATURE FEFnPrecondOrigFeature(FEFN_PRECOND fefn_precond)
{
  return fefn_precond->fefn_feature;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN FEFnPrecondManifest(FEFN_PRECOND fefn_precond, CONTEXT cxt,      */
/*    TYPE self_type)                                                        */
/*                                                                           */
/*  Manifest this creation function default value.  It is an outer level     */
/*  function in the back end.  It also gathers hidden parameters.            */
/*                                                                           */
/*  The context when this function is called will be the context of the      */
/*  body of the enclosing feature.  The feature's parameters will all be     */
/*  in scope, and, if invoked, will be converted into hidden parameters      */
/*  of this precondition.                                                    */
/*                                                                           */
/*  Also pass the back end version of this function to the back end          */
/*  version of the original feature.  This can't be done at creation         */
/*  time since the back end feature is not known then.                       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN FEFnPrecondManifest(FEFN_PRECOND fefn_precond, CONTEXT cxt,
  TYPE self_type)
{
  BOOLEAN success;  BEFN befn;
  if( DEBUG1 )
  {
    fprintf(stderr, "[ FEFnPrecondManifest([");
    FEFnPrecondDebug(fefn_precond, cxt, stderr);
    fprintf(stderr, "], cxt, %s)\n", TypeShow(self_type, cxt));
  }

  /* manifest the body */
  success = TRUE;
  ContextPushEmpty(cxt, (FEFN) fefn_precond, FALSE);
  success = ExprManifest(&fefn_precond->body, cxt, self_type,
    (BEFN) fefn_precond->befn_precond);
  ContextPop(cxt, TRUE);

  /* pass the back end version of this precond to the appropriate feature */
  befn = FEFnBEFn((FEFN) fefn_precond->fefn_feature);
  assert(befn != NULL && NamedKind((NAMED) befn) == KIND_BEFN_FEATURE);
  BEFnAddInnerBEFn(befn, (BEFN) fefn_precond->befn_precond);

  if( DEBUG1 )
  {
    fprintf(stderr, "] FEFnPrecondManifest returning %s, fefn_precond now: ",
      bool(success));
    FEFnPrecondDebug(fefn_precond, cxt, stderr);
    fprintf(stderr, "\n");
  }
  return success;
}


/*****************************************************************************/
/*                                                                           */
/*  void FEFnPrecondDebug(FEFN_PRECOND fefn_precond, FILE *fp)               */
/*                                                                           */
/*  Debug print of fefn_precond.                                             */
/*                                                                           */
/*****************************************************************************/

void FEFnPrecondDebug(FEFN_PRECOND fefn_precond, CONTEXT cxt, FILE *fp)
{
  fprintf(fp, "%s%s := ", NameShow(fefn_precond->name),
    FEFnSignatureShow((FEFN) fefn_precond, cxt));
  ExprDebug(fefn_precond->body, cxt, FALSE, fp, SINGLE_LINE);
}
