/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         iset.c                                                     */
/*  DESCRIPTION:  Interval sets, such as {0-6, 9-11, 12-65535}.              */
/*                                                                           */
/*****************************************************************************/
#include "iset.h"
#include "memory.h"
#include "array.h"
#include <assert.h>
#include <string.h>
#define min(a, b) ((a) < (b) ? (a) : (b))
#define max(a, b) ((a) > (b) ? (a) : (b))

struct iset_rec {
  ARRAY(int)	elems;			/* even-length array of endpoints */
};


/*****************************************************************************/
/*                                                                           */
/*  ISET ISetNew()                                                           */
/*                                                                           */
/*  Return a new, empty ISet.                                                */
/*                                                                           */
/*****************************************************************************/

ISET ISetNew()
{
  ISET res;
  GetMemory(res, ISET);
  ArrayInit(&res->elems);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  ISET ISetInterval(int from, int to)                                      */
/*                                                                           */
/*  Return a new ISet representing the interval [from, to].                  */
/*                                                                           */
/*****************************************************************************/

ISET ISetInterval(int from, int to)
{
  ISET res;
  assert(from >= 0);
  assert(from <= to);
  res = ISetNew();
  ArrayAddLast(res->elems, from);
  ArrayAddLast(res->elems, to);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN IntervalsIntersect(int from1, int to1, int from2, int to2,       */
/*    int *common_point)                                                     */
/*                                                                           */
/*  Return TRUE if [from1, to1] and [from2, to2] have a non-empty            */
/*  intersection, setting *common_point to one element of that intersection. */
/*  Otherwise return FALSE.                                                  */
/*                                                                           */
/*  Implementation note.  There are six cases, depending on where            */
/*  [from2, to2] falls relative to [from1, to1]:                             */
/*                                                                           */
/*                                     [-----1-----]                         */
/*                                                                           */
/*  Case 1: to2 < from1       [--2--]                                        */
/*  Case 2:                   [------2------]                                */
/*  Case 3:                   [-----------2-----------]                      */
/*  Case 4:                             [--2--]                              */
/*  Case 5:                             [------2------]                      */
/*  Case 6: to1 < from2                               [--2--]                */
/*                                                                           */
/*  Only cases 1 and 6 have an empty intersection, and in the other cases    */
/*  max(from1, from2) is clearly a common point.                             */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN IntervalsIntersect(int from1, int to1, int from2, int to2,
  int *common_point)
{
  if( to1 < from2 )
    return FALSE;
  else if( to2 < from1 )
    return FALSE;
  else
  {
    *common_point = max(from1, from2);
    return TRUE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ISetNonEmptyIntersection(ISET is1, ISET is2, int *common_point)  */
/*                                                                           */
/*  Return TRUE if is1 and is2 have a non-empty intersection, and set        */
/*  *common_point to one element of that intersection.  Otherwise return     */
/*  FALSE.  Either way, make no changes to is1 and is2.                      */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ISetNonEmptyIntersection(ISET is1, ISET is2, int *common_point)
{
  int i1, i2;
  i1 = i2 = 0;
  while( i1 < ArraySize(is1->elems) && i2 < ArraySize(is2->elems) )
  {
    if( IntervalsIntersect(ArrayGet(is1->elems, i1), ArrayGet(is1->elems, i1+1),
	ArrayGet(is2->elems, i2), ArrayGet(is2->elems, i2+1), common_point) )
      return TRUE;
    if( ArrayGet(is1->elems, i1) < ArrayGet(is2->elems, i2) )
      i1 += 2;
    else
      i2 += 2;
  }
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  void ISetUnion(ISET is1, ISET is2)                                       */
/*                                                                           */
/*  Change is1 to be the ordinary set union of is1 with is2.                 */
/*                                                                           */
/*****************************************************************************/

void ISetUnion(ISET is1, ISET is2)
{
  static ARRAY_INT res;
  int i1, i2, start, stop;
  if( ArraySize(is2->elems) > 0 )
  {
    ArrayFresh(res);

    /* loop invt: remainder of res is determined by is1[i1..] and is2[i2..] */
    i1 = i2 = 0;
    while( i1 < ArraySize(is1->elems) || i2 < ArraySize(is2->elems) )
    {
      /* find the smallest start point */
      if( i1 >= ArraySize(is1->elems) )
	start = ArrayGet(is2->elems, i2);
      else if( i2 >= ArraySize(is2->elems) )
	start = ArrayGet(is1->elems, i1);
      else
	start = min(ArrayGet(is1->elems, i1), ArrayGet(is2->elems, i2));
      ArrayAddLast(res, start);

      /* loop invt: [start, stop] is in the union */
      stop = start;
      for(;;)
      {
	if( i1 < ArraySize(is1->elems) && ArrayGet(is1->elems, i1) <= stop+1 )
	{
	  stop = max(stop, ArrayGet(is1->elems, i1+1));
	  i1 += 2;
	}
	else if( i2 < ArraySize(is2->elems) && ArrayGet(is2->elems,i2)<=stop+1 )
	{
	  stop = max(stop, ArrayGet(is2->elems, i2+1));
	  i2 += 2;
	}
	else
	  break;
      }
      ArrayAddLast(res, stop);
    }
    ArrayClear(is1->elems);
    ArrayAppend(is1->elems, res);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ISetDisjointUnion(ISET is1, ISET is2, int *common_point)         */
/*                                                                           */
/*  Similar to ISetUnion, except that the union is expected to be disjoint.  */
/*  TRUE is returned if it is and is1 is updated as usual.  Otherwise FALSE  */
/*  is returned and *common_point contains an integer common to is1 and is2. */
/*                                                                           */
/*****************************************************************************/

BOOLEAN ISetDisjointUnion(ISET is1, ISET is2, int *common_point)
{
  if( ISetNonEmptyIntersection(is1, is2, common_point) )
    return FALSE;
  else
  {
    ISetUnion(is1, is2);
    return TRUE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void ISetInsertInterval(ISET is1, int from, int to)                      */
/*                                                                           */
/*  Update is1 by replacing it with its union with [from, to].               */
/*                                                                           */
/*****************************************************************************/

void ISetInsertInterval(ISET is1, int from, int to)
{
  assert(from <= to);
  if( ArraySize(is1->elems) == 0 )
  {
    /* set is empty, just add [from, to] */
    ArrayAddLast(is1->elems, from);
    ArrayAddLast(is1->elems, to);
  }
  else if( ArrayLast(is1->elems) < from )
  {
    /* a simple change to the end of is1->elems does the job */
    if( ArrayLast(is1->elems) == from - 1 )
      ArrayPut(is1->elems, ArraySize(is1->elems) - 1, to);
    else
    {
      ArrayAddLast(is1->elems, from);
      ArrayAddLast(is1->elems, to);
    }
  }
  else
  {
    /* fall back on a general union operation */
    ISetUnion(is1, ISetInterval(from, to));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  int ISetMinElement(ISET is)                                              */
/*                                                                           */
/*  Return the minimum element of is, or 0 if empty.                         */
/*                                                                           */
/*****************************************************************************/

int ISetMinElement(ISET is)
{
  return ArraySize(is->elems) == 0 ? 0 : ArrayFirst(is->elems);
}


/*****************************************************************************/
/*                                                                           */
/*  int ISetMaxElement(ISET is)                                              */
/*                                                                           */
/*  Return the maximum element of is, or 0 if empty.                         */
/*                                                                           */
/*****************************************************************************/

int ISetMaxElement(ISET is)
{
  return ArraySize(is->elems) == 0 ? 0 : ArrayLast(is->elems);
}


/*****************************************************************************/
/*                                                                           */
/*  int ISetGap(ISET is, int width, int align)                               */
/*                                                                           */
/*  Find the smallest non-negative x such that none of [x, x+width-1] are    */
/*  in is, and such that x is a multiple of align.                           */
/*                                                                           */
/*****************************************************************************/

int ISetGap(ISET is, int width, int align)
{
  int i, start;

  /* if array is empty or there is space before the first interval, return 0 */
  if( ArraySize(is->elems) == 0 || width <= ArrayFirst(is->elems) )
    return 0;

  /* if there is at least one interval, try the gap between i-1 and i */
  for( i = 2;  i < ArraySize(is->elems);  i += 2 )
  {
    start = ArrayGet(is->elems, i-1) + 1;
    if( start % align != 0 )
      start += align - start % align;
    assert(start % align == 0);
    if( start + width <= ArrayGet(is->elems, i) )
      return start;
  }

  /* if nothing done by now, use the gap after the last interval */
  start = ArrayLast(is->elems) + 1;
  if( start % align != 0 )
    start += align - start % align;
  assert(start % align == 0);
  return start;
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING ISetShow(ISET is)                                                */
/*                                                                           */
/*  Return a static string representation of is.                             */
/*                                                                           */
/*****************************************************************************/
#define end_buff &buff[bp][strlen(buff[bp])]

ASTRING ISetShow(ISET is)
{
  int i, e1, e2, bp;
  static char buff[3][300];
  bp = (bp + 1) % 3;
  sprintf(buff[bp], "{");
  for( i = 0;  i < ArraySize(is->elems);  i += 2 )
  {
    e1 = ArrayGet(is->elems, i);
    e2 = ArrayGet(is->elems, i+1);
    if( e1 == e2 )
      sprintf(end_buff, "%s%d", i > 0 ? ", " : "", e2);
    else
      sprintf(end_buff, "%s%d..%d", i > 0 ? ", " : "", e1, e2);
  }
  sprintf(end_buff, "}");
  return buff[bp];
}


/*****************************************************************************/
/*                                                                           */
/*  void ISetTestUnion(ISET is1, ISET is2, int indent, FILE *fp)             */
/*                                                                           */
/*  Test ISetUnion(is1, is2) and print the results.                          */
/*                                                                           */
/*****************************************************************************/

static void ISetTestUnion(ISET is1, ISET is2, int indent, FILE *fp)
{
  fprintf(fp, "%*sISetUnion(%s, %s) = ", indent, "",
    ISetShow(is1), ISetShow(is2));
  ISetUnion(is1, is2);
  fprintf(fp, "%s\n", ISetShow(is1));
}


/*****************************************************************************/
/*                                                                           */
/*  void ISetTestGap(ISET is, int width, int align, int indent, FILE *fp)    */
/*                                                                           */
/*  Test ISetGap(is, width, align) and print the results.                    */
/*                                                                           */
/*****************************************************************************/

static void ISetTestGap(ISET is, int width, int align, int indent, FILE *fp)
{
  fprintf(fp, "%*sISetGap(%s, %d, %d) = %d\n", indent, "",
    ISetShow(is), width, align, ISetGap(is, width, align));
}


/*****************************************************************************/
/*                                                                           */
/*  void ISetTest(FILE *fp)                                                  */
/*                                                                           */
/*  Test this module, printing results to fp.                                */
/*                                                                           */
/*****************************************************************************/

void ISetTest(FILE *fp)
{
  ISET is1, is2;
  fprintf(fp, "[ ISetTest():\n");

  /* build {0..2, 4..7, 12..15, 20} */
  is1 = ISetNew();
  ISetUnion(is1, ISetInterval(0, 2));
  ISetUnion(is1, ISetInterval(4, 7));
  ISetUnion(is1, ISetInterval(12, 15));
  ISetUnion(is1, ISetInterval(20, 20));

  /* build {0..1, 5..6, 9..11, 15..17} */
  is2 = ISetNew();
  ISetUnion(is2, ISetInterval(0, 1));
  ISetUnion(is2, ISetInterval(5, 6));
  ISetUnion(is2, ISetInterval(9, 11));
  ISetUnion(is2, ISetInterval(15, 17));

  /* find the union, which should be {0..2, 4..7, 9..17, 20} */
  ISetTestUnion(is1, is2, 2, fp);
  fprintf(fp, "]\n");

  /* do some gap finding in is1 */
  ISetTestGap(is1, 4, 4, 2, fp);
  ISetTestGap(is1, 2, 4, 2, fp);
  ISetTestGap(is1, 4, 1, 2, fp);
  ISetTestGap(is1, 2, 2, 2, fp);
}
