
/****************************************************************************/
/*                                                                          */
/*  npsys.c                                                                 */
/*                                                                          */
/*  Nonpareil runtime system                                                */
/*                                                                          */
/*  Jeffrey H. Kingston                                                     */
/*  March 2005                                                              */
/*                                                                          */
/*  This software has been placed in the public domain by its author.       */
/*                                                                          */
/****************************************************************************/

#include <assert.h>
#include <string.h>
#include <stdio.h>
#include "npsys.h"
#include "system.h"
#define OBJ_CACHE_INIT_SIZE 150001		/* prime no., not near 2^n  */
#define FUN_CACHE_INIT_SIZE 150001		/* prime no., not near 2^n  */
#define	TRUE	1
#define	FALSE	0
#define DEBUG1  1


/****************************************************************************/
/*                                                                          */
/*  Submodule "casts to and from float".                                    */
/*                                                                          */
/****************************************************************************/

int float_to_int(float f)
{
  union { float f; int i; } u;
  u.f = f;
  return u.i;
}

void *float_to_voidp(float f)
{
  union { float f; void *v; } u;
  u.f = f;
  return u.v;
}

float int_to_float(int i)
{
  union { float f; int i; } u;
  u.i = i;
  return u.f;
}

float voidp_to_float(void *v)
{
  union { float f; void *v; } u;
  u.v = v;
  return u.f;
}


/****************************************************************************/
/*                                                                          */
/*  Submodule "memory system".                                              */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/*  void mem_new_chunk(MEM_SYS mem_sys, size_t size, bool exact)            */
/*                                                                          */
/*  Add a new chunk of memory to mem_sys, of at least the given size        */
/*  (or exactly that size if exact is TRUE).                                */
/*                                                                          */
/****************************************************************************/

static void mem_new_chunk(MEM_SYS mem_sys, size_t size, bool exact)
{
  MEM_CHUNK new_chunk;

  if( !exact && mem_sys->free_chunks != NULL &&
      (mem_sys->free_chunks->top - mem_sys->free_chunks->first) >= size )
  {
    /* use a free chunk, since it's available and suitable */
    new_chunk = mem_sys->free_chunks;
    mem_sys->free_chunks = new_chunk->prev_chunk;
    new_chunk->prev_chunk = mem_sys->used_chunks;
    mem_sys->used_chunks = new_chunk;
  }
  else
  {
    /* otherwise get a chunk from malloc */
    size = (exact || size > 1048576 ? size : 1048576);
    new_chunk = (MEM_CHUNK) malloc(sizeof(struct mem_chunk_rec) + size);
    if( new_chunk == NULL )
    {
      fprintf(stderr, "out of memory\n");
      exit(1);
    }
    new_chunk->first = (char *) new_chunk + sizeof(struct mem_chunk_rec);
    new_chunk->free = new_chunk->first;
    new_chunk->top = new_chunk->first + size;
    new_chunk->prev_chunk = mem_sys->used_chunks;
    mem_sys->used_chunks = new_chunk;
  }
}


/****************************************************************************/
/*                                                                          */
/*  void mem_free_chunk(MEM_SYS mem_sys, MEM_CHUNK chunk)                   */
/*                                                                          */
/*  Free chunk, which must be one of the used chunks of mem_sys.            */
/*                                                                          */
/****************************************************************************/

static void mem_free_chunk(MEM_SYS mem_sys, MEM_CHUNK chunk)
{
  MEM_CHUNK c;
  if( mem_sys->used_chunks == chunk )
    mem_sys->used_chunks = chunk->prev_chunk;
  else
  {
    for(c = mem_sys->used_chunks;  c->prev_chunk != chunk;  c = c->prev_chunk);
    c->prev_chunk = chunk->prev_chunk;
  }
  chunk->free = chunk->first;
  chunk->prev_chunk = mem_sys->free_chunks;
  mem_sys->free_chunks = chunk;
}


/****************************************************************************/
/*                                                                          */
/*  void *mem_new(MEM_SYS mem_sys, size_t size)                             */
/*                                                                          */
/*  Return a pointer to size bytes of memory, obtained from mem_sys.        */
/*                                                                          */
/****************************************************************************/

static void *mem_new(MEM_SYS mem_sys, size_t size)
{
  void *res;

  /* if chunk is full, get another */
  if( mem_sys->used_chunks->free + size > mem_sys->used_chunks->top )
    mem_new_chunk(mem_sys, size, FALSE);

  /* allocate from used_chunks */
  res = (void *) mem_sys->used_chunks->free;
  mem_sys->used_chunks->free += size;
  return res;
}


/****************************************************************************/
/*                                                                          */
/*  void mem_free(MEM_SYS mem_sys, void *obj, size_t size)                  */
/*                                                                          */
/*  Free obj, whose size is size; but only if this can be done easily.      */
/*                                                                          */
/****************************************************************************/

static void mem_free(MEM_SYS mem_sys, void *obj, size_t size)
{
  if( mem_sys->used_chunks->free == (char *) obj + size )
    mem_sys->used_chunks->free -= size;
}


/****************************************************************************/
/*                                                                          */
/*  MEM_SYS mem_init(int init_len)                                          */
/*                                                                          */
/*  Create and return a new memory system containing just an empty cache    */
/*  with the given initial cache length.                                    */
/*                                                                          */
/****************************************************************************/

static MEM_SYS mem_init(int init_len)
{
  MEM_SYS res;  int i;
  res = (MEM_SYS) malloc(sizeof(struct mem_sys_rec));
  if( res == NULL )
  {
    fprintf(stderr, "out of memory\n");
    exit(1);
  }
  res->used_chunks = res->free_chunks = NULL;
  mem_new_chunk(res, init_len * sizeof(void *), TRUE);
  res->cache = (CACHE_OBJ *) mem_new(res, init_len * sizeof(void *));
  for( i = 0;  i < init_len;  i++ )
    res->cache[i] = NULL;
  res->cache_len = init_len;
  res->cache_count = 0;
  res->cache_maxcount = 0.8 * init_len;
  res->cache_chunk = res->used_chunks;
  return res;
}


/****************************************************************************/
/*                                                                          */
/*  void mem_rehash(MEM_SYS mem_sys, int new_len)                           */
/*                                                                          */
/*  Rehash the cache of this memory system, so that its length is now       */
/*  new_len.                                                                */
/*                                                                          */
/****************************************************************************/

static void mem_rehash(MEM_SYS mem_sys, int new_len)
{
  CACHE_OBJ *new_cache;  CACHE_OBJ obj;  int i, j;

  /* get a new chunk, and a new cache initialized to all null */
  mem_new_chunk(mem_sys, new_len * sizeof(void *), TRUE);
  new_cache = (CACHE_OBJ *) mem_new(mem_sys, new_len * sizeof(void *));
  for( i = 0;  i < new_len;  i++ )
    new_cache[i] = NULL;

  /* insert the old cache's entries into the new cache */
  for( i = 0;  i < mem_sys->cache_len;  i++ )
  {
    obj = mem_sys->cache[i];
    for(j = obj->signature % new_len; new_cache[j]!=NULL; j = (j+1) % new_len);
    new_cache[j] = obj;
  }

  /* free old cache and set the cache variables to refer to the new cache */
  mem_free_chunk(mem_sys, mem_sys->cache_chunk);
  mem_sys->cache = new_cache;
  mem_sys->cache_len = new_len;
  mem_sys->cache_maxcount =  0.8 * new_len;
  mem_sys->cache_chunk = mem_sys->used_chunks;
}


/****************************************************************************/
/*                                                                          */
/*  int mem_offset(MEM_SYS mem_sys, CACHE_OBJ obj)                          */
/*                                                                          */
/*  Return the non-zero offset in mem_sys of obj, or -1 if not present.     */
/*  The cache chunk does not count when working out offsets.                */
/*                                                                          */
/****************************************************************************/

int mem_offset(MEM_SYS mem_sys, CACHE_OBJ obj)
{
  int res;  MEM_CHUNK chunk;
  res = 0;
  for(chunk = mem_sys->used_chunks;  chunk != NULL;  chunk = chunk->prev_chunk)
  {
    if( chunk == mem_sys->cache_chunk )
      continue;
    if( chunk->first <= (char *) obj && (char *) obj < chunk->free )
      return res + ((char *) obj - chunk->first);
    res += chunk->free - chunk->first;
  }
  return -1;
}


/****************************************************************************/
/*                                                                          */
/*  Submodule "object memory systems".                                      */
/*                                                                          */
/****************************************************************************/

MEM_SYS npsys_obj1;
MEM_SYS npsys_obj2;


/*****************************************************************************/
/*                                                                           */
/*  bool mem_is_bigendian()                                                  */
/*                                                                           */
/*  Return true if this program is running on a big-endian machine.          */
/*                                                                           */
/*****************************************************************************/

bool mem_is_bigendian()
{
  int i;
  union {
    unsigned int whole_word;
    char         part_word[sizeof(int)];
  } test;
  for( i = 0;  i < sizeof(int);  i++ )
    test.part_word[i] = i;
  if( test.whole_word % 256 == 0 )
    return TRUE;
  else if( test.whole_word % 256 == sizeof(int) - 1 )
    return FALSE;
  else
    assert(FALSE);
}


/****************************************************************************/
/*                                                                          */
/*  Submodule "current object memory system".                               */
/*                                                                          */
/****************************************************************************/

MEM_SYS npsys_obj;			/* current object memory system     */
CACHE_OBJ *npsys_obj_cache;		/* current object mem sys cache     */
int npsys_obj_cache_len;		/* current object mem sys cache len */


/****************************************************************************/
/*                                                                          */
/*  void npsys_obj_init()                                                   */
/*                                                                          */
/*  Initialize current object memory system.                                */
/*                                                                          */
/****************************************************************************/

void npsys_obj_init(MEM_SYS *mem_sys)
{
  *mem_sys = mem_init(OBJ_CACHE_INIT_SIZE);
  npsys_obj = *mem_sys;
  npsys_obj_cache = (CACHE_OBJ *) (*mem_sys)->cache;
  npsys_obj_cache_len = (*mem_sys)->cache_len;
}


/****************************************************************************/
/*                                                                          */
/*  CACHE_OBJ npsys_obj_new(size_t size)                                    */
/*                                                                          */
/*  Return a new object record of the given size.                           */
/*                                                                          */
/****************************************************************************/

CACHE_OBJ npsys_obj_new(size_t size)
{
  return (CACHE_OBJ) mem_new(npsys_obj, size);
}


/****************************************************************************/
/*                                                                          */
/*  void npsys_obj_free(void *obj, size_t size)                             */
/*                                                                          */
/*  Free obj, whose size is size; but only if this can be done easily.      */
/*                                                                          */
/****************************************************************************/

void npsys_obj_free(void *obj, size_t size)
{
  mem_free(npsys_obj, obj, size);
}


/****************************************************************************/
/*                                                                          */
/*  void npsys_obj_cache_insert(void *obj, int pos)                         */
/*                                                                          */
/*  Insert obj into object cache at pos, resizing if needed.  Its true      */
/*  type is CACHE_OBJ, not void *.                                          */
/*                                                                          */
/****************************************************************************/

void npsys_obj_cache_insert(void *obj, int pos)
{
  npsys_obj->cache[pos] = (CACHE_OBJ) obj;
  if( ++npsys_obj->cache_count > npsys_obj->cache_maxcount )
  {
    mem_rehash(npsys_obj, 3 * npsys_obj->cache_len);
    npsys_obj_cache = (CACHE_OBJ *) npsys_obj->cache;
    npsys_obj_cache_len = npsys_obj->cache_len;
  }
}


/****************************************************************************/
/*                                                                          */
/*  Submodule "function call memory system".                                */
/*                                                                          */
/****************************************************************************/

MEM_SYS npsys_fun;			/* current function memory system   */
CACHE_FUN *npsys_fun_cache;		/* current function mem cache       */
int npsys_fun_cache_len;		/* current function mem cache len   */


/****************************************************************************/
/*                                                                          */
/*  void npsys_fun_init()                                                   */
/*                                                                          */
/*  Initialize current function memory system.                              */
/*                                                                          */
/****************************************************************************/

void npsys_fun_init(MEM_SYS *mem_sys)
{
  *mem_sys = mem_init(FUN_CACHE_INIT_SIZE);
  npsys_fun = *mem_sys;
  npsys_fun_cache = (CACHE_FUN *) (*mem_sys)->cache;
  npsys_fun_cache_len = (*mem_sys)->cache_len;
}


/****************************************************************************/
/*                                                                          */
/*  CACHE_FUN npsys_fun_new(size_t size)                                    */
/*                                                                          */
/*  Return a new function record of the given size.                         */
/*                                                                          */
/****************************************************************************/

CACHE_FUN npsys_fun_new(size_t size)
{
  return (CACHE_FUN) mem_new(npsys_fun, size);
}


/****************************************************************************/
/*                                                                          */
/*  void npsys_fun_free(void *fun, size_t size)                             */
/*                                                                          */
/*  Free fun, whose size is size; but only if this can be done easily.      */
/*                                                                          */
/****************************************************************************/

void npsys_fun_free(void *fun, size_t size)
{
  mem_free(npsys_fun, fun, size);
}


/****************************************************************************/
/*                                                                          */
/*  void npsys_fun_cache_insert(void *fun, int pos)                         */
/*                                                                          */
/*  Insert fun into function cache at pos, resizing if needed.  Its true    */
/*  type is CACHE_FUN, not void *.                                          */
/*                                                                          */
/****************************************************************************/

void npsys_fun_cache_insert(void *fun, int pos)
{
  npsys_fun->cache[pos] = (CACHE_OBJ) fun;
  if( ++npsys_fun->cache_count > npsys_fun->cache_maxcount )
  {
    mem_rehash(npsys_fun, 3 * npsys_fun->cache_len);
    npsys_fun_cache = (CACHE_FUN *) npsys_fun->cache;
    npsys_fun_cache_len = npsys_fun->cache_len;
  }
}


/****************************************************************************/
/*                                                                          */
/*  Submodule "array and string creation functions".                        */
/*                                                                          */
/****************************************************************************/

array array_create1(int length)
{
  /* allocates memory but does not set elems or check cache */
  array self;  size_t len;
  len = sizeof(struct array_rec) + sizeof(self->elems[0])*(length - 1);
  self = (array) npsys_obj_new(len);
  self->type_tag = array_tag;
  self->length = length;
  return self;
}

array array_create2(array self)
{
  /* assumes elems set; hashes and checks cache */
  unsigned int sig, i, j, len;

  /* calculate hash signature */
  assert(self->type_tag == array_tag);
  sig = (array_tag << 16) + self->length;
  len = (self->length <= 12 ? self->length : 12);
  for( i = 0;  i < len;  i++ )
    sig += ((int) self->elems[i]) << i;

  /* if in cache, unallocate self and return other */
  i = sig % npsys_obj_cache_len;
  while( npsys_obj_cache[i] != NULL )
  {
    if( npsys_obj_cache[i]->signature == sig )
    {
      array other = (array) npsys_obj_cache[i];
      if( other->type_tag == self->type_tag && other->length == self->length )
      {
	/* decide whether the array is already present or not */
        for( j = 0;  j < self->length;  j++ )
          if( self->elems[j] != other->elems[j] )
            goto NOT_OTHER;

	/* if already present, free it, first marking it to avoid confusion */
	if( self->length > 0 )
	  self->elems[0] = (void *) DFT_PARAM_VAL;
        npsys_obj_free((void *) self,
          sizeof(struct array_rec) + sizeof(self->elems[0])*(self->length-1));
        return other;
      }
    }
    NOT_OTHER:
    i = (i + 1) % npsys_obj_cache_len;
  }

  /* not present: set sig, insert into cache, and return */
  self->signature = sig;
  npsys_obj_cache_insert((CACHE_OBJ) self, i);
  return self;
}


string string_create1(int length)
{
  /* allocates memory but does not set elems or check cache */
  string self;  size_t len;
  len = sizeof(struct string_rec) + sizeof(self->elems[0])*(length - 1);
  self = (string) npsys_obj_new(len);
  self->type_tag = string_tag;
  self->length = length;
  return self;
}

string string_create2(string self)
{
  /* assumes elems set; hashes and checks cache */
  unsigned int sig, i, j, len;

  /* calculate hash signature */
  assert(self->type_tag == string_tag);
  sig = (string_tag << 16) + self->length;
  len = (self->length <= 12 ? self->length : 12);
  for( i = 0;  i < len;  i++ )
    sig += ((int) self->elems[i]) << i;

  /* if in cache, unallocate self and return other */
  i = sig % npsys_obj_cache_len;
  while( npsys_obj_cache[i] != NULL )
  {
    if( npsys_obj_cache[i]->signature == sig )
    {
      string other = (string) npsys_obj_cache[i];
      if( other->type_tag == self->type_tag && other->length == self->length )
      {

	/* decide whether the string is already present or not */
        for( j = 0;  j < self->length;  j++ )
          if( self->elems[j] != other->elems[j] )
            goto NOT_OTHER;

	/* if already present, free it, first marking it to avoid confusion */
	if( self->length > 0 )
	  self->elems[0] = 0;
        npsys_obj_free((void *) self,
          sizeof(struct string_rec) + sizeof(self->elems[0])*(self->length-1));
        return other;
      }
    }
    NOT_OTHER:
    i = (i + 1) % npsys_obj_cache_len;
  }

  /* not present: set sig, insert into cache, and return */
  self->signature = sig;
  npsys_obj_cache_insert((CACHE_OBJ) self, i);
  return self;
}
