
/****************************************************************************/
/*                                                                          */
/*  npsys_init.c                                                            */
/*                                                                          */
/*  Nonpareil runtime system (initialization of predefined objects)         */
/*                                                                          */
/*  Jeffrey H. Kingston                                                     */
/*  March 2005                                                              */
/*                                                                          */
/*  This software has been placed in the public domain by its author.       */
/*                                                                          */
/****************************************************************************/

#include <assert.h>
#include <string.h>
#include <stdio.h>
#include "npsys.h"
#include "npsys_init.h"
#include "system.h"
#include "system_init.h"
#define DEBUG1  0


/****************************************************************************/
/*                                                                          */
/*  Submodule "swizzling"                                                   */
/*                                                                          */
/****************************************************************************/

int array_swizzle(array self, array other)
{
  int offset1, offset2, i;
  assert(self->type_tag == other->type_tag);
  self->swizzle_bits = 0;
  if( self->length > 0 )
  {
    offset1 = mem_offset(npsys_obj1, (CACHE_OBJ) self->elems[0]);
    offset2 = mem_offset(npsys_obj2, (CACHE_OBJ) other->elems[0]);
    assert(offset1 == offset2);
    if( offset1 >= 0 )
    {
      self->swizzle_bits = 1;
      for( i = 0;  i < self->length;  i++ )
        self->elems[i] =
	  (void *) mem_offset(npsys_obj1, (CACHE_OBJ) self->elems[i]);
    }
  }
  return sizeof(struct array_rec) + sizeof(self->elems[0])*(self->length - 1);
}


int string_swizzle(string self, string other)
{
  /* no actual swizzling in strings */
  assert(self->type_tag == other->type_tag);
  return sizeof(struct string_rec) + sizeof(self->elems[0])*(self->length - 1);
}


/****************************************************************************/
/*                                                                          */
/*  void npsys_swizzle(MEM_SYS npsys_obj1, MEM_SYS npsys_obj2)              */
/*                                                                          */
/*  Swizzle all the objects in the npsys_obj1 memory system.  Ignore        */
/*  the cache chunk (we don't swizzle the cache).  Use npsys_obj2, which    */
/*  should be an exact duplicate of npsys_obj1 but in different memory,     */
/*  as an aid in working out which fields need to be swizzled and which     */
/*  do not.                                                                 */
/*                                                                          */
/*  This function calls system_swizzle(p1, p2), a function generated into   */
/*  file system.c by the Nonpareil compiler, to swizzle individual objects. */
/*                                                                          */
/****************************************************************************/

static void npsys_swizzle(MEM_SYS npsys_obj1, MEM_SYS npsys_obj2)
{
  MEM_CHUNK chunk1, chunk2;
  char *p1, *p2;  int inc;
  if( DEBUG1 )
    fprintf(stderr, "[ npsys_swizzle(%p, %p)\n", (void *) npsys_obj1,
      (void *) npsys_obj2);
  chunk2 = npsys_obj2->used_chunks;
  for( chunk1=npsys_obj1->used_chunks; chunk1!=NULL; chunk1=chunk1->prev_chunk )
  {
    if( chunk1 != npsys_obj1->cache_chunk )
    {
      assert(chunk2 != NULL && chunk2 != npsys_obj2->cache_chunk);
      if( DEBUG1 )
	fprintf(stderr, "  npsys_swizzle chunk %d\n",
	  chunk1->free-chunk1->first);
      p1 = chunk1->first;
      p2 = chunk2->first;
      assert((chunk1->free - p1) == (chunk2->free - p2));
      while( p1 < chunk1->free )
      {
	inc = system_swizzle((object) p1, (object) p2);
	p1 += inc;
	p2 += inc;
	if( DEBUG1 )
	  fprintf(stderr, "    npsys_swizzle inc %d\n", inc);
      }
    }
    chunk2 = chunk2->prev_chunk;
  }
  if( DEBUG1 )
    fprintf(stderr, "] npsys_swizzle returning\n");
}


/****************************************************************************/
/*                                                                          */
/*  Submodule "helper functions for creating small strings".                */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/*  string string_make0()                                                   */
/*                                                                          */
/*  Make and cache a string of length 0.                                    */
/*                                                                          */
/****************************************************************************/

string string_make0()
{
  string res;
  res = string_create1(0);
  return string_create2(res);
}


/****************************************************************************/
/*                                                                          */
/*  string string_make1(uchar c1)                                           */
/*                                                                          */
/*  Make and cache a string of length 1.                                    */
/*                                                                          */
/****************************************************************************/

string string_make1(uchar c1)
{
  string res;
  res = string_create1(1);
  res->elems[0] = c1;
  return string_create2(res);
}


/****************************************************************************/
/*                                                                          */
/*  string string_make2(uchar c1, uchar c2)                                 */
/*                                                                          */
/*  Make and cache a string of length 2.                                    */
/*                                                                          */
/****************************************************************************/

string string_make2(uchar c1, uchar c2)
{
  string res;
  res = string_create1(2);
  res->elems[0] = c1;
  res->elems[1] = c2;
  return string_create2(res);
}


/****************************************************************************/
/*                                                                          */
/*  string string_make3(uchar c1, uchar c2, uchar c3)                       */
/*                                                                          */
/*  Make and cache a string of length 3.                                    */
/*                                                                          */
/****************************************************************************/

string string_make3(uchar c1, uchar c2, uchar c3)
{
  string res;
  res = string_create1(3);
  res->elems[0] = c1;
  res->elems[1] = c2;
  res->elems[2] = c3;
  return string_create2(res);
}


/****************************************************************************/
/*                                                                          */
/*  string string_make4(uchar c1, uchar c2, uchar c3, uchar c4)             */
/*                                                                          */
/*  Make and cache a string of length 4.                                    */
/*                                                                          */
/****************************************************************************/

string string_make4(uchar c1, uchar c2, uchar c3, uchar c4)
{
  string res;
  res = string_create1(4);
  res->elems[0] = c1;
  res->elems[1] = c2;
  res->elems[2] = c3;
  res->elems[3] = c4;
  return string_create2(res);
}


/****************************************************************************/
/*                                                                          */
/*  string string_make5(uchar c1, uchar c2, uchar c3, uchar c4, uchar c5)   */
/*                                                                          */
/*  Make and cache a string of length 5.                                    */
/*                                                                          */
/****************************************************************************/

string string_make5(uchar c1, uchar c2, uchar c3, uchar c4, uchar c5)
{
  string res;
  res = string_create1(5);
  res->elems[0] = c1;
  res->elems[1] = c2;
  res->elems[2] = c3;
  res->elems[3] = c4;
  res->elems[4] = c5;
  return string_create2(res);
}


/****************************************************************************/
/*                                                                          */
/*  null tries of height 1..4                                               */
/*                                                                          */
/****************************************************************************/

array null_trie1;
array null_trie2;
array null_trie3;
array null_trie4;

static void null_tries_init()
{
  int i;

  /* initialize null_trie1 */
  null_trie1 = array_create1(256);
  for( i = 0;  i < 256;  i++ )
    null_trie1->elems[i] = NULL;
  null_trie1 = array_create2(null_trie1);

  /* initialize null_trie2 */
  null_trie2 = array_create1(256);
  for( i = 0;  i < 256;  i++ )
    null_trie2->elems[i] = (void *) null_trie1;
  null_trie2 = array_create2(null_trie2);

  /* initialize null_trie3 */
  null_trie3 = array_create1(256);
  for( i = 0;  i < 256;  i++ )
    null_trie3->elems[i] = (void *) null_trie2;
  null_trie3 = array_create2(null_trie3);

  /* initialize null_trie4 */
  null_trie4 = array_create1(256);
  for( i = 0;  i < 256;  i++ )
    null_trie4->elems[i] = (void *) null_trie3;
  null_trie4 = array_create2(null_trie4);
}


/****************************************************************************/
/*                                                                          */
/*  void npsys_init()                                                       */
/*                                                                          */
/*  Initialize to file.                                                     */
/*                                                                          */
/*  This function calls system_init(), generated by the Nonpareil           */
/*  compiler into file system.c, to do a single initialization of the       */
/*  system into the current memory system.  It does this twice, making      */
/*  two copies of the same initialized system, then swizzles the first      */
/*  one, utilizing the second to help decide which fields to swizzle        */
/*  (see comment on npsys_swizzle).  It then saves the swizzled memory      */
/*  system to a file.                                                       */
/*                                                                          */
/****************************************************************************/

static void npsys_init()
{
  array all_all1, all_all2;
  char *file_name;  FILE *fp;  int sys_sig;
  MEM_CHUNK chunk;  size_t nmemb;  int all_all_offset;

  /* initialize function call memory system (just once) */
  npsys_fun_init(&npsys_fun);

  /* first init, into npsys_obj1 memory system */
  npsys_obj_init(&npsys_obj1);
  null_tries_init();
  all_all1 = system_init();

  /* second init, into npsys_obj2 memory system */
  npsys_obj_init(&npsys_obj2);
  null_tries_init();
  all_all2 = system_init();

  /* swizzle */
  npsys_swizzle(npsys_obj1, npsys_obj2);

  /* save: get file name and open the file for writing */
  if( mem_is_bigendian() )
    file_name = NP_DATA_DIR NP_DIR_SEP "data_be";
  else
    file_name = NP_DATA_DIR NP_DIR_SEP "data_le";
  fp = fopen(file_name, "wb");
  if( fp == NULL )
  {
    fprintf(stderr, "cannot write to data file \"%s\"\n", file_name);
    exit(1);
  }

  /* save: write the magic number */
  sys_sig = NP_SYS_SIG;
  if( fwrite((void *) &sys_sig, sizeof(int), 1, fp) != 1 )
  {
    fprintf(stderr, "could not write magic number to data file \"%s\"\n",
      file_name);
    exit(1);
  }

  /* save: swizzle and write all_all1 */
  all_all_offset = mem_offset(npsys_obj1, (CACHE_OBJ) all_all1);
  if( fwrite((void *) &all_all_offset, sizeof(int), 1, fp) != 1 )
  {
    fprintf(stderr, "could not write all_all offset to data file \"%s\"\n",
      file_name);
    exit(1);
  }

  /* write the chunks (but not the cache chunk) and close the file */
  for( chunk=npsys_obj1->used_chunks; chunk!=NULL; chunk = chunk->prev_chunk )
  {
    if( chunk == npsys_obj1->cache_chunk )
      continue;
    nmemb = chunk->free - chunk->first;
    if( fwrite((void *) chunk->first, sizeof(char), nmemb, fp) != nmemb )
    {
      fprintf(stderr, "opened but could not write to data file \"%s\"\n",
	file_name);
      exit(1);
    }
  }
  fclose(fp);
}


/****************************************************************************/
/*                                                                          */
/*  int main(int argc, char *argv[])                                        */
/*                                                                          */
/*  Main function (for initialization of predefined objects).               */
/*                                                                          */
/****************************************************************************/

int main(int argc, char *argv[])
{
  int i;

  /* read command line arguments and initialize */
  for( i = 1;  i < argc;  i++ )
  {
    if( strcmp(argv[i], "-v") == 0 )
    {
      /* print version and module information and exit */
      system_version();
      exit(0);
    }
    else
    {
      fprintf(stderr, "%s: unknown command line option \"%s\"\n",
	argv[0], argv[i]);
      exit(1);
    }
  }

  /* initialize */
  npsys_init();

  /* that's all we do on this run */
  exit(0);
}
