
/****************************************************************************/
/*                                                                          */
/*  npsys_load.c                                                            */
/*                                                                          */
/*  Nonpareil runtime system (loading of predefined objects)                */
/*                                                                          */
/*  Jeffrey H. Kingston                                                     */
/*  March 2005                                                              */
/*                                                                          */
/*  This software has been placed in the public domain by its author.       */
/*                                                                          */
/****************************************************************************/

#include <assert.h>
#include <string.h>
#include <stdio.h>
#include "npsys.h"
#include "npsys_load.h"
#include "system.h"
#include "system_load.h"
#define DEBUG1 0
#define DEBUG2 1


/****************************************************************************/
/*                                                                          */
/*  Submodule "array and string unswizzling functions".                     */
/*                                                                          */
/****************************************************************************/

int array_unswizzle(array self, char *mem)
{
  unsigned i, sig, len;

  if( self->swizzle_bits == 1 )
  {
    /* unswizzle every element */
    for( i = 0;  i < self->length;  i++ )
      self->elems[i] = (void *) (mem + (int) self->elems[i]);

    /* recalculate signature */
    sig = (array_tag << 16) + self->length;
    len = (self->length <= 12 ? self->length : 12);
    for( i = 0;  i < len;  i++ )
      sig += ((int) self->elems[i]) << i;
    self->signature = sig;
  }

  /* insert into cache (cannot be already present) */
  i = self->signature % npsys_obj_cache_len;
  while( npsys_obj_cache[i] != NULL )
    i = (i + 1) % npsys_obj_cache_len;
  npsys_obj_cache_insert((CACHE_OBJ) self, i);

  /* return record length */
  return sizeof(struct array_rec) + sizeof(self->elems[0])*(self->length - 1);
}


int string_unswizzle(string self, char *mem)
{
  int i;

  /* insert into cache (cannot be already present) */
  i = self->signature % npsys_obj_cache_len;
  while( npsys_obj_cache[i] != NULL )
    i = (i + 1) % npsys_obj_cache_len;
  npsys_obj_cache_insert((CACHE_OBJ) self, i);

  /* return record length */
  return sizeof(struct string_rec) + sizeof(self->elems[0])*(self->length - 1);
}


/****************************************************************************/
/*                                                                          */
/*  void npsys_load()                                                       */
/*                                                                          */
/*  This function parallels npsys_init().  It loads a file previously       */
/*  saved by npsys_init() and unswizzles it.                                */
/*                                                                          */
/****************************************************************************/

static void npsys_load()
{
  char *file_name;  FILE *fp;  long file_len;
  int chunk_len;  char *mem, *mem_top, *p;
  int sys_sig;  array all_all;  int all_all_offset;

  /* get file name, open file for reading */
  if( mem_is_bigendian() )
    file_name = NP_DATA_DIR NP_DIR_SEP "data_be";
  else
    file_name = NP_DATA_DIR NP_DIR_SEP "data_le";
  fp = fopen(file_name, "rb");
  if( fp == NULL )
  {
    fprintf(stderr, "cannot open data file \"%s\" for reading\n", file_name);
    exit(1);
  }

  /* find the file's length */
  fseek(fp, 0L, SEEK_END);
  file_len = ftell(fp);
  rewind(fp);

  /* read and check the magic number */
  if( fread(&sys_sig, sizeof(int), 1, fp) != 1 )
  {
    fprintf(stderr, "could not read magic number from data file \"%s\"\n",
      file_name);
    exit(1);
  }
  if( sys_sig != NP_SYS_SIG )
  {
    if( DEBUG2 )
      fprintf(stderr, "found magic number %d, expected %d\n", sys_sig,
	NP_SYS_SIG);
    fprintf(stderr, "wrong magic number in data file \"%s\"\n", file_name);
    fprintf(stderr, "  (an initializing run now should fix this problem)\n");
    exit(1);
  }

  /* read the offset of all_all in the following chunk */
  if( fread(&all_all_offset, sizeof(int), 1, fp) != 1 )
  {
    fprintf(stderr, "could not read all_all offset from data file \"%s\"\n",
      file_name);
    exit(1);
  }

  /* initialize the current memory system and get space for the chunk */
  npsys_obj_init(&npsys_obj1);
  chunk_len = file_len - 2*sizeof(int);
  mem = (char *) npsys_obj_new(chunk_len);
  if( fread(mem, sizeof(char), chunk_len, fp) != chunk_len )
  {
    fprintf(stderr, "internal error: failed to read data chunk\n");
    exit(1);
  }
  
  /* unswizzle all_all_offset into all_all */
  all_all = (array) (mem + all_all_offset);

  /* unswizzle mem */
  mem_top = mem + chunk_len;
  p = mem;
  while( p < mem_top )
    p += system_unswizzle((object) p, mem);

  /* initialize all_predefined etc. arrays from all_all */
  system_load(all_all);

  /* initialize function call memory system */
  npsys_fun_init(&npsys_fun);
}


/****************************************************************************/
/*                                                                          */
/*  int main(int argc, char *argv[])                                        */
/*                                                                          */
/*  Main function (for loading of predefined objects from file).            */
/*                                                                          */
/****************************************************************************/

int main(int argc, char *argv[])
{
  int i;

  /* read command line arguments and initialize */
  for( i = 1;  i < argc;  i++ )
  {
    if( strcmp(argv[i], "-v") == 0 )
    {
      /* print version and module information and exit */
      system_version();
      exit(0);
    }
    else
    {
      fprintf(stderr, "%s: unknown command line option \"%s\"\n",
	argv[0], argv[i]);
      exit(1);
    }
  }

  /* load from file */
  npsys_load();

  /* that's all so far */
  fprintf(stderr, "If you see this, %s loaded the file of\n", argv[0]);
  fprintf(stderr, "predefined data without crashing.  That's\n");
  fprintf(stderr, "all it does so far.  Have a nice day.\n");
  exit(0);
}
