/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         main.c                                                     */
/*  DESCRIPTION:  Main program (implementation)                              */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#define DFT_COMPILER "gcc -g -ansi -pedantic -Wall"


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_KIND                                                             */
/*                                                                           */
/*  The type of code to generate.  At present there is only one choice,      */
/*  namely C.                                                                */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  CODEGEN_UNKNOWN,			/* code gen type not know yet        */
  CODEGEN_C				/* generate C code                   */
} CODEGEN_KIND;


/*****************************************************************************/
/*                                                                           */
/*  void test_strings()                                                      */
/*                                                                           */
/*  Test the ustring module.                                                 */
/*                                                                           */
/*****************************************************************************/

void test_strings()
{
  USTRING s1;
  ASTRING a1;
  fprintf(stderr, "[ test_strings\n");

  /* Test 1: from ascii to USTRING and back again */
  a1 = "abcdefg";
  s1 = AStringToUString(a1);
  printf("Test 1:AStringToUString(\"%s\") = \"%s\" (%d)\n", a1,
    UStringToUTF8(s1), UStringLength(s1));

  /* Test 2: escape characters */
  a1 = "a\t\nb";
  s1 = AStringToUString(a1);
  printf("Test 2: AStringToUString(\"%s\") = \"%s\" (%d)\n", a1,
    UStringToUTF8(s1), UStringLength(s1));
  fprintf(stderr, "] end test_strings\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void test_arrays()                                                       */
/*                                                                           */
/*  Test the arrays module.                                                  */
/*                                                                           */
/*****************************************************************************/

void test_arrays()
{
  ARRAY(int) a;
  ARRAY(float) b;
  int i, x;

  /* printf("%s\n", VERSION); */

  /* test of arrays code */
  fprintf(stderr, "[ test_arrays\n");
  ArrayInit(&a);

  printf("Test 1: printing empty array\n");
  for( i = 0; i < ArraySize(a); i++ )
    printf("ArrayGet(a, %d) = %d\n", i, ArrayGet(a, i));
  printf("End Test 1\n\n");

  printf("Test 2: appending some numbers\n");
  for( i = 0;  i < 10;  i++ )
    ArrayAddLast(a, 2*i);
  for( i = 0; i < ArraySize(a); i++ )
    printf("ArrayGet(a, %d) = %d\n", i, ArrayGet(a, i));
  printf("End Test 2\n\n");


  printf("Test 3: removing some numbers\n");
  for( i = 0;  i < 5;  i++ )
    x = ArrayRemoveLast(a);
  for( i = 0; i < ArraySize(a); i++ )
    printf("ArrayGet(a, %d) = %d\n", i, ArrayGet(a, i));
  printf("End Test 3\n\n");

  ArrayInit(&b);

  printf("Test 4: printing empty float array\n");
  for( i = 0; i < ArraySize(b); i++ )
    printf("ArrayGet(b, %d) = %f\n", i, ArrayGet(b, i));
  printf("End Test 4\n\n");

  printf("Test 5: appending some numbers\n");
  for( i = 0;  i < 10;  i++ )
    ArrayAddLast(b, 2*i + 0.5);
  for( i = 0; i < ArraySize(b); i++ )
    printf("ArrayGet(b, %d) = %f\n", i, ArrayGet(b, i));
  printf("End Test 5\n\n");

  printf("Test 6: replacing some numbers\n");
  ArrayPut(b, 0, 0.0001);
  ArrayPut(b, 9, 99.0001);
  for( i = 0; i < ArraySize(b); i++ )
    printf("ArrayGet(b, %d) = %f\n", i, ArrayGet(b, i));
  printf("End Test 5\n\n");

  fprintf(stderr, "] end test_arrays\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void test_symtab()                                                       */
/*                                                                           */
/*  Test the symbol table module.                                            */
/*                                                                           */
/*****************************************************************************/

void test_symtab()
{
  SYMTAB_VOIDP s;
  void *v1;

  /* initialize and debug print */
  fprintf(stderr, "[ test_symtab\n");
  SymInit(&s);
  SymDebugPrint(s);

  /* do some insertions, and debug print */
  SymInsert(s, AStringToUString("abc"), NULL, &v1);
  SymInsert(s, AStringToUString("a"), NULL, &v1);
  SymDebugPrint(s);

  /* do some more insertions, and debug print */
  SymInsert(s, AStringToUString("abc"), NULL, &v1);
  SymInsert(s, AStringToUString("a"), NULL, &v1);
  SymDebugPrint(s);

  /* do some more insertions, and debug print */
  SymInsert(s, AStringToUString("def"), NULL, &v1);
  SymInsert(s, AStringToUString("d"), NULL, &v1);
  SymDebugPrint(s);

  /* do some more insertions, and debug print */
  SymInsert(s, AStringToUString("ghi"), NULL, &v1);
  SymInsert(s, AStringToUString("g"), NULL, &v1);
  SymDebugPrint(s);

  /* do some more insertions, and debug print */
  SymInsert(s, AStringToUString("jkl"), NULL, &v1);
  SymInsert(s, AStringToUString("j"), NULL, &v1);
  SymInsert(s, AStringToUString("mno"), NULL, &v1);
  SymInsert(s, AStringToUString("m"), NULL, &v1);
  SymInsert(s, AStringToUString("pqr"), NULL, &v1);
  SymInsert(s, AStringToUString("p"), NULL, &v1);
  SymInsert(s, AStringToUString("stu"), NULL, &v1);
  SymInsert(s, AStringToUString("s"), NULL, &v1);
  SymDebugPrint(s);

  /* now try some retrievals */
  printf("SymRetrieve(s, \"%s\"): %s\n", "def", 
    SymRetrieve(s, AStringToUString("def"), &v1) ? "present" : "not present");
  printf("SymRetrieve(s, \"%s\"): %s\n", "d", 
    SymRetrieve(s, AStringToUString("d"), &v1) ? "present" : "not present");
  printf("SymRetrieve(s, \"%s\"): %s\n", "xyz", 
    SymRetrieve(s, AStringToUString("xyz"), &v1) ? "present" : "not present");
  printf("SymRetrieve(s, \"%s\"): %s\n", "x", 
    SymRetrieve(s, AStringToUString("x"), &v1) ? "present" : "not present");
  fprintf(stderr, "] end test_symtab\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void LoadSystem(CODEGEN be, ASTRING sys_dir, SYSTEM *system,             */
/*    SYSTEM_VIEW *sv)                                                       */
/*                                                                           */
/*  Load the system and its system module.                                   */
/*                                                                           */
/*****************************************************************************/

static void LoadSystem(CODEGEN be, ASTRING sys_dir, SYSTEM *system,
  SYSTEM_VIEW *sv)
{
  *system = NULL;
  if( !SystemCompile(be, sys_dir, system) )
    *sv = NULL;
  else
    *sv = SystemSystemView(*system);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN InitCodeGen()                                                    */
/*                                                                           */
/*  Return a newly initialized CODEGEN object of the kind requested by       */
/*  codegen_type.  Also initialize the NPBack module.                        */
/*                                                                           */
/*****************************************************************************/

static CODEGEN_KIND codegen_type;

static CODEGEN InitCodeGen()
{
  CODEGEN res;
  switch( codegen_type )
  {
    case CODEGEN_C:

      res = CodeGen_C(AStringToUString("./ccode"), NPC_C_COMPILER);
      break;

    default:

      assert(FALSE);
      res = NULL;  /* keep compiler happy */
      break;
  }
  NPBackInit(res);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void Interact(SYSTEM sys, SYSTEM_VIEW curr_sv, CLASS_VIEW curr_cv,       */
/*    TYPE_VARS curr_type_vars, TYPE curr_type, CONTEXT curr_cxt,            */
/*    FILE *in_fp, FILE *out_fp, ASTRING sys_dir, CODEGEN be)                */
/*                                                                           */
/*  Do interactive tests, reading them from in_fp, and writing them to       */
/*  out_fp, which are both open.                                             */
/*                                                                           */
/*****************************************************************************/
#define FF "%-26s %s\n"
#define MAX_BUFF 100
#define pr(a, b) fprintf(out_fp, FF, (a), (b))
#define str_eq(a, b) (strcmp((a), (b)) == 0)

static void heading(FILE *fp, ASTRING str)
{
  int i;
  fprintf(fp, "\n%s\n", str);
  for( i = 0;  i < strlen(str);  i++ )
    fprintf(fp, "-");
  fprintf(fp, "\n");
}

static void skip_space(FILE *fp)
{
  int ch = getc(fp);
  while( ch == ' ' || ch == '\t' )
    ch = getc(fp);
  ungetc(ch, fp);
}

static void Interact(SYSTEM sys, SYSTEM_VIEW curr_sv, CLASS_VIEW curr_cv,
  TYPE_VARS curr_type_vars, TYPE curr_type, CONTEXT curr_cxt,
  FILE *in_fp, FILE *out_fp, ASTRING sys_dir, CODEGEN be)
{
  TOKEN token_list; EXPR expr;
  char command[MAX_BUFF], buff[MAX_BUFF];
  FILE *new_fp;

  /* print session header if fully interactive */
  if( in_fp == stdin && out_fp == stdout )
  {
    fprintf(out_fp, "\nNonpareil compiler; type help for help\n");
    if( sys == NULL )
      fprintf(out_fp, "No system owing to previous error(s) loading it\n");
  }

  /* prompt for and execute commands, one line at a time */
  fprintf(out_fp, "npc: ");
  for( ; fscanf(in_fp, "%s", command) == 1; fprintf(out_fp, "npc: ") )  
  {
    /* echo command unless fully interactive */
    if( in_fp != stdin || out_fp != stdout )
      fprintf(out_fp, "%s\n", command);

    /* help command */
    if( str_eq(command, "help") || str_eq(command, "?") )
    {
      if( sys != NULL )
      {
	heading(out_fp, "Systems and modules");
	pr("setsystem", "Reload system");
	pr("system", "Show current system");
	pr("setmodule <modulename>", "Set current module");
	pr("module", "Show current module");

	heading(out_fp, "Classes and features");
	pr("setclass <classname>", "Set current class");
	pr("class", "Show current class");
	pr("classtypes", "Show class types");
	pr("features", "Show list of features of current class");
	pr("feature <featurename>", "Show feature");

	heading(out_fp, "Types and feature signatures");
	pr("settype <type>", "Parse, flatten, echo and set current type");
	pr("type", "Show current type");
	pr("sig <featurename>", "Find <featurename> in current type");

	heading(out_fp, "Commands to exercise system without changing state");
	pr("lex <tokens>", "Lex these tokens");
	pr("expr <expr>", "Parse and echo expression <expr>");
	pr("texpr <expr>", "Like expr, but with full type info");

	heading(out_fp, "Miscellaneous commands");
	pr("limits", "Print INT_MIN, INT_MAX etc.");
	pr("infile <filename>", "Take commands from <file>");
	pr("outfile <filename>", "Place results on <file>");
	pr("sh <command>", "Shell escape");
	pr("copyright", "Print copyright notice");
	pr("help or ?", "Print this help message");
	pr("quit", "Quit (or end most recent unfinished infile/outfile)");
	fprintf(out_fp, "\n");
      }
      else
      {
	pr("limits", "Print INT_MIN, INT_MAX etc.");
	pr("infile <filename>", "Take commands from <file>");
	pr("outfile <filename>", "Place results on <file>");
	pr("setsystem", "Reload system");
	pr("copyright", "Print copyright notice");
	pr("help or ?", "Print this help message");
	pr("quit", "Quit");
	fprintf(out_fp, "\n");
	fprintf(out_fp, "Other commands unavailable since no system\n");
	fprintf(out_fp, "\n");
      }
    }

    /************************************************************************
    heading(out_fp, "Systems and modules");
    pr("setsystem", "Reload system");
    pr("system", "Show current system");
    pr("setmodule <modulename>", "Set current module");
    pr("module", "Show current module");
    *************************************************************************/

    else if( str_eq(command, "setsystem") )
    {
      sys = NULL;
      curr_cxt = NULL;
      curr_cv = NULL;
      be = InitCodeGen();
      if( !SystemCompile(be, sys_dir, &sys) )
        fprintf(out_fp, "no system owing to error(s) loading it\n");
      else if( (curr_sv = SystemSystemView(sys)) == NULL )
        fprintf(out_fp, "could not find system module\n");
      else
      {
        fprintf(out_fp, "system loaded; current module is system module; %s\n",
	  "no current class");
	curr_cxt = ContextNew(SystemViewClassViews(curr_sv));
      }
    }

    else if( str_eq(command, "system") )
    {
      if( sys == NULL )
	fprintf(out_fp, "no current system\n");
      else
      {
	SystemDebugSystemViewNames(sys, curr_sv, out_fp);
	fprintf(out_fp, "\n");
      }
    }

    else if( str_eq(command, "setmodule") )
    {
      if( sys == NULL )
	fprintf(stderr, "no system; type help for available commands\n");
      else if( fscanf(in_fp, "%s", buff) != 1 )
	fprintf(out_fp, "cannot read module name\n");
      else if( !SystemRetrieveView(sys, AStringToUString(buff), &curr_sv) )
	fprintf(out_fp, "cannot find module \"%s\" in system\n", buff);
      else
      {
	fprintf(out_fp, "switching to module \"%s\"\n", buff);
	curr_cxt = ContextNew(SystemViewClassViews(curr_sv));
      }
      curr_cv = NULL;
    }

    else if( str_eq(command, "module") )
    {
      if( curr_sv == NULL )
	fprintf(out_fp, "no current module\n");
      else
      {
	SystemViewDebug(curr_sv, curr_cxt, out_fp, 0);
	fprintf(out_fp, "\n");
      }
    }

    /************************************************************************
    heading(out_fp, "Classes and features");
    pr("setclass <classname>", "Set current class");
    pr("class", "Show current class");
    pr("classtypes", "Show class types");
    pr("features", "Show list of features of current class");
    pr("feature <featurename>", "Show feature (with body if original)");
    *************************************************************************/

    else if( str_eq(command, "setclass") )
    {
      if( curr_sv == NULL )
	fprintf(stderr, "no current module\n");
      else if( fscanf(in_fp, "%s", buff) != 1 )
	fprintf(stderr, "cannot read class name\n");
      else if( !SystemViewRetrieveClassView(curr_sv, AStringToUString(buff),
	  &curr_cv) )
	fprintf(stderr, "class \"%s\" not in current module %s\n",
	  buff, NameShow(SystemViewName(curr_sv)));
    }

    else if( str_eq(command, "class") )
    {
      if( curr_cv == NULL )
	fprintf(out_fp, "no current class\n");
      else
      {
	ClassViewDebug(curr_cv, curr_cxt, out_fp, 0);
	fprintf(out_fp, "\n");
      }
    }

    else if( str_eq(command, "classtypes") )
    {
      if( curr_cv == NULL )
	fprintf(out_fp, "no current class\n");
      else
      {
	ClassDebugTypes(ClassViewClass(curr_cv), curr_cxt, out_fp, 0);
	fprintf(out_fp, "\n");
      }
    }

    else if( str_eq(command, "features") )
    {
      if( curr_cv == NULL )
	fprintf(out_fp, "no current class\n");
      else
      {
	ClassViewDebug(curr_cv, curr_cxt, out_fp, 0);
	fprintf(out_fp, "\n");
      }
    }

    else if( str_eq(command, "feature") )
    {
      FEFN_FEATURE_SET fvs;
      if( fscanf(in_fp, "%s", buff) != 1 )
	fprintf(stderr, "cannot read feature name\n");
      else if( curr_cv == NULL )
	fprintf(out_fp, "no current class\n");
      else if( !ClassViewRetrieveFEFnFeatureSet(curr_cv,
	AStringToUString(buff), &fvs) )
	fprintf(out_fp, "no feature called \"%s\" in current class \"%s\"",
	  buff, NameShow(ClassViewName(curr_cv)));
      else
      {
	FEFnFeatureSetDebug(fvs, curr_cxt, out_fp, 0);
	fprintf(out_fp, "\n");
      }
    }

    /************************************************************************
    heading(out_fp, "Types and feature signatures");
    pr("settype <type>", "Parse, flatten, echo and set current type");
    pr("type", "Show current type");
    pr("sig <featurename>", "Find <featurename> in current type");
    *************************************************************************/

    else if( str_eq(command, "settype") )
    {
      TYPE type;  TYPE_VARS type_vars;
      skip_space(in_fp);
      if( sys == NULL )
	fprintf(stderr, "no system; type help for available commands\n");
      else if( !LexLine(in_fp, &token_list) )
	fprintf(stderr, "could not lex that line\n");
      else if(
	!TypeOptionalFormalGenericsParse(&token_list, FEATURE_NONCREATION,
	  FALSE, &type_vars)
	|| !TypeParse(&token_list, &type) )
	fprintf(stderr, "could not parse that line\n");
      else if( !TypeWithVarsLevelOneValid(type, type_vars, curr_cxt) )
	fprintf(stderr, "could not instantiate that type\n");
      else if( !TypeLevelThreeValid(type, curr_cxt) )
	fprintf(stderr, "type failed consistency check\n");
      else
      {
	if( type_vars != NULL )
	  fprintf(out_fp, "%s %s\n", TypeVarsShow(type_vars, curr_cxt),
	    TypeShow(type, curr_cxt));	
	else
	  fprintf(out_fp, "%s\n", TypeShow(type, curr_cxt));	
	curr_type_vars = type_vars;
	curr_type = type;
      }
    }

    else if( str_eq(command, "type") )
    {
      if( curr_type == NULL )
	fprintf(out_fp, "no current type\n");
      else if( curr_type_vars != NULL )
	fprintf(out_fp, "%s %s\n", TypeVarsShow(curr_type_vars, curr_cxt),
	  TypeShow(curr_type, curr_cxt));	
      else
	fprintf(out_fp, "%s\n", TypeShow(curr_type, curr_cxt));	
    }

    else if( str_eq(command, "sig") )
    {
      FEFN_FEATURE_SET fvs;
      if( fscanf(in_fp, "%s", buff) != 1 )
	fprintf(stderr, "cannot read feature name\n");
      else if( curr_type == NULL )
	fprintf(out_fp, "no current type\n");
      else if( !TypeRetrieveFEFnFeatureSet(curr_type, curr_cxt,
	  TypeFilePos(curr_type), AStringToUString(buff), &fvs) )
	fprintf(out_fp, "no feature %s in type %s\n", buff,
	  TypeShow(curr_type, curr_cxt));
      else
      {
	FEFnFeatureSetDebug(fvs, curr_cxt, out_fp, 0);
	fprintf(out_fp, "\n");
      }
    }

    /************************************************************************
    heading(out_fp, "Commands to exercise the system without changing state");
    pr("lex <tokens>", "Lex these tokens");
    pr("expr <expr>", "Parse and echo expression <expr>");
    pr("texpr <expr>", "Like expr, but with full type info");
    *************************************************************************/

    else if( str_eq(command, "lex") )
    {
      skip_space(in_fp);
      if( sys == NULL )
	fprintf(stderr, "no system; type help for available commands\n");
      else if( !LexLine(in_fp, &token_list) )
	fprintf(stderr, "could not lex that line\n");
      else
      {
        LexDebug(token_list, out_fp);
	fprintf(out_fp, "\n");
      }
    }

    else if( str_eq(command, "expr") || str_eq(command, "texpr") )
    {
      skip_space(in_fp);
      if( sys == NULL )
	fprintf(stderr, "no system; type help for available commands\n");
      else if( !LexLine(in_fp, &token_list) )
	fprintf(stderr, "could not lex that line\n");
      else if( !ExprParse(&token_list, curr_sv, &expr) )
	fprintf(stderr, "could not parse that line\n");
      else
      {
	if( !ExprManifest(&expr, curr_cxt, NULL, NULL) )
	  fprintf(stderr, "could not manifest that line\n");	
	else
	{
	  /* print expression and actual type (may be null if not done/known) */
	  if( str_eq(command, "texpr") )
	    ExprDebug(expr, curr_cxt, TRUE, out_fp, 0);
	  else
	  {
	    ExprDebug(expr, curr_cxt, FALSE, out_fp, SINGLE_LINE);
	    fprintf(out_fp, " : %s", TypeShow(ExprType(expr), curr_cxt));
	  }
	  fprintf(out_fp, "\n");
	}
      }
    }


    /************************************************************************
    heading(out_fp, "Miscellaneous commands");
    pr("limits", "Print INT_MIN, INT_MAX etc.");
    pr("infile <filename>", "Take commands from <file>");
    pr("outfile <filename>", "Place results on <file>");
    pr("sh <command>", "Shell escape");
    pr("help or ?", "Print this help message");
    pr("setsystem", "Reload system");
    pr("copyright", "Print copyright notice");
    pr("quit", "Quit (or end most recent unfinished infile/outfile)");
    *************************************************************************/

    else if( str_eq(command, "limits") )
    {
      fprintf(stdout, "%-10s: %d\n", "CHAR_BIT", CHAR_BIT);
      fprintf(stdout, "%-10s: %d\n", "CHAR_MAX", CHAR_MAX);
      fprintf(stdout, "%-10s: %d\n", "CHAR_MIN", CHAR_MIN);
      fprintf(stdout, "%-10s: %d\n", "INT_MAX", INT_MAX);
      fprintf(stdout, "%-10s: %d\n", "INT_MIN", INT_MIN);
      fprintf(stdout, "%-10s: %d\n", "SHRT_MAX", SHRT_MAX);
      fprintf(stdout, "%-10s: %d\n", "SHRT_MIN", SHRT_MIN);
    }
    else if( str_eq(command, "infile") )
    {
      if( fscanf(in_fp, "%s", buff) != 1 )
	fprintf(stderr, "cannot read file name\n");
      else if( (new_fp = fopen(buff, "r")) == NULL )
	fprintf(stderr, "cannot read file \"%s\"\n", buff);
      else
      {
	fprintf(out_fp, "reading commands from file %s\n", buff);
	Interact(sys, curr_sv, curr_cv, curr_type_vars,
	  curr_type, curr_cxt, new_fp, out_fp, sys_dir, be);
	fprintf(out_fp, "finished reading commands from file %s\n", buff);
      }
    }

    else if( str_eq(command, "outfile") )
    {
      if( fscanf(in_fp, "%s", buff) != 1 )
	fprintf(stderr, "cannot read file name\n");
      else if( (new_fp = fopen(buff, "w")) == NULL )
	fprintf(stderr, "cannot write file \"%s\"\n", buff);
      else
      {
	fprintf(stdout, "results will go to \"%s\" until next quit\n", buff);
	Interact(sys, curr_sv, curr_cv, curr_type_vars,
	  curr_type, curr_cxt, in_fp, new_fp, sys_dir, be);
	fprintf(out_fp, "\nfinished writing to file %s\n", buff);
      }
    }

    else if( str_eq(command, "sh") )
    {
      char buff[200];
      if( fgets(buff, 200, in_fp) )
        system(buff);
    }

    else if( str_eq(command, "copyright") )
    {
      MainCopyrightNotice(out_fp);
    }

    else if( str_eq(command, "quit") )
    {
      break;
    }

    /* unknown command */
    else
    {
      int ch;
      fprintf(stderr, "unknown command %s\n", command);
      ch = getc(in_fp);
      while( ch != '\n' && ch != '\0' )
	ch = getc(in_fp);
    }
  }
}


/*****************************************************************************/
/*                                                                           */
/* void MainCopyrightNotice(FILE *fp)                                        */
/*                                                                           */
/* Print copyright notice as required by GNU General Public License.         */
/*                                                                           */
/*****************************************************************************/

void MainCopyrightNotice(FILE *fp)
{
  fprintf(fp, "Copyright 2003, 2005 Jeffrey H. Kingston\n");
  fprintf(fp, "\n");
  fprintf(fp, "Nonpareil is free software, covered by the GNU General\n");
  fprintf(fp, "Public License, and you are welcome to change it and/or\n");
  fprintf(fp, "distribute copies of it under certain conditions.  There\n");
  fprintf(fp, "is absolutely no warranty for Nonpareil.  For details\n");
  fprintf(fp, "about both these issues consult the file \"gpl_license\"\n");
  fprintf(fp, "which is distributed with this software.\n");
}


/*****************************************************************************/
/*                                                                           */
/*  int PrimeAfter(int from)                                                 */
/*                                                                           */
/*  Return the smallest prime number greater than from.                      */
/*                                                                           */
/*****************************************************************************/

static int PrimeAfter(int from)
{
  int n, n2, i;
  for( n = from + 1;  ;  n = n + 1 )
  {
    n2 = n/2;
    for( i = 3;  i <= n2;  i++ )
      if( n % i == 0 )
	break;
    if( i > n2 )
      return n;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void UsageMessage(FILE *fp)                                              */
/*                                                                           */
/*  Print a usage message on fp.                                             */
/*                                                                           */
/*****************************************************************************/

void UsageMessage(FILE *fp)
{
  fprintf(fp, "%s\n\n", NPC_VERSION);
  fprintf(fp, "  -b<backend>    back end (default -bc, no others yet)\n");
  fprintf(fp, "  -i             enter interactive mode after compilation\n");
  fprintf(fp, "  -m             print various sizeof() values\n");
  fprintf(fp, "  -p<int>        print first prime number after <int>\n");
  fprintf(fp, "  -s<directory>  (optional) full name of system dirctory\n");
  fprintf(fp, "  -u             print this message\n");
  fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  main(argc, argv)                                                         */
/*                                                                           */
/*  Read command line and run.                                               */
/*                                                                           */
/*****************************************************************************/

int main(int argc, ASTRING argv[])
{
  SYSTEM sys;  SYSTEM_VIEW curr_sv;  CONTEXT curr_cxt;  int i;
  BOOLEAN interactive;  CODEGEN be;  ASTRING sys_dir;

  /* interpret command line arguments */
  codegen_type = CODEGEN_UNKNOWN;
  be = NULL;
  interactive = FALSE;
  sys_dir = NULL;
  for( i = 1;  i < argc;  i++ )
  {
    /* -b option: set back end */
    if( AStringBeginsWith(argv[i], "-b") )
    {
      if( codegen_type != CODEGEN_UNKNOWN )
      {
	fprintf(stderr, "%s: -b option given twice\n", argv[0]);
	exit(1);
      }
      if( AStringEqual(argv[i], "-bc") )
	codegen_type = CODEGEN_C;
      else
      {
	fprintf(stderr, "%s: unknown backend %s\n", argv[0], argv[i]);
	exit(1);
      }
    }

    /* -i option: enter interactive phase after compilation */
    else if( AStringEqual(argv[i], "-i") )
    {
      if( interactive )
      {
	fprintf(stderr, "%s: -i option given twice\n", argv[0]);
	exit(1);
      }
      interactive = TRUE;
    }

    /* -m option: prints various sizeofs and exits */
    else if( AStringEqual(argv[i], "-s") )
    {
      fprintf(stderr, "sizeof(char) = %d\n", sizeof(char));
      fprintf(stderr, "sizeof(short) = %d\n", sizeof(short));
      fprintf(stderr, "sizeof(int) = %d\n", sizeof(int));
      fprintf(stderr, "sizeof(void *) = %d\n", sizeof(void *));
      fprintf(stderr, "sizeof(long unsigned int) = %d\n",
	sizeof(long unsigned int));
      exit(0);
    }

    /* -p option: calculates first prime after a given number then exits */
    else if( AStringEqual(argv[i], "-p") )
    {
      int from;
      if( i == argc-1 || sscanf(argv[i+1], "%d", &from) != 1 || from < 1 )
	from = 1;
      fprintf(stderr, "first prime after %d is %d\n", from, PrimeAfter(from));
      exit(0);
    }

    /* -s option: full name of system directory */
    else if( AStringEqual(argv[i], "-s") )
    {
      if( sys_dir != NULL )
      {
	fprintf(stderr, "%s: -s option given twice\n", argv[0]);
	exit(1);
      }
      sys_dir = &argv[i][2];
      if( sys_dir[0] != NPC_DIR_SEP[0] )
      {
	fprintf(stderr, "%s: value of -d option does not begin with %s\n",
	  argv[0], NPC_DIR_SEP);
	exit(1);
      }
    }

    /* -u option: version and usage information print */
    else if( AStringEqual(argv[i], "-u") )
    {
      UsageMessage(stderr);
      MainCopyrightNotice(stderr);
      exit(0);
    }

    /* else unknown command line argument */
    else
    {
      fprintf(stderr, "%s: unknown command line argument \"%s\"\n",
	argv[0], argv[i]);
      exit(1);
    }
  }

  /* default sys directory is PWD */
  if( sys_dir == NULL )
  {
    sys_dir = getenv("PWD");
    if( sys_dir == NULL )
    {
      fprintf(stderr, "%s: no PWD variable in env, please use -s flag\n",
	argv[0]);
      exit(1);
    }
  }

  /* default code generation type is C */
  if( codegen_type == CODEGEN_UNKNOWN )
    codegen_type = CODEGEN_C;

  /* create the appropriate codegen object for codegen_type */
  be = InitCodeGen();

  /* load the system, either interactively or not */
  if( interactive )
  {
    MainCopyrightNotice(stderr);
    UCharInit(AStringToUString(NPC_LIB_DIR));
    LoadSystem(be, sys_dir, &sys, &curr_sv);
    if( curr_sv != NULL )
      curr_cxt = ContextNew(SystemViewClassViews(curr_sv));
    else
      curr_cxt = NULL;
    Interact(sys, curr_sv, NULL, NULL, NULL, curr_cxt, stdin, stdout,
      sys_dir, be);
  }
  else
  {
    UCharInit(AStringToUString(NPC_LIB_DIR));
    LoadSystem(be, sys_dir, &sys, &curr_sv);
  }

  /* quit */
  exit(0);
  return 0;
}
