/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         name.c                                                     */
/*  DESCRIPTION:  Nonpareil name definitions (implementation)                */
/*                                                                           */
/*****************************************************************************/
#include <stdlib.h>
#include <string.h>
#include "externs.h"


/*****************************************************************************/
/*                                                                           */
/*  NAME                                                                     */
/*                                                                           */
/*  A name is the name of something, holding: the name itself (stored        */
/*  twice, in fields key and rep); whether it is an operator and if so       */
/*  what kind and precedence; and optionally a previous named object that    */
/*  was renamed with this name.                                              */
/*                                                                           */
/*  Usually, key and rep are the same, but in the case of a prefix           */
/*  operator which is a punctuation sequence, key is rep with an             */
/*  underscore prepended.  We always store such prefix operators in          */
/*  symbol tables with the underscore, to ensure they don't clash with       */
/*  postfix or infix operators with the same name.                           */
/*                                                                           */
/*  Storing a previous named object allows a name to be printed in           */
/*  the full format                                                          */
/*                                                                           */
/*      <name> (from <pos>, <pos> ... )                                      */
/*                                                                           */
/*  Or if there is renaming along the way:                                   */
/*                                                                           */
/*      <name> (from <name> at <pos>, ... )                                  */
/*                                                                           */
/*  This allows for a clear record of how a name came to be in a module.     */
/*                                                                           */
/*****************************************************************************/

typedef enum {
  ORDINARY,
  PREFIX,
  INFIX,
  INFIXR,
  POSTFIX 
} NAME_KIND;

struct name_rec {
  USTRING	key;		/* key under which this name is stored       */
  USTRING	rep;		/* representation of name in source files    */
  NAME_KIND	kind;		/* the kind of name (see NAME_KIND above)    */
  int		precedence;	/* its precedence if kind is INFIX or INFIXR */
  NAMED		previous;	/* the entity this is a rename of, if any    */
};

struct named_rec {
  KIND_TAG	kind_tag;	/* type tag (must be TYPE_NAMED)             */
  FILE_POS	file_pos;	/* position of entity name in input          */
  NAME		name;		/* name of entity                            */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule "construction and query".                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  NAME NameMake(USTRING rep, USTRING key, NAME_KIND kind,                  */
/*    int precedence, NAMED previous)                                        */
/*                                                                           */
/*  Return a new name with these attribures.  Since the key field differs    */
/*  from the rep field only in optionally having a preceding underscore,     */
/*  we don't pass the key field, just a boolean which is TRUE when we        */
/*  want to construct this different key.                                    */
/*                                                                           */
/*****************************************************************************/

static NAME NameMake(USTRING rep, USTRING key, NAME_KIND kind,
  int precedence, NAMED previous)
{
  NAME res;
  GetMemory(res, NAME);
  res->key = key;
  res->rep = rep;
  res->kind = kind;
  res->precedence = precedence;
  res->previous = previous;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  NAME NameNew(USTRING rep)                                                */
/*                                                                           */
/*  Return a new ordinary (non-operator) name with these attributes.         */
/*                                                                           */
/*****************************************************************************/

NAME NameNew(USTRING rep)
{
  return NameMake(rep, rep, ORDINARY, 0, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  NAME NameNewInfix(USTRING rep, int precedence)                           */
/*                                                                           */
/*  Make a new infix name with this rep.                                     */
/*                                                                           */
/*****************************************************************************/

NAME NameNewInfix(USTRING rep, int precedence)
{
  return NameMake(rep, rep, INFIX, precedence, NULL);
}


/*****************************************************************************/
/*                                                                           */
/*  NAME NameNewFromPrev(NAME value, NAMED prev)                             */
/*                                                                           */
/*  Make a new name which contains the representation etc. from value,       */
/*  with prev for its previous named object.                                 */
/*                                                                           */
/*****************************************************************************/

NAME NameNewFromPrev(NAME value, NAMED prev)
{
  return NameMake(value->rep, value->key, value->kind,value->precedence,prev);
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING NameRep(NAME name)                                               */
/*                                                                           */
/*  Return the representation of name.                                       */
/*                                                                           */
/*****************************************************************************/

USTRING NameRep(NAME name)
{
  return name->rep;
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING NameKey(NAME name)                                               */
/*                                                                           */
/*  Return the key of this name.                                             */
/*                                                                           */
/*****************************************************************************/

USTRING NameKey(NAME name)
{
  return name->key;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NameIsOperator(NAME name)                                        */
/*  BOOLEAN NameIsPrefixOperator(NAME name)                                  */
/*  BOOLEAN NameIsInfixOperator(NAME name)                                   */
/*  BOOLEAN NameIsPostfixOperator(NAME name)                                 */
/*                                                                           */
/*  TRUE if this name is an operator, or an operator of a particular type.   */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NameIsOperator(NAME name)
{
  NAME_KIND k = name->kind;
  return k == PREFIX || k == POSTFIX || k == INFIX || k == INFIXR;
}

BOOLEAN NameIsPrefixOperator(NAME name)
{
  return name->kind == PREFIX;
}

BOOLEAN NameIsInfixOperator(NAME name)
{
  return name->kind == INFIX || name->kind == INFIXR;
}

BOOLEAN NameIsPostfixOperator(NAME name)
{
  return name->kind == POSTFIX;
}


/*****************************************************************************/
/*                                                                           */
/*  int NamePrecedence(NAME name)                                            */
/*                                                                           */
/*  Return the precedence of this name, without checking anything.           */
/*                                                                           */
/*****************************************************************************/

int NamePrecedence(NAME name)
{
  return name->precedence;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "parsing".                                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NameParse(TOKEN *t, TOKEN *token)                                */
/*                                                                           */
/*  Parse a name (identifier or punctseq), and set *token to the name token. */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NameParse(TOKEN *t, TOKEN *token)
{
  *token = curr_token;
  if( LexType(curr_token) == TK_IDENTIFIER )
    next_token;
  else
    skip(TK_PUNCTSEQ, "name (identifier or punctuation sequence)");
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NameDefParse(*t, SYMTAB_NAMED op_table, NAME *res)               */
/*                                                                           */
/*  Parse a name definition, and return it in *res.                          */
/*                                                                           */
/*  Usually, the name stored in *res, (*res)->key, is equal                  */
/*  to the token string found, (*res)->token->value.  However, if            */
/*  this is a name definition of a prefix operator whose value is a          */
/*  punctuation sequence, (*res)->key has an underscore character            */
/*  prepended to it, to allow it to share symbol tables with infix and       */
/*  postfix operators with the same name, without clashing.                  */
/*                                                                           */
/*  NameDefParse has the unique attribute of being called during the         */
/*  lexical analysis as well as during the parse.  The lexical analyser      */
/*  uses it to construct name definitions of all the operators it finds,     */
/*  for insertion into the public operator table which is later used by      */
/*  the parser stage proper to determine whether the names it finds are      */
/*  operators, and if so their types and precedences.                        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NameDefParse(TOKEN *t, SYMTAB_NAMED op_table, NAME *res)
{
  TOKEN name_token;  NAMED named;  NAME_KIND kind;
  int precedence;  USTRING key;

  /* find out what type of name def this is, and the precedence if infix */
  precedence = -1;  /* temporary, see below */
  switch( LexType(curr_token) )
  {
    case TK_PREFIX:

      /* prefix operator */
      kind = PREFIX;
      next_token;
      break;


    case TK_POSTFIX:

      /* postfix operator */
      kind = POSTFIX;
      next_token;
      break;


    case TK_INFIX:
    case TK_INFIXR:

      /* infix operator */
      kind = (LexType(curr_token) == TK_INFIX ? INFIX : INFIXR);
      next_token;
      if( LexType(curr_token) == TK_LIT_INTEGER )
      {
        precedence = UStringToInteger(LexValue(curr_token));
	if( precedence < MIN_INFIX_PRECEDENCE )
	{
	  fprintf(stderr, "%s: infix precedence too small (min is %d)\n",
	    LexPos(curr_token), MIN_INFIX_PRECEDENCE);
	  return FALSE;
	}
	if( precedence > MAX_INFIX_PRECEDENCE )
	{
	  fprintf(stderr, "%s: infix precedence too large (max is %d)\n",
	    LexPos(curr_token), MAX_INFIX_PRECEDENCE);
	  return FALSE;
	}
	if( kind == INFIX && precedence % 2 != 0 )
	{
	  fprintf(stderr, "%s: infix precedence must be an even number\n",
	    LexPos(curr_token));
	  return FALSE;
	}
	if( kind == INFIXR && precedence % 2 != 1 )
	{
	  fprintf(stderr, "%s: infixr precedence must be an odd number\n",
	    LexPos(curr_token));
	  return FALSE;
	}
        next_token;
      }
      break;


    case TK_IDENTIFIER:

      /* ordinary identifier */
      kind = ORDINARY;
      break;


    default:

      /* this skip will fail */
      skip(TK_IDENTIFIER,
        "start of name definition (identifier or operator keyword)\n");
      break;

  }

  /* get the name and key (which may need an underscore prefixed to it */
  if( !NameParse(t, &name_token) )
    return FALSE;
  key = (kind == PREFIX && LexType(name_token) == TK_PUNCTSEQ ?
    UStringCat(AStringToUString("_"), LexValue(name_token)) :
    LexValue(name_token));
  *res = NameMake(LexValue(name_token), key, kind, precedence, NULL);

  /* sort out any missing precedence by reference to op_table */
  if( kind == INFIX || kind == INFIXR )
  {
    if( precedence == -1 )
    {
      if( !SymRetrieve(op_table, (*res)->key, &named) ||
	(named->name->kind != INFIX && named->name->kind != INFIXR) )
      {
	fprintf(stderr,
	  "%s: no default precedence available for %s operator \"%s\"\n",
	  LexPos(name_token), NameKindShow(*res), LexShow(name_token));
	return FALSE;
      }
      else if( kind != named->name->kind )
      {
	fprintf(stderr,
	  "%s: %s operator \"%s\" previously defined as %s at %s\n",
	  LexPos(name_token), NameKindShow(*res), LexShow(name_token),
	  NameKindShow(named->name), FilePosShow(named->file_pos));
	return FALSE;
      }
      (*res)->precedence = named->name->precedence;
    }
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NameDefListParse(TOKEN *t, SYMTAB_NAMED op_table,                */
/*    ARRAY_NAME *names)                                                     */
/*                                                                           */
/*  Parse a sequence of name definitions (class names, feature names, or     */
/*  parameter names) and return them in a new array, *names.                 */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NameDefListParse(TOKEN *t, SYMTAB_NAMED op_table, ARRAY_NAME *names)
{
  NAME name;

  /* initialize result array */
  ArrayInit(names);

  /* parse and store first name definition */
  if( !NameDefParse(t, op_table, &name) )
    return FALSE;
  ArrayAddLast(*names, name);

  /* parse other name definitions */
  while( LexType(curr_token) == TK_COMMA )
  {
    next_token;
    if( !NameDefParse(t, op_table, &name) )
      return FALSE;
    ArrayAddLast(*names, name);
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "miscellaneous tests".                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NameEqual(NAME n1, NAME n2)                                      */
/*                                                                           */
/*  Check that names n1 and n2 are equal in rep, kind, and precedence.       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NameEqual(NAME n1, NAME n2)
{
  if( !UStringEqual(n1->rep, n2->rep) )
    return FALSE;
  if( n1->kind != n2->kind )
    return FALSE;
  if( (n1->kind == INFIX || n1->kind == INFIXR) &&
      n1->precedence != n2->precedence )
    return FALSE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NameKindEqual(NAME n1, NAME n2)                                  */
/*                                                                           */
/*  Check that names n1 and n2 are equal in kind.                            */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NameKindEqual(NAME n1, NAME n2)
{
  return n1->kind == n2->kind;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NameCompatible(NAME n1, FILE_POS p1, NAME n2, FILE_POS p2,       */
/*    ASTRING str)                                                           */
/*                                                                           */
/*  Return TRUE if names are compatible, i.e. they are already known to      */
/*  have the same key, and the question is whether they have the same kind   */
/*  (and precedence if infix).                                               */
/*                                                                           */
/*  If they are not compatible, print an error message including str and     */
/*  return FALSE.                                                            */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NameCompatible(NAME n1, FILE_POS p1, NAME n2, FILE_POS p2, ASTRING str)
{
  if( !NameKindEqual(n1, n2) )
  {
    /* conflicting name kinds */
    fprintf(stderr,"conflicting name kinds in %s:\n", str);
    fprintf(stderr, "  %s: %s\n", FilePosShow(p1), NameFullShow(n1));
    fprintf(stderr, "  %s: %s\n", FilePosShow(p2), NameFullShow(n2));
    return FALSE;
  }
  else if( NameIsInfixOperator(n2) && NamePrecedence(n2) != NamePrecedence(n1) )
  {
    /* conflicting precedences */
    fprintf(stderr, "conflicting operator precedences in %s:\n", str);
    fprintf(stderr, "  %s: %s\n", FilePosShow(p1), NameFullShow(n1));
    fprintf(stderr, "  %s: %s\n", FilePosShow(p2), NameFullShow(n2));
    return FALSE;
  }
  else
  {
    /* same kind and precedence, so compatible */
    return TRUE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NameConsistentWithParameters(NAME name, int param_count,         */
/*    FILE_POS err_pos)                                                      */
/*                                                                           */
/*  Check whether name is consistent with the number of parameters (1 for    */
/*  prefix and postfix, 2 for infix, don't care for ordinary).  Return       */
/*  TRUE if OK, or print an error message and return FALSE if not.           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NameConsistentWithParameters(NAME name, int param_count,
  FILE_POS err_pos)
{
  switch( name->kind )
  {
    case PREFIX:
    case POSTFIX:

      if( param_count != 1 )
      {
	fprintf(stderr, "%s: \"%s\" not suitable name when %d parameters\n",
	  FilePosShow(err_pos), NameShow(name), param_count);
	return FALSE;
      }
      break;


    case INFIX:
    case INFIXR:

      if( param_count != 2 )
      {
	fprintf(stderr, "%s: \"%s\" not suitable name when %d parameters\n",
	  FilePosShow(err_pos), NameShow(name), param_count);
	return FALSE;
      }
      break;


    case ORDINARY:

      /* don't care */
      break;


    default:

      assert(FALSE);
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NameOperatorConsistent(NAME name, FILE_POS pos,                  */
/*    SYMTAB_NAMED op_table, ASTRING str)                                    */
/*                                                                           */
/*  Check that name is not incompatible with any operator in op_table, and   */
/*  add it to op_table if it is itself an operator.                          */
/*                                                                           */
/*  Print an error message and return FALSE if there is an incompatibility.  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NameOperatorConsistent(NAME name, FILE_POS pos,
  SYMTAB_NAMED op_table, ASTRING str)
{
  NAMED n2;
  if( SymRetrieve(op_table, NameKey(name), &n2) )
  {
    /* name is present; check that it is compatible */
    return NameCompatible(name, pos, n2->name, n2->file_pos, str);
  }
  else
  {
    /* name not present; insert it if it is an operator */
    if( NameIsOperator(name) )
      SymInsert(op_table, NameKey(name), NamedNew(pos, name), &n2);
    return TRUE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NameIsInfix(USTRING str, SYMTAB_NAMED op_table, NAME *name)      */
/*                                                                           */
/*  Return TRUE if str is an infix operator in op_table, or else FALSE.      */
/*  If it is, also set *name to the name.                                    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NameIsInfix(USTRING str, SYMTAB_NAMED op_table, NAME *name)
{
  NAMED named;
  if( SymRetrieve(op_table, str, &named) && named->name->kind == INFIX )
  {
    *name = named->name;
    return TRUE;
  }
  else
  {
    *name = NULL;
    return FALSE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "display and debug".                                           */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void NameAdd(NAME name, AFACTORY af)                                     */
/*                                                                           */
/*  Add to AString factory af an ASCII string representation of this name    */
/*  as it appears in source files, with "prefix" etc.  Not efficient, but    */
/*  used mostly for error messages.                                          */
/*                                                                           */
/*****************************************************************************/

void NameAdd(NAME name, AFACTORY af)
{
  char buff[50];
  if( name == NULL )
    AStringAddAString(af, "[noname]");
  else switch( name->kind )
  {
    case PREFIX:
    case POSTFIX:

      AStringAddFmt2(af, "%s %s", NameKindShow(name),
	(ASTRING) UStringToUTF8(name->rep));
      break;


    case INFIX:
    case INFIXR:

      sprintf(buff, "%s %d %s", NameKindShow(name), name->precedence,
	(ASTRING) UStringToUTF8(name->rep));
      AStringAddAString(af, buff);
      break;


    case ORDINARY:

      AStringAddAString(af, (ASTRING) UStringToUTF8(name->rep));
      break;


    default:

      assert(FALSE);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  UTF8 NameShow(NAME name)                                                 */
/*                                                                           */
/*  Return a UTF8 string representation of this name as it appears in        */
/*  source files, with "prefix" etc.  Not efficient, but used mostly only    */
/*  for error messages.                                                      */
/*                                                                           */
/*****************************************************************************/

UTF8 NameShow(NAME name)
{
  AFACTORY af = AStringBegin();
  NameAdd(name, af);
  return (UTF8) AStringEnd(af);
}


/*****************************************************************************/
/*                                                                           */
/*  void NameFullAdd(NAME n, AFACTORY af)                                    */
/*                                                                           */
/*  Add to af a string containing the fully shown name, e.g.                 */
/*                                                                           */
/*     layout/module:10:5: "prefix -" (from lang/int:5:8)                    */
/*                                                                           */
/*****************************************************************************/

static void NameFullAdd(NAME n, AFACTORY af)
{
  NAMED nc;  NAME prev_name;

  /* start off with current name */
  AStringAddChar(af, '"');
  NameAdd(n, af);
  AStringAddChar(af, '"');

  /* carry on with the rest, if any */
  if( n->previous != NULL )
  {
    AStringAddAString(af, " (");
    prev_name = n;
    for( nc = n->previous;  nc != NULL;  nc = nc->name->previous )
    {
      /* print comma if not first */
      if( nc != n->previous ) AStringAddAString(af, ", ");

      AStringAddAString(af, "from ");

      /* print prior name if different */
      if( nc->name->rep != prev_name->rep )
      {
	AStringAddChar(af, '"');
	NameAdd(nc->name, af);
	AStringAddAString(af, "\" at ");
      }

      /* print position */
      LexAddPos(nc->file_pos, af);
      prev_name = nc->name;
    }
    AStringAddChar(af, ')');
  }
}


/*****************************************************************************/
/*                                                                           */
/*  UTF8 NameFullShow(NAME n)                                                */
/*                                                                           */
/*  Return a static string containing the fully shown name, e.g.             */
/*                                                                           */
/*     layout/module:10:5: "prefix -" (from lang/int:5:8)                    */
/*                                                                           */
/*****************************************************************************/

UTF8 NameFullShow(NAME n)
{
  AFACTORY af = AStringBegin();
  NameFullAdd(n, af);
  return (UTF8) AStringEnd(af);
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING NameKindShow(NAME name)                                          */
/*                                                                           */
/*  Return the string corresponding to the kind of this name, either         */
/*  "prefix", "postfix", "infix", or "ordinary".                             */
/*                                                                           */
/*****************************************************************************/

ASTRING NameKindShow(NAME name)
{
  NAME_KIND k = name->kind;
  return k == PREFIX ? "prefix" : k == POSTFIX  ? "postfix"  :
	 k == INFIX  ? "infix"  : k == INFIXR ? "infixr" :
	 k == ORDINARY ? "ordinary" : "<illegal!>";
}


/*****************************************************************************/
/*                                                                           */
/*  void NameDefListDebug(ARRAY_NAME names, FILE *fp)                        */
/*                                                                           */
/*  Debug name definition list onto fp.                                      */
/*                                                                           */
/*****************************************************************************/

void NameDefListDebug(ARRAY_NAME names, FILE *fp)
{
  NAME name;
  ArrayForEach(names, name)
  {
    if( name != ArrayFirst(names) )
      fprintf(fp, ", ");
    fprintf(fp, "%s", NameShow(name));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "named entities".                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  NAMED NamedNew(FILE_POS file_pos, NAME name)                             */
/*                                                                           */
/*  Return a new NAMED object with these attributes.                         */
/*                                                                           */
/*****************************************************************************/

NAMED NamedNew(FILE_POS file_pos, NAME name)
{
  NAMED res;
  GetMemory(res, NAMED);
  res->kind_tag = KIND_NAMED;
  res->file_pos = file_pos;
  res->name = name;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  KIND_TAG NamedKind(NAMED named)                                          */
/*                                                                           */
/*  Return the kind tag of this named entity.                                */
/*                                                                           */
/*****************************************************************************/

KIND_TAG NamedKind(NAMED named)
{
  return named->kind_tag;
}


/*****************************************************************************/
/*                                                                           */
/*  FILE_POS NamedFilePos(NAMED named)                                       */
/*                                                                           */
/*  Return the file position of named entity named.                          */
/*                                                                           */
/*****************************************************************************/

FILE_POS NamedFilePos(NAMED named)
{
  return named->file_pos;
}


/*****************************************************************************/
/*                                                                           */
/*  NAME NamedName(NAMED named)                                              */
/*                                                                           */
/*  Return the name of named entity "named".                                 */
/*                                                                           */
/*****************************************************************************/

NAME NamedName(NAMED named)
{
  return named->name;
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING KindShow(KIND_TAG kind_tag)                                      */
/*                                                                           */
/*  Return a string showing the type of this named object.                   */
/*                                                                           */
/*****************************************************************************/

ASTRING KindShow(KIND_TAG kind_tag)
{
  switch( kind_tag )
  {
    case KIND_LOCATED:			return "located entity";
      case KIND_TYPEV:			return "type (var invocation)";
      case KIND_TYPEC:			return "type (class or meet)";
      case KIND_CTYPE:			return "ctype";
      case KIND_EXPR:			return "expression";
        case KIND_EXPR_LIT_BOOLEAN:	return "boolean literal expression";
        case KIND_EXPR_LIT_CHAR:	return "character literal expression";
        case KIND_EXPR_LIT_STRING:	return "string literal expression";
        case KIND_EXPR_LIT_INTEGER:	return "integer literal expression";
        case KIND_EXPR_LIT_REAL:	return "real num literal expression";
        case KIND_EXPR_CALL:		return "call expression";
        case KIND_EXPR_PAREN:		return "parenthesized expression";
        case KIND_EXPR_TUPLE:		return "manifest tuple expression";
        case KIND_EXPR_LIST:		return "manifest list expression";
        case KIND_EXPR_ARRAY:		return "manifest array expression";
        case KIND_EXPR_LET:		return "let expression";
        case KIND_EXPR_FUN:		return "anonymous function expression";
        case KIND_EXPR_IF:		return "if expression";
        case KIND_EXPR_CASE:		return "case expression";
        case KIND_EXPR_FNHEAD:		return "fnhead expression";
        case KIND_EXPR_PRECOND:		return "precondition expression";
        case KIND_EXPR_DEFAULT:		return "default expression";
      case KIND_NAMED:			return "named entity";
        case KIND_CLASS_VIEW:		return "class view";
        case KIND_FEFN_FEATURE_SET:	return "feature view set";
        case KIND_TYPE_VAR:		return "type variable";
	/* front-end function types */
          case KIND_FEFN_CREATION:	return "front-end creation function";
          case KIND_FEFN_CREDFT:	return "front-end creation feature dft";
          case KIND_FEFN_FEATURE:	return "front-end feature";
          case KIND_FEFN_PRECOND:	return "front-end precondition";
          case KIND_FEFN_BUILTIN:	return "front-end builtin function";
          case KIND_FEFN_LETDEF:	return "front-end let def";
          case KIND_FEFN_DOWNDEF:	return "front-end downcast variable";
	  case KIND_FEFN_PARAM:		return "front-end parameter";
          case KIND_FEFN_INVARIANT:	return "front-end invariant";
	/* back-end function types */
          case KIND_BEFN_CREATION:	return "back-end creation function";
          case KIND_BEFN_CREDFT:        return "back-end creation feature dft";
          case KIND_BEFN_FEATURE:	return "back-end feature";
          case KIND_BEFN_PRECOND:	return "back-end precondition";
          case KIND_BEFN_BUILTIN:	return "back-end builtin function";
          case KIND_BEFN_LETDEF:	return "back-end let def";
          case KIND_BEFN_DOWNDEF:	return "back-end downcast variable";
	  case KIND_BEFN_PARAM:		return "back-end function parameter";
          case KIND_BEFN_INVARIANT:	return "back-end function invariant";
          case KIND_BEFN_CLASS_INIT:	return "back-end class init function";
          case KIND_BEFN_ENUM_INIT:	return "back-end enum_init function";
          case KIND_BEFN_SYSTEM_INIT:	return "back-end system init function";
  }
  return "??";  /* keep compiler happy */
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN NamedIsLocal(NAMED named)                                        */
/*                                                                           */
/*  Return TRUE if named is a local definition (a letdef, downcast           */
/*  variable, or parameter).                                                 */
/*                                                                           */
/*****************************************************************************/

BOOLEAN NamedIsLocal(NAMED named)
{
  return named->kind_tag >= KIND_FEFN_LETDEF &&
    named->kind_tag <= KIND_FEFN_PARAM;
}
