/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         system.c                                                   */
/*  DESCRIPTION:  A system.                                                  */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#define DEBUG1	0
#define DEBUG2	0
#define DEBUG3	0
#define DEBUG4	0
#define DEBUG5  0
#define DEBUG6  0


/*****************************************************************************/
/*                                                                           */
/*  SYSTEM - a system                                                        */
/*                                                                           */
/*****************************************************************************/

struct system_rec {
  NAME				name;		/* "system"                  */
  SYSTEM_VIEW			view;		/* system's view of system   */
  						/*   (includes *all* views)  */
  ASTRING			data_dir;	/* system/data directory     */
  ARRAY_CLASS			classes;	/* all classes of system     */
  ARRAY_CLASS			enum_classes;	/* enum classes of system    */
  BEFN_SYSTEM_INIT		init_fn;	/* "system_init" (predefs)   */
  CODEGEN_EXEC			be_init_exec;	/* initializing executable   */
  CODEGEN_EXEC			be_load_exec;	/* loading executable        */
  CODEGEN_FILE			be_init_file;	/* "system_init"             */
  CODEGEN_FILE			be_load_file;	/* "system_load"             */
  CODEGEN_FILE			be_file;	/* "system"                  */
  CODEGEN_FILE			typedefs_file;	/* "typedefs.h"              */
  CODEGEN_FILE			npsys_typedefs_file; /* "npsys_typedefs.h"   */
  CODEGEN_FILE			npsys_init_file;/* "npsys_init.h"            */
  CODEGEN_FILE			npsys_load_file;/* "npsys_load.h"            */
  CODEGEN_FILE			npsys_main_file;/* "npsys.h"                 */
  CODEGEN_OBJ			load_be_obj;	/* "system_load"             */
  CODEGEN_OBJ			version_be_obj;	/* "system_version"          */
  CODEGEN_OBJ			swizzle_be_obj;	/* "system_swizzle"          */
  CODEGEN_OBJ			unswizzle_be_obj;/* "system_unswizzle"       */
};


/*****************************************************************************/
/*                                                                           */
/*  SYSTEM SystemNew(TOKEN token)                                            */
/*                                                                           */
/*  Return a new, empty SYSTEM object.                                       */
/*                                                                           */
/*****************************************************************************/

static SYSTEM SystemNew(TOKEN token, ASTRING data_dir)
{
  SYSTEM res;
  GetMemory(res, SYSTEM);
  res->name = NameNew(AStringToUString("system"));
  res->view = SystemViewNew(LexFilePos(token), res->name, res);
  res->data_dir = data_dir;
  ArrayInit(&res->classes);
  ArrayInit(&res->enum_classes);
  res->init_fn = BEFnSystemInitMake(res);
  res->be_init_file = NULL;
  res->be_load_file = NULL;
  res->be_file = NULL;
  res->typedefs_file = NULL;
  res->load_be_obj = NULL;
  res->version_be_obj = NULL;
  res->swizzle_be_obj = NULL;
  res->unswizzle_be_obj = NULL;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemRegisterClass(SYSTEM sys, CLASS c)                            */
/*                                                                           */
/*  Register c with sys.                                                     */
/*                                                                           */
/*****************************************************************************/

void SystemRegisterClass(SYSTEM sys, CLASS c)
{
  ArrayAddLast(sys->classes, c);
  if( ClassIsEnum(c) )
    ArrayAddLast(sys->enum_classes, c);
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_CLASS SystemClasses(SYSTEM sys)                                    */
/*                                                                           */
/*  Return the classes of sys.                                               */
/*                                                                           */
/*****************************************************************************/

ARRAY_CLASS SystemClasses(SYSTEM sys)
{
  return sys->classes;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemRetrieveView(SYSTEM sys, USTRING str, SYSTEM_VIEW *sv)     */
/*                                                                           */
/*  If a view with the given name is known to sys, set *sv to it and         */
/*  return TRUE.  Otherwise return FALSE.                                    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemRetrieveView(SYSTEM sys, USTRING str, SYSTEM_VIEW *sv)
{
  return SystemViewRetrieveSystemView(sys->view, str, sv);
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemCodeGenModuleFunctionRange(ARRAY_SYSTEM_VIEW concrete_views,  */
/*    int from, int to, ARRAY_CODEGEN_OBJ params,                            */
/*    CODEGEN_OBJ (*name_in_sv)(SYSTEM_VIEW sv), CODEGEN be)                 */
/*                                                                           */
/*  Generate that part of the body of SystemCodeGenModuleFunction (see       */
/*  below) that handles modules concrete_views[from..to].                    */
/*                                                                           */
/*****************************************************************************/

static void SystemCodeGenModuleFunctionRange(ARRAY_SYSTEM_VIEW concrete_views,
  int from, int to, ARRAY_CODEGEN_OBJ params,
  CODEGEN_OBJ (*name_in_sv)(SYSTEM_VIEW sv), CODEGEN be)
{
  SYSTEM_VIEW sv;  CODEGEN_OBJ param;  int mid, i;  CLASS c;
  assert(from <= to);
  if( from == to )
  {
    /* dispatch to a single module */
    be->IndentBegin();
    sv = ArrayGet(concrete_views, from);
    be->VarAsstBegin(be->ret);
    be->CallBegin(name_in_sv(sv));
    for( i = 0;  i < ArraySize(params);  i++ )
    {
      param = ArrayGet(params, i);
      if( i > 0 )
	be->CallContinue(name_in_sv(sv), i);
      Var(param);
    }
    be->CallEnd(name_in_sv(sv));
    be->AsstEnd();
    be->IndentEnd();
  }
  else
  {
    /* dispatch to two or more modules */
    mid = (from + to) / 2;
    sv = ArrayGet(concrete_views, mid);
    SystemViewLastClassWithTypeTag(sv, &c);
    IfElse( Call2(be->le, Call1(NPBack_Type_Tag, Var(ArrayFirst(params))),
	Var(ClassTypeTag(c))),
      SystemCodeGenModuleFunctionRange(concrete_views, from, mid,
	params, name_in_sv, be),
      SystemCodeGenModuleFunctionRange(concrete_views, mid+1, to,
	params, name_in_sv, be)
    );
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemCodeGenModuleFunction(SYSTEM sys, CODEGEN_TYPE res_be_type,   */
/*    CODEGEN_OBJ be_obj, ARRAY_CODEGEN_OBJ params, ARRAY_BE_TYPE p_types,   */
/*    CODEGEN_OBJ (*name_in_sv)(SYSTEM_VIEW sv), CODEGEN be)                 */
/*                                                                           */
/*  Generate on be a module function with the given name and parameters      */
/*  (all of type object), which works by dispatching on the first parameter  */
/*  to the various modules, calling name_in_sv(sv)(params) for classes       */
/*  lying in that module.                                                    */
/*                                                                           */
/*  This code dispatches using nested if statements, so that it will work    */
/*  on extremely large dispatches that have a limb for every class in the    */
/*  system (currently swizzle and unswizzle).  The code within individual    */
/*  modules (not generated here) dispatches by case statement.               */
/*                                                                           */
/*****************************************************************************/

static void SystemCodeGenModuleFunction(SYSTEM sys, CODEGEN_TYPE res_be_type,
  CODEGEN_OBJ be_obj, ARRAY_CODEGEN_OBJ params, ARRAY_CODEGEN_TYPE param_types,
  CODEGEN_OBJ (*name_in_sv)(SYSTEM_VIEW sv), CODEGEN be)
{
  CODEGEN_OBJ param;  CODEGEN_TYPE param_type;
  ARRAY_SYSTEM_VIEW all_views;  SYSTEM_VIEW sv;  CLASS c;  int i;
  static ARRAY_SYSTEM_VIEW concrete_views = NULL;
  assert(params != NULL && ArraySize(params) >= 1);

  /* only want system views containing at least one class with a type tag */
  if( concrete_views == NULL )
  {
    all_views = SystemViewSystemViews(sys->view);
    ArrayInit(&concrete_views);
    ArrayForEach(all_views, sv)
      if( SystemViewLastClassWithTypeTag(sv, &c) )
	ArrayAddLast(concrete_views, sv);
  }

  /* codegen the function */
  be->FunctionBegin(be_obj, res_be_type);
  for( i = 0;  i < ArraySize(params);  i++ )
  {
    param = ArrayGet(params, i);
    param_type = ArrayGet(param_types, i);
    be->FunctionFormal(be_obj, param, param_type);
  }
  be->FunctionContinue(be_obj);
  if( ArraySize(concrete_views) >= 1 )
    SystemCodeGenModuleFunctionRange(concrete_views, 0,
      ArraySize(concrete_views) - 1, params, name_in_sv, be);
  be->FunctionEnd(be_obj);
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemModuleInitFileHeaderIncludes(SYSTEM sys, CODEGEN be)          */
/*                                                                           */
/*  Generate on be all the include files that go into the header (.h) files  */
/*  of the _init files of modules.                                           */
/*                                                                           */
/*****************************************************************************/

void SystemModuleInitFileHeaderIncludes(SYSTEM sys, CODEGEN be)
{
  be->HeaderFileInclude(sys->typedefs_file);
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemModuleInitFileCodeIncludes(SYSTEM sys, CODEGEN be)            */
/*                                                                           */
/*  Generate on be all the include files that go into the code (.c) files    */
/*  of the _init files of modules.                                           */
/*                                                                           */
/*****************************************************************************/

void SystemModuleInitFileCodeIncludes(SYSTEM sys, CODEGEN be)
{
  ARRAY_SYSTEM_VIEW all_views;  SYSTEM_VIEW sv;  int i;

  /* the usual system include files */
  be->CodeFileInclude(NPBack_File_Stdio);
  be->CodeFileInclude(NPBack_File_Limits);
  be->CodeFileInclude(NPBack_File_Float);
  be->CodeFileInclude(NPBack_File_Assert);

  /* *** still wondering about these ones
  be->CodeFileInclude(NPBack_File_Stdlib);
  be->CodeFileInclude(NPBack_File_String);
  be->CodeFileInclude(NPBack_File_Math);
  be->CodeFileInclude(sys->be_file);
  *** */

  /* nonpareil header files */
  be->CodeFileInclude(sys->npsys_init_file);
  be->CodeFileInclude(sys->npsys_main_file);

  /* needed for child all_predefined objects, plus own header file */
  all_views = SystemViewSystemViews(sys->view);
  for( i = 0;  i < ArraySize(all_views);  i++ )
  {
    /* NB can't use internal cursor here */
    sv = ArrayGet(all_views, i);
    be->CodeFileInclude(SystemViewBEFile(sv));
  }
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemModuleLoadFileHeaderIncludes(SYSTEM sys, CODEGEN be)          */
/*                                                                           */
/*  Generate on be all the include files that go into the header (.h) files  */
/*  of the _load files of modules.                                           */
/*                                                                           */
/*****************************************************************************/

void SystemModuleLoadFileHeaderIncludes(SYSTEM sys, CODEGEN be)
{
  be->HeaderFileInclude(sys->typedefs_file);
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemModuleLoadFileCodeIncludes(SYSTEM sys, CODEGEN be)            */
/*                                                                           */
/*  Generate on be all the include files that go into the code (.c) files    */
/*  of the _load files of modules.                                           */
/*                                                                           */
/*****************************************************************************/

void SystemModuleLoadFileCodeIncludes(SYSTEM sys, CODEGEN be)
{
  /* ***
  ARRAY_SYSTEM_VIEW all_views;
  SYSTEM_VIEW sv;  int i;
  *** */

  /* the usual system include files */
  be->CodeFileInclude(NPBack_File_Stdio);
  be->CodeFileInclude(NPBack_File_Assert);

  /* *** still wondering about these ones
  be->CodeFileInclude(NPBack_File_Stdlib);
  be->CodeFileInclude(NPBack_File_String);
  be->CodeFileInclude(NPBack_File_Limits);
  be->CodeFileInclude(NPBack_File_Float);
  be->CodeFileInclude(NPBack_File_Math);
  *** */

  /* nonpareil header files */
  be->CodeFileInclude(sys->npsys_load_file);
  be->CodeFileInclude(sys->npsys_main_file);

  /* *** still wondering about these ones
  all_views = SystemViewSystemViews(sys->view);
  for( i = 0;  i < ArraySize(all_views);  i++ )
  {
    ** NB can't use internal cursor here **
    sv = ArrayGet(all_views, i);
    be->CodeFileInclude(SystemViewBEFile(sv));
  }
  be->CodeFileInclude(sys->be_file);
  *** */
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemModuleMainFileHeaderIncludes(SYSTEM sys, CODEGEN be)          */
/*                                                                           */
/*  Generate on be all the include files that go into the header (.h) files  */
/*  of the main files of modules.                                            */
/*                                                                           */
/*****************************************************************************/

void SystemModuleMainFileHeaderIncludes(SYSTEM sys, CODEGEN be)
{
  be->HeaderFileInclude(sys->typedefs_file);
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemModuleMainFileCodeIncludes(SYSTEM sys, CODEGEN be)            */
/*                                                                           */
/*  Generate on be all the include files that go into the code (.c) files    */
/*  of the main files of modules.                                            */
/*                                                                           */
/*****************************************************************************/

void SystemModuleMainFileCodeIncludes(SYSTEM sys, CODEGEN be)
{
  ARRAY_SYSTEM_VIEW all_views;
  SYSTEM_VIEW sv;  int i;

  /* the usual system include files */
  be->CodeFileInclude(NPBack_File_Stdlib);
  be->CodeFileInclude(NPBack_File_Stdio);
  be->CodeFileInclude(NPBack_File_String);
  be->CodeFileInclude(NPBack_File_Limits);
  be->CodeFileInclude(NPBack_File_Float);
  be->CodeFileInclude(NPBack_File_Math);
  be->CodeFileInclude(NPBack_File_Assert);

  /* nonpareil header files */
  be->CodeFileInclude(sys->npsys_main_file);
  all_views = SystemViewSystemViews(sys->view);
  for( i = 0;  i < ArraySize(all_views);  i++ )
  {
    /* NB can't use internal cursor here */
    sv = ArrayGet(all_views, i);
    be->CodeFileInclude(SystemViewBEFile(sv));
  }
  /* be->CodeFileInclude(sys->be_file); */
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemCodeGenSysInitFile(SYSTEM sys, CODEGEN be)                 */
/*                                                                           */
/*  Code gen the system_init.h and system_init.c files.                      */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN SystemCodeGenSysInitFile(SYSTEM sys, CODEGEN be)
{
  ARRAY_CODEGEN_OBJ params;  ARRAY_CODEGEN_TYPE param_types;
  ARRAY_SYSTEM_VIEW all_views;  SYSTEM_VIEW sv;  int i;

  /* open system_init.h and system_init.c */
  be->FileBegin(sys->be_init_file, NPC_VERSION, TimingTimeStamp());

  /* start system_init.h with include files */
  SystemModuleInitFileHeaderIncludes(sys, be);

  /* include all _init.h header files into system_init.c */
  all_views = SystemViewSystemViews(sys->view);
  for( i = 0;  i < ArraySize(all_views);  i++ )
  {
    /* NB can't use internal cursor here */
    sv = ArrayGet(all_views, i);
    be->CodeFileInclude(SystemViewBEInitFile(sv));
  }
  be->CodeFileInclude(sys->npsys_main_file);

  /* system_init function */
  BEFnFinalize((BEFN) sys->init_fn, be);
  BEFnCodeGen((BEFN) sys->init_fn, be);

  /* system swizzle function */
  ArrayInit(&params);
  ArrayInit(&param_types);
  ArrayAddLast(params, NPBack_Self);
  ArrayAddLast(param_types, ClassBEType(ClassObject));
  ArrayAddLast(params, NPBack_Other);
  ArrayAddLast(param_types, ClassBEType(ClassObject));
  SystemCodeGenModuleFunction(sys, be->int_type, sys->swizzle_be_obj, params,
    param_types, &SystemViewSwizzleFn, be);

  /* close files */
  be->FileEnd(sys->be_init_file);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemCodeGenSysLoadFile(SYSTEM sys, CODEGEN be)                 */
/*                                                                           */
/*  Code gen the system_load.h and system_load.c files.                      */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN SystemCodeGenSysLoadFile(SYSTEM sys, CODEGEN be)
{
  ARRAY_CODEGEN_OBJ params;  ARRAY_CODEGEN_TYPE param_types;
  ARRAY_SYSTEM_VIEW all_views;  SYSTEM_VIEW sv;  int i;

  /* open system_load.h and system_load.c */
  be->FileBegin(sys->be_load_file, NPC_VERSION, TimingTimeStamp());

  /* start system_load.h with include files */
  SystemModuleLoadFileCodeIncludes(sys, be);

  /* include all _load.h header files into system_load.c */
  all_views = SystemViewSystemViews(sys->view);
  for( i = 0;  i < ArraySize(all_views);  i++ )
  {
    /* NB can't use internal cursor here */
    sv = ArrayGet(all_views, i);
    be->CodeFileInclude(SystemViewBELoadFile(sv));
  }

  /* system unswizzle function */
  ArrayInit(&params);
  ArrayInit(&param_types);
  ArrayAddLast(params, NPBack_Self);
  ArrayAddLast(param_types, ClassBEType(ClassObject));
  ArrayAddLast(params, NPBack_Mem);
  ArrayAddLast(param_types, be->charp_type);
  SystemCodeGenModuleFunction(sys, be->int_type, sys->unswizzle_be_obj, params,
    param_types, &SystemViewUnSwizzleFn, be);

  /* system_load function */
  BEFnSystemInitCodeGenLoadFn(sys->init_fn, sys->load_be_obj, be);

  /* close files */
  be->FileEnd(sys->be_load_file);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemCodeGenSysMainFile(SYSTEM sys, CODEGEN be)                 */
/*                                                                           */
/*  Code gen the system.h and system.c files.                                */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN SystemCodeGenSysMainFile(SYSTEM sys, CODEGEN be)
{
  ARRAY_SYSTEM_VIEW all_views;  SYSTEM_VIEW sv;  int i, sys_sig;

  /* open system.h and system.c */
  be->FileBegin(sys->be_file, NPC_VERSION, TimingTimeStamp());

  /* start system.h with include files */
  SystemModuleMainFileHeaderIncludes(sys, be);

  /* add NP_DIR_SEP, NP_DATA_DIR, and NP_SYS_SIG to system.h */
  be->CommentSmall("miscellaneous global constants", TRUE, FALSE);
  be->ConstDeclare(be->ConstMakeAString("NP_DIR_SEP", NULL, NPC_DIR_SEP,TRUE));
  be->ConstDeclare(be->ConstMakeAString("NP_DATA_DIR",NULL,sys->data_dir,TRUE));
  sys_sig = time(NULL) % (1 << 30);
  be->ConstDeclare(be->ConstMakeInt("NP_SYS_SIG", NULL, sys_sig, TRUE));

  /* add include files to system.c */
  SystemModuleMainFileCodeIncludes(sys, be);

  /* add system_version function to .c file */
  be->CommentSmall("system functions", TRUE, FALSE);
  be->FunctionBegin(sys->version_be_obj, be->void_type);
  be->FunctionContinue(sys->version_be_obj);
  Stmt( Call3(be->fprintf_fn, Var(be->stderr_fn), String("%s\\n  Modules:"),
    String(NPC_VERSION)));
  all_views = SystemViewSystemViews(sys->view);
  for( i = 0;  i < ArraySize(all_views);  i++ )
  {
    sv = ArrayGet(all_views, i);
    Stmt( Call3(be->fprintf_fn, Var(be->stderr_fn), String(" %s"),
      String((ASTRING) NameShow(SystemViewName(sv)))));
  }
  Stmt(Call2(be->fprintf_fn, Var(be->stderr_fn), String("\\n")));
  be->FunctionEnd(sys->version_be_obj);

  /* close files */
  be->FileEnd(sys->be_file);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemCodeGenTypeDefsFile(SYSTEM sys, CODEGEN be)                   */
/*                                                                           */
/*  Code gen the typedefs file of sys onto be.                               */
/*                                                                           */
/*****************************************************************************/

void SystemCodeGenTypeDefsFile(SYSTEM sys, CODEGEN be)
{
  ARRAY_SYSTEM_VIEW all_views;  SYSTEM_VIEW sv;

  /* create and open the typedefs file */
  all_views = SystemViewSystemViews(sys->view);
  be->FileBegin(sys->typedefs_file, NPC_VERSION, TimingTimeStamp());

  /* codegen the typedefs */
  be->CommentLarge("Typedefs", "",
    "One for each class except int, whose Nonpareil and C names are equal.",
    TRUE, FALSE);
  ArrayForEach(all_views, sv)
    SystemViewCodeGenTypeDefs(sv, be);

  /* codegen the type tags */
  be->CommentLarge("Type tags", "",
    "One for each non-abstract, non-builtin class, plus array and string.",
    TRUE, FALSE);
  ArrayForEach(all_views, sv)
    SystemViewCodeGenTypeTags(sv, be);

  /* code gen the struct declarations */
  be->CommentLarge("Structs", "",
    "One for each non-builtin class, plus array and string.", TRUE, FALSE);
  ArrayForEach(all_views, sv)
    SystemViewCodeGenTypeStructs(sv, be);

  be->FileEnd(sys->typedefs_file);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_EXEC SystemBEInitExec(SYSTEM sys)                                */
/*                                                                           */
/*  Return the backend executable that initializes predefined objects.       */
/*                                                                           */
/*****************************************************************************/

CODEGEN_EXEC SystemBEInitExec(SYSTEM sys)
{
  return sys->be_init_exec;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_EXEC SystemBELoadExec(SYSTEM sys)                                */
/*                                                                           */
/*  Return the backend executable that loads predefined objects and runs.    */
/*                                                                           */
/*****************************************************************************/

CODEGEN_EXEC SystemBELoadExec(SYSTEM sys)
{
  return sys->be_load_exec;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemCodeGen(SYSTEM sys, CODEGEN be)                            */
/*                                                                           */
/*  Carry out code generation of sys, including compilation of the           */
/*  generated C code, if we get that far.                                    */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN SystemCodeGen(SYSTEM sys, CODEGEN be)
{
  ARRAY_SYSTEM_VIEW all_views;  SYSTEM_VIEW sv;  CLASS c;  USTRING npc_lib_dir;
  if( DEBUG2 )
    fprintf(stderr, "[ SystemCodeGen(sys)\n");

  /* initialize the two executables to be made, "np_init" and "np" */
  sys->be_init_exec = be->ExecMake(AStringToUString("np_init"));
  sys->be_load_exec = be->ExecMake(AStringToUString("np"));

  /* typedefs file, etc. */
  sys->load_be_obj = be->FunctionMake2(NameRep(sys->name),
    AStringToUString("load"), TRUE, TRUE);
  sys->version_be_obj = be->FunctionMake2(NameRep(sys->name),
    AStringToUString("version"), TRUE, TRUE);

  /* backend initialization of system views (must be done after the above) */
  all_views = SystemViewSystemViews(sys->view);
  ArrayForEach(all_views, sv)
    SystemViewCodeGenInit(sv, be);
  SystemViewCodeGenInit(sys->view, be);

  /* swizzle objects are copied from sys->view (seems a bit odd) */
  sys->swizzle_be_obj = SystemViewSwizzleFn(sys->view);
  sys->unswizzle_be_obj = SystemViewUnSwizzleFn(sys->view);

  /* declare files; some are copied from sys->view (seems a bit odd) */
  sys->be_init_file = SystemViewBEInitFile(sys->view);
  sys->be_load_file = SystemViewBELoadFile(sys->view);
  sys->be_file = SystemViewBEFile(sys->view);
  sys->typedefs_file = be->FileMake(AStringToUString("typedefs"), FALSE, TRUE);
  sys->npsys_typedefs_file = be->FileMake(AStringToUString("npsys_typedefs"),
    FALSE, TRUE);
  sys->npsys_init_file = be->FileMake(AStringToUString("npsys_init"),
    FALSE, FALSE);
  sys->npsys_load_file = be->FileMake(AStringToUString("npsys_load"),
    FALSE, FALSE);
  sys->npsys_main_file = be->FileMake(AStringToUString("npsys"), FALSE, FALSE);

  /* backend initialization of classes */
  ClassCodeGenInitAll(sys->classes, be);

  /* backend initialization of builtin functions */
  FEFnBuiltinFinalizeAll(be);

  /* backend initialization of creation features */
  ArrayForEach(sys->classes, c)
    ClassCodeGenInitCreationFeatures(c, be);

  /* backend initialization of noncreation features etc. */
  ArrayForEach(sys->classes, c)
    ClassCodeGenInitFunctions(c, be);

  /* copy Nonpareil runtime sytem file npsys_typedefs from library directory */
  npc_lib_dir = AStringToUString(NPC_LIB_DIR);
  be->FileCopy(npc_lib_dir, sys->npsys_typedefs_file, NPC_VERSION,
    TimingTimeStamp());

  /* copy Nonpareil runtime sytem file npsys_init from library directory */
  be->FileCopy(npc_lib_dir, sys->npsys_init_file, NPC_VERSION,
    TimingTimeStamp());
  be->ExecAddFile(sys->be_init_exec, sys->npsys_init_file);

  /* copy Nonpareil runtime sytem file npsys_load from library directory */
  be->FileCopy(npc_lib_dir, sys->npsys_load_file, NPC_VERSION,
    TimingTimeStamp());
  be->ExecAddFile(sys->be_load_exec, sys->npsys_load_file);

  /* copy Nonpareil runtime sytem file npsys from library directory */
  be->FileCopy(npc_lib_dir, sys->npsys_main_file, NPC_VERSION,
    TimingTimeStamp());
  be->ExecAddFile(sys->be_init_exec, sys->npsys_main_file);
  be->ExecAddFile(sys->be_load_exec, sys->npsys_main_file);

  /* code gen the typedefs file */
  SystemCodeGenTypeDefsFile(sys, be);

  /* code gen the modules */
  all_views = SystemViewSystemViews(sys->view);
  ArrayForEach(all_views, sv)
  {
    if( !SystemViewCodeGenInitFile(sv, be) )
      db_return(DEBUG2, "SystemCodeGen (module init file)", FALSE);
    if( !SystemViewCodeGenLoadFile(sv, be) )
      db_return(DEBUG2, "SystemCodeGen (module load file)", FALSE);
    if( !SystemViewCodeGenMainFile(sv, be) )
      db_return(DEBUG2, "SystemCodeGen (module main file)", FALSE);
  }

  /* code gen system_init.c, system_load.c, and system.c */
  if( !SystemCodeGenSysInitFile(sys, be) )
    db_return(DEBUG2, "SystemCodeGen (system init file)", FALSE);
  if( !SystemCodeGenSysLoadFile(sys, be) )
    db_return(DEBUG2, "SystemCodeGen (system load file)", FALSE);
  if( !SystemCodeGenSysMainFile(sys, be) )
    db_return(DEBUG2, "SystemCodeGen (system file)", FALSE);

  /* do the compile */
  db_return(DEBUG2, "SystemCodeGen", be->Compile());
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemDSDebug(SYSTEM sys, FILE *fp)                                 */
/*                                                                           */
/*  Debug print of disjoint sets of classes in system.                       */
/*                                                                           */
/*****************************************************************************/

static void SystemDSDebug(SYSTEM sys, FILE *fp)
{
  CLASS c;
  fprintf(fp, "[ System disjoint sets:\n");
  ArrayForEach(sys->classes, c)
    ClassDSDebug(c, fp);
  fprintf(fp, "]\n");
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemPreparePredefs(SYSTEM sys)                                 */
/*                                                                           */
/*  Prepare the predefined objects and enumerated classes of sys, by         */
/*  promoting child objects into parent lists, assigning code numbers,       */
/*  and checking initialization order.                                       */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN SystemPreparePredefs(SYSTEM sys)
{
  CLASS c;

  if( DEBUG4 )
    fprintf(stderr, "[ SystemPreparePredefs(sys)\n");

  /* build complete sets of predefined objects in all classes */
  ArrayForEach(sys->classes, c)
    ClassPromotePredefinedObjects(c);

  /* initialize code sets in all enumerated classes */
  ArrayForEach(sys->enum_classes, c)
    ClassEnumCodeSetInit(c);

  /* assign code sets in all enumerated classes */
  ArrayForEach(sys->enum_classes, c)
    if( !ClassEnumAssignCodes(c) )
      db_return(DEBUG4, "SystemPreparePredefs assign codes", FALSE);

  /* find a suitable initialization order for the classes */
  ArrayForEach(sys->classes, c)
    if( !ClassFindInitializationOrder(c, sys->init_fn) )
      db_return(DEBUG4, "SystemPreparePredefs initialization order", FALSE);

  db_return(DEBUG4, "SystemPreparePredefs", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemCompile(CODEGEN be, ASTRING data_dir, SYSTEM *res)         */
/*                                                                           */
/*  Create a system by reading the system file, its modules, and their       */
/*  class files, and carrying out all the compilation stages up to and       */
/*  including code generation.                                               */
/*                                                                           */
/*  Parameter data_dir is the full name of the data directory of this        */
/*  system (i.e. /.../system/data).                                          */
/*                                                                           */
/*  If no errors were encountered, return TRUE and set *res to the           */
/*  resulting system.  Otherwise return FALSE, leaving *res unset.           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemCompile(CODEGEN be, ASTRING data_dir, SYSTEM *res)
{
  TOKEN token_list, *t;  ARRAY_SYSTEM_VIEW all_views;  CLASS c;
  BOOLEAN first_module, is_private;  SYSTEM_VIEW sv;
  USTRING ustr_data_be, ustr_data_le, ustr_ccode;
  if( DEBUG3 )
    fprintf(stderr, "[ SystemCompile()\n");

  /* must clear builtin module, else get state hang over when recompiling */
  ustr_data_be = AStringToUString("data_be");
  ustr_data_le = AStringToUString("data_le");
  ustr_ccode = AStringToUString("ccode");
  *res = NULL;
  FEFnBuiltinInitialize(be);

  /* lexical analysis of system file */
  if( !LexFile(NULL, AStringToUString("system"), &token_list, NULL, NULL) )
    db_return(DEBUG3, "SystemCompile lex", FALSE);
  t = &token_list;

  /* initialize system object */
  *res = SystemNew(curr_token, data_dir);

  /* skip system keyword */
  skip(TK_SYSTEM, "\"system\" keyword");

  /* parse module uses */
  first_module = TRUE;
  while( LexType(curr_token) == TK_IDENTIFIER ||
         LexType(curr_token) == TK_PRIVATE )
  {
    /* find out if private usage or not */
    is_private = FALSE;
    if( LexType(curr_token) == TK_PRIVATE )
    {
      is_private = TRUE;
      next_token;
      check(TK_IDENTIFIER, "identifier");
    }

    /* make sure the module name is not "ccode", "data_be", or "data_le" */
    if( UStringEqual(LexValue(curr_token), ustr_data_be) ||
	UStringEqual(LexValue(curr_token), ustr_data_le) ||
    	UStringEqual(LexValue(curr_token), ustr_ccode) )
    {
      fprintf(stderr, "%s: \"%s\" is not allowed as a module name\n",
	LexPos(curr_token), LexShow(curr_token));
      db_return(DEBUG3, "SystemCompile bad module name", FALSE);
    }

    /* establish the module; do some special things for the first module */
    if( first_module )
    {
      /* auto-generate "tuples", "functs", and "char" files if not present */
      USTRING module_dir_name = LexValue(curr_token);
      if( !AutoTupleClasses(module_dir_name) )
	db_return(DEBUG3, "SystemCompile AutoTupleClasses", FALSE);
      if( !AutoFunctionClasses(module_dir_name) )
	db_return(DEBUG3, "SystemCompile AutoFunctionClasses", FALSE);
      if( !AutoCharClass(module_dir_name) )
	db_return(DEBUG3, "SystemCompile AutoCharClass", FALSE);
    }
    if( !SystemViewEstablish(curr_token, *res, &sv) )
      db_return(DEBUG3, "SystemCompile module parse", FALSE);
    if( first_module )
    {
      /* add void class; initialize predefined classes and builtin functions */
      if( !ClassViewVoidClassMake(sv) )
	db_return(DEBUG3, "SystemCompile void class create", FALSE);
      if( !ClassInitPredefined(sv) )
	db_return(DEBUG3, "SystemCompile class init", FALSE);
      FEFnBuiltinInitFeatures(be);
      first_module = FALSE;
    }

    /* carry out later stages */
    if( !SystemViewValidateInterfaceTypes(sv) )
      db_return(DEBUG3, "SystemCompile validate type expressions", FALSE);
    if( !SystemViewInheritFeatures(sv) )
      db_return(DEBUG3, "SystemCompile inherit features", FALSE);

    /* import sv into *res's system view, with renaming */
    next_token;
    if( !SystemViewImport((*res)->view, sv, is_private, t) )
      db_return(DEBUG3, "SystemCompile import", FALSE);
  }

  /* finish parsing the system file */
  skip(TK_END, "module name (identifier) or \"end\" keyword");
  skip(TK_END_FILE, "end of file");

  /* instantiate the system's view */
  sv = (*res)->view;
  if( !SystemViewValidateInterfaceTypes(sv) )
    db_return(DEBUG3, "SystemCompile validate type expressions", FALSE);
  if( !SystemViewInheritFeatures(sv) )
    db_return(DEBUG3, "SystemCompile inherit features", FALSE);

  /* validate expressions in all modules */
  all_views = SystemViewSystemViews((*res)->view);
  ArrayForEach(all_views, sv)
    if( !SystemViewValidateExpressions(sv) )
      db_return(DEBUG3, "SystemCompile validate expressions", FALSE);

  /* finalize disjoint sets components */
  ArrayForEach((*res)->classes, c)
    ClassDSFinalizeComponent(c);

  /* prepare predefined objects */
  if( !SystemPreparePredefs(*res) )
    db_return(DEBUG3, "SystemCompile prepare predefs", FALSE);

  /* handle disjoint sets and cloning */
  if( DEBUG6 )
    SystemDSDebug(*res, stderr);

  /* code generation */
  if( !SystemCodeGen(*res, be) )
    db_return(DEBUG3, "SystemCompile code gen", FALSE);

  db_return(DEBUG3, "SystemCompile", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  SYSTEM_VIEW SystemSystemView(SYSTEM sys)                                 */
/*                                                                           */
/*  Return this system's view of the system.                                 */
/*                                                                           */
/*****************************************************************************/

SYSTEM_VIEW SystemSystemView(SYSTEM sys)
{
  return sys->view;
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemDebug(SYSTEM s, CONTEXT cxt, FILE *fp, int print_style)       */
/*                                                                           */
/*  Debug print of system s on file fp with the given print style.           */
/*                                                                           */
/*****************************************************************************/

void SystemDebug(SYSTEM s, CONTEXT cxt, FILE *fp, int print_style)
{
  fprintf(fp, "[ system");
  begin_indent;
  SystemViewDebug(s->view, cxt, fp, print_style);
  fprintf(fp, "] end system");
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemDebugSystemViewNames(SYSTEM s, SYSTEM_VIEW curr_sv, FILE *fp) */
/*                                                                           */
/*  Print a list of the system views currently in s.                         */
/*                                                                           */
/*****************************************************************************/

void SystemDebugSystemViewNames(SYSTEM s, SYSTEM_VIEW curr_sv, FILE *fp)
{
  SystemViewDebugSystemViewNames(s->view, curr_sv, fp);
}
