/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         system_view.c                                              */
/*  DESCRIPTION:  A view of a system, essentially a module.                  */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#define DEBUG1	0
#define DEBUG2	0
#define DEBUG3	0
#define DEBUG4	0
#define DEBUG5	0
#define DEBUG6	0


/*****************************************************************************/
/*                                                                           */
/*  SYSTEM_VIEW                                                              */
/*                                                                           */
/*  A view of a system, containing views of some of its classes, and         */
/*  also a symbol context for when reading within classes.                   */
/*                                                                           */
/*****************************************************************************/

struct system_view_rec {
  FILE_POS		    file_pos;		/* file position of view     */
  NAME			    name;		/* view (module) name        */
  SYSTEM		    system;		/* the enclosing system      */
  SYMTAB_CLASS_VIEW	    class_views;	/* class views visible in m  */
  ARRAY_CLASS		    orig_classes;	/* classes originating in m  */
  CONTEXT		    context;		/* cxt, holding class_views  */
  SYMTAB_NAMED		    op_table;		/* operators in view         */
  ARRAY_SYSTEM_VIEW	    view_array;		/* views incorporated        */
  SYMTAB_SYSTEM_VIEW	    view_table;		/* views incorporated        */

  /* code generation fields */
  CODEGEN_FILE		    be_file;		/* backend file              */
  CODEGEN_FILE		    be_init_file;	/* backend init file         */
  CODEGEN_FILE		    be_load_file;	/* backend load file         */
  ARRAY_CLASS		    concrete_classes;	/* concrete classes          */
  ASTRING		    code_file_stem;	/* e.g. "ccode/lang"         */
  CODEGEN_OBJ		    swizzle_be_obj;	/* swizzle backend obj       */
  CODEGEN_OBJ		    unswizzle_be_obj;	/* unswizzle backend obj     */
  SYMTAB_BEFN_FEATURE	    predef_strings;	/* literal strings table     */
};


/*****************************************************************************/
/*                                                                           */
/*  CLASS_RENAME (private)                                                   */
/*                                                                           */
/*  A rename of one class, with optional feature renaming.                   */
/*  Actually to_name is optional too.                                        */
/*                                                                           */
/*  Note that CLASS_RENAME inherits from NAMED, with the to position         */
/*  and name available for use in the previous field of a NAMED object.      */
/*                                                                           */
/*****************************************************************************/

typedef struct class_rename_rec {
  FILE_POS		to_pos;
  NAME			to_name;
  FILE_POS		from_pos;
  NAME			from_name;
  CLASS_VIEW_UPDATE	class_view_update;
} *CLASS_RENAME;

typedef ARRAY(CLASS_RENAME)			ARRAY_CLASS_RENAME;
typedef SYMTAB(ARRAY_CLASS_RENAME)		SYMTAB_CLASS_RENAME;


/*****************************************************************************/
/*                                                                           */
/*  Submodule "establish"                                                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  SYSTEM_VIEW SystemViewNew(FILE_POS file_pos, NAME name, SYSTEM system)   */
/*                                                                           */
/*  Return a new, empty system view with these attributes.                   */
/*                                                                           */
/*****************************************************************************/

SYSTEM_VIEW SystemViewNew(FILE_POS file_pos, NAME name, SYSTEM system)
{
  SYSTEM_VIEW res;
  GetMemory(res, SYSTEM_VIEW);
  res->file_pos = file_pos;
  res->name = name;
  res->system = system;
  SymInit(&res->class_views);
  ArrayInit(&res->orig_classes);
  res->context = NULL;
  SymInit(&res->op_table);
  ArrayInit(&res->view_array);
  SymInit(&res->view_table);
  res->be_file = NULL;
  res->be_init_file = NULL;
  res->be_load_file = NULL;
  res->swizzle_be_obj = NULL;
  res->unswizzle_be_obj = NULL;
  res->concrete_classes = NULL;
  res->code_file_stem = NULL;
  SymInit(&res->predef_strings);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  FILE_POS SystemViewFilePos(SYSTEM_VIEW sv)                               */
/*                                                                           */
/*  Return the file position of sv.                                          */
/*                                                                           */
/*****************************************************************************/

FILE_POS SystemViewFilePos(SYSTEM_VIEW sv)
{
  return sv->file_pos;
}


/*****************************************************************************/
/*                                                                           */
/*  NAME SystemViewName(SYSTEM_VIEW sv)                                      */
/*                                                                           */
/*  Return the name of sv.                                                   */
/*                                                                           */
/*****************************************************************************/

NAME SystemViewName(SYSTEM_VIEW sv)
{
  return sv->name;
}


/*****************************************************************************/
/*                                                                           */
/*  SYSTEM SystemViewSystem(SYSTEM_VIEW sv)                                  */
/*                                                                           */
/*  Return the system that sv lies in.                                       */
/*                                                                           */
/*****************************************************************************/

SYSTEM SystemViewSystem(SYSTEM_VIEW sv)
{
  return sv->system;
}


/*****************************************************************************/
/*                                                                           */
/*  SYMTAB_NAMED SystemViewOpTable(SYSTEM_VIEW sv)                           */
/*                                                                           */
/*  Return the op table of sv.                                               */
/*                                                                           */
/*****************************************************************************/

SYMTAB_NAMED SystemViewOpTable(SYSTEM_VIEW sv)
{
  return sv->op_table;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewInsertClassView(SYSTEM_VIEW sv, CLASS_VIEW cv,         */
/*    CLASS_VIEW *cv2)                                                       */
/*                                                                           */
/*  If cv can be inserted into sv, do so and return TRUE.  Otherwise         */
/*  return FALSE and set *cv2 to the preexisting class view with the         */
/*  same name.                                                               */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewInsertClassView(SYSTEM_VIEW sv, CLASS_VIEW cv,
  CLASS_VIEW *cv2)
{
  BOOLEAN res;
  if( DEBUG6 )
    fprintf(stderr, "[ SystemViewInsertClassView(%s, %s)\n",
      NameShow(sv->name), NameShow(ClassViewName(cv)));
  res = SymInsert(sv->class_views, NameKey(ClassViewName(cv)), cv, cv2);
  if( DEBUG6 )
    fprintf(stderr, "] SystemViewInsertClassView returning %s\n", bool(res));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewRetrieveClassView(SYSTEM_VIEW sv, USTRING key,         */
/*    CLASS_VIEW *cv)                                                        */
/*                                                                           */
/*  If key is the name of a class view in sv, return TRUE and set *cv        */
/*  to that view; otherwise return FALSE.                                    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewRetrieveClassView(SYSTEM_VIEW sv, USTRING key,
  CLASS_VIEW *cv)
{
  return SymRetrieve(sv->class_views, key, cv);
}


/*****************************************************************************/
/*                                                                           */
/*  SYMTAB_CLASS_VIEW SystemViewClassViews(SYSTEM_VIEW sv)                   */
/*                                                                           */
/*  Return the class views of sv.                                            */
/*                                                                           */
/*****************************************************************************/

SYMTAB_CLASS_VIEW SystemViewClassViews(SYSTEM_VIEW sv)
{
  return sv->class_views;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewRetrieveSystemView(SYSTEM_VIEW sv, USTRING key,        */
/*    SYSTEM_VIEW *sv2)                                                      */
/*                                                                           */
/*  Retrieve the system view with the given key from sv, or return FALSE     */
/*  if no such view.                                                         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewRetrieveSystemView(SYSTEM_VIEW sv, USTRING key,
  SYSTEM_VIEW *sv2)
{
  return SymRetrieve(sv->view_table, key, sv2);
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_SYSTEM_VIEW SystemViewSystemViews(SYSTEM_VIEW sv)                  */
/*                                                                           */
/*  Return all system views in the order they were read.                     */
/*                                                                           */
/*****************************************************************************/

ARRAY_SYSTEM_VIEW SystemViewSystemViews(SYSTEM_VIEW sv)
{
  return sv->view_array;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ParseOneClassRename(TOKEN *t, SYMTAB_NAMED op_table,             */
/*    CLASS_RENAME *res)                                                     */
/*                                                                           */
/*  Parse one class rename and return it in *res.                            */
/*  Return TRUE if there were no errors in the parse, else FALSE.            */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ParseOneClassRename(TOKEN *t, SYMTAB_NAMED op_table,
  CLASS_RENAME *res)
{
  GetMemory(*res, CLASS_RENAME);

  /* name of class being renamed */
  (*res)->from_pos = LexFilePos(curr_token);
  if( !NameDefParse(t, op_table, &(*res)->from_name) ) return FALSE;

  /* optional feature rename list */
  if( !ParseOptionalClassViewUpdate(t, op_table, &(*res)->class_view_update) )
    return FALSE;

  /* optional "as" part of class rename */
  if( LexType(curr_token) == TK_AS )
  {
    next_token;
    (*res)->to_pos = LexFilePos(curr_token);
    if( !NameDefParse(t, op_table, &(*res)->to_name) ) return FALSE;
  }
  else
  {
    (*res)->to_pos = NULL;
    (*res)->to_name = NULL;
  }

  /* but you can't leave both out, since that would be useless */
  if( (*res)->class_view_update == NULL && (*res)->to_name == NULL )
  {
    fprintf(stderr, "%s: rename of class \"%s\" does nothing\n",
      FilePosShow((*res)->from_pos), NameShow((*res)->from_name));
    return FALSE;
  }

  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN ModuleOptionalRenameClauseParse(TOKEN *t, SYMTAB_NAMED op_table, */
/*    SYMTAB_CLASS_RENAME *rename_table)                                     */
/*                                                                           */
/*  Parse an optional rename clause, and store the renames in rename_table,  */
/*  or set it to NULL if there is no rename clause.                          */
/*  Return TRUE if there were no errors in the parse, else FALSE.            */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN ModuleOptionalRenameClauseParse(TOKEN *t, SYMTAB_NAMED op_table,
  SYMTAB_CLASS_RENAME *rename_table)
{
  CLASS_RENAME cr, cr2;
  if( LexType(curr_token) == TK_RENAME )
  {
    SymInit(rename_table);
    next_token;
    skip(TK_LEFT_PAREN, "opening ( of rename clause");
    if( !ParseOneClassRename(t, op_table, &cr) )  return FALSE;
    SymInsert(*rename_table, NameKey(cr->from_name), cr, &cr2);
    while( LexType(curr_token) == TK_COMMA )
    {
      next_token;
      if( !ParseOneClassRename(t, op_table, &cr) )  return FALSE;
      if( !SymInsert(*rename_table, NameKey(cr->from_name), cr, &cr2) )
      {
	fprintf(stderr, "%s already renamed, at %s\n",
	  NameFullShow(cr->from_name), FilePosShow(cr2->from_pos));
	return FALSE;
      }
    }
    skip(TK_RIGHT_PAREN, "comma or closing ) of rename clause");
  }
  else
    *rename_table = NULL;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewDoImport(SYSTEM_VIEW target, SYSTEM_VIEW source,       */
/*    BOOLEAN is_private, SYMTAB_CLASS_RENAME rename_table, FILE_POS mpos)   */
/*                                                                           */
/*  Import system view source into system view target.  Every public class   */
/*  from source is imported into target, unless it is already present there  */
/*  (in which case it must have the same name).  Every public original       */
/*  feature of every class is also imported, again unless it is already      */
/*  present under the same name.                                             */
/*                                                                           */
/*  These imports create new class views and new feature views, which        */
/*  need file positions of their own as well as names.  If an import         */
/*  includes a rename, then the appropriate file position is the point       */
/*  of the rename.  If an import does not include a rename, then the         */
/*  appropriate point is the nearest point.  For a class import this         */
/*  will be the point where its module name was mentioned to be imported;    */
/*  for a feature, this will be the point where its class was mentioned      */
/*  to be imported, or if not mentioned, as for the class import.            */
/*  Parameter module_pos is the position where the module was mentioned.     */
/*                                                                           */
/*  If is_private is TRUE, this is a private import, so the class views      */
/*  it creates are marked private.                                           */
/*                                                                           */
/*  This code keeps target->op_table up to date as well as everything else.  */
/*  Every imported class and feature name is checked against the table, and  */
/*  inserted into it if it is an operator.                                   */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN SystemViewDoImport(SYSTEM_VIEW target, SYSTEM_VIEW source,
  BOOLEAN is_private, SYMTAB_CLASS_RENAME rename_table, FILE_POS module_pos)
{
  USTRING key;  NAME from_name, new_name;  FILE_POS from_pos, new_pos;
  CLASS_RENAME class_rename;
  CLASS_VIEW source_cv, target_cv, cv2;
  CLASS_VIEW_UPDATE class_view_update;

  /* make sure that every rename does actually apply and is given accurately */
  if( rename_table != NULL )
  {
    SymForEach(rename_table, &key, &class_rename)
    {
      from_name = class_rename->from_name;
      from_pos = class_rename->from_pos;
      if( !SymRetrieve(source->class_views, key, &source_cv) )
      {
	/* can't rename an unknown class */
	fprintf(stderr, "%s: %s not known in %s\n", FilePosShow(from_pos),
	  NameFullShow(from_name), NameShow(source->name));
	return FALSE;
      }
      else if( ClassViewIsPrivate(source_cv) )
      {
	/* can't rename a private class */
	fprintf(stderr, "%s: %s is private in %s\n", FilePosShow(from_pos),
	  NameFullShow(from_name), NameShow(source->name));
	return FALSE;
      }
      else if( ClassViewIsNoRename(source_cv) )
      {
	/* can't rename a norename class */
	fprintf(stderr, "%s: %s is norename in %s\n", FilePosShow(from_pos),
	  NameFullShow(from_name), NameShow(source->name));
	return FALSE;
      }
      else if( !NameCompatible(from_name, from_pos, ClassViewName(source_cv),
	  ClassViewFilePos(source_cv), "class rename") )
	return FALSE;
    }
  }

  /* meld each non-private class of source, possibly renamed, into target */
  SymForEach(source->class_views, &key, &source_cv)
  {
    if( !ClassViewIsPrivate(source_cv) )
    {
      /* get feature rename_table and new_name for this class */
      /* still to do: review how these names are got at, could be better */
      if( rename_table==NULL || !SymRetrieve(rename_table,key,&class_rename) )
      {
	/* class is not renamed, use null or existing values */
	class_view_update = NULL;
	new_name = NameNewFromPrev(ClassViewName(source_cv), (NAMED) source_cv);
	new_pos = module_pos;
      }
      else if( class_rename->to_name == NULL )
      {
	/* class itself is not being renamed, but features are */
	class_view_update = class_rename->class_view_update;
	new_name = NameNewFromPrev(ClassViewName(source_cv), (NAMED) source_cv);
	new_pos = class_rename->from_pos;
      }
      else
      {
	/* class is being renamed, and possibly features too */
	class_view_update = class_rename->class_view_update;
	new_name = NameNewFromPrev(class_rename->to_name, (NAMED) source_cv);
	new_pos = class_rename->to_pos;
      }

      /* check new_name against target op table, possibly adding to op_table */
      if( !NameOperatorConsistent(new_name, new_pos, target->op_table,
	  "class import") )
	return FALSE;

      /* make and insert a new class view, or retrieve existing one */
      if( SymRetrieve(target->class_views, NameKey(new_name), &target_cv) )
      {
	/* already present, must denote the same class */
	if( ClassViewClass(source_cv) != ClassViewClass(target_cv) )
	{
	  fprintf(stderr, "class name clash:\n");
	  fprintf(stderr, "  %s: %s\n",
	    FilePosShow(new_pos), NameFullShow(new_name));
	  fprintf(stderr, "  %s: %s\n",
	    FilePosShow(ClassViewFilePos(target_cv)),
	    NameFullShow(ClassViewName(target_cv)));
	  return FALSE;
	}

	/* privacy must be equal */
	if( ClassViewIsPrivate(source_cv) != ClassViewIsPrivate(target_cv) )
	{
	  fprintf(stderr, "class privacy clash:\n");
	  fprintf(stderr, "  %s: %s\n",
	    FilePosShow(new_pos), NameFullShow(new_name));
	  fprintf(stderr, "  %s: %s\n",
	    FilePosShow(ClassViewFilePos(target_cv)),
	    NameFullShow(ClassViewName(target_cv)));
	  return FALSE;
	}
      }
      else
      {
	/* not present, so make one */
	target_cv = ClassViewImportCopy(source_cv, new_pos, new_name,
	  is_private, target);
	SymInsert(target->class_views, NameKey(new_name), target_cv, &cv2);
      }

      /* meld public original features into feature context */
      if( !ClassViewImportUninheritedFEFnFeatures(target_cv, source_cv,
	  class_view_update, target->op_table, new_pos) )
	return FALSE;
    }
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewImport(SYSTEM_VIEW target, SYSTEM_VIEW source,         */
/*    BOOLEAN is_private, TOKEN *t)                                          */
/*                                                                           */
/*  Import source into target.  TOKEN t points the the beginning of an       */
/*  optional rename table which is to be applied to source during the        */
/*  import.  If is_private is TRUE, this is a private import.                */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewImport(SYSTEM_VIEW target, SYSTEM_VIEW source,
  BOOLEAN is_private, TOKEN *t)
{
  SYMTAB_CLASS_RENAME rename_table;  SYSTEM_VIEW sv2;  FILE_POS file_pos;

  /* insert source view into target's table, making sure not already present */
  if( !SymInsert(target->view_table, NameKey(source->name), source, &sv2) )
  {
    fprintf(stderr, "%s: %s already used, at %s\n",
      LexPos(curr_token), LexShow(curr_token), FilePosShow(sv2->file_pos));
    return FALSE;
  }
  ArrayAddLast(target->view_array, source);
  file_pos = LexFilePos(curr_token);

  /* parse the optional rename clause to obtain the rename table */
  if( !ModuleOptionalRenameClauseParse(t, target->op_table, &rename_table) )
    return FALSE;

  /* do the import */
  if( !SystemViewDoImport(target, source, is_private, rename_table, file_pos) )
    return FALSE;

  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewEstablish(TOKEN module_name_token, SYSTEM system,      */
/*    SYSTEM_VIEW *res)                                                      */
/*                                                                           */
/*  Establish one module.  If successful, return TRUE and set *res to the    */
/*  resulting system view.                                                   */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewEstablish(TOKEN module_name_token, SYSTEM system,
  SYSTEM_VIEW *res)
{
  ARRAY_TOKEN extend_token_lists;   /* temp storage of extension token lists */
  ARRAY_FILE_POS extend_file_poses; /* temp storage of extension file names  */
  ARRAY_TOKEN class_token_lists;    /* temp storage of class token lists     */
  ARRAY_FILE_POS class_file_poses;  /* temp storage of extension file names  */

  TOKEN token_list, token_list_2, *t;  USTRING module_name;
  SYSTEM_VIEW sv;  FILE_POS file_pos;  int i;  BOOLEAN is_private;
  if( DEBUG2 )
    fprintf(stderr, "[ SystemViewEstablish(%s)\n", LexShow(module_name_token));

  /* lex the module file */
  module_name = LexValue(module_name_token);
  if( !LexFile(module_name, AStringToUString("module"), &token_list, NULL,
	LexFilePos(module_name_token)) )
      db_return(DEBUG4, "SystemViewEstablish lex", FALSE);
  t = &token_list;

  /* skip header and initialize result object */
  skip(TK_MODULE, "\"module\" keyword");
  *res = SystemViewNew(LexFilePos(curr_token), NameNew(LexValue(curr_token)),
    system);
  skip(TK_IDENTIFIER, "module name (identifier)");

  /* make sure that module name equals file name */
  if( !UStringEqual(LexValue(module_name_token), NameKey((*res)->name)) )
  {
    fprintf(stderr, "%s: module name \"%s\" %s \"%s\" at %s\n",
      FilePosShow((*res)->file_pos), NameShow((*res)->name),
      "here disagrees with\n  module directory name",
      LexShow(module_name_token), FilePosShow(LexFilePos(module_name_token)));
    db_return(DEBUG2, "SystemViewEstablish module name", FALSE);
  }

  /* parse uses clause and store module uses */
  if( LexType(curr_token) == TK_USE )
  {
    next_token;
    while( LexType(curr_token) == TK_IDENTIFIER ||
	   LexType(curr_token) == TK_PRIVATE )
    {
      /* find out if module use is private */
      is_private = FALSE;
      if( LexType(curr_token) == TK_PRIVATE )
      {
	is_private = TRUE;
	next_token;
	check(TK_IDENTIFIER, "identifier");
      }

      /* module should already be known in system, so just retrieve its view */
      if( !SystemRetrieveView(system, LexValue(curr_token), &sv) )
      {
	fprintf(stderr, "%s: attempt to use module \"%s\", which is\n%s\n",
	  LexPos(curr_token), LexShow(curr_token),
	  "  unknown or does not precede this module in system file");
	db_return(DEBUG2, "SystemViewEstablish retrieve view", FALSE);
      }

      /* incorporate module into sv, with renaming */
      next_token;
      if( !SystemViewImport(*res, sv, is_private, t) )
	db_return(DEBUG2, "SystemViewEstablish view import", FALSE);
    }
  }
  else
  {
    /* did not find use, but make sure found something sensible */
    if( LexType(curr_token) != TK_EXTEND &&
	LexType(curr_token) != TK_INTRODUCE && LexType(curr_token) != TK_END )
    {
      skip(TK_END, "\"use\", \"extend\", \"introduce\" or \"end\" keyword");
    }
  }

  /* lex class extensions, build op_table, store names and token lists */
  ArrayInit(&extend_token_lists);
  ArrayInit(&extend_file_poses);
  if( LexType(curr_token) == TK_EXTEND )
  {
    next_token;
    while( LexType(curr_token) == TK_IDENTIFIER )
    {
      file_pos = LexFilePos(curr_token);
      if( !LexFile(NameKey((*res)->name), LexValue(curr_token), &token_list_2,
	  (*res)->op_table, (*res)->file_pos) )
	return FALSE;
      ArrayAddLast(extend_token_lists, token_list_2);
      ArrayAddLast(extend_file_poses, file_pos);

      /* next class extension */
      next_token;
    }
  }
  else if( LexType(curr_token)!=TK_INTRODUCE && LexType(curr_token)!=TK_END )
  {
    /* did not find something sensible */
    skip(TK_END, "\"extend\", \"introduce\" or \"end\" keyword");
  }

  /* lex classes, build op_table, store names and token lists */
  ArrayInit(&class_token_lists);
  ArrayInit(&class_file_poses);
  if( LexType(curr_token) == TK_INTRODUCE )
  {
    next_token;
    while( LexType(curr_token) == TK_IDENTIFIER )
    {
      file_pos = LexFilePos(curr_token);
      if( !LexFile(NameKey((*res)->name), LexValue(curr_token), &token_list_2,
	  (*res)->op_table, (*res)->file_pos) )
	db_return(DEBUG2, "SystemViewEstablish", FALSE);
      ArrayAddLast(class_token_lists, token_list_2);
      ArrayAddLast(class_file_poses, file_pos);

      /* next class */
      next_token;
    }
  }
  else if( LexType(curr_token) != TK_END )
  {
    /* did not find something sensible */
    skip(TK_END, "\"introduce\" or \"end\" keyword");
  }

  /* finish parse of module file */
  skip(TK_END, "class name (identifier) or \"end\" keyword");
  skip(TK_END_FILE, "end of file");
  LexTokenListFree(token_list);

  /* (*res)->op_table is complete; parse the extension files */
  for( i = 0;  i < ArraySize(extend_token_lists);  i++ )
  {
    token_list_2 = ArrayGet(extend_token_lists, i);
    file_pos = ArrayGet(extend_file_poses, i);
    if( !ClassViewListParse(&token_list_2, *res, file_pos, TRUE) )
      db_return(DEBUG2, "SystemViewEstablish extension list parse", FALSE);
    LexTokenListFree(token_list_2);
  }

  /* parse the class files */
  for( i = 0;  i < ArraySize(class_token_lists);  i++ )
  {
    token_list_2 = ArrayGet(class_token_lists, i);
    file_pos = ArrayGet(class_file_poses, i);
    if( !ClassViewListParse(&token_list_2, *res, file_pos, FALSE) )
      db_return(DEBUG2, "SystemViewEstablish class list parse", FALSE);
    LexTokenListFree(token_list_2);
  }

  db_return(DEBUG2, "SystemViewEstablish", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "validate interface types"                                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewValidateInterfaceTypes(SYSTEM_VIEW sv)                 */
/*                                                                           */
/*  Validate the interface types of system view sv.  This takes five         */
/*  steps: instantiate, build ancestor sets, topological sort, Level         */
/*  3 validity, and feature types.                                           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewValidateInterfaceTypes(SYSTEM_VIEW sv)
{
  USTRING key;  CLASS_VIEW cv;  CLASS c;

  if( DEBUG1 )
    fprintf(stderr, "[ SystemViewValidateInterfaceTypes(%s)\n",
      NameShow(sv->name));

  /* Stage 0: add an all_predefined feature to each non-generic class */
  if( DEBUG1 ) fprintf(stderr, "SystemViewValidateInterfaceTypes (Stage 0):\n");
  SymForEach(sv->class_views, &key, &cv)
    if( !ClassViewValidateInterfaceTypes0(cv) )
      db_return(DEBUG1, "SystemViewValidateInterfaceTypes (Stage 0)", FALSE);

  /* make a context containing the class views plus an empty level */
  sv->context = ContextNew(sv->class_views);
  ContextPushEmpty(sv->context, NULL, TRUE);

  /* Stage 1: instantiate generic parameters, class types, and inherit types */
  if( DEBUG1 ) fprintf(stderr, "SystemViewValidateInterfaceTypes (Stage 1):\n");
  SymForEach(sv->class_views, &key, &cv)
  {
    if( !ClassViewValidateInterfaceTypes1(cv, sv->context) )
      db_return(DEBUG1, "SystemViewValidateInterfaceTypes (Stage 1)", FALSE);
    ContextClearTopLevel(sv->context);
  }

  /* Stage 2: build ancestor set */
  if( DEBUG1 ) fprintf(stderr, "SystemViewValidateInterfaceTypes (Stage 2):\n");
  ArrayForEach(sv->orig_classes, c)
    if( !ClassValidateInterfaceTypes2(c, sv->context) )
      db_return(DEBUG1, "SystemViewValidateInterfaceTypes (Stage 2)", FALSE);

  /* Stage 3: find topological order and set eq_class_id and eq_class_offset */
  if( DEBUG1 ) fprintf(stderr, "SystemViewValidateInterfaceTypes (Stage 3):\n");
  ArrayForEach(sv->orig_classes, c)
    if( !ClassValidateInterfaceTypes3(c, sv->context) )
      db_return(DEBUG1, "SystemViewValidateInterfaceTypes (Stage 3)", FALSE);

  /* Stage 4: establish Level 3 validity of types */
  if( DEBUG1 ) fprintf(stderr, "SystemViewValidateInterfaceTypes (Stage 4):\n");
  ArrayForEach(sv->orig_classes, c)
    if( !ClassValidateInterfaceTypes4(c, sv->context) )
      db_return(DEBUG1, "SystemViewValidateInterfaceTypes (Stage 4)", FALSE);

  /* Stage 5: get feature types */
  if( DEBUG1 ) fprintf(stderr, "SystemViewValidateInterfaceTypes (Stage 5):\n");
  SymForEach(sv->class_views, &key, &cv)
    if( !ClassViewValidateInterfaceTypes5(cv, sv->context) )
      db_return(DEBUG1, "SystemViewValidateInterfaceTypes (Stage 5)", FALSE);

  /* long-term, the context holds just the class views level */
  ContextPop(sv->context, TRUE);

  db_return(DEBUG1, "SystemViewValidateInterfaceTypes", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "inherit features"                                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewInheritFeatures(SYSTEM_VIEW sv)                        */
/*                                                                           */
/*  Carry out inheritance of feature views throughout sv.                    */
/*                                                                           */
/*  Implementation note.  The context here is used only for lookup, it       */
/*  does not change.                                                         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewInheritFeatures(SYSTEM_VIEW sv)
{
  USTRING key;  CLASS_VIEW cv;

  /* Stage 1: uncoerced inherited features */
  assert(sv->context != NULL);
  if( DEBUG1 ) fprintf(stderr, "SystemViewInheritFeatures (Stage 1):\n");
  SymForEach(sv->class_views, &key, &cv)
    if( !ClassViewInheritFeatures1(cv, sv->context) )
      db_return(DEBUG1, "SystemViewInheritFeatures (Stage 1)", FALSE);

  /* Stage 2: coerced inherited features */
  if( DEBUG1 ) fprintf(stderr, "SystemViewInheritFeatures (Stage 2):\n");
  SymForEach(sv->class_views, &key, &cv)
    if( !ClassViewInheritFeatures2(cv, sv->context) )
      db_return(DEBUG1, "SystemViewInheritFeatures (Stage 2)", FALSE);

  db_return(DEBUG1, "SystemViewInheritFeatures", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "validate expressions"                                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewValidateExpressions(SYSTEM_VIEW sv)                    */
/*                                                                           */
/*  Validate the expressions of sv.                                          */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewValidateExpressions(SYSTEM_VIEW sv)
{
  USTRING key;  CLASS_VIEW cv;

  if( DEBUG5 )
    fprintf(stderr, "[ SystemViewValidateExpressions(%s)\n",NameShow(sv->name));
  SymForEach(sv->class_views, &key, &cv)
    if( !ClassViewValidateExpressions(cv, sv->context) )
      db_return(DEBUG5, "SystemViewValidateExpressions", FALSE);
  db_return(DEBUG5, "SystemViewValidateExpressions", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "code generation"                                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_FILE SystemViewBEFile(SYSTEM_VIEW sv)                            */
/*                                                                           */
/*  Return the backend file of this system view.                             */
/*                                                                           */
/*****************************************************************************/

CODEGEN_FILE SystemViewBEFile(SYSTEM_VIEW sv)
{
  assert(sv->be_file != NULL);
  return sv->be_file;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_FILE SystemViewBEInitFile(SYSTEM_VIEW sv)                        */
/*                                                                           */
/*  Return the backend file of this system view.                             */
/*                                                                           */
/*****************************************************************************/

CODEGEN_FILE SystemViewBEInitFile(SYSTEM_VIEW sv)
{
  assert(sv->be_init_file != NULL);
  return sv->be_init_file;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_FILE SystemViewBELoadFile(SYSTEM_VIEW sv)                        */
/*                                                                           */
/*  Return the backend file of this system view.                             */
/*                                                                           */
/*****************************************************************************/

CODEGEN_FILE SystemViewBELoadFile(SYSTEM_VIEW sv)
{
  assert(sv->be_load_file != NULL);
  return sv->be_load_file;
}


/*****************************************************************************/
/*                                                                           */
/*  MODULE_INIT_FN SystemViewInitBEFn(SYSTEM_VIEW sv)                        */
/*                                                                           */
/*  Return the C function that initializes this module.                      */
/*                                                                           */
/*****************************************************************************/

/* ***
MODULE_INIT_FN SystemViewInitBEFn(SYSTEM_VIEW sv)
{
  assert(sv->init_fn != NULL);
  return sv->init_fn;
}
*** */


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ SystemViewSwizzleFn(SYSTEM_VIEW sv)                          */
/*                                                                           */
/*  Return the C name of the swizzle function of this module.                */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ SystemViewSwizzleFn(SYSTEM_VIEW sv)
{
  assert(sv->swizzle_be_obj != NULL);
  return sv->swizzle_be_obj;
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_OBJ SystemViewUnSwizzleFn(SYSTEM_VIEW sv)                        */
/*                                                                           */
/*  Return the C name of the unswizzle function of this module.              */
/*                                                                           */
/*****************************************************************************/

CODEGEN_OBJ SystemViewUnSwizzleFn(SYSTEM_VIEW sv)
{
  assert(sv->unswizzle_be_obj != NULL);
  return sv->unswizzle_be_obj;
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemViewRegisterClass(SYSTEM_VIEW sv, CLASS class)                */
/*                                                                           */
/*  Register class with sv.                                                  */
/*                                                                           */
/*****************************************************************************/

void SystemViewRegisterClass(SYSTEM_VIEW sv, CLASS class)
{
  ArrayAddLast(sv->orig_classes, class);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewLastClassWithTypeTag(SYSTEM_VIEW sv, CLASS *c)         */
/*                                                                           */
/*  If at least one concrete class is defined in sv, return TRUE and         */
/*  set *c to that class.  Otherwise return FALSE.                           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewLastClassWithTypeTag(SYSTEM_VIEW sv, CLASS *c)
{
  ArrayForEachReverse(sv->orig_classes, *c)
    if( ClassHasTypeTag(*c) )
      return TRUE;
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  BEFN_FEATURE SystemViewRegisterLiteralString(SYSTEM_VIEW sv,             */
/*    USTRING str, FILE_POS file_pos)                                        */
/*                                                                           */
/*  Each literal string in module sv calls this function, which converts     */
/*  the string into a unique predefined object feature and inserts that      */
/*  feature into the module's predefs.  Or if the string has already         */
/*  occurred in this module, it merely returns the existing feature.         */
/*                                                                           */
/*****************************************************************************/

BEFN_FEATURE SystemViewRegisterLiteralString(SYSTEM_VIEW sv, USTRING str,
  FILE_POS file_pos)
{
  BEFN_FEATURE res, r2;
  if( SymRetrieve(sv->predef_strings, str, &res) )
    return res;
  res = BEFnFeatureNewAnon(file_pos, ClassString,
    ExprArrayFromString(file_pos, str));
  ClassAddBEFnFeature(ClassString, res, FALSE, TRUE);
  /* BEFnSystemInitRegisterPredef(SystemBEFnSystemInit(sv->system), res); */
  /* SystemRegisterPredef(sv->system, res); */
  SymInsert(sv->predef_strings, str, res, &r2);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  void comment(CODEGEN be, BOOLEAN header, BOOLEAN code,                  */
/*    BOOLEAN *control, char *str, char *str2)                               */
/*                                                                           */
/*  Add a comment containing str and optionally str2 to be, but only if      */
/*  *control is FALSE.  Then set *control to TRUE so that the comment won't  */
/*  be added again.                                                          */
/*                                                                           */
/*****************************************************************************/

static void comment(CODEGEN be, BOOLEAN header, BOOLEAN code,
  BOOLEAN *control, char *str, char *str2)
{
  if( !*control )
  {
    be->CommentLarge(str, str2 == NULL ? NULL : "", str2, header, code);
    *control = TRUE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  char *TimingTimeStamp()                                                  */
/*                                                                           */
/*  Return the usual Unix time stamp string describing the time right now.   */
/*                                                                           */
/*****************************************************************************/

char *TimingTimeStamp()
{
  time_t save_time;  struct tm *local_time;  char *str;
  time(&save_time);
  local_time = localtime(&save_time);
  str = asctime(local_time);
  if( str[strlen(str) - 1] == '\n' )  /* which apparently it always will be */
    str[strlen(str) - 1] = '\0';
  return str;
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemViewCodeGenModuleFunction(SYSTEM_VIEW sv,                     */
/*    CODEGEN_TYPE res_be_type, CODEGEN_OBJ be_obj, ARRAY_CODEGEN_OBJ params,*/
/*    ARRAY_CODEGEN_TYPE param_types,                                        */
/*    CODEGEN_OBJ (*name_in_class)(CLASS c), CODEGEN be)                     */
/*                                                                           */
/*  Generate on be a function that implements sv's contribution to the       */
/*  module function with this name and parameters.  This function is just    */
/*  a big switch on all the concrete classes, including array and string     */
/*  if present.                                                              */
/*                                                                           */
/*****************************************************************************/

static void SystemViewCodeGenModuleFunction(SYSTEM_VIEW sv,
  CODEGEN_TYPE res_be_type, CODEGEN_OBJ be_obj, ARRAY_CODEGEN_OBJ params,
  ARRAY_CODEGEN_TYPE param_types,
  CODEGEN_OBJ (*name_in_class)(CLASS c), CODEGEN be)
{
  CLASS class;  CODEGEN_OBJ param;  CODEGEN_TYPE param_type;
  CODEGEN_OBJ return_be_obj;  int i;

  /* initialize various backend variables */
  assert(params != NULL && ArraySize(params) >= 1);
  return_be_obj = res_be_type == be->void_type ? NULL : be->ret;

  /* interested in concrete, non-builtin classes, plus array and string */
  if( sv->concrete_classes == NULL )
  {
    ArrayInit(&sv->concrete_classes);
    ArrayForEach(sv->orig_classes, class)
      if( (!ClassIsAbstract(class) && !ClassIsBuiltin(class))
	  || class == ClassArray || class == ClassString )
	ArrayAddLast(sv->concrete_classes, class);
  }

  /* print header */
  be->FunctionBegin(be_obj, res_be_type);
  for( i = 0;  i < ArraySize(params);  i++ )
  {
    param = ArrayGet(params, i);
    param_type = ArrayGet(param_types, i);
    be->FunctionFormal(be_obj, param, param_type);
  }
  be->FunctionContinue(be_obj);
  if( ArraySize(sv->concrete_classes) >= 1 )
  {
    be->SwitchBegin();
    Call1(NPBack_Type_Tag, Var(ArrayFirst(params)));
    be->SwitchContinue();
    ArrayForEach(sv->concrete_classes, class)
    {
      be->SwitchCase(ClassTypeTag(class));
      be->SwitchActionBegin();
      be->VarAsstBegin(return_be_obj);
      be->CallBegin(name_in_class(class));
      for( i = 0;  i < ArraySize(params);  i++ )
      {
	param = ArrayGet(params, i);
	param_type = ArrayGet(param_types, i);
	if( i > 0 )
	  be->CallContinue(name_in_class(class), i);
	if( param_type == be->charp_type )
	  Var(param);
	else if( ClassIsEnum(class) )
	  /* ClassEnumCodeGenTrieAccess(class, param, be); */
	  Cast(ClassBEContentType(class), NULL, Var(param));
	else
	  Cast(ClassBEType(class), NULL, Var(param));
      }
      be->CallEnd(name_in_class(class));
      be->AsstEnd();
      be->SwitchActionEnd(return_be_obj != be->ret);
    }
    be->SwitchCase(NULL);  /* default case */
    be->SwitchActionBegin();
    Stmt(Call4(be->fprintf_fn, Var(be->stderr_fn),
      String("%s failing on %u\\n"), String(be->VarShow(be_obj)),
    Call1(NPBack_Type_Tag, Var(ArrayFirst(params)))));
    be->Fail();
    be->SwitchActionEnd(FALSE);
    be->SwitchEnd();
  }
  be->FunctionEnd(be_obj);
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemViewCodeGenInit(SYSTEM_VIEW sv, CODEGEN be)                   */
/*                                                                           */
/*  Initialize sv for code generation.                                       */
/*                                                                           */
/*****************************************************************************/

void SystemViewCodeGenInit(SYSTEM_VIEW sv, CODEGEN be)
{
  USTRING name = NameRep(sv->name);
  static USTRING init_str = NULL;
  static USTRING load_str = NULL;
  static USTRING swizzle_str = NULL;
  static USTRING unswizzle_str = NULL;
  if( init_str == NULL )
  {
    init_str = AStringToUString("_init");
    load_str = AStringToUString("_load");
    swizzle_str = AStringToUString("swizzle");;
    unswizzle_str = AStringToUString("unswizzle");;
  }

  /* backend init file, part of init executable only */
  sv->be_init_file = be->FileMake(UStringCat(name, init_str), FALSE, FALSE);
  be->ExecAddFile(SystemBEInitExec(sv->system), sv->be_init_file);

  /* backend load file, part of load executable only */
  sv->be_load_file = be->FileMake(UStringCat(name, load_str), FALSE, FALSE);
  be->ExecAddFile(SystemBELoadExec(sv->system), sv->be_load_file);

  /* backend file, part of both executables */
  sv->be_file = be->FileMake(name, FALSE, FALSE);
  be->ExecAddFile(SystemBEInitExec(sv->system), sv->be_file);
  be->ExecAddFile(SystemBELoadExec(sv->system), sv->be_file);

  /* swizzle functions */
  sv->swizzle_be_obj = be->FunctionMake2(name, swizzle_str, TRUE, TRUE);
  sv->unswizzle_be_obj = be->FunctionMake2(name, unswizzle_str, TRUE, TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemViewCodeGenTypeDefs(SYSTEM_VIEW sv, CODEGEN be)               */
/*                                                                           */
/*  Code gen the type defs originating in sv.                                */
/*                                                                           */
/*****************************************************************************/

void SystemViewCodeGenTypeDefs(SYSTEM_VIEW sv, CODEGEN be)
{
  char buff[100];  CLASS class;
  sprintf(buff, "typedefs originating in module %s", NameShow(sv->name));
  be->CommentSmall(buff, TRUE, FALSE);
  ArrayForEach(sv->orig_classes, class)
    ClassTypeDefPrint(class, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemViewCodeGenTypeTags(SYSTEM_VIEW sv, CODEGEN be)               */
/*                                                                           */
/*  Code gen the type tags originating in sv.                                */
/*                                                                           */
/*****************************************************************************/

void SystemViewCodeGenTypeTags(SYSTEM_VIEW sv, CODEGEN be)
{
  char buff[100];  CLASS class;
  sprintf(buff, "type tags originating in module %s", NameShow(sv->name));
  be->CommentSmall(buff, TRUE, FALSE);
  ArrayForEach(sv->orig_classes, class)
    if( ClassHasTypeTag(class) )
      ClassTypeTagPrint(class, be);
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemViewCodeGenTypeStructs(SYSTEM_VIEW sv, CODEGEN be)            */
/*                                                                           */
/*  Code gen the type structs originating in sv.                             */
/*                                                                           */
/*****************************************************************************/

void SystemViewCodeGenTypeStructs(SYSTEM_VIEW sv, CODEGEN be)
{
  char buff[100];  CLASS class;
  sprintf(buff, "structs originating in module %s", NameShow(sv->name));
  be->CommentSmall(buff, TRUE, FALSE);
  ArrayForEach(sv->orig_classes, class)
    ClassTypeStructPrint(class, be);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewCodeGenInitFile(SYSTEM_VIEW sv, CODEGEN be)            */
/*                                                                           */
/*  Generate code for this system view's init file onto be.                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewCodeGenInitFile(SYSTEM_VIEW sv, CODEGEN be)
{
  CLASS class;  BOOLEAN control;  ARRAY_CODEGEN_OBJ params;
  ARRAY_CODEGEN_TYPE param_types;
  if( DEBUG3 )
    fprintf(stderr, "[ SystemViewCodeGenInitFile(%s)\n",
      NameShow(SystemViewName(sv)));

  /* open the file */
  be->FileBegin(sv->be_init_file, NPC_VERSION, TimingTimeStamp());

  /* add #includes */
  SystemModuleInitFileHeaderIncludes(sv->system, be);
  /* be->CodeFileInclude(sv->be_file); included with next call */
  SystemModuleInitFileCodeIncludes(sv->system, be);
  be->CodeFileInclude(sv->be_init_file);

  /* print class functions */
  ArrayForEach(sv->orig_classes, class)
    ClassCodeGenFunctionsInitFile(class, be);

  /* header */
  control = FALSE;
  comment(be, TRUE, TRUE, &control, "Module init functions", NULL);

  /* module_swizzle */
  ArrayInit(&params);
  ArrayInit(&param_types);
  ArrayAddLast(params, NPBack_Self);
  ArrayAddLast(param_types, ClassBEType(ClassObject));
  ArrayAddLast(params, NPBack_Other);
  ArrayAddLast(param_types, ClassBEType(ClassObject));
  SystemViewCodeGenModuleFunction(sv, be->int_type, sv->swizzle_be_obj,
    params, param_types, &ClassSwizzleFn, be);

  /* close file */
  be->FileEnd(sv->be_init_file);
  db_return(DEBUG3, "SystemViewCodeGenInitFile", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewCodeGenLoadFile(SYSTEM_VIEW sv, CODEGEN be)            */
/*                                                                           */
/*  Generate code for this system view's load file onto be.                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewCodeGenLoadFile(SYSTEM_VIEW sv, CODEGEN be)
{
  CLASS class;  BOOLEAN control;
  ARRAY_CODEGEN_OBJ params;  ARRAY_CODEGEN_TYPE param_types;
  if( DEBUG3 )
    fprintf(stderr, "[ SystemViewCodeGenLoadFile(%s)\n",
      NameShow(SystemViewName(sv)));

  /* open the file */
  be->FileBegin(sv->be_load_file, NPC_VERSION, TimingTimeStamp());

  /* add #includes */
  SystemModuleLoadFileHeaderIncludes(sv->system, be);
  be->CodeFileInclude(sv->be_file);
  be->CodeFileInclude(sv->be_load_file);
  SystemModuleLoadFileCodeIncludes(sv->system, be);

  /* print class load functions */
  ArrayForEach(sv->orig_classes, class)
    ClassCodeGenFunctionsLoadFile(class, be);

  /* print module functions */
  control = FALSE;
  comment(be, TRUE, TRUE, &control, "Module load functions", NULL);

  /* module_unswizzle */
  ArrayInit(&params);
  ArrayInit(&param_types);
  ArrayAddLast(params, NPBack_Self);
  ArrayAddLast(param_types, ClassBEType(ClassObject));
  ArrayAddLast(params, NPBack_Mem);
  ArrayAddLast(param_types, be->charp_type);
  SystemViewCodeGenModuleFunction(sv, be->int_type, sv->unswizzle_be_obj,
    params, param_types, &ClassUnSwizzleFn, be);

  /* close file */
  be->FileEnd(sv->be_load_file);
  db_return(DEBUG3, "SystemViewCodeGenLoadFile", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SystemViewCodeGenMainFile(SYSTEM_VIEW sv, CODEGEN be)            */
/*                                                                           */
/*  Generate code for this system view's main file onto be.                  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN SystemViewCodeGenMainFile(SYSTEM_VIEW sv, CODEGEN be)
{
  CLASS class;
  if( DEBUG3 )
    fprintf(stderr, "[ SystemViewCodeGenMainFile(%s)\n",
      NameShow(SystemViewName(sv)));

  /* open the file */
  be->FileBegin(sv->be_file, NPC_VERSION, TimingTimeStamp());

  /* add #includes */
  SystemModuleMainFileHeaderIncludes(sv->system, be);
  SystemModuleMainFileCodeIncludes(sv->system, be);

  /* print class functions (including creation and clone) */
  ArrayForEach(sv->orig_classes, class)
    ClassCodeGenFunctionsMainFile(class, be);

  /* print module functions */
  /* (none at present) */

  /* close file */
  be->FileEnd(sv->be_file);
  db_return(DEBUG3, "SystemViewCodeGenMainFile", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "debug".                                                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void SystemViewDebug(SYSTEM_VIEW sv, CONTEXT cxt, FILE *fp,              */
/*    int print_style)                                                       */
/*                                                                           */
/*  Debug print of sv onto fp with the given print_style.                    */
/*                                                                           */
/*****************************************************************************/

void SystemViewDebug(SYSTEM_VIEW sv, CONTEXT cxt, FILE *fp, int print_style)
{
  CLASS_VIEW cv;  USTRING key;  NAMED named;  BOOLEAN found;
  SYSTEM_VIEW sv2;

  /* header line including names of other views incorporated */
  fprintf(fp, "[ system view %s", NameFullShow(sv->name));
  found = FALSE;
  ArrayForEach(sv->view_array, sv2)
  {
    fprintf(fp, found ? ", %s" : " (incorporating %s",
      NameShow(SystemViewName(sv2)));
    found = TRUE;
  }
  if( found )
    fprintf(fp, ")");
  begin_indent;

  /* debug class views */
  SymForEach(sv->class_views, &key, &cv)
  {
    next_line;
    ClassViewDebug(cv, cxt, fp, print_style);
  }

  /* debug operator table */
  next_line;
  fprintf(fp, "[ operator table:");
  begin_indent;
  SymForEach(sv->op_table, &key, &named)
  {
    next_line;
    fprintf(fp, "%s: %s", UStringToUTF8(key), NameFullShow(NamedName(named)));
  }
  end_indent;
  next_line;
  fprintf(fp, "]");

  /* wrapup */
  end_indent;
  next_line;
  fprintf(fp, "] end system view %s", NameShow(sv->name));
}


/*****************************************************************************/
/*                                                                           */
/*  void SystemViewDebugSystemViewNames(SYSTEM_VIEW sv, SYSTEM_VIEW curr_sv, */
/*    FILE *fp)                                                              */
/*                                                                           */
/*  Debug print of the names of the system views incorporated into sv.       */
/*  If any are equal to curr_sv, indicate that in the print.                 */
/*                                                                           */
/*****************************************************************************/

void SystemViewDebugSystemViewNames(SYSTEM_VIEW sv, SYSTEM_VIEW curr_sv,
  FILE *fp)
{
  SYSTEM_VIEW sv2;
  fprintf(fp, "[ modules in %s:\n", NameShow(SystemViewName(sv)));
  ArrayForEach(sv->view_array, sv2)
    fprintf(fp, "  %s%s\n", NameShow(SystemViewName(sv2)),
      sv2 == curr_sv ? " (current)" : "");
  fprintf(fp, "]\n");
}
