/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         trie.c                                                     */
/*  DESCRIPTION:  Four-level trie.                                           */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <limits.h>
#include "trie.h"
#include "memory.h"
#include "array.h"
#define DEBUG1 0
#define DEBUG2 0

/*****************************************************************************/
/*                                                                           */
/*  INIT_TRIE - an initializer for a trie.                                   */
/*                                                                           */
/*  Even the initializer, which does not have to be efficient, has to        */
/*  avoid an array with one element for each possible key, since keys are    */
/*  (at least potentially) 32-bit quantities; so we use a trie for it too.   */
/*                                                                           */
/*****************************************************************************/

typedef struct init_node_rec {
  short	elements[256];
  int	ref_count;
  int	unique_id;
} *INIT_NODE;

typedef ARRAY(INIT_NODE) ARRAY_INIT_NODE;
typedef ARRAY(ARRAY_INIT_NODE) ARRAY_ARRAY_INIT_NODE;

typedef struct init_elem_rec {
  int	unique_id;
  void	*value;
} *INIT_ELEM;

typedef ARRAY(INIT_ELEM) ARRAY_INIT_ELEM;

struct init_trie_rec {
  int		elem_size;				/* size of elements  */
  unsigned int	max_key;				/* largest key       */
  int		(*compar)(const void *, const void *);	/* element compare   */
  void		(*value_debug)(void *, FILE *);		/* element debug     */
  void		*default_value;				/* default value     */
  ARRAY_ARRAY_INIT_NODE	init_nodes;
  ARRAY_INIT_ELEM	init_elems;
};


/*****************************************************************************/
/*                                                                           */
/*  TRIE - a four-level trie.                                                */
/*                                                                           */
/*****************************************************************************/

struct trie_node {
  short elements[256];
};

struct trie_rec {
  int			node_count;			/* number of nodes   */
  int			elem_size;			/* size of elements  */
  unsigned int		max_key;			/* largest key       */
  char			*first_elem;			/* first element     */
  struct trie_node 	nodes[1];			/* trie nodes        */
};


/*****************************************************************************/
/*                                                                           */
/*  Submodule InitNode - one node of the initializing trie.                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  INIT_NODE InitNodeNew(short elem, int ref_count)                         */
/*                                                                           */
/*  Return a new initial trie node with these attributes.                    */
/*                                                                           */
/*****************************************************************************/

static INIT_NODE InitNodeNew(short elem, int ref_count)
{
  INIT_NODE res;  int i;
  GetMemory(res, INIT_NODE);
  for( i = 0;  i < 256;  i++ )
    res->elements[i] = elem;
  res->ref_count = ref_count;
  res->unique_id = 0;  /* set up later */
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  INIT_NODE InitNodeCopy(INIT_NODE node, INIT_TRIE trie, int level)        */
/*                                                                           */
/*  Return a fresh copy of this node, with reference count set to 1.         */
/*  Make sure the reference counts of everything node points to are          */
/*  increased by 1, unless level is 3.                                       */
/*                                                                           */
/*****************************************************************************/

static INIT_NODE InitNodeCopy(INIT_NODE node, INIT_TRIE trie, int level)
{
  INIT_NODE res;  int i;
  ARRAY_INIT_NODE level_nodes;  INIT_NODE n;

  /* make a copy of node */
  GetMemory(res, INIT_NODE);
  for( i = 0;  i < 256;  i++ )
    res->elements[i] = node->elements[i];
  res->ref_count = 1;
  res->unique_id = 0;  /* set up later */

  /* increment the reference counts of all nodes pointed to */
  if( level < 3 )
  {
    level_nodes = ArrayGet(trie->init_nodes, level + 1);
    for( i = 0;  i < 256;  i++ )
    {
      n = ArrayGet(level_nodes, res->elements[i]);
      n->ref_count++;
    }
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int InitNodeCmp(const void *t1, const void *t2)                          */
/*                                                                           */
/*  Compare two INIT_NODE objects for equality in such a way that equal      */
/*  nodes are brought together.                                              */
/*                                                                           */
/*****************************************************************************/

static int InitNodeCmp(const void *t1, const void *t2)
{
  INIT_NODE in1 = * (INIT_NODE *) t1;
  INIT_NODE in2 = * (INIT_NODE *) t2;
  int i;
  for( i = 0;  i < 256;  i++ )
    if( in1->elements[i] != in2->elements[i] )
      return in1->elements[i] - in2->elements[i];
  return 0;
  
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN InitNodeEqual(INIT_NODE n1, INIT_NODE n2)                        */
/*                                                                           */
/*  Return TRUE if n1 and n2 have equal value.                               */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN InitNodeEqual(INIT_NODE n1, INIT_NODE n2)
{
  INIT_NODE in1 = n1;
  INIT_NODE in2 = n2;
  return InitNodeCmp(&in1, &in2) == 0;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule InitElem - one element of the initializing trie.               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  INIT_ELEM InitElemNew(void *value)                                       */
/*                                                                           */
/*  Return a INIT_ELEM with the given value.                                 */
/*                                                                           */
/*****************************************************************************/

static INIT_ELEM InitElemNew(void *value)
{
  INIT_ELEM res;
  GetMemory(res, INIT_ELEM);
  res->unique_id = 0;  /* set up later */
  res->value = value;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int InitElemCmp(const void *t1, const void *t2)                          */
/*                                                                           */
/*  Compare two INIT_ELEM objects based on comparing their values using      */
/*  static_compare, so as to bring records with equal values together.       */
/*                                                                           */
/*****************************************************************************/

static int (*static_compar)(const void *, const void *);

static int InitElemCmp(const void *t1, const void *t2)
{
  INIT_ELEM ne1 = * (INIT_ELEM *) t1;
  INIT_ELEM ne2 = * (INIT_ELEM *) t2;
  return static_compar(&ne1->value, &ne2->value);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN InitElemEqual(INIT_ELEM e1, INIT_ELEM e2)                        */
/*                                                                           */
/*  Return TRUE if e1 and e2 have equal value, again using static_compar.    */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN InitElemEqual(INIT_ELEM e1, INIT_ELEM e2)
{
  return static_compar(&e1->value, &e2->value) == 0;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule Trie.                                                          */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  INIT_TRIE TrieNew(size_t size, int(*compar)(const void *, const void *), */
/*    void *default_value)                                                   */
/*                                                                           */
/*  Return a new, empty four-level trie initializer.  Elements of the trie   */
/*  will have the given size, and when an array of them is sorted by compar, */
/*  identical ones will be brought together.  The default value if no        */
/*  element is given a particular index is default_value.                    */
/*                                                                           */
/*****************************************************************************/

INIT_TRIE TrieNew(size_t size, int(*compar)(const void *, const void *),
  void(*value_debug)(void *, FILE *), void *default_value)
{
  INIT_TRIE res;  ARRAY_INIT_NODE nodes;  int level, ref_count;

  /* make the trie object and initialize size etc. fields */
  GetMemory(res, INIT_TRIE);
  res->elem_size = size;
  if( res->elem_size % sizeof(int) != 0 )
    res->elem_size += sizeof(int) - res->elem_size % sizeof(int);
  assert(res->elem_size % sizeof(int) == 0);
  res->max_key = 0;
  res->compar = compar;
  res->value_debug = value_debug;
  res->default_value = default_value;

  /* build initial 4-level trie */
  ArrayInit(&res->init_nodes);
  for( level = 0;  level <= 3;  level++ )
  {
    ref_count = (level == 0 ? 1 : 256);
    ArrayInit(&nodes);
    ArrayAddLast(nodes, InitNodeNew(0, ref_count));
    ArrayAddLast(res->init_nodes, nodes);
  }

  /* only element is element 0, the default value */
  ArrayInit(&res->init_elems);
  ArrayAddLast(res->init_elems, InitElemNew(default_value));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  short insert(INIT_TRIE trie, int level, short index, short keys[],       */
/*    short new_val)                                                         */
/*                                                                           */
/*  Helper function for TrieInsert which does one level of the insertion     */
/*  and recurses.  Its parameters are:                                       */
/*                                                                           */
/*     trie       The trie we are inserting into                             */
/*     level      The level we are currently at, from 0 to 3 inclusive       */
/*     index      The index in trie->init_nodes[level] of the insert node    */
/*     keys       The trie key, split into 4 elements, one for each level    */
/*     new_val    The value to be inserted at level 3                        */
/*                                                                           */
/*  Any node requiring updating that is shared by more than one parent (as   */
/*  indicated by its reference count) must be copied before updating.  This  */
/*  will increase the reference counts of the nodes it points to, if it      */
/*  points to nodes (i.e. if its level is less than 3).                      */
/*                                                                           */
/*****************************************************************************/

static short insert(INIT_TRIE trie, int level, short index, short keys[],
  short new_val)
{
  ARRAY_INIT_NODE level_nodes;  INIT_NODE node;  short curr_val;
  level_nodes = ArrayGet(trie->init_nodes, level);
  node = ArrayGet(level_nodes, index);
  curr_val = node->elements[keys[level]];
  assert(curr_val >= 0);
  if( level < 3 )
    new_val = insert(trie, level+1, curr_val, keys, new_val);
  if( new_val != curr_val )
  {
    if( node->ref_count > 1 )
    {
      node->ref_count--;
      node = InitNodeCopy(node, trie, level);
      if( ArraySize(level_nodes) >= SHRT_MAX )
      {
	fprintf(stderr, "TrieInsert: too many nodes at level %d\n", level);
	exit(1);
      }
      ArrayAddLast(level_nodes, node);
      index = ArraySize(level_nodes) - 1;
    }
    node->elements[keys[level]] = new_val;
  }
  return index;
}


/*****************************************************************************/
/*                                                                           */
/*  void TrieInsert(INIT_TRIE trie, unsigned int first_key,                  */
/*    unsigned int last_key, void *value)                                    */
/*                                                                           */
/*  Associate every key in trie in the range first_key .. last_key with      */
/*  value.  Any existing associations for keys in this range are silently    */
/*  overwritten.  It is done this way because initially every key has the    */
/*  default value.                                                           */
/*                                                                           */
/*****************************************************************************/
#define index0(key) (((key) >> 24) & 0x00FF)
#define index1(key) (((key) >> 16) & 0x00FF)
#define index2(key) (((key) >>  8) & 0x00FF)
#define index3(key) ((key)         & 0x00FF)

void TrieInsert(INIT_TRIE trie, unsigned int first_key,
  unsigned int last_key, void *value)
{
  short keys[4], root;  unsigned int key;
  int db_init_nodes_count1, db_init_nodes_count2;
  ARRAY_INIT_NODE level_nodes;

  if( DEBUG1 )
  {
    fprintf(stderr, "TrieInsert(trie, 0x%X, 0x%X, -)\n", first_key, last_key);
    db_init_nodes_count1 = 0;
    ArrayForEach(trie->init_nodes, level_nodes)
      db_init_nodes_count1 += ArraySize(level_nodes);

  }

  /* keep a record of the largest key inserted */
  if( last_key > trie->max_key )
    trie->max_key = last_key;

  /* insert the value into the init_elems array */
  if( ArraySize(trie->init_elems) >= SHRT_MAX )
  {
    fprintf(stderr, "TrieInsert: too many elements for init_elems\n");
    exit(1);
  }
  ArrayAddLast(trie->init_elems, InitElemNew(value));

  /* do the key insertions */
  for( key = first_key;  key <= last_key;  key++ )
  {
    keys[0] = index0(key);
    keys[1] = index1(key);
    keys[2] = index2(key);
    keys[3] = index3(key);
    if( DEBUG1 && key == last_key )
      fprintf(stderr, "  insert(0, %d:%d:%d:%d, %d)\n",
	keys[0], keys[1], keys[2], keys[3], ArraySize(trie->init_elems)-1);
    root = insert(trie, 0, 0, keys, ArraySize(trie->init_elems) - 1);
  }
  if( DEBUG1 )
  {
    db_init_nodes_count2 = 0;
    ArrayForEach(trie->init_nodes, level_nodes)
      db_init_nodes_count2 += ArraySize(level_nodes);
    if( db_init_nodes_count1 != db_init_nodes_count2 )
    {
      fprintf(stderr, "  init_nodes now");
      ArrayForEach(trie->init_nodes, level_nodes)
	fprintf(stderr, " %d", ArraySize(level_nodes));
      fprintf(stderr, "\n");
    }
  }
  assert(root == 0);
}


/*****************************************************************************/
/*                                                                           */
/*  void *TrieInitRetrieve(INIT_TRIE trie, unsigned int key)                 */
/*                                                                           */
/*  Retrieve the entry with the given key from this initial trie.            */
/*                                                                           */
/*****************************************************************************/

void *TrieInitRetrieve(INIT_TRIE trie, unsigned int key)
{
  short keys[4], index, level;
  ARRAY_INIT_NODE level_nodes;  INIT_NODE node;
  keys[0] = index0(key);
  keys[1] = index1(key);
  keys[2] = index2(key);
  keys[3] = index3(key);
  level_nodes = ArrayGet(trie->init_nodes, 0);
  node = ArrayGet(level_nodes, 0);
  for( level = 0;  level < 3;  level++ )
  {
    index = node->elements[keys[level]];
    level_nodes = ArrayGet(trie->init_nodes, level + 1);
    node = ArrayGet(level_nodes, index);
  }
  index = node->elements[keys[3]];
  return ArrayGet(trie->init_elems, index)->value;
}


/*****************************************************************************/
/*                                                                           */
/*  void TrieSave(INIT_TRIE trie, USTRING file_name)                         */
/*                                                                           */
/*  Convert this initial trie into a real one and save its binary image      */
/*  under file_name for fast restoring later.                                */
/*                                                                           */
/*****************************************************************************/

void TrieSave(INIT_TRIE trie, USTRING file_name)
{
  ARRAY_INIT_ELEM sorted_init_elems, unique_init_elems;
  INIT_ELEM ne, prev_ne;  int i, node_count, level_num, offset, n;
  ARRAY_INIT_NODE upper_level, lower_level, sorted_level, unique_level;
  ARRAY_ARRAY_INIT_NODE unique_levels;  char *p;
  INIT_NODE node, prev_node, node2;  UTF8 utf8_filename;
  size_t node_mem, elem_mem, total_mem;
  TRIE otrie;  FILE *fp;

  if( DEBUG2 )
  {
    fprintf(stderr, "[ TrieSave(init_trie, %s)\n", UStringToUTF8(file_name));
    TrieInitDebug(trie, stderr);
  }

  /* initialize the array of levels of unique nodes */
  ArrayInit(&unique_levels);
  for( i = 0;  i <= 3;  i++ )
  {
    ArrayInit(&unique_level);
    ArrayAddLast(unique_levels, unique_level);
  }

  /* uniqueify the new elements and renumber them */
  ArrayCopy(sorted_init_elems, trie->init_elems);
  static_compar = trie->compar;
  ArraySort(sorted_init_elems, &InitElemCmp);
  ArrayInit(&unique_init_elems);
  prev_ne = NULL;
  ArrayForEach(sorted_init_elems, ne)
  {
    if( prev_ne == NULL || !InitElemEqual(prev_ne, ne) )
    {
      ArrayAddLast(unique_init_elems, ne);
      prev_ne = ne;
    }
    ne->unique_id = ArraySize(unique_init_elems) - 1;
  }

  /* change the indexes out of level 3 to uniqueified indexes */
  upper_level = ArrayGet(trie->init_nodes, 3);
  ArrayForEach(upper_level, node)
    for( i = 0;  i < 256;  i++ )
    {
      ne = ArrayGet(trie->init_elems, node->elements[i]);
      node->elements[i] = ne->unique_id;
    }

  /* uniquify level level_num and change indexes at level level_num - 1 */
  node_count = 0;
  for( level_num = 3;  level_num >= 1;  level_num-- )
  {
    lower_level = ArrayGet(trie->init_nodes, level_num);
    upper_level = ArrayGet(trie->init_nodes, level_num - 1);

    /* uniqueify lower level and renumber its nodes */
    ArrayCopy(sorted_level, lower_level);
    ArraySort(sorted_level, &InitNodeCmp);
    unique_level = ArrayGet(unique_levels, level_num);
    prev_node = NULL;
    ArrayForEach(sorted_level, node)
    {
      if( prev_node == NULL || !InitNodeEqual(prev_node, node) )
      {
	ArrayAddLast(unique_level, node);
	prev_node = node;
      }
      node->unique_id = ArraySize(unique_level) - 1;
    }
    node_count += ArraySize(unique_level);

    /* change the indexes out of upper level to uniqueified indexes */
    ArrayForEach(upper_level, node)
      for( i = 0;  i < 256;  i++ )
      {
	node2 = ArrayGet(lower_level, node->elements[i]);
	node->elements[i] = node2->unique_id;
      }
  }

  /* uniqueify level 0 and renumber its nodes (but in fact only one node) */
  lower_level = ArrayGet(trie->init_nodes, 0);
  assert(ArraySize(lower_level) == 1);
  unique_level = ArrayGet(unique_levels, 0);
  ArrayAddLast(unique_level, ArrayFirst(lower_level));
  node_count += ArraySize(unique_level);

  /* work out how much memory is required for the optimized trie */
  node_mem = node_count * sizeof(struct trie_node);
  elem_mem = ArraySize(unique_init_elems) * trie->elem_size;
  total_mem = node_mem + elem_mem + 3*sizeof(int) + sizeof(char *);
  otrie = (TRIE) malloc(total_mem);
  otrie->node_count = node_count;
  otrie->elem_size = trie->elem_size;
  otrie->max_key = trie->max_key;
  otrie->first_elem =
    ((char *) otrie) + 3*sizeof(int) + sizeof(char *) + node_mem;

  /* copy the trie nodes into their part, with adjusted offsets */
  n = 0;
  offset = 0;
  for( level_num = 0;  level_num <= 3;  level_num++ )
  {
    lower_level = ArrayGet(unique_levels, level_num);
    offset = level_num < 3 ? offset + ArraySize(lower_level) : 0;
    ArrayForEach(lower_level, node)
    {
      for( i = 0;  i < 256;  i++ )
      {
	if( node->elements[i] + offset > SHRT_MAX )
	{
	  fprintf(stderr, "TrieSave: otrie->nodes[%d].elements[%d] too large\n",
	    n, i);
	  exit(1);
	}
	otrie->nodes[n].elements[i] = node->elements[i] + offset;
      }
      n++;
    }
  }

  /* copy the elements into their part */
  p = otrie->first_elem;
  ArrayForEach(unique_init_elems, ne)
  {
    memcpy(p, ne->value, trie->elem_size);
    p += trie->elem_size;
  }
  
  /* open the file and write it */
  utf8_filename = UStringToUTF8(file_name);
  fp = fopen((ASTRING) utf8_filename, "wb");
  if( fp == NULL )
  {
    fprintf(stderr, "cannot save trie to file %s\n", utf8_filename);
    exit(1);
  }
  fwrite(otrie, total_mem, 1, fp);
  fclose(fp);

  if( DEBUG2 )
    fprintf(stderr, "] TrieSave returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  TRIE TrieRestore(USTRING file_name)                                      */
/*                                                                           */
/*  Load the trie stored at *file_name, or return NULL if file does not      */
/*  open.                                                                    */
/*                                                                           */
/*****************************************************************************/

TRIE TrieRestore(USTRING file_name)
{
  FILE *fp;  size_t total_mem;  TRIE res;

  /* open the file and get its length */
  fp = fopen((ASTRING) UStringToUTF8(file_name), "rb");
  if( fp == NULL )
    return NULL;
  fseek(fp, 0, SEEK_END);
  total_mem = ftell(fp);
  rewind(fp);

  /* read in the file */
  res = malloc(total_mem);
  fread(res, total_mem, 1, fp);
  fclose(fp);

  /* set the sole pointer in the record and return */
  res->first_elem = ((char *) res) + 3*sizeof(int) + sizeof(char *) +
    res->node_count * sizeof(struct trie_node);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  unsigned int TrieMaxKey(TRIE trie)                                       */
/*                                                                           */
/*  Return the largest key that was inserted into trie.                      */
/*                                                                           */
/*****************************************************************************/

unsigned int TrieMaxKey(TRIE trie)
{
  return trie->max_key;
}


/*****************************************************************************/
/*                                                                           */
/*  void *TrieRetrieve(TRIE trie, unsigned int key)                          */
/*                                                                           */
/*  Retrieve the void * value associated with key in trie.  Every key        */
/*  has an associated value; it is the default value if not specifically     */
/*  inserted.                                                                */
/*                                                                           */
/*****************************************************************************/

void *TrieRetrieve(TRIE trie, unsigned int key)
{
  int index;
  index = trie->nodes[0].elements[index0(key)];
  index = trie->nodes[index].elements[index1(key)];
  index = trie->nodes[index].elements[index2(key)];
  index = trie->nodes[index].elements[index3(key)];
  return trie->first_elem + index * trie->elem_size;
}


/*****************************************************************************/
/*                                                                           */
/*  void TrieInitLastLevelRangeDebug(INIT_TRIE trie, INIT_NODE n,            */
/*    int from_key, int from_index, int to_index, FILE *fp)                  */
/*                                                                           */
/*  Print one range of equal objects from the last level of trie.            */
/*                                                                           */
/*****************************************************************************/

static void TrieInitLastLevelRangeDebug(INIT_TRIE trie, INIT_NODE n,
  int from_key, int from_index, int to_index, FILE *fp)
{
  INIT_ELEM ie = ArrayGet(trie->init_elems, n->elements[from_index]);
  if( from_index == to_index )
    fprintf(fp, "%*s0x%X: id%d ", 8, "", from_key + from_index, ie->unique_id);
  else
    fprintf(fp, "%*s0x%X .. 0x%X: id%d ", 8, "", from_key + from_index,
      from_key + to_index, ie->unique_id);
  trie->value_debug(ie->value, fp);
  fprintf(fp, "\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void TrieInitNodeDebug(INIT_TRIE trie, int node, int prev_node,          */
/*    int from_key, int to_key, int level, int inc, FILE *fp)                */
/*                                                                           */
/*  Debug print of trie node "node" and its subtree.                         */
/*                                                                           */
/*****************************************************************************/

static void TrieInitNodeDebug(INIT_TRIE trie, int node, int prev_node,
  int from_key, int to_key, int level, int inc, FILE *fp)
{
  int i, prev_i, sub_node, prev_sub_node, key;
  ARRAY_INIT_NODE level_nodes;  INIT_NODE n;
  fprintf(fp, "%*s0x%X..0x%X: ", level * 2, "", from_key, to_key);

  /* if copy of previous node, just say so and return */
  if( node == prev_node )
  {
    fprintf(fp, "ditto\n");
    return;
  }

  /* get node record and print header */
  level_nodes = ArrayGet(trie->init_nodes, level);
  n = ArrayGet(level_nodes, node);
  fprintf(fp, "[ <%d;%d> ref%d id%d\n", level, node, n->ref_count,
    n->unique_id);

  if( level < 3 )
  {
    /* ordinary node */
    prev_sub_node = -1;
    key = from_key;
    for( i = 0;  i < 256 && key <= to_key;  i++ )
    {
      sub_node = n->elements[i];
      TrieInitNodeDebug(trie, sub_node, prev_sub_node, key, key + inc - 1,
	level + 1, inc / 256, fp);
      prev_sub_node = sub_node;
      key += inc;
    }
  }
  else
  {
    /* last level nodes, holding pointers to values */
    assert(inc == 1);
    key = from_key;
    prev_i = 0;
    for( i = 1;  i < 256;  i++ )
    {
      sub_node = n->elements[i];
      if( n->elements[i] != n->elements[prev_i] )
      {
        TrieInitLastLevelRangeDebug(trie, n, from_key, prev_i, i - 1, fp);
	prev_i = i;
      }
    }
    TrieInitLastLevelRangeDebug(trie, n, from_key, prev_i, i - 1, fp);
  }

  /* print footer and exit */
  fprintf(fp, "%*s]\n", level * 2, "");
}


/*****************************************************************************/
/*                                                                           */
/*  void TrieInitDebug(INIT_TRIE trie, FILE *fp)                             */
/*                                                                           */
/*  Debug print of initializing trie onto fp.                                */
/*                                                                           */
/*****************************************************************************/

void TrieInitDebug(INIT_TRIE trie, FILE *fp)
{
  TrieInitNodeDebug(trie, 0, -1, 0, trie->max_key, 0, 1 << 24, fp);
}


/*****************************************************************************/
/*                                                                           */
/*  void TrieDebug(TRIE trie, FILE *fp)                                      */
/*                                                                           */
/*  Debug print of trie onto fp.                                             */
/*                                                                           */
/*****************************************************************************/

void TrieDebug(TRIE trie, FILE *fp)
{
  /* still to do */
}
