/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         type.c                                                     */
/*  DESCRIPTION:  Nonpareil actual types and type variables                  */
/*                                                                           */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "externs.h"
#define	NO_SWIZZLE -2
#define	NEED_SWIZZLE -1
#define FAIL_ON_NOT_KNOWN 0

#define DEBUG1 0
#define DEBUG2 0
#define DEBUG3 0
#define DEBUG4 0
#define DEBUG5 0
#define DEBUG6 0
#define DEBUG7 0
#define DEBUG8 0
#define DEBUG9 0
#define DEBUG10 0
#define DEBUG11 0
#define DEBUG12 0
#define DEBUG13 0
#define DEBUG14 0
#define DEBUG15 0
#define DEBUG16 0
#define DEBUG17 0
#define DEBUG18 0
#define DEBUG19 0
#define DEBUG20 0
#define DEBUG21 0
#define DEBUG22 0
#define DEBUG23 0
#define DEBUG24 0
#define DEBUG25 0
#define DEBUG26 0
#define DEBUG27 0
#define DEBUG28 0
#define DEBUG29 0
#define DEBUG30 0
#define DEBUG31 0

/* static BOOLEAN DebugOn = FALSE; */


/*****************************************************************************/
/*                                                                           */
/*  TYPE, TYPE_VAR, and TYPE_VARS                                            */
/*                                                                           */
/*  These three types are exported to the compiler as a whole by this        */
/*  module.  TYPE represents an actual type, TYPE_VAR is a type variable,    */
/*  optionally constrained, and TYPE_VARS is a sequence of type variables.   */
/*                                                                           */
/*  A TYPE is either a variable or else it is a meet of one or more class    */
/*  types.  No type value may ever be NULL, although some type fields are    */
/*  optional (notably upper and lower constraints), so they can be null.     */
/*                                                                           */
/*  There is no way to go directly from a TYPE to the set of variables it    */
/*  uses, but these sets are known; for types stored in classes, their       */
/*  variables are in mc->type_vars, and so on.                               */
/*                                                                           */
/*****************************************************************************/
#define IsVar(type) ((type)->kind_tag == KIND_TYPEV)

typedef enum { Ordinary, Unify, Range } VAR_STATE;

struct type_var_rec {
  KIND_TAG	kind_tag;		/* type tag (KIND_TYPE_VAR)          */
  FILE_POS	file_pos;		/* position in input file            */
  NAME		name;			/* name of type variable             */
  int		seq_num;		/* for debugging only                */
  int		occurrences;		/* used during printing only         */
  int		swizzle_index;		/* index of swizzzle bit, if non-neg */
  TYPE		resolved_to;		/* resolved value if non-null        */
  TYPE		lower_constraint;	/* lower constraint if non-null      */
  TYPE		upper_constraint;	/* upper constraint if non-null      */
  VAR_STATE	state;			/* state of the variable             */
};

typedef struct ctype_rec *CTYPE;
typedef ARRAY(CTYPE) ARRAY_CTYPE;

struct type_rec {
  KIND_TAG	kind_tag;		/* tag (KIND_TYPEV or KIND_TYPEC)    */
  FILE_POS	file_pos;		/* position in input file            */
  union {
    TYPE_VAR	variable;		/* the variable being invoked        */
    ARRAY_CTYPE	ctypes;			/* the classes meeting               */
  } u;
};

struct ctype_rec {
  KIND_TAG	kind_tag;		/* type tag (KIND_CTYPE)             */
  FILE_POS	file_pos;		/* position in input file            */
  USTRING	uninstantiated;		/* uninstantiated class name         */
  CLASS		class;			/* the class being invoked           */
  ARRAY_TYPE	generics;		/* its actual generic parameters     */
};

struct itype_rec {
  CTYPE		ctype;			/* the CTYPE                         */
  int		weight;			/* the weight                        */
  COERCION	coercion;		/* the coercion required to get here */
};



/*****************************************************************************/
/*                                                                           */
/*  The "Resolve" submodule.                                                 */
/*                                                                           */
/*  This module contains only the resolve method.                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  static TYPE TypeResolve(TYPE t)                                          */
/*                                                                           */
/*  Return the resolved value of t; t could be NULL.                         */
/*                                                                           */
/*****************************************************************************/

static TYPE TypeResolve(TYPE t)
{
  /* debugging version */
  if( DEBUG9 )
  {
    int count = 0;
    if( DEBUG9 )
      fprintf(stderr, "[ TypeResolve()\n");
    while( count++ < 20 && t != NULL && IsVar(t) &&
	t->u.variable->resolved_to != NULL )
      t = t->u.variable->resolved_to;
    assert(count < 20);
    if( DEBUG9 )
      fprintf(stderr, "] resolve returning\n");
  }
  else

  /* somewhat faster version */
  if( t != NULL )
    while( IsVar(t) && t->u.variable->resolved_to != NULL )
      t = t->u.variable->resolved_to;

  return t;
}


/*****************************************************************************/
/*                                                                           */
/*  The "Show" submodule.                                                    */
/*                                                                           */
/*  This submodule is concerned with displaying Types and variables.         */
/*                                                                           */
/*  The functions all come with an cxt parameter that determines which       */
/*  view of the system the types are to be displayed with respect to.        */
/*  For debugging it is permissible to supply a null value for cxt, and      */
/*  then the original class names will be used.                              */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void TypeDebugOn(void)                                                   */
/*  void TypeDebugOff(void)                                                  */
/*                                                                           */
/*  Turn debugging on and off.                                               */
/*                                                                           */
/*****************************************************************************/

/* ***
void TypeDebugOn(void)
{
  DebugOn = TRUE;
}


void TypeDebugOff(void)
{
  DebugOn = FALSE;
}
*** */

static void do_type_show(TYPE type, AFACTORY af, CONTEXT cxt);
static void do_ctype_show(CTYPE ctype, AFACTORY af, CONTEXT cxt);

static void do_var_name_show(TYPE_VAR v, AFACTORY af)
{
  if( v->state == Unify )
    AStringAddChar(af, '!');
  NameAdd(v->name, af);
  if( DEBUG14 )
  {
    AStringAddChar(af, '_');
    AStringAddInt(af, v->seq_num);
  }
}

static void do_var_show(TYPE_VAR v, AFACTORY af, CONTEXT cxt)
{
  /* show variable name, preceded by ! if unification */
  do_var_name_show(v, af);

  /* show lower constraint if non-null */
  if( v->lower_constraint != NULL )
  {
    AStringAddAString(af, " isabove ");
    do_type_show(v->lower_constraint, af, cxt);
  }

  /* show upper constraint if non-null */
  if( v->upper_constraint != NULL )
  {
    AStringAddAString(af, " is ");
    do_type_show(v->upper_constraint, af, cxt);
  }

  /* show resolved_to if non-null */
  if( v->resolved_to != NULL )
  {
    AStringAddAString(af, " := ");
    do_type_show(v->resolved_to, af, cxt);
  }
}

static void do_type_array_show(ARRAY_TYPE agen, AFACTORY af, CONTEXT cxt)
{
  TYPE gen;
  if( agen != NULL )
  {
    AStringAddAString(af, "{");
    ArrayForEach(agen, gen)
    {
      if( gen != ArrayFirst(agen) )
	AStringAddAString(af, ", ");
      do_type_show(gen, af, cxt);
    }
    AStringAddAString(af, "}");
  }
}

static void do_ctype_show(CTYPE ctype, AFACTORY af, CONTEXT cxt)
{
  assert(ctype->class != NULL || ctype->uninstantiated != NULL);
  if( ctype->class == NULL )
  {
    AStringAddAString(af, "'");
    AStringAddAString(af, (ASTRING) UStringToUTF8(ctype->uninstantiated));
    AStringAddAString(af, "'");
  }
  else if( cxt == NULL )
    AStringAddAString(af,
      (ASTRING) NameShow(ClassViewName(ClassOrigClassView(ctype->class))));
  else
    NameAdd(ClassName(ctype->class, cxt), af);
  do_type_array_show(ctype->generics, af, cxt);
}

static void do_ctypes_show(ARRAY_CTYPE ctypes, AFACTORY af, CONTEXT cxt)
{
  CTYPE ctype;  BOOLEAN first;
  first = TRUE;
  ArrayForEach(ctypes, ctype)
  {
      /* separate by "meet" if not first */
      if( !first )
	AStringAddAString(af, " meet ");

      /* show the component */
      do_ctype_show(ctype, af, cxt);
      first = FALSE;
  }
}

static void do_itype_show(ITYPE itype, AFACTORY af, CONTEXT cxt)
{
  do_ctype_show(itype->ctype, af, cxt);
  if( itype->ctype->class != NULL )
  {
    AStringAddChar(af, '$');
    AStringAddInt(af, ClassSortKey(itype->ctype->class));
  }
  else
    AStringAddAString(af, "$NULL");
  if( itype->weight > 0 )
  {
    AStringAddChar(af, '[');
    AStringAddInt(af, itype->weight);
    AStringAddChar(af, ']');
  }
  if( itype->coercion != NULL )
    AStringAddFmt1(af, "[%s]", (ASTRING) CoercionShow(itype->coercion));
}

static void do_type_show(TYPE type, AFACTORY af, CONTEXT cxt)
{
  assert(type != NULL);
  type = TypeResolve(type);
  if( IsVar(type) )
  {
    if( type->u.variable->state == Range && type->u.variable->occurrences == 1 )
    {
      if( type->u.variable->lower_constraint != NULL )
	do_type_show(type->u.variable->lower_constraint, af, cxt);
      AStringAddAString(af, "..");
      if( type->u.variable->upper_constraint != NULL )
        do_type_show(type->u.variable->upper_constraint, af, cxt);
    }
    else
      do_var_name_show(type->u.variable, af);
    if( DEBUG11 )
    {
      AStringAddAString(af, " <");
      do_var_show(type->u.variable, af, cxt);
      AStringAddAString(af, ">");
    }
  }
  else
    do_ctypes_show(type->u.ctypes, af, cxt);
}

UTF8 TypeVarsShow(TYPE_VARS tv, CONTEXT cxt)
{
  TYPE_VAR v;
  if( tv == NULL )
    return (UTF8) "";
  else
  {
    AFACTORY af = AStringBegin();
    AStringAddAString(af, "{");
    ArrayForEach(tv, v)
    {
      if( v != ArrayFirst(tv) )
	AStringAddAString(af, ", ");
      do_var_show(v, af, cxt);
    }
    AStringAddAString(af, "}");
    return (UTF8) AStringEnd(af);
  }
}

static UTF8 TypeVarShow(TYPE_VAR v, CONTEXT cxt)
{
  AFACTORY af = AStringBegin();
  do_var_show(v, af, cxt);
  return (UTF8) AStringEnd(af);
}

static void TypeShowClearOccurrences(TYPE type)
{
  CTYPE ctype;  TYPE t;
  type = TypeResolve(type);
  if( IsVar(type) )
  {
    type->u.variable->occurrences = 0;
    if( type->u.variable->upper_constraint != NULL )
      TypeShowClearOccurrences(type->u.variable->upper_constraint);
    if( type->u.variable->lower_constraint != NULL )
      TypeShowClearOccurrences(type->u.variable->lower_constraint);
  }
  else
    ArrayForEach(type->u.ctypes, ctype)
      if( ctype->generics != NULL )
	ArrayForEach(ctype->generics, t)
	  TypeShowClearOccurrences(t);
}

static void TypeShowCountOccurrences(TYPE type)
{
  CTYPE ctype;  TYPE t;
  type = TypeResolve(type);
  if( IsVar(type) )
  {
    type->u.variable->occurrences++;
    if( type->u.variable->upper_constraint != NULL )
      TypeShowCountOccurrences(type->u.variable->upper_constraint);
    if( type->u.variable->lower_constraint != NULL )
      TypeShowCountOccurrences(type->u.variable->lower_constraint);
  }
  else
    ArrayForEach(type->u.ctypes, ctype)
      if( ctype->generics != NULL )
	ArrayForEach(ctype->generics, t)
	  TypeShowCountOccurrences(t);
}

UTF8 TypeShow(TYPE type, CONTEXT cxt)
{
  if( type == NULL )
    return (UTF8) "<null>";
  else
  {
    AFACTORY af = AStringBegin();
    TypeShowClearOccurrences(type);
    TypeShowCountOccurrences(type);
    do_type_show(type, af, cxt);
    return (UTF8) AStringEnd(af);
  }
}

UTF8 TypeArrayShow(ARRAY_TYPE agen, CONTEXT cxt)
{
  AFACTORY af = AStringBegin();
  do_type_array_show(agen, af, cxt);
  return (UTF8) AStringEnd(af);
}

UTF8 TypeITypesShow(ITYPES itypes, CONTEXT cxt)
{
  ITYPE itype;
  AFACTORY af = AStringBegin();
  AStringAddChar(af, '{');
  if( itypes != NULL ) ArrayForEach(itypes, itype)
  {
    if( itype != ArrayFirst(itypes) )
      AStringAddAString(af, ", ");
    do_itype_show(itype, af, cxt);
  }
  AStringAddAString(af, "}");
  return (UTF8) AStringEnd(af);
}

static UTF8 CTypeShow(CTYPE ctype, CONTEXT cxt)
{
  AFACTORY af = AStringBegin();
  do_ctype_show(ctype, af, cxt);
  return (UTF8) AStringEnd(af);
}

/* *** currently not used
static UTF8 CTypesShow(ARRAY_CTYPE ctypes, CONTEXT cxt)
{
  AFACTORY af = AStringBegin();
  do_ctypes_show(ctypes, af, cxt);
  return (UTF8) AStringEnd(af);
}
*** */

FILE_POS TypeFilePos(TYPE type)
{
  return type->file_pos;
}


/*****************************************************************************/
/*                                                                           */
/*  The "Make" submodule.                                                    */
/*                                                                           */
/*  This module makes various kinds of types.                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  CTYPE TypeMakeCType(FILE_POS file_pos, USTRING uninstantiated,           */
/*    CLASS class, ARRAY_TYPE generics)                                      */
/*                                                                           */
/*  Make a new CTYPE with these attributes.  The "uninstantiated" attribute  */
/*  is only for before the class referred to by the name is identified,      */
/*  and may be NULL if class is not.                                         */
/*                                                                           */
/*****************************************************************************/

static CTYPE TypeMakeCType(FILE_POS file_pos, USTRING uninstantiated,
  CLASS class, ARRAY_TYPE generics)
{
  CTYPE res;
  assert(class != NULL || uninstantiated != NULL);
  GetMemory(res, CTYPE);
  res->kind_tag = KIND_CTYPE;
  res->file_pos = file_pos;
  res->uninstantiated = uninstantiated;
  res->class = class;
  res->generics = generics;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeMakeFromCType(CTYPE ctype)                                      */
/*                                                                           */
/*  Make a new TYPE containing just this CTYPE.                              */
/*                                                                           */
/*****************************************************************************/

static TYPE TypeMakeFromCType(CTYPE ctype)
{
  TYPE res;
  GetMemory(res, TYPE);
  res->kind_tag = KIND_TYPEC;
  res->file_pos = ctype->file_pos;
  ArrayInit(&res->u.ctypes);
  ArrayAddLast(res->u.ctypes, ctype);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE_VAR TypeVarMake(FILE_POS file_pos, NAME name)                       */
/*                                                                           */
/*  Return a new TYPE_VAR (initially Ordinary, with empty constraints).      */
/*                                                                           */
/*****************************************************************************/

TYPE_VAR TypeVarMake(FILE_POS file_pos, NAME name)
{
  static int seq_num = 0;		/* unique seq num for each variable */
  TYPE_VAR res;
  GetMemory(res, TYPE_VAR);
  res->kind_tag = KIND_TYPE_VAR;
  res->file_pos = file_pos;
  res->name = name;
  res->seq_num = ++seq_num;
  res->occurrences = 0;
  res->swizzle_index = NO_SWIZZLE;
  res->resolved_to = NULL;
  res->lower_constraint = NULL;
  res->upper_constraint = NULL;
  res->state = Ordinary;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  NAME TypeVarName(TYPE_VAR tv)                                            */
/*                                                                           */
/*  Return the name of tv.                                                   */
/*                                                                           */
/*****************************************************************************/

NAME TypeVarName(TYPE_VAR tv)
{
  return tv->name;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeMakeFromVar(FILE_POS file_pos, TYPE_VAR v)                      */
/*                                                                           */
/*  Return a new type which is just an invocation of this variable.          */
/*                                                                           */
/*****************************************************************************/

TYPE TypeMakeFromVar(FILE_POS file_pos, TYPE_VAR v)
{
  TYPE res;
  GetMemory(res, TYPE);
  res->kind_tag = KIND_TYPEV;
  res->file_pos = file_pos;
  res->u.variable = v;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeMakeUninstantiatedClassType(FILE_POS file_pos,                  */
/*    USTRING class_name, ARRAY_TYPE generics)                               */
/*                                                                           */
/*  Make a TYPE with this one class name and its generics.                   */
/*                                                                           */
/*****************************************************************************/

TYPE TypeMakeUninstantiatedClassType(FILE_POS file_pos,
  USTRING class_name, ARRAY_TYPE generics)
{
  return TypeMakeFromCType(TypeMakeCType(file_pos, class_name, NULL, generics));
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeMakeInstantiatedClassType(FILE_POS file_pos, CLASS c,           */
/*    TYPE_VARS c_vars)                                                      */
/*                                                                           */
/*  Return a new type which is an instantiation of this class with these     */
/*  type variables.                                                          */
/*                                                                           */
/*****************************************************************************/

TYPE TypeMakeInstantiatedClassType(FILE_POS file_pos, CLASS c, TYPE_VARS c_vars)
{
  CTYPE ctype;  TYPE_VAR v;
  ctype = TypeMakeCType(file_pos, NULL, c, NULL);
  if( c_vars != NULL )
  {
    ArrayInit(&ctype->generics);
    ArrayForEach(c_vars, v)
    {
      TYPE vt = TypeMakeFromVar(file_pos, v);
      ArrayAddLast(ctype->generics, vt);
    }
  }
  return TypeMakeFromCType(ctype);
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeMakeInstantiatedClassType2(FILE_POS file_pos, CLASS c,          */
/*    ARRAY_TYPE generics)                                                   */
/*                                                                           */
/*  Like TypeMakeInstantiatedClassType except we are given the actual        */
/*  generic parameters, not variables to be made into types.                 */
/*                                                                           */
/*****************************************************************************/

TYPE TypeMakeInstantiatedClassType2(FILE_POS file_pos, CLASS c,
  ARRAY_TYPE generics)
{
  return TypeMakeFromCType(TypeMakeCType(file_pos, NULL, c, generics));
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeMakeFunctionType(ARRAY_TYPE generics, FILE_POS file_pos)        */
/*                                                                           */
/*  Return a new type which is fun{t1, t2, ... tn}.                          */
/*                                                                           */
/*****************************************************************************/

TYPE TypeMakeFunctionType(ARRAY_TYPE generics, FILE_POS file_pos)
{
  CTYPE fun_ctype;

  /* precondition check */
  assert(generics != NULL && ArraySize(generics) >= 1);
  if( ArraySize(generics)-1 > MAX_FUN_PARAMS )
  {
    /* array size minus one is number of params */
    fprintf(stderr, "%s: too many function parameters (max is %d)\n",
      FilePosShow(file_pos), MAX_FUN_PARAMS);
    exit(1);
  }

  /* make the ctype fun{t1, t2, ... tn} */
  fun_ctype = TypeMakeCType(file_pos, NULL, ClassFun[ArraySize(generics)-1],
    generics); 

  /* make it the sole occupant of res */
  return TypeMakeFromCType(fun_ctype);
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeMakeUnconstrainedRange(FILE_POS file_pos, NAME name)            */
/*                                                                           */
/*  Make an unconstrained range type.                                        */
/*                                                                           */
/*****************************************************************************/

TYPE TypeMakeUnconstrainedRange(FILE_POS file_pos, NAME name)
{
  TYPE_VAR tv;
  tv = TypeVarMake(file_pos, name);
  tv->state = Range;
  return TypeMakeFromVar(file_pos, tv);
}


/*****************************************************************************/
/*                                                                           */
/*  The "Parse" submodule.                                                   */
/*                                                                           */
/*  Parsing of types.                                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeOptionalActualGenericsParse(TOKEN *t, ARRAY_TYPE *generics)  */
/*                                                                           */
/*  Parse optional actual generic parameters and set *generics to the        */
/*  resulting array of types, or to NULL if there are no generics.           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeOptionalActualGenericsParse(TOKEN *t, ARRAY_TYPE *generics)
{
  TYPE type;
  if( LexType(curr_token) == TK_LEFT_BRACE )
  {
    /* skip opening brace and initialize result */
    next_token;
    ArrayInit(generics);

    /* parse list of types */
    if( !TypeParse(t, &type) )
      return FALSE;
    ArrayAddLast(*generics, type);
    while( LexType(curr_token) == TK_COMMA )
    {
      next_token;
      if( !TypeParse(t, &type) )
        return FALSE;
      ArrayAddLast(*generics, type);
    }

    /* skip closing right brace */
    skip(TK_RIGHT_BRACE, "comma or closing brace of actual generic parameters");
  }
  else
    *generics = NULL;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  static BOOLEAN TypeClassParse(TOKEN *t, CTYPE *res)                      */
/*                                                                           */
/*  Parse a class type setting *res to the resulting CTYPE object.           */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeClassParse(TOKEN *t, CTYPE *res)
{
  TOKEN token;  ARRAY_TYPE generics;

  /* parse the class name */
  if( !NameParse(t, &token) )
    return FALSE;

  /* parse optional generic parameters */
  if( !TypeOptionalActualGenericsParse(t, &generics) )
    return FALSE;

  /* set up result and return */
  *res = TypeMakeCType(LexFilePos(token), LexValue(token), NULL, generics);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeParse(TOKEN *t, TYPE *res)                                   */
/*                                                                           */
/*  Parse a type, setting *res to the resulting type.                        */
/*  This assumes that all names are class names; later on, if it turns       */
/*  out that a name was in fact a variable name, there will have to be       */
/*  a conversion of the type record from class type to variable type.        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeParse(TOKEN *t, TYPE *res)
{
  CTYPE class_type;

  /* parse compulsory first class type and build *res with it */
  if( !TypeClassParse(t, &class_type) )
    return FALSE;
  *res = TypeMakeFromCType(class_type);

  /* parse optional subsequent class types separated by "meet" */
  while( LexType(curr_token) == TK_MEET )
  {
    next_token;
    if( !TypeClassParse(t, &class_type) )
      return FALSE;
    ArrayAddLast((*res)->u.ctypes, class_type);
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeOptionalFormalGenericsParse(TOKEN *t, FEATURE_TYPE ftype,    */
/*    BOOLEAN is_coerce, TYPE_VARS *res)                                     */
/*                                                                           */
/*  Parse formal generic parameters and set *res to the resulting array.     */
/*  If ftype is FEATURE_CREATION, or is_coerce is TRUE, these are the        */
/*  generic parameters of a creation or coercion feature and hence there     */
/*  should be none; print an error message and return FALSE if found.        */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeParseOneFormalGeneric(TOKEN *t, TYPE_VAR *res)
{
  *res = TypeVarMake(LexFilePos(curr_token), NameNew(LexValue(curr_token)));
  skip(TK_IDENTIFIER, "identifier (name of formal generic type parameter)");
  if( LexType(curr_token) == TK_IS )
  {
    next_token;
    if( !TypeParse(t, &(*res)->upper_constraint) )
      return FALSE;
  }
  return TRUE;
}


BOOLEAN TypeOptionalFormalGenericsParse(TOKEN *t, FEATURE_TYPE ftype,
  BOOLEAN is_coerce, TYPE_VARS *res)
{
  TYPE_VAR formal;

  *res = NULL;
  if( LexType(curr_token) == TK_LEFT_BRACE )
  {
    /* error if is_creation, is_coerce, or is_predef */
    if( ftype == FEATURE_CREATION )
    {
      fprintf(stderr, "%s: creation feature has generic parameter(s)\n",
	LexPos(curr_token));
      return FALSE;
    }
    else if( is_coerce )
    {
      fprintf(stderr, "%s: coerce feature has generic parameter(s)\n",
	LexPos(curr_token));
      return FALSE;
    }
    else if( ftype == FEATURE_PREDEF )
    {
      fprintf(stderr,"%s: predefined object feature has generic parameter(s)\n",
	LexPos(curr_token));
      return FALSE;
    }

    /* skip opening left brace */
    ArrayInit(res);
    next_token;

    /* parse the formal generic parameters */
    if( !TypeParseOneFormalGeneric(t, &formal) )
      return FALSE;
    ArrayAddLast(*res, formal);
    while( LexType(curr_token) == TK_COMMA )
    {
      next_token;
      if( !TypeParseOneFormalGeneric(t, &formal) )
	return FALSE;
      ArrayAddLast(*res, formal);
    }

    /* skip closing right brace */
    skip(TK_RIGHT_BRACE, "\",\" or closing \"}\" of formal generics");
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  The "LevelOneValid" submodule.                                           */
/*                                                                           */
/*  This submodule is concerned with instantiating types, i.e. resolving     */
/*  their names to actual type variables and classes.                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeVarsBegin(TYPE_VARS agen, CONTEXT cxt)                       */
/*                                                                           */
/*  Set up generic parameters agen in cxt.  If failure, leave everything as  */
/*  it was on entry.                                                         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeVarsBegin(TYPE_VARS agen, CONTEXT cxt)
{
  TYPE_VAR v;  int i, j;
  if( DEBUG24 )
    fprintf(stderr, "[ TypeVarsBegin(%s, cxt)\n", TypeVarsShow(agen, cxt));
  if( agen != NULL )
  {
    for( i = 0;  i < ArraySize(agen);  i++ )
    {
      v = ArrayGet(agen, i);
      if( !ContextInsertGeneric(cxt, v) )
      {
	/* reset context and fail */
	for( j = i-1;  j >= 0;  j-- )
	{
	  v = ArrayGet(agen, j);
	  ContextDelete(cxt, (NAMED) v);
	}
	db_return(DEBUG24, "TypeVarsBegin", FALSE);
      }
    }
  }
  db_return(DEBUG24, "TypeVarsBegin", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  void TypeVarsEnd(TYPE_VARS agen, CONTEXT cxt)                            */
/*                                                                           */
/*  Take away generic parameters agen, cancelling a previous successful      */
/*  call to TypeVarsBegin.                                                   */
/*                                                                           */
/*****************************************************************************/

void TypeVarsEnd(TYPE_VARS agen, CONTEXT cxt)
{
  TYPE_VAR v;
  if( DEBUG24 )
    fprintf(stderr, "[ TypeVarsEnd(%s, cxt)\n", TypeVarsShow(agen, cxt));
  if( agen != NULL )
    ArrayForEachReverse(agen, v)
      ContextDelete(cxt, (NAMED) v);
  if( DEBUG24 )
    fprintf(stderr, "] TypeVarsEnd returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN CheckGenericsCount(TYPE_VARS formals, CTYPE ctype,               */
/*    CONTEXT cxt)                                                           */
/*                                                                           */
/*  Check that the number of formal generics equals the number of actual     */
/*  generics (in ctype->generics), and print an error message if not.        */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN CheckGenericsCount(TYPE_VARS formals, CTYPE ctype,
  CONTEXT cxt)
{
  int g1, g2;  NAME name;
  g1 = formals == NULL ? 0 : ArraySize(formals);
  g2 = ctype->generics == NULL ? 0 : ArraySize(ctype->generics);
  if( g1 != g2 )
  {
    fprintf(stderr, "%s: expected %d actual %s but found %d in %s\n",
      FilePosShow(ctype->file_pos), g1, g1 == 1 ? "generic" : "generics",
      g2, CTypeShow(ctype, cxt));
    name = ClassName(ctype->class, cxt);
    fprintf(stderr, "  %s\n", NameFullShow(name));
    return FALSE;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeLevelOneValid(TYPE t, CONTEXT cxt, BOOLEAN is_public)        */
/*                                                                           */
/*  Establish the Level 1 validity of type t wrt the given cxt.              */
/*  It may already be at Level 1, in which case it is not touched.           */
/*                                                                           */
/*  If is_public is TRUE, this type is part of a public interface, i.e.      */
/*  it is visible outside the module it occurs in.  In that case, all the    */
/*  classes it mentions must not be private, and this code checks for that.  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeLevelOneValid(TYPE t, CONTEXT cxt, BOOLEAN is_public)
{
  CTYPE ctype;  TYPE gt;  NAMED named;  CLASS_VIEW cv;
  if( DEBUG29 )
    fprintf(stderr, "[ TypeLevelOneValid(%s, cxt)\n", TypeShow(t, cxt));
  assert(t != NULL);

  /* uninstantiated types are stored as class types, so if t */
  /* is a variable it must already be instantiated */
  if( IsVar(t) )
    db_return(DEBUG29, "TypeLevelOneValid (var)", TRUE);
  ArrayForEach(t->u.ctypes, ctype)
  {
    /* add class or variable, error if neither, error if variable in meet */
    if( ctype->class != NULL )
    {
      /* already instantiated, do nothing */
    }
    else if( !ContextRetrieve(cxt, ctype->uninstantiated, &named) )
    {
      fprintf(stderr, "%s: %s not known\n", FilePosShow(ctype->file_pos),
	UStringToUTF8(ctype->uninstantiated));
      if( FAIL_ON_NOT_KNOWN )
	assert(FALSE);
      db_return(DEBUG29, "TypeLevelOneValid (not known)", FALSE);
    }
    else switch( NamedKind(named) )
    {
      case KIND_CLASS_VIEW:

	/* make sure class view has the right kind of privacy */
	cv = (CLASS_VIEW) named;
	if( is_public && ClassViewIsPrivate(cv) )
	{
	  fprintf(stderr, "%s: private class %s in public interface\n",
	    FilePosShow(ctype->file_pos), NameShow(ClassViewName(cv)));
	  db_return(DEBUG29, "TypeLevelOneValid (private class)", FALSE);
	}

	/* instantiate ctype and check for the right number of generics */
	ctype->class = ClassViewClass((CLASS_VIEW) named);
	if( ctype->generics != NULL ) ArrayForEach(ctype->generics, gt)
	  if( !TypeLevelOneValid(gt, cxt, is_public) )
	    db_return(DEBUG29, "TypeLevelOneValid (generic)", FALSE);
	if( !CheckGenericsCount(ClassVars(ctype->class), ctype, cxt) )
	  db_return(DEBUG29, "TypeLevelOneValid (generics count)", FALSE);
	break;

      case KIND_TYPE_VAR:

	/* make sure not part of a meet, then convert to variable invocn. */
	if( ArraySize(t->u.ctypes) > 1 )
	{
	  fprintf(stderr, "%s: type variable %s (defined at %s) in meet\n",
	    FilePosShow(ctype->file_pos), UStringToUTF8(ctype->uninstantiated),
	    FilePosShow(NamedFilePos(named)));
	  db_return(DEBUG29, "TypeLevelOneValid (var in meet)", FALSE);
	}
	t->kind_tag = KIND_TYPEV;
	t->u.variable = (TYPE_VAR) named;
	/* *** actually it can be already resolved, in a class extension 
	assert(t->u.variable->resolved_to == NULL);
	*** */
	db_return(DEBUG29, "TypeLevelOneValid (var)", TRUE);

      default:

	fprintf(stderr, "%s: %s %s (defined at %s) where type expected\n",
	  FilePosShow(ctype->file_pos), KindShow(NamedKind(named)),
	  UStringToUTF8(ctype->uninstantiated),
	  FilePosShow(NamedFilePos(named)));
	db_return(DEBUG29, "TypeLevelOneValid (unknown)", FALSE);
    }
  }
  db_return(DEBUG29, "TypeLevelOneValid", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  static BOOLEAN TypeVarOccurFree(TYPE_VAR v, TYPE t)                      */
/*                                                                           */
/*  Return TRUE if type t does not contain any occurrence of type variable   */
/*  v in itself or in any constraint on any variable invoked within t.       */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeVarOccurFree(TYPE_VAR v, TYPE t)
{
  BOOLEAN res;  TYPE save_constraint;  CTYPE ctype;  TYPE t2;
  if( IsVar(t) )
  {
    /* fail if this variable is the one we are trying to be free of */
    if( t->u.variable == v )
      return FALSE;

    /* temporarily delete upper_constraint to prevent cycles, then check it */
    if( t->u.variable->upper_constraint != NULL )
    {
      save_constraint = t->u.variable->upper_constraint;
      t->u.variable->upper_constraint = NULL;
      res = TypeVarOccurFree(v, save_constraint);
      t->u.variable->upper_constraint = save_constraint;
    }

    /* temporarily delete lower_constraint to prevent cycles, then check it */
    if( t->u.variable->lower_constraint != NULL )
    {
      save_constraint = t->u.variable->lower_constraint;
      t->u.variable->lower_constraint = NULL;
      res = TypeVarOccurFree(v, save_constraint);
      t->u.variable->lower_constraint = save_constraint;
    }

    return res;
  }
  else
  {
    /* check the actual parameters of each ctype */
    ArrayForEach(t->u.ctypes, ctype)
      if( ctype->generics != NULL )
	ArrayForEach(ctype->generics, t2)
	  if( !TypeVarOccurFree(v, t2) )
	    return FALSE;
    return TRUE;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  static BOOLEAN TypeVarsOccurFree(TYPE_VARS tv)                           */
/*                                                                           */
/*  Returns TRUE if no type variable in tv occurs within its own             */
/*  constraints, either directly or indirectly within the constraints of     */
/*  other variables invoked within that constraint.  Else print an error     */
/*  and return FALSE.                                                        */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeVarsOccurFree(TYPE_VARS tv)
{
  TYPE_VAR v;
  ArrayForEach(tv, v)
  {
    if( v->upper_constraint!=NULL && !TypeVarOccurFree(v, v->upper_constraint) )
    {
      fprintf(stderr, "%s: type variable %s occurs in own constraint\n",
	FilePosShow(v->file_pos), NameShow(v->name));
      return FALSE;
    }
    if( v->lower_constraint!=NULL && !TypeVarOccurFree(v, v->lower_constraint) )
    {
      fprintf(stderr, "%s: type variable %s occurs in own constraint\n",
	FilePosShow(v->file_pos), NameShow(v->name));
      return FALSE;
    }
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeVarsBeginLevelOneValid(TYPE_VARS tv, CONTEXT cxt,            */
/*    BOOLEAN is_public)                                                     */
/*                                                                           */
/*  Check that type variables tv are Level 1 valid with respect to the       */
/*  given cxt.  On success, the variables are left in cxt; on failure,       */
/*  everything is reset to how it was on entry.                              */
/*                                                                           */
/*  If the user wants these variables to shadow outer names, the user has    */
/*  to push and empty context level with a shadow limit before calling       */
/*  this function.                                                           */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeVarsBeginLevelOneValid(TYPE_VARS tv, CONTEXT cxt, BOOLEAN is_public)
{
  int i;  TYPE_VAR v;
  if( DEBUG8 )
    fprintf(stderr, "[ TypeVarsBeginLevelOneValid(%s, cxt)\n",
      TypeVarsShow(tv,cxt));
  if( tv != NULL )
  {
    /* add variables to cxt */
    if( !TypeVarsBegin(tv, cxt) )
      db_return(DEBUG8, "TypeVarsBeginLevelOneValid", FALSE);

    /* instantiate constraints */
    for( i = 0;  i < ArraySize(tv);  i++ )
    {
      v = ArrayGet(tv, i);
      if( v->upper_constraint != NULL )
        if( !TypeLevelOneValid(v->upper_constraint, cxt, is_public) )
	{
	  TypeVarsEnd(tv, cxt);
	  db_return(DEBUG8, "TypeVarsBeginLevelOneValid", FALSE);
	}
    }

    /* make occurs check */
    if( !TypeVarsOccurFree(tv) )
    {
      TypeVarsEnd(tv, cxt);
      db_return(DEBUG8, "TypeVarsBeginLevelOneValid", FALSE);
    }
  }

  if( DEBUG8 )
    fprintf(stderr, "] TypeVarsBeginLevelOneValid returning TRUE (%s)\n",
      TypeVarsShow(tv, cxt));
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeWithVarsLevelOneValid(TYPE t, TYPE_VARS tv, CONTEXT cxt)     */
/*                                                                           */
/*  Like TypeLevelOneValid except that variables are supplied as well.       */
/*  This function is called only from main() so its evident inefficiency     */
/*  does not matter.                                                         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeWithVarsLevelOneValid(TYPE t, TYPE_VARS tv, CONTEXT cxt)
{
  BOOLEAN res;
  assert(t != NULL);
  ContextPushEmpty(cxt, NULL, FALSE);
  res = TypeVarsBeginLevelOneValid(tv, cxt, FALSE) && 
    TypeLevelOneValid(t, cxt, FALSE);
  ContextPop(cxt, TRUE);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  The "Variable" submodule.                                                */
/*                                                                           */
/*  This submodule contains code for resolving and forwarding variables.     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  static void TypeVarsResolvedBegin(TYPE_VARS tv1, ARRAY_TYPE actuals)     */
/*                                                                           */
/*  Begin resolving type vars tv1 to the given actual types.                 */
/*                                                                           */
/*****************************************************************************/

static void TypeVarsResolvedBegin(TYPE_VARS tv1, ARRAY_TYPE actuals)
{
  int i, s1, s2;  TYPE_VAR v1;  TYPE t2;
  if( DEBUG21 )
    fprintf(stderr, "[ TypeVarsResolvedBegin(%s, %s)\n",
      TypeVarsShow(tv1, NULL), TypeArrayShow(actuals, NULL));
  s1 = (tv1 == NULL ? 0 : ArraySize(tv1));
  s2 = (actuals == NULL ? 0 : ArraySize(actuals));
  assert(s1 == s2);
  for( i = 0;  i < s1;  i++ )
  {
    v1 = ArrayGet(tv1, i);
    assert(v1->resolved_to == NULL);
    t2 = TypeResolve(ArrayGet(actuals, i));
    if( !IsVar(t2) || t2->u.variable != v1 )
      v1->resolved_to = t2;  /* only if not resolving to self */
  }
  if( DEBUG21 )
    fprintf(stderr, "] TypeVarsResolvedBegin returning\n");
}


/*****************************************************************************/
/*                                                                           */
/*  static void TypeVarsResolvedEnd(TYPE_VARS tv1)                           */
/*                                                                           */
/*  End resolving type vars tv1 to other values.                             */
/*                                                                           */
/*****************************************************************************/

static void TypeVarsResolvedEnd(TYPE_VARS tv)
{
  TYPE_VAR v;
  if( tv != NULL )  ArrayForEach(tv, v)
    v->resolved_to = NULL;
  if( DEBUG7 )
    fprintf(stderr, "[] TypeVarsResolvedEnd\n");
}


/*****************************************************************************/
/*                                                                           */
/*  void TypeVarsForwardBegin(TYPE_VARS tv1, TYPE_VARS tv2, TOKEN token)     */
/*                                                                           */
/*  Resolve each variable in tv1 to a type whose value is the corresponding  */
/*  variable in tv2.  It is acceptable for a variable to be forwarded to     */
/*  itself, but in that case it is simply left unresolved.                   */
/*                                                                           */
/*****************************************************************************/

void TypeVarsForwardBegin(TYPE_VARS tv1, TYPE_VARS tv2, FILE_POS file_pos)
{
  int s1, s2, i;  TYPE_VAR v1, v2;  TYPE t2;

  if( DEBUG3 )
    fprintf(stderr, "[ TypeVarsForwardBegin(%s, %s)\n",
      TypeVarsShow(tv1, NULL), TypeVarsShow(tv2, NULL));

  /* find number of variables and ensure equal */
  s1 = tv1 == NULL ? 0 : ArraySize(tv1);
  s2 = tv2 == NULL ? 0 : ArraySize(tv2);
  assert(s1 == s2);

  /* do the resolving */
  for( i = 0;  i < s1;  i++ )
  {
    v1 = ArrayGet(tv1, i);
    assert(v1->state == Ordinary);
    v2 = ArrayGet(tv2, i);
    if( v1 != v2 )
    {
      t2 = TypeMakeFromVar(file_pos, v2);
      v1->resolved_to = t2;
    }
  }
}

/*****************************************************************************/
/*                                                                           */
/*  void TypeVarsForwardEnd(TYPE_VARS tv1)                                   */
/*                                                                           */
/*  Cancel a preceding TypeVarsForwardBegin, including freeing the types     */
/*  created by that call.  These had better not have become referred to      */
/*  by others in the meantime!                                               */
/*                                                                           */
/*****************************************************************************/

void TypeVarsForwardEnd(TYPE_VARS tv1)
{
  int s1, i;  TYPE_VAR v1;
  s1 = tv1 == NULL ? 0 : ArraySize(tv1);
  for( i = 0;  i < s1;  i++ )
  {
    v1 = ArrayGet(tv1, i);
    if( v1->resolved_to != NULL )
    {
      /* free(v1->resolved_to); */
      v1->resolved_to = NULL;
    }
  }
  if( DEBUG3 )
    fprintf(stderr, "] TypeVarsForwardEnd(%s)\n", TypeVarsShow(tv1, NULL));
}


/*****************************************************************************/
/*                                                                           */
/*  The "Copy" submodule.                                                    */
/*                                                                           */
/*  Instantiated types are copied, basically, to freeze previous decisions   */
/*  to resolve variables to certain values.  Uninstantiated types are        */
/*  copied as part of disentangling features declared together from each     */
/*  other.                                                                   */
/*                                                                           */
/*****************************************************************************/


/*****************************************************************************/
/*                                                                           */
/*  CTYPE TypeCTypeCopyUninstantiated(CTYPE ctype)                           */
/*                                                                           */
/*  Return a fresh copy of uninstantiated ctype.                             */
/*                                                                           */
/*****************************************************************************/

static CTYPE TypeCTypeCopyUninstantiated(CTYPE ctype)
{
  CTYPE res;  TYPE gen;
  res = TypeMakeCType(ctype->file_pos, ctype->uninstantiated, ctype->class,
    NULL);
  if( ctype->generics != NULL )
  {
    ArrayInit(&res->generics);
    ArrayForEach(ctype->generics, gen)
      ArrayAddLast(res->generics, TypeCopyUninstantiated(gen));
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeCopyUninstantiated(TYPE type)                                   */
/*                                                                           */
/*  Make a copy of uninstantiated type.                                      */
/*                                                                           */
/*****************************************************************************/

TYPE TypeCopyUninstantiated(TYPE type)
{
  TYPE res;  CTYPE ctype;
  if( DEBUG28 )
    fprintf(stderr, "[ TypeCopyUninstantiated(%s)\n", TypeShow(type, NULL));
  assert(type != NULL);
  assert(type->kind_tag == KIND_TYPEC);
  GetMemory(res, TYPE);
  res->kind_tag = type->kind_tag;
  res->file_pos = type->file_pos;
  ArrayInit(&res->u.ctypes);
  ArrayForEach(type->u.ctypes, ctype)
    ArrayAddLast(res->u.ctypes, TypeCTypeCopyUninstantiated(ctype));
  if( DEBUG28 )
    fprintf(stderr, "] TypeCopyUninstantiated returning %s\n",
      TypeShow(res, NULL));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE_VARS TypeVarsCopyUninstantiated(TYPE_VARS type_vars)                */
/*                                                                           */
/*  Make a copy of a set of uninstantiated type variables.  These just       */
/*  contain strings, not pointers to types, so copying is trivial.           */
/*                                                                           */
/*****************************************************************************/

TYPE_VARS TypeVarsCopyUninstantiated(TYPE_VARS type_vars)
{
  TYPE_VARS res = NULL;  TYPE_VAR v, rv;
  if( type_vars != NULL )
  {
    ArrayInit(&res);
    ArrayForEach(type_vars, v)
    {
      assert(v->state == Ordinary);
      rv = TypeVarMake(v->file_pos, v->name);
      assert(v->lower_constraint == NULL);
      if( v->upper_constraint != NULL )
	rv->upper_constraint = TypeCopyUninstantiated(v->upper_constraint);
      rv->state = v->state;
      ArrayAddLast(res, rv);
    }
  }
  return res;
}

/*****************************************************************************/
/*                                                                           */
/*  static CTYPE TypeCTypeCopy(CTYPE ctype)                                  */
/*                                                                           */
/*  Return a copy of ctype.                                                  */
/*                                                                           */
/*****************************************************************************/

static CTYPE TypeCTypeCopy(CTYPE ctype)
{
  CTYPE res;  TYPE gen;
  res = TypeMakeCType(ctype->file_pos, ctype->uninstantiated,
    ctype->class, NULL);
  if( ctype->generics != NULL )
  {
    ArrayInit(&res->generics);
    ArrayForEach(ctype->generics, gen)
      ArrayAddLast(res->generics, TypeCopy(gen));
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeCopy(TYPE type)                                                 */
/*                                                                           */
/*  Return a copy of type, implementing any forwarding and resolving that    */
/*  has been set up prior to the call.                                       */
/*                                                                           */
/*  No variables are every created by TypeCopy; copying a variable           */
/*  invocation produces a new invocation of that same variable.  If new      */
/*  variables are wanted, these must be set up prior to the copying and      */
/*  the old variables forwarded to the new ones.  Then the copy will         */
/*  replace the old ones by the new ones.                                    */
/*                                                                           */
/*****************************************************************************/

TYPE TypeCopy(TYPE type)
{
  TYPE res;  CTYPE ctype;
  assert(type != NULL);
  if( DEBUG8 )
    fprintf(stderr, "[ TypeCopy(%s)\n", TypeShow(type, NULL));
  type = TypeResolve(type);
  GetMemory(res, TYPE);
  res->kind_tag = type->kind_tag;
  res->file_pos = type->file_pos;
  if( IsVar(type) )
    res->u.variable = type->u.variable;
  else
  {
    ArrayInit(&res->u.ctypes);
    ArrayForEach(type->u.ctypes, ctype)
      ArrayAddLast(res->u.ctypes, TypeCTypeCopy(ctype));
  }
  db_return_val(DEBUG8, "TypeCopy", TypeShow(res, NULL), res);
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE_VARS TypeVarsCopy(TYPE_VARS type_vars, FILE_POS file_pos)           */
/*                                                                           */
/*  Return a copy of type_vars, with constraints copied correctly.           */
/*                                                                           */
/*****************************************************************************/

TYPE_VARS TypeVarsCopy(TYPE_VARS type_vars, FILE_POS file_pos)
{
  TYPE_VARS res = NULL;  TYPE_VAR v, rv;  int i;
  if( type_vars != NULL )
  {
    /* make new variables res, copied from type_vars */
    ArrayInit(&res);
    ArrayForEach(type_vars, v)
    {
      assert(v->state == Ordinary);
      rv = TypeVarMake(file_pos, v->name);
      rv->state = v->state;
      ArrayAddLast(res, rv);
    }

    /* copy constraints */
    TypeVarsForwardBegin(type_vars, res, file_pos);
    for( i = 0;  i < ArraySize(type_vars);  i++ )
    {
      v = ArrayGet(type_vars, i);
      rv = ArrayGet(res, i);
      if( v->lower_constraint != NULL )
        rv->lower_constraint = TypeCopy(v->lower_constraint);
      if( v->upper_constraint != NULL )
        rv->upper_constraint = TypeCopy(v->upper_constraint);
    }
    TypeVarsForwardEnd(type_vars);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  The "Range" submodule.                                                   */
/*                                                                           */
/*  This submodule contains code for handling range variables and inference  */
/*  of actual generic parameters.                                            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void TypeRangeMarkBegin()                                                */
/*  static void TypeRangeMarkAssign(TYPE_VAR v, TYPE resolved_value)         */
/*  void TypeRangeMarkEnd(BOOLEAN confirm)                                   */
/*                                                                           */
/*  Calls to TypeRangeMarkBegin() and TypeRangeMarkEnd() are meant to        */
/*  bracket regions of code where resolutions of type range variables may    */
/*  occur which may need to be undone at the end of the region.  These       */
/*  regions cannot be dynamically nested, as marked_range_active verifies.   */
/*                                                                           */
/*  Within a bracketed region, calls to TypeRangeMarkAssign resolve a type   */
/*  range to a type.  At the end of the bracket, these resolutions           */
/*  are undone, unless confirm is TRUE in which case they are retained.      */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN	 marked_range_active = FALSE;
static TYPE_VARS marked_range_variables = NULL;

void TypeRangeMarkBegin()
{
  assert(!marked_range_active);
  if( marked_range_variables == NULL )
    ArrayInit(&marked_range_variables);
  assert(ArraySize(marked_range_variables) == 0);
  marked_range_active = TRUE;
}

static void TypeRangeMarkAssign(TYPE_VAR v, TYPE resolved_value)
{
  assert(marked_range_active);
  assert(v->resolved_to == NULL);
  v->resolved_to = resolved_value;
  ArrayAddLast(marked_range_variables, v);
}

void TypeRangeMarkEnd(BOOLEAN confirm)
{
  TYPE_VAR v;
  assert(marked_range_active);
  while( ArraySize(marked_range_variables) > 0 )
  {
    v = ArrayRemoveLast(marked_range_variables);
    assert(v->resolved_to != NULL);
    if( !confirm )
      v->resolved_to = NULL;
  }
  marked_range_active = FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeVarsInferBegin(TYPE_VARS tv, ARRAY_TYPE *actuals,            */
/*    TOKEN token, BOOLEAN confirm)                                          */
/*                                                                           */
/*  Similar to TypeVarsResolvedBegin in that we resolve type variables tv    */
/*  to actual types actuals.  However, here we check that the actuals obey   */
/*  any constraints on tv after the assignment, and we convert some of the   */
/*  variables of tv to unification variables if actuals is not long enough,  */
/*  lengthening actuals to add invocations of these new variables.           */
/*                                                                           */
/*  If confirm is TRUE, we are doing this for real and any errors should be  */
/*  printed.  If confirm is FALSE, we are just trying out one of a set of    */
/*  overloaded function signatures, so fail silently if it does not match.   */
/*                                                                           */
/*  If this function fails (returns FALSE) there is no need to call the      */
/*  matching TypeVarsInferEnd because it will have undone all changes.       */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN	infer_current = FALSE;	/* whether inferring now or not      */
static int	infer_params;		/* how many unification vars put in  */

BOOLEAN TypeVarsInferBegin(TYPE_VARS tv, ARRAY_TYPE *actuals,
  FILE_POS file_pos, BOOLEAN confirm, CONTEXT cxt)
{
  int sa, sv, i;  TYPE_VAR v;  TYPE vt, junk;  COERCION c;

  /* change state from not inferring to inferring */
  assert(!infer_current);
  infer_current = TRUE;
  infer_params = 0;

  /* make sure we have at least as many variables as actuals */
  sv = (tv == NULL ? 0 : ArraySize(tv));
  sa = (*actuals == NULL ? 0 : ArraySize(*actuals));
  if( sv < sa )
  {
    if( confirm )
    {
      if( sv == 0 )
	fprintf(stderr, "%s: too many generics (wanted none, found %d)\n",
	  FilePosShow(file_pos), sa);
      else
	fprintf(stderr, "%s: too many generics (wanted at most %d, found %d)\n",
	  FilePosShow(file_pos), sv, sa);
    }
    infer_current = FALSE;
    return FALSE;
  }

  /* resolve each variable, or convert it to a unification variable */
  /* store in static variable infer_params the number of variables inserted */
  for( i = 0;  i < sv;  i++ )
  {
    v = ArrayGet(tv, i);
    assert(v->state == Ordinary);
    if( i >= sa )
    {
      /* convert v to unification and add an invocation of v to *actuals */
      v->state = Unify;
      vt = TypeMakeFromVar(file_pos, v);
      if( *actuals == NULL )
	ArrayInit(actuals);
      ArrayAddLast(*actuals, vt);
      if( DEBUG12 )
	fprintf(stderr, "    adding unification variable %s\n",
	  TypeShow(vt, cxt));
      infer_params++;
    }
    else v->resolved_to = ArrayGet(*actuals, i);
  }

  /* check that subtype relations hold */
  for( i = 0;  i < sa;  i++ )
  {
    v = ArrayGet(tv, i);
    if( v->upper_constraint != NULL &&
	!TypeIsSubType(v->resolved_to, v->upper_constraint, &c, cxt) )
    {
      if( confirm )
      {
	fprintf(stderr, "%s: generic parameter %d violates constraint %s %s\n",
	  FilePosShow(file_pos), i, "on corresponding formal",
	  NameShow(v->name));
	fprintf(stderr, "  (formal generic parameter %s is defined at %s)\n",
	  NameShow(v->name), FilePosShow(v->file_pos));
      }
      for( i = 0;  i < infer_params;  i++ )
	junk = ArrayRemoveLast(*actuals);
      infer_current = FALSE;
      return FALSE;
    }
  }
  return TRUE;
}

/*****************************************************************************/
/*                                                                           */
/*  void TypeVarsInferEnd(TYPE_VARS tv, ARRAY_TYPE *actuals, BOOLEAN confirm)*/
/*                                                                           */
/*  End the type inference begun by the corresponding TypeVarsInferBegin.    */
/*                                                                           */
/*****************************************************************************/

void TypeVarsInferEnd(TYPE_VARS tv, ARRAY_TYPE *actuals, BOOLEAN confirm)
{
  int i;  TYPE junk, t;
  assert(infer_current);
  infer_current = FALSE;

  if( confirm )
  {
    /* confirming, so convert unresolved unification variables to range vars */
    for( i = 0;  i < infer_params;  i++ )
    { t = ArrayGet(*actuals, ArraySize(*actuals) - infer_params + i);
      assert(IsVar(t) && t->u.variable->state == Unify);
      if( t->u.variable->resolved_to == NULL )
	t->u.variable->state = Range;
    }
  }
  else
  {
    /* not confirming, so remove inserted unification variables */
    for( i = 0;  i < infer_params;  i++ )
      junk = ArrayRemoveLast(*actuals);
    if( *actuals != NULL && ArraySize(*actuals) == 0 )
      *actuals = NULL;
  }
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeFixRange(TYPE type)                                             */
/*                                                                           */
/*  Convert type range to its lower limit or anyway drop upper limit.        */
/*                                                                           */
/*****************************************************************************/

TYPE TypeFixRange(TYPE type)
{
  /* do nothing if not a range variable; can't be a unify variable */
  TYPE_VAR v;
  type = TypeResolve(type);
  if( !IsVar(type) || type->u.variable->state == Ordinary )
    return type;
  assert(type->u.variable->state == Range);

  /* lower constraint if there is one */
  if( type->u.variable->lower_constraint != NULL )
    return type->u.variable->lower_constraint;

  /* as is if no upper constraint */
  if( type->u.variable->upper_constraint != NULL )
    return type->u.variable->upper_constraint;

  /* otherwise need to get rid of upper constraint */
  v = TypeVarMake(type->file_pos, type->u.variable->name);
  v->state = Range;
  return TypeMakeFromVar(type->file_pos, v);
}


/*****************************************************************************/
/*                                                                           */
/*  The "Equality" submodule.                                                */
/*                                                                           */
/*  This submodule handles testing types for equality.                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeCTypeEqual(CTYPE c1, CTYPE c2, CONTEXT cxt)                  */
/*                                                                           */
/*  Return true if these two CTYPE types are equal.                          */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeCTypeEqual(CTYPE c1, CTYPE c2, CONTEXT cxt)
{
  int g1, g2, i;

  if( DEBUG13 )
  {
    fprintf(stderr, "[ TypeCTypeEqual(%s, %s)\n", CTypeShow(c1, cxt),
      CTypeShow(c2, cxt));
    fflush(stderr);
  }

  /* must be the same class */
  if( c1->class != c2->class )
    db_return(DEBUG13, "TypeCTypeEqual (classes differ)", FALSE);

  /* need same number of generic parameters */
  g1 = c1->generics == NULL ? 0 : ArraySize(c1->generics);
  g2 = c2->generics == NULL ? 0 : ArraySize(c2->generics);
  if( g1 != g2 )
    db_return(DEBUG13, "TypeCTypeEqual (generic count)", FALSE);

  /* need same values for generic parameters */
  for( i = 0;  i < g1;  i++ )
    if( !TypeEqual(ArrayGet(c1->generics, i), ArrayGet(c2->generics, i), cxt) )
    {
      if( DEBUG13 )
	fprintf(stderr, "] TypeCTypeEqual returning FALSE (i = %d)\n", i);
      return FALSE;
    }

  db_return(DEBUG13, "TypeCTypeEqual", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeEqualMeetWithMeet(TYPE t1, TYPE t2, CONTEXT cxt)             */
/*                                                                           */
/*  Return true if these two types are equal.  Both are meets of one or      */
/*  more classes, which do not have to appear in the same order in the       */
/*  two types for equality.                                                  */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeEqualMeetWithMeet(TYPE t1, TYPE t2, CONTEXT cxt)
{
  int s1, s2, i, j;  CTYPE t1c;
  assert(!IsVar(t1));
  assert(!IsVar(t2));

  /* number of classes must be equal */
  s1 = ArraySize(t1->u.ctypes);
  s2 = ArraySize(t2->u.ctypes);
  if( s1 != s2 )
    return FALSE;

  /* must be able to find each class in the other */
  for( i = 0;  i < s1;  i++ )
  {
    t1c = ArrayGet(t1->u.ctypes, i);
    for( j = 0;  j < s2;  j++ )
      if( TypeCTypeEqual(t1c, ArrayGet(t2->u.ctypes, j), cxt) )
	break;
    if( j >= s2 )
      return FALSE;
  }
  return TRUE;
}

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeEqualUnifyVarWithOrdinaryVarOrMeet(TYPE t1, TYPE t2,         */
/*    CONTEXT cxt)                                                           */
/*                                                                           */
/*  Return true if these two types are equal.  The first is an invocation    */
/*  of a unification variable, the second is an ordinary variable or meet.   */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeEqualUnifyVarWithOrdinaryVarOrMeet(TYPE t1, TYPE t2,
  CONTEXT cxt)
{
  TYPE_VAR v1;  COERCION c;
  assert(IsVar(t1) && t1->u.variable->state == Unify);
  assert(!IsVar(t2) || t2->u.variable->state == Ordinary);
  v1 = t1->u.variable;

  /* t2 must satisfy any lower constraint on v1 */
  if( v1->lower_constraint != NULL &&
      !TypeIsSubType(v1->lower_constraint, t2, &c, cxt) )
    return FALSE;

  /* t2 must satisfy any upper constraint on v1 */
  if( v1->upper_constraint != NULL &&
      !TypeIsSubType(t2, v1->upper_constraint, &c, cxt) )
    return FALSE;

  /* resolve v1 to t2 */
  v1->resolved_to = t2;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeEqualUnifyVarWithUnifyVar(TYPE t1, TYPE t2, CONTEXT cxt)     */
/*                                                                           */
/*  Return true if these two types are equal.  Both are invocations of       */
/*  unification variables.                                                   */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeEqualUnifyVarWithUnifyVar(TYPE t1, TYPE t2, CONTEXT cxt)
{
  TYPE_VAR v1, v2;  TYPE v2low, v2upp;  COERCION c;

  /* get the variables */
  assert(IsVar(t1) && t1->u.variable->state == Unify);
  assert(IsVar(t2) && t2->u.variable->state == Unify);
  v1 = t1->u.variable;
  v2 = t2->u.variable;

  /* nothing to do if the variables are the same */
  if( v1 == v2 )
    return TRUE;

  /* v1's lower constraint is the join with v2's (if present and not v1) */
  v2low = v2->lower_constraint;
  if( v2low != NULL && !(IsVar(v2low) && v2low->u.variable == v1) &&
      !TypeJoin(&v1->lower_constraint, v2low, cxt) )
    db_return(DEBUG15, "UnifyVariables (2)", FALSE);

  /* v1's upper constraint is the meet with v2's (if present and not v1) */
  v2upp = v2->upper_constraint;
  if( v2upp != NULL && !(IsVar(v2upp) && v2upp->u.variable == v1) &&
      !TypeMeet(&v1->upper_constraint, v2upp, cxt) )
    db_return(DEBUG15, "UnifyVariables (3)", FALSE);

  /* ensure lower(v1) <: upper(v1) */
  if( v1->lower_constraint != NULL && v1->upper_constraint != NULL &&
      !TypeIsSubType(v1->lower_constraint,v1->upper_constraint,&c,cxt) )
    db_return(DEBUG15, "UnifyVariables (4)", FALSE);

  /* resolve v2 to v1 */
  v2->resolved_to = TypeMakeFromVar(t1->file_pos, v1);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeEqualUnifyVarWithRange(TYPE t1, TYPE t2, CONTEXT cxt)        */
/*                                                                           */
/*  Return true if these two types are equal.  The first is an invocation    */
/*  of a unification variable, the second is a range type.                   */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeEqualUnifyVarWithRange(TYPE t1, TYPE t2, CONTEXT cxt)
{
  TYPE_VAR v1, v2;  COERCION c;
  assert(IsVar(t1) && t1->u.variable->state == Unify);
  assert(IsVar(t2) && t2->u.variable->state == Range);
  v1 = t1->u.variable;
  v2 = t2->u.variable;

  /* if v2 has a lower constraint, it goes into v1's lower constraint */
  if( v2->lower_constraint != NULL )
    if( !TypeJoin(&v1->lower_constraint, v2->lower_constraint, cxt) )
      return FALSE;

  /* if v2 has an upper constraint, it goes into v1's upper constraint */
  if( v2->upper_constraint != NULL )
    if( !TypeMeet(&v1->upper_constraint, v2->upper_constraint, cxt) )
      return FALSE;

  /* ensure lower(v1) <: upper(v1) */
  if( v1->lower_constraint != NULL && v1->upper_constraint != NULL &&
      !TypeIsSubType(v1->lower_constraint,v1->upper_constraint,&c,cxt) )
    return FALSE;

  /* and v2 is resolved to t1 */
  TypeRangeMarkAssign(v2, t1);
  return TRUE;
}

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeEqualRangeWithOrdinaryVarOrMeet(TYPE t1, TYPE t2,            */
/*    CONTEXT cxt)                                                           */
/*                                                                           */
/*  Return true if these two types are equal.  The first is a type range,    */
/*  these second is an invocation of an ordinary variable, or a meet.        */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeEqualRangeWithOrdinaryVarOrMeet(TYPE t1, TYPE t2,
  CONTEXT cxt)
{
  TYPE_VAR v1;  COERCION c;
  assert(IsVar(t1) && t1->u.variable->state == Range);
  assert(!IsVar(t2) || t2->u.variable->state == Ordinary);
  v1 = t1->u.variable;

  /* t2 must lie within the range of t1 */
  if( v1->lower_constraint != NULL &&
      !TypeIsSubType(v1->lower_constraint, t2, &c, cxt) )
    return FALSE;
  if( v1->upper_constraint != NULL &&
      !TypeIsSubType(t2, v1->upper_constraint, &c, cxt) )
    return FALSE;

  /* now resolve v1 to t2 */
  TypeRangeMarkAssign(v1, t2);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeIsSubRange(TYPE_VAR v1, TYPE_VAR v2, CONTEXT cxt)            */
/*                                                                           */
/*  Here v1 and v2 are type ranges.  Return TRUE if v1 is a subrange of v2.  */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeIsSubRange(TYPE_VAR v1, TYPE_VAR v2, CONTEXT cxt)
{
  COERCION c;

  /* check that v2's lower limit is at or below v1's */
  if( v2->lower_constraint != NULL && (v1->lower_constraint == NULL ||
    !TypeIsSubType(v2->lower_constraint, v1->lower_constraint, &c, cxt)) )
    return FALSE;

  /* check that v2's upper limit is at or above v1's */
  if( v2->upper_constraint != NULL && (v1->upper_constraint == NULL ||
    !TypeIsSubType(v1->upper_constraint, v2->upper_constraint, &c, cxt)) )
    return FALSE;

  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeEqualRangeWithRange(TYPE t1, TYPE t2, CONTEXT cxt)           */
/*                                                                           */
/*  Return true if these two types are equal.  Both are type ranges.         */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeEqualRangeWithRange(TYPE t1, TYPE t2, CONTEXT cxt)
{
  assert(IsVar(t1) && t1->u.variable->state == Range);
  assert(IsVar(t2) && t2->u.variable->state == Range);

  /* if t1 is a subrange of t2, resolve t2 to t1 and succeed */
  if( TypeIsSubRange(t1->u.variable, t2->u.variable, cxt) )
  {
    TypeRangeMarkAssign(t2->u.variable, t1);
    return TRUE;
  }

  /* if t2 is a subrange of t1, resolve t1 to t2 and succeed */
  if( TypeIsSubRange(t2->u.variable, t1->u.variable, cxt) )
  {
    TypeRangeMarkAssign(t1->u.variable, t2);
    return TRUE;
  }

  /* otherwise, since we can only resolve, not unify, must fail */
  return FALSE;
}

typedef enum { MeetType, OrdinaryVarType, UnifyVarType, RangeType } TYPE_KIND;

static TYPE_KIND TypeKind(TYPE t)
{
  return !IsVar(t) ? MeetType :
    t->u.variable->state == Ordinary ? OrdinaryVarType :
    t->u.variable->state == Unify ? UnifyVarType : RangeType;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeEqual(TYPE t1, TYPE t2, CONTEXT cxt)                         */
/*                                                                           */
/*  Return true if these two types are equal.                                */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeEqual(TYPE t1, TYPE t2, CONTEXT cxt)
{
  BOOLEAN res;
  assert(t1 != NULL && t2 != NULL);
  if( DEBUG17 )
    fprintf(stderr, "[ TypeEqual(%s, %s):\n", TypeShow(t1, cxt),
      TypeShow(t2, cxt));

  /* resolve t1 and t2; equal if identical */
  t1 = TypeResolve(t1);
  t2 = TypeResolve(t2);
  if( t1 == t2 )
    db_return(DEBUG17, "TypeEqual (exact)", TRUE);

  /* two-level dispatch */
  switch( TypeKind(t1) )
  {
    case MeetType:

      switch( TypeKind(t2) )
      {
	case MeetType:

	  res = TypeEqualMeetWithMeet(t1, t2, cxt);
	  break;

	case OrdinaryVarType:

	  res = FALSE;
	  break;

	case UnifyVarType:

	  res = TypeEqualUnifyVarWithOrdinaryVarOrMeet(t2, t1, cxt);
	  break;

	case RangeType:

	  res = TypeEqualRangeWithOrdinaryVarOrMeet(t2, t1, cxt);
	  break;
      }
      break;


    case OrdinaryVarType:

      switch( TypeKind(t2) )
      {
	case MeetType:

	  res = FALSE;
	  break;

	case OrdinaryVarType:

	  res = (t1->u.variable == t2->u.variable);
	  break;

	case UnifyVarType:

	  res = TypeEqualUnifyVarWithOrdinaryVarOrMeet(t2, t1, cxt);
	  break;

	case RangeType:

	  res = TypeEqualRangeWithOrdinaryVarOrMeet(t2, t1, cxt);
	  break;
      }
      break;


    case UnifyVarType:

      switch( TypeKind(t2) )
      {
	case MeetType:
	case OrdinaryVarType:

	  res = TypeEqualUnifyVarWithOrdinaryVarOrMeet(t1, t2, cxt);
	  break;

	case UnifyVarType:

	  res = TypeEqualUnifyVarWithUnifyVar(t1, t2, cxt);
	  break;

	case RangeType:

	  res = TypeEqualUnifyVarWithRange(t1, t2, cxt);
	  break;
      }
      break;


    case RangeType:

      switch( TypeKind(t2) )
      {
	case MeetType:
	case OrdinaryVarType:

	  res = TypeEqualRangeWithOrdinaryVarOrMeet(t1, t2, cxt);
	  break;

	case UnifyVarType:

	  res = TypeEqualUnifyVarWithRange(t2, t1, cxt);
	  break;

	case RangeType:

	  res = TypeEqualRangeWithRange(t1, t2, cxt);
	  break;
      }
      break;

  }
  db_return(DEBUG17, "TypeEqual", res);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeEqualOptional(TYPE t1, TYPE t2, CONTEXT cxt)                 */
/*                                                                           */
/*  Like TypeEqual but the two parameters may be NULL and are considered     */
/*  equal if they are both NULL, or both non-NULL and equal as types.        */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeEqualOptional(TYPE t1, TYPE t2, CONTEXT cxt)
{
  if( t1 == NULL || t2 == NULL )
    return t1 == NULL && t2 == NULL;
  else
    return TypeEqual(t1, t2, cxt);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeVarsEquivalent(TYPE_VARS tv1, TYPE_VARS tv2, CONTEXT cxt)    */
/*                                                                           */
/*  Decide whether type variables tv1 and tv2 are equivalent - whether they  */
/*  are the same, with the same constraints, apart from the change of name.  */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeVarsEquivalent(TYPE_VARS tv1, TYPE_VARS tv2, CONTEXT cxt,
  FILE_POS file_pos)
{
  int g1, g2, i;  TYPE_VAR v1, v2;  BOOLEAN res;
  if( DEBUG16 )
    fprintf(stderr, "[ TypeVarsEquivalent(%s, %s):\n", TypeVarsShow(tv1, cxt),
      TypeVarsShow(tv2, cxt));

  /* ensure same number of generics */
  g1 = tv1 == NULL ? 0 : ArraySize(tv1);
  g2 = tv2 == NULL ? 0 : ArraySize(tv2);
  if( g1 != g2 )
    db_return(DEBUG16, "[TypeVarsEquivalent (1)", FALSE);

  /* add forwarding addresses from tv1's variables to tv2's */
  TypeVarsForwardBegin(tv1, tv2, file_pos);

  /* if generics, check equality of their constraints */
  res = TRUE;
  for( i = 0;  i < g1;  i++ )
  {
    /* not expecting to do this for unification variables */
    v1 = ArrayGet(tv1, i);
    v2 = ArrayGet(tv2, i);
    assert(v1->state != Unify && v2->state != Unify);

    /* check lower constraints equal */
    if( !TypeEqualOptional(v1->lower_constraint, v2->lower_constraint, cxt) )
      res = FALSE;

    /* check upper constraints equal */
    if( !TypeEqualOptional(v1->upper_constraint, v2->upper_constraint, cxt) )
      res = FALSE;
  }

  /* take away forwarding addresses */
  TypeVarsForwardEnd(tv1);
  db_return(DEBUG16, "[TypeVarsEquivalent (2)", res);
}


/*****************************************************************************/
/*                                                                           */
/*  The "Subtype" submodule.                                                 */
/*                                                                           */
/*  This submodule handles testing whether one type is a subtype of another. */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeAncestorsContainIType(ITYPES ancestors, CLASS c, ITYPE *res) */
/*                                                                           */
/*  Return TRUE if ancestor set ancestors contains an invocation of class    */
/*  c, and return the ITYPE that is that invocation in *res; else FALSE.     */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeAncestorsContainIType(ITYPES ancestors, CLASS c, ITYPE *res)
{
  ITYPE itype;
  ArrayForEach(ancestors, itype)
  {
    if( itype->ctype->class == c )
    {
      *res = itype;
      return TRUE;
    }
  }
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeCSubType(CTYPE lower, CTYPE upper, COERCION *coercion,       */
/*    CONTEXT cxt)                                                           */
/*                                                                           */
/*  Return TRUE if CTYPE lower is a subtype of CTYPE upper.                  */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeCSubType(CTYPE lower, CTYPE upper, COERCION *coercion,
  CONTEXT cxt)
{
  ITYPES ancestors;  ITYPE itype;  TYPE_VARS type_vars;  BOOLEAN res;
  if( DEBUG31 )
    fprintf(stderr, "[ TypeCSubType(%s, %s)\n", CTypeShow(lower, cxt),
      CTypeShow(upper, cxt));

  /* true if identical or equal */
  *coercion = NULL;
  if( lower == upper || TypeCTypeEqual(lower, upper, cxt) )
    db_return(DEBUG31, "TypeCSubType (simple case)", TRUE);

  /* retrieve type_vars and class ancestors from lower */
  type_vars = ClassVars(lower->class);
  ancestors = ClassAncestorSet(lower->class);

  /* search for upper in ancestors of lower */
  res = FALSE;
  if( TypeAncestorsContainIType(ancestors, upper->class, &itype) )
  {
    TypeVarsResolvedBegin(type_vars, lower->generics);
    if( DEBUG31 )
      fprintf(stderr, "  TypeCSubType calling TypeCTypeEqual\n");
    res = TypeCTypeEqual(itype->ctype, upper, cxt);
    TypeVarsResolvedEnd(type_vars);
    if( res )
      *coercion = itype->coercion;
  }
  db_return(DEBUG31, "TypeCSubType", res);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeCArraySubType(ARRAY_CTYPE lower, ARRAY_CTYPE upper,          */
/*    COERCION *coercion, CONTEXT cxt)                                       */
/*                                                                           */
/*  Return TRUE if for every CTYPE in upper there is a CTYPE in lower        */
/*  which is a subtype.  This is the subtype condition between meet types.   */
/*                                                                           */
/*  As for TypeIsSubType, if TRUE then set *coercion to the coercion         */
/*  required to get there.  If there is a coercion, it must be from a        */
/*  single class type to another single class type.                          */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeCArraySubType(ARRAY_CTYPE lower, ARRAY_CTYPE upper,
  COERCION *coercion, CONTEXT cxt)
{
  CTYPE a, b;  BOOLEAN b_found;  COERCION c;  int i, j;
  *coercion = NULL;
  for( i = 0;  i < ArraySize(upper);  i++ )
  {
    b = ArrayGet(upper, i);
    b_found = FALSE;
    for( j = 0;  j < ArraySize(lower);  j++ )
    {
      a = ArrayGet(lower, j);
      if( TypeCSubType(a, b, &c, cxt) )
      {
	if( c != NULL && (ArraySize(lower) > 1 || ArraySize(upper) > 1) )
	{
	  /* coercion of meet, or double coercion, not allowed! */
	  b_found = FALSE;
	}
	else
	{
	  *coercion = c;
	  b_found = TRUE;
	}
	break;
      }
    }
    if( !b_found )
      return FALSE;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeVarUnifyAddUpperConstraint(TYPE_VAR v, TYPE t, CONTEXT cxt)  */
/*                                                                           */
/*  Add t to the upper constraint of unification variable v, if possible.    */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeVarUnifyAddUpperConstraint(TYPE_VAR v, TYPE t, CONTEXT cxt)
{
  COERCION c;
  if( DEBUG15 )
    fprintf(stderr, "  [ AddUpper(%s, %s):\n", TypeVarShow(v, cxt),
      TypeShow(t, cxt));
  assert(v->state == Unify);

  /* if t is just v, no need to add anything */
  if( IsVar(t) && t->u.variable == v )
    db_return(DEBUG15, "AddLower (1)", TRUE);

  /* upper constraint is the meet of what it was with t */
  if( !TypeMeet(&v->upper_constraint, t, cxt) )
    db_return(DEBUG15, "AddUpper (2)", FALSE);

  /* must maintain invariant that lower <: higher */
  if( v->lower_constraint != NULL && v->upper_constraint != NULL &&
      !TypeIsSubType(v->lower_constraint, v->upper_constraint, &c, cxt) )
    db_return(DEBUG15, "AddUpper (3)", FALSE);

  if( DEBUG15 )
    fprintf(stderr, "  ] AddUpper returning TRUE (v = %s)\n",
      TypeVarShow(v, cxt));
  return TRUE;
}

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeVarUnifyAddLowerConstraint(TYPE_VAR v, TYPE t, CONTEXT cxt)  */
/*                                                                           */
/*  Add t to the lower constraint of unification variable v, if possible.    */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeVarUnifyAddLowerConstraint(TYPE_VAR v, TYPE t, CONTEXT cxt)
{
  COERCION c;
  if( DEBUG15 )
    fprintf(stderr, "  [ AddLower(%s, %s):\n", TypeVarShow(v, cxt),
      TypeShow(t, cxt));
  assert(v->state == Unify);

  /* if t is just v, no need to add anything */
  if( IsVar(t) && t->u.variable == v )
    db_return(DEBUG15, "AddLower (1)", TRUE);

  /* lower constraint is the join of what it was with t */
  if( !TypeJoin(&v->lower_constraint, t, cxt) )
    db_return(DEBUG15, "AddLower (2)", FALSE);

  /* must maintain invariant that lower <: higher */
  if( v->lower_constraint != NULL && v->upper_constraint != NULL &&
      !TypeIsSubType(v->lower_constraint, v->upper_constraint, &c, cxt) )
    db_return(DEBUG15, "AddLower (3)", FALSE);

  if( DEBUG15 )
    fprintf(stderr, "  ] AddLower returning TRUE (v = %s)\n",
      TypeVarShow(v, cxt));
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeIsSubType(TYPE lower, TYPE upper, COERCION *coercion,        */
/*    CONTEXT cxt)                                                           */
/*                                                                           */
/*  Decide whether or not lower is a subtype of higher.  If it is, set       */
/*  *coercion to the coercion required to bridge the gap, or NULL if none.   */
/*                                                                           */
/*  Note: the rule "a <= b implies a meet c <= b meet c" is only true        */
/*  when the subtype relationship between a and b does not rely on a         */
/*  coercion.  This code understands that.  Hence, there can be at most      */
/*  one coercion involved in the subtype relation between lower and upper.   */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeIsSubType(TYPE lower, TYPE upper, COERCION *coercion, CONTEXT cxt)
{
  BOOLEAN res;
  assert(lower != NULL && upper != NULL);
  if( DEBUG4 )
    fprintf(stderr, "[ TypeIsSubType(%s, %s)\n",
      TypeShow(lower, cxt), TypeShow(upper, cxt));
  lower = TypeResolve(lower);
  upper = TypeResolve(upper);
  *coercion = NULL;
  if( lower == upper )
    return TRUE;

  /* upper cannot be a range variable */
  assert( !IsVar(upper) || upper->u.variable->state != Range);

  /* if lower is a range variable, it must be totally unconstrained */
  if( IsVar(lower) && lower->u.variable->state == Range )
  {
    assert(lower->u.variable->lower_constraint == NULL);
    assert(lower->u.variable->upper_constraint == NULL);
    TypeRangeMarkAssign(lower->u.variable, upper);
    res = TRUE;
  }

  /* if lower is a unification variable, add upper to its upper constraints */
  /* unless upper equals lower, in which case it adds nothing new           */
  else if( IsVar(lower) && lower->u.variable->state == Unify )
  {
    if( DEBUG4 )
      fprintf(stderr, "  TypeIsSubType adding upper constraint\n");
    res = IsVar(upper) && upper->u.variable == lower->u.variable ?
      TRUE : TypeVarUnifyAddUpperConstraint(lower->u.variable, upper, cxt);
  }

  /* if upper is a unification variable, add lower to its lower constraints */
  /* unless lower equals upper, in which case it adds nothing new           */
  else if( IsVar(upper) && upper->u.variable->state == Unify )
  {
    if( DEBUG4 )
      fprintf(stderr, "  TypeIsSubType adding lower constraint\n");
    res = IsVar(lower) && lower->u.variable == upper->u.variable ?
      TRUE : TypeVarUnifyAddLowerConstraint(upper->u.variable, lower, cxt);
  }

  /* otherwise follow the usual analysis */
  else if( IsVar(upper) )
  {
    if( DEBUG4 )
      fprintf(stderr, "  TypeIsSubType handling case of variable upper\n");
    res = !IsVar(lower) ? FALSE :
      lower->u.variable == upper->u.variable ? TRUE :
      lower->u.variable->upper_constraint == NULL ? FALSE :
      TypeIsSubType(lower->u.variable->upper_constraint, upper, coercion, cxt);
  }
  else
  {
    if( DEBUG4 )
      fprintf(stderr, "  TypeIsSubType handling case of non-variable upper\n");
    res = !IsVar(lower) ?
      TypeCArraySubType(lower->u.ctypes, upper->u.ctypes, coercion, cxt) :
      lower->u.variable->upper_constraint == NULL ? FALSE :
      TypeIsSubType(lower->u.variable->upper_constraint, upper, coercion, cxt);
  }
  if( DEBUG4 )
    fprintf(stderr, "] TypeIsSubType(%s, %s) returning %s (coercion %s)\n",
      TypeShow(lower, cxt), TypeShow(upper, cxt), bool(res),
      CoercionShow(*coercion));
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  The "JoinAndMeet" submodule.                                             */
/*                                                                           */
/*  This submodule handles joins and meets, and the algorithms on weighted   */
/*  ancestor sets that support those operations.                             */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int TypeITypeCompare(ITYPE *it1, ITYPE *it2)                             */
/*                                                                           */
/*  Return negative, 0, or positive accordingly as the sort key of it1 is    */
/*  greater than, equal to, or less than the sort key of it2.                */
/*                                                                           */
/*****************************************************************************/

static int TypeITypeCompare(ITYPE *it1, ITYPE *it2)
{
  return
    ClassSortKey((*it1)->ctype->class) - ClassSortKey((*it2)->ctype->class);
}

static int VoidTypeITypeCompare(const void *it1, const void *it2)
{
  return
    ClassSortKey((* (ITYPE *)it1)->ctype->class) -
    ClassSortKey((* (ITYPE *)it2)->ctype->class);
}

/*****************************************************************************/
/*                                                                           */
/*  void TypeITypeSort(ITYPES itypes)                                        */
/*                                                                           */
/*  Sort itypes in place according to the sort key values of its classes.    */
/*                                                                           */
/*****************************************************************************/

void TypeITypeSort(ITYPES itypes)
{
  ArraySort(itypes, &VoidTypeITypeCompare);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeAncestorSetUnion(ITYPES *anc1, ITYPES anc2,                  */
/*    CTYPE *c1, CTYPE *c2)                                                  */
/*                                                                           */
/*  Set sorted ancestor set *anc1 to its union with anc2 and return TRUE,    */
/*  or else return FALSE and set *c1 and *c2 if there is inconsistent        */
/*  genericity.  The algorithm here is just sorted list union.               */
/*  Although *anc1 is reset in this algorithm, the initial value of *anc1    */
/*  is not harmed by this operation; both sets are treated as immutable.     */
/*  If *anc1 == NULL initially, the union is just anc2.                      */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeAncestorSetUnion(ITYPES *anc1, ITYPES anc2,
  CTYPE *c1, CTYPE *c2, CONTEXT cxt)
{
  int pos1, pos2, cmp;  ITYPE it1, it2;  ITYPES res;
  if( DEBUG23 )
    fprintf(stderr, "[ TypeAncestorSetUnion(%s, %s)\n",
      TypeITypesShow(*anc1, cxt), TypeITypesShow(anc2, cxt));
  assert(anc2 != NULL);
  if( *anc1 == NULL ) { *anc1 = anc2;  return TRUE; }
  ArrayInit(&res);  pos1 = pos2 = 0;
  while( pos1 < ArraySize(*anc1) && pos2 < ArraySize(anc2) )
  {
    it1 = ArrayGet(*anc1, pos1);  it2 = ArrayGet(anc2, pos2);
    cmp = TypeITypeCompare(&it1, &it2);
    if( cmp == 0 )
    {
      if( TypeCTypeEqual(it1->ctype, it2->ctype, cxt) )
      { ArrayAddLast(res, it1->weight >= it2->weight ? it1 : it2);
	pos1++;  pos2++;
      }
      else
      { fprintf(stderr, "%s: type has inconsistent genericity %s and %s\n",
	 FilePosShow(it1->ctype->file_pos), CTypeShow(it1->ctype, cxt),
	 CTypeShow(it2->ctype, cxt));
        db_return(DEBUG23, "TypeAncestorSetUnion", FALSE);
      }
    }
    else if( cmp < 0 )
    { /* it1 is present but it2 is not, so add it1 to growing result */
      ArrayAddLast(res, it1);
      pos1++;
    }
    else
    { /* it2 is present but it1 is not, so add it2 to growing result */
      ArrayAddLast(res, it2);
      pos2++;
    }
  }

  /* now one of *anc1 and anc2 is exhausted, so finish off with a copy */
  while( pos1 < ArraySize(*anc1) )
  { it1 = ArrayGet(*anc1, pos1);
    ArrayAddLast(res, it1);
    pos1++;
  }
  while( pos2 < ArraySize(anc2) )
  { it2 = ArrayGet(anc2, pos2);
    ArrayAddLast(res, it2);
    pos2++;
  }
  *anc1 = res;
  db_return(DEBUG23, "TypeAncestorSetUnion", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeAncestorSetIntersection(ITYPES *anc1, ITYPES anc2,           */
/*    CONTEXT cxt)                                                           */
/*                                                                           */
/*  Set sorted ancestor set *anc1 to its intersection with anc2 and return   */
/*  TRUE, or return FALSE if the result is empty.  The algorithm here is     */
/*  just sorted list intersection.                                           */
/*                                                                           */
/*  Although *anc1 is reset in this algorithm, the initial value of *anc1    */
/*  is not harmed by this operation.  Bother ancestor sets are treated as    */
/*  immutable.  This is necessary in case either is an uncopied ancestor     */
/*  set from a class type.                                                   */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeAncestorSetIntersection(ITYPES *anc1, ITYPES anc2,
  CONTEXT cxt)
{
  int pos1, pos2, cmp;  ITYPE it1, it2;  ITYPES res;
  ArrayInit(&res);

  pos1 = pos2 = 0;
  while( pos1 < ArraySize(*anc1) && pos2 < ArraySize(anc2) )
  {
    it1 = ArrayGet(*anc1, pos1);
    it2 = ArrayGet(anc2, pos2);
    cmp = TypeITypeCompare(&it1, &it2);
    if( cmp == 0 )
    {
      if( TypeCTypeEqual(it1->ctype, it2->ctype, cxt) )
      {
	/* types are equal, add heaviest weight one to result */
	if( it1->weight >= it2->weight )
	  ArrayAddLast(res, it1);
	else
	  ArrayAddLast(res, it2);
      }
      pos1++;  pos2++;
    }
    else if( cmp < 0 )
    {
      /* it1 is present in *anc1 but not in anc2, so skip it */
      pos1++;
    }
    else
    {
      /* it2 is present in anc2 but not in *anc1, so skip it */
      pos2++;
    }
  }
  *anc1 = res;
  return ArraySize(res) > 0;
}


/*****************************************************************************/
/*                                                                           */
/*  ITYPES TypeITypesCopy(ITYPES itypes)                                     */
/*                                                                           */
/*  Copy itypes.                                                             */
/*                                                                           */
/*****************************************************************************/

static ITYPES TypeITypesCopy(ITYPES itypes)
{
  ITYPES res;  ITYPE itype, it2;
  ArrayInit(&res);
  ArrayForEach(itypes, itype)
  {
    GetMemory(it2, ITYPE);
    it2->ctype = TypeCTypeCopy(itype->ctype);
    it2->weight = itype->weight;
    it2->coercion = itype->coercion;
    ArrayAddLast(res, it2);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeToAncestorSet(TYPE type, ITYPES *res, CTYPE *c1, CTYPE *c2)  */
/*                                                                           */
/*  Find the ancestor set of the given type.  The result type must be        */
/*  treated as immutable, because it will be shared if type is a class       */
/*  type with no generic parameters.                                         */
/*                                                                           */
/*  If FALSE is returned, it will be because the type is a meet type and     */
/*  it does not have consistent genericity.  In that case, *c1 and *c2       */
/*  will be set to the inconsistent ctypes.                                  */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeToAncestorSet(TYPE type, ITYPES *res, CTYPE *c1, CTYPE *c2,
  CONTEXT cxt)
{
  ITYPES anc;  CTYPE ctype;  TYPE_VARS tv;
  assert(!IsVar(type));
  *res = NULL;
  ArrayForEach(type->u.ctypes, ctype)
  {
    /* get ctype's ancestor set, copying it if substitution is necessary */
    tv = ClassVars(ctype->class);
    anc = ClassAncestorSet(ctype->class);
    if( tv != NULL )
    {
      TypeVarsResolvedBegin(tv, ctype->generics);
      anc = TypeITypesCopy(anc);
      TypeVarsResolvedEnd(tv);
    }

    /* merge with *res */
    if( !TypeAncestorSetUnion(res, anc, c1, c2, cxt) )
      return FALSE;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE TypeFromAncestorSet(ITYPES itypes)                                  */
/*                                                                           */
/*  Return a type with this ancestor set.                                    */
/*                                                                           */
/*  The algorithm here chooses the heaviest weighted representative from     */
/*  each equivalence class that generates the type.  See separate doc.       */
/*                                                                           */
/*****************************************************************************/

static TYPE TypeFromAncestorSet(ITYPES itypes, CONTEXT cxt)
{
  int current_class_id, i, di, j;  ITYPE itype, best_itype;  TYPE res;
  BOOLEAN gen;  COERCION c;
  GetMemory(res, TYPE);
  res->kind_tag = KIND_TYPEC;
  res->file_pos = NULL;
  ArrayInit(&res->u.ctypes);
  for( i = 0;  i < ArraySize(itypes);  i = di )
  {
    /* find best_itype, the heaviest weighted representative */
    best_itype = ArrayGet(itypes, i);
    current_class_id = ClassEquivalenceId(best_itype->ctype->class);
    for( di = i + 1;  di < ArraySize(itypes);  di++ )
    {
      itype = ArrayGet(itypes, di);
      if( ClassEquivalenceId(itype->ctype->class) != current_class_id )
	break;
      if( itype->weight > best_itype->weight )
        best_itype = itype;
    }

    /* check whether any later itype is a subtype of best_itype  */
    gen = TRUE;
    for( j = di;  gen && j < ArraySize(itypes);  j++ )
    {
      itype = ArrayGet(itypes, j);
      if( TypeCSubType(itype->ctype, best_itype->ctype, &c, cxt) )
	gen = FALSE;
    }

    /* if best_itype is a generator, add it to the growing result */
    if( gen )
      ArrayAddLast(res->u.ctypes, best_itype->ctype);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeJoin(TYPE *t1, TYPE t2)                                      */
/*                                                                           */
/*  Join type t2 into type *t1, replacing *t1 but leaving t2 unharmed.  If   */
/*  *t1 == NULL, set *t1 to t2.  Fail if the two types cannot be joined.     */
/*  When joining it's convenient to view the two types as a constraint chain */
/*  of zero or more variables plus a terminating meet of 0 or more classes:  */
/*                                                                           */
/*      v1 <: v2 <: ... <: vn <: A                                           */
/*      w1 <: w2 <: ... <: wn <: B                                           */
/*                                                                           */
/*  The join is then the first variable vi = wj where the two chains         */
/*  converge, or else the maximum subset of classes common to A and B.       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeJoin(TYPE *type, TYPE t2, CONTEXT cxt)
{
  TYPE t1;  ITYPES anc1, anc2;  CTYPE c1, c2;  COERCION c;
  assert(t2 != NULL);

  /* t2 cannot be a type range */
  t2 = TypeResolve(t2);
  assert(!IsVar(t2) || t2->u.variable->state != Range);

  /* if no *type, or *type is a subtype of t2, the join is just t2 */
  if( *type == NULL || TypeIsSubType(*type, t2, &c, cxt) )
  { *type = t2;
    return TRUE;
  }

  /* if t2 is a subtype of *type, then *type is the join */
  if( TypeIsSubType(t2, *type, &c, cxt) )
    return TRUE;

  /* for each variable on t2's chain, see if that variable is the join */
  while( t2 != NULL && IsVar(t2) )
  {
    /* search for t2 on *type's chain and return TRUE if found */
    t1 = TypeResolve(*type);
    while( t1 != NULL && IsVar(t1) )
    {
      if( t1 == t2 ) { *type = t1; return TRUE; }
      t1 = TypeResolve(t1->u.variable->upper_constraint);
    }

    /* continue on up t2's chain */
    t2 = TypeResolve(t2->u.variable->upper_constraint);
  }

  /* no luck with variables; traverse *type's chain to the end */
  t1 = TypeResolve(*type);
  while( t1 != NULL && IsVar(t1) )
    t1 = TypeResolve(t1->u.variable->upper_constraint);
  if( t1 == NULL || t2 == NULL )
    return FALSE;

  /* now find and intersect the two ancestor sets then map back to a type */
  if( !TypeToAncestorSet(t1, &anc1, &c1, &c2, cxt) ||
      !TypeToAncestorSet(t2, &anc2, &c1, &c2, cxt) ||
      !TypeAncestorSetIntersection(&anc1, anc2, cxt) )
    return FALSE;
  *type = TypeFromAncestorSet(anc1, cxt);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeMeet(TYPE *type, TYPE t2, CONTEXT cxt)                       */
/*                                                                           */
/*  Meet type t2 into type *t1, replacing *t1 but leaving t2 unharmed.  If   */
/*  *t1 == NULL, set *t1 to t2.  Fail if the two types cannot be met.        */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeMeet(TYPE *type, TYPE t2, CONTEXT cxt)
{
  TYPE t1;  CTYPE c1, c2;  COERCION c;
  ITYPES anc1, anc2;
  assert(t2 != NULL);

  /* t2 cannot be a type range */
  t2 = TypeResolve(t2);
  assert(!IsVar(t2) || t2->u.variable->state != Range);

  /* if no *type, the meet is just t2 */
  if( *type == NULL )
  {
    *type = t2;
    return TRUE;
  }

  /* if either is a subtype of the other, the lower is the meet */
  /* this will also handle the case of equal variables          */
  t1 = TypeResolve(*type);
  if( TypeIsSubType(t2, t1, &c, cxt) )
  {
    *type = t2;
    return TRUE;
  }
  if( TypeIsSubType(t1, t2, &c, cxt) )
    return TRUE;

  /* if either is avariable and we have got this far, then no meet */
  if( IsVar(t1) || IsVar(t2) )
    return FALSE;

  /* now find and union the two ancestor sets then map back to a type */
  if( !TypeToAncestorSet(t1, &anc1, &c1, &c2, cxt) ||
      !TypeToAncestorSet(t2, &anc2, &c1, &c2, cxt) ||
      !TypeAncestorSetUnion(&anc1, anc2, &c1, &c2, cxt) )
    return FALSE;
  *type = TypeFromAncestorSet(anc1, cxt);
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  The "LevelTwoValid" submodule.                                           */
/*                                                                           */
/*  This submodule is concerned with ancestors, i.e. tracing back through    */
/*  inheritance and coercion paths and making all the implied types by them  */
/*  explicit.                                                                */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeCheckInheritance(TYPE type, CLASS c, BOOLEAN coerce,         */
/*    CONTEXT cxt)                                                           */
/*                                                                           */
/*  Check that type is suitable to appear as an inherit type for class c,    */
/*  or, if coerce is TRUE, as the result of a coercion.                      */
/*                                                                           */
/*  If any error messages need printing, the appropriate view of the         */
/*  classes involved may be found in cxt.                                    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeCheckInheritance(TYPE type, CLASS c, BOOLEAN coerce,
  CONTEXT cxt)
{
  CTYPE ctype;  NAME name;
  assert(type != NULL);
  if( IsVar(type) )
  {
    name = ClassName(c, cxt);
    fprintf(stderr, "%s: class %s %s type variable %s\n",
      FilePosShow(type->file_pos), NameShow(name),
      coerce ? "coerces to" : "inherits",
      NameShow(type->u.variable->name));
    return FALSE;
  }
  else ArrayForEach(type->u.ctypes, ctype)
  {
    if( ctype->class == c )
    {
      name = ClassName(c, cxt);
      fprintf(stderr, "%s: class %s %s itself\n", FilePosShow(type->file_pos),
	NameShow(name), coerce ? "coerces to" : "inherits");
      return FALSE;
    }
    if( !coerce && ClassIsEnum(c) && !ClassIsEnum(ctype->class) )
    {
      fprintf(stderr, "%s: enum class %s inherits non-enum class %s\n",
	FilePosShow(ctype->file_pos), NameShow(ClassName(c, cxt)),
	NameShow(ClassName(ctype->class, cxt)));
      return FALSE;
    }
    else if( !coerce && !ClassIsEnum(c) && ClassIsEnum(ctype->class) )
    {
      fprintf(stderr, "%s: non-enum class %s inherits enum class %s\n",
	FilePosShow(ctype->file_pos), NameShow(ClassName(c, cxt)),
	NameShow(ClassName(ctype->class, cxt)));
      return FALSE;
    }
    else if( !coerce && ClassIsBuiltin(ctype->class) )
    {
      if( ClassIsBuiltin(c) )
      {
	if( !ClassCheckBuiltinInheritance(c, ctype->class) )
	  return FALSE;
      }
      else
      {
	NAME n = ClassName(ctype->class, cxt);
	name = ClassName(c, cxt);
	fprintf(stderr, "%s: class %s inherits builtin class %s\n",
	  FilePosShow(ctype->file_pos), NameShow(name), NameShow(n));
	fprintf(stderr, "  (%s is defined at %s)\n", NameShow(n),
	  FilePosShow(ClassFilePos(ctype->class)));
	return FALSE;
      }
    }
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  CLASS TypeITypeClass(ITYPE itype)                                        */
/*                                                                           */
/*  Return the class of this itype.                                          */
/*                                                                           */
/*****************************************************************************/

CLASS TypeITypeClass(ITYPE itype)
{
  return itype->ctype->class;
}


/*****************************************************************************/
/*                                                                           */
/*  COERCION TypeITypeCoercion(ITYPE itype)                                  */
/*                                                                           */
/*  Return the coercion of itype.  This will be NULL if no coercion.         */
/*                                                                           */
/*****************************************************************************/

COERCION TypeITypeCoercion(ITYPE itype)
{
  return itype->coercion;
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_TYPE TypeITypeGenerics(ITYPE itype)                                */
/*                                                                           */
/*  Return the generics of itype.                                            */
/*                                                                           */
/*****************************************************************************/

ARRAY_TYPE TypeITypeGenerics(ITYPE itype)
{
  return itype->ctype->generics;
}


/*****************************************************************************/
/*                                                                           */
/*  void TypeAddITypes(ITYPES itypes, TYPE type, int weight, COERCION co)    */
/*                                                                           */
/*  Add the ctypes of type to the ITYPES of itypes, without checking for     */
/*  redundancy or anything else, just the basic conversion from ctype to     */
/*  itype.                                                                   */
/*                                                                           */
/*****************************************************************************/

void TypeAddITypes(ITYPES itypes, TYPE type, int weight, COERCION co)
{
  CTYPE ctype;  ITYPE itype;
  assert(!IsVar(type));
  ArrayForEach(type->u.ctypes, ctype)
  {
    GetMemory(itype, ITYPE);
    itype->ctype = ctype;
    itype->weight = weight;
    itype->coercion = co;
    ArrayAddLast(itypes, itype);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeBuildAncestorSet(CLASS c, ITYPES *ancestor_set, CONTEXT cxt) */
/*                                                                           */
/*  Build the ancestor set for class c, returning it in *ancestor_set.       */
/*  If any error messages need printing, the appropriate view of the         */
/*  classes involved is cxt.                                                 */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeBuildAncestorSet(CLASS c, ITYPES *ancestor_set, CONTEXT cxt)
{
  ITYPES res, parent_set;  ITYPE itype, it1, it2;
  TYPE_VARS type_vars;  TYPE_VAR v;
  TYPE type;  CTYPE ctype;
  NAME name, n1, n2;  FILE_POS file_pos1, file_pos2;
  int i, j, len;  BOOLEAN found;

  /* initialize res to contain an itype holding just the class itself */
  type_vars = ClassVars(c);
  ctype = TypeMakeCType(ClassFilePos(c), NULL, c, NULL);
  if( type_vars != NULL )
  {
    ArrayInit(&ctype->generics);
    ArrayForEach(type_vars, v)
    {
      type = TypeMakeFromVar(ClassFilePos(c), v);
      ArrayAddLast(ctype->generics, type);
    }
  }
  GetMemory(itype, ITYPE);
  itype->ctype = ctype;
  itype->weight = 1;
  itype->coercion = NULL;
  ArrayInit(&res);
  ArrayAddLast(res, itype);

  /* build up the ancestors by doing a breadth-first search from c */
  for( i = 0;  i < ArraySize(res);  i++ )
  {
    itype = ArrayGet(res, i);
    len = CoercionLength(itype->coercion);

    /* retrieve, copy and substitute the parent set of ith itype */
    if( DEBUG22 && itype->ctype->class == NULL )
      fprintf(stderr, "TypeBuildAncestorSet(%s, -, -) fail at itype %d in %s\n",
	NameShow(ClassName(c, cxt)), i, TypeITypesShow(res, cxt));
    type_vars = ClassVars(itype->ctype->class);
    parent_set = ClassParentSet(itype->ctype->class);
    TypeVarsResolvedBegin(type_vars, itype->ctype->generics);
    parent_set = TypeITypesCopy(parent_set);
    TypeVarsResolvedEnd(type_vars);

    /* add each new element of parent_set to res, taking shortest coercion */
    ArrayForEach(parent_set, it1)
    {
      /* search for it1->ctype in res */
      found = FALSE;
      for( j = 0;  j < ArraySize(res);  j++ )
      {
	it2 = ArrayGet(res, j);
	if( it2->ctype->class == it1->ctype->class )
	{
	  found = TRUE;
	  break;
	}
      }

      if( found )
      {
	/* found; make sure generic parameters are the same */
	if( !TypeCTypeEqual(it1->ctype, it2->ctype, cxt) )
	{
	  name = ClassName(c, cxt);
	  n1 = ClassName(it1->ctype->class, cxt);
	  file_pos1 = ClassFilePos(it1->ctype->class);
	  file_pos2 = ClassFilePos(it2->ctype->class);
	  n2 = ClassName(it2->ctype->class, cxt);
	  fprintf(stderr,
	    "%s: class %s inherits or coerces to inconsistent types:\n",
	    FilePosShow(ClassFilePos(c)), NameShow(name));
	  fprintf(stderr, "  %s: %s\n", FilePosShow(file_pos1),
	    CTypeShow(it1->ctype,cxt));
	  fprintf(stderr, "  %s: %s\n", FilePosShow(file_pos2),
	    CTypeShow(it2->ctype,cxt));
	  return FALSE;
	}

	/* keep the coercion with the smallest length */
	if( len + CoercionLength(it1->coercion) < CoercionLength(it2->coercion))
	  it2->coercion = CoercionCopyAndAppend(itype->coercion, it1->coercion);
      }
      else
      {
	/* not found, so add it (with appended coercion) to res */
	if( itype->coercion != NULL )
	  it1->coercion = CoercionCopyAndAppend(itype->coercion, it1->coercion);
	assert(it1->weight == 0);
	ArrayAddLast(res, it1);
      }
    }
  }

  *ancestor_set = res;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  The "LevelThreeValid" submodule.                                         */
/*                                                                           */
/*  This submodule is concerned with establishing that a type has Level 3    */
/*  validity.  Specifically, it checks that:                                 */
/*                                                                           */
/*  *  Every actual generic parameter satisfies any subtype constraint       */
/*     on its corresponding formal generic parameter;                        */
/*                                                                           */
/*  *  Every meet type has consistent genericity and non-redundancy.         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeMeetCheck(TYPE type, CONTEXT cxt)                            */
/*                                                                           */
/*  Check that meet type type has consistent genericity and non-redundancy,  */
/*  by expanding it to an ancestor set and back again.                       */
/*                                                                           */
/*  This code also initiates disjoint sets merges between the elements       */
/*  of the meet, as a service to the CLASS module.                           */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN TypeMeetCheck(TYPE type, CONTEXT cxt)
{
  ITYPES ancestor_set;  CTYPE c1, c2;  TYPE t;  int i;
  assert(!IsVar(type));
  assert(ArraySize(type->u.ctypes) > 1);
  if( !TypeToAncestorSet(type, &ancestor_set, &c1, &c2, cxt) )
  {
    fprintf(stderr, "%s: inconsistent genericity in meet type %s\n",
      FilePosShow(type->file_pos), TypeShow(type, cxt));
    fprintf(stderr, "  (the inconsistent class types were %s and %s)\n",
      CTypeShow(c1, cxt), CTypeShow(c2, cxt));
    db_return(DEBUG5, "TypeMeetCheck (1)", FALSE);
  }
  t = TypeFromAncestorSet(ancestor_set, cxt);
  assert(!IsVar(t));
  if( ArraySize(t->u.ctypes) < ArraySize(type->u.ctypes) )
  {
    fprintf(stderr, "%s: redundancy in meet type %s\n", 
      FilePosShow(type->file_pos), TypeShow(type, cxt));
    fprintf(stderr, "  (a non-redundant alternative is %s)\n", TypeShow(t,cxt));
    db_return(DEBUG5, "TypeMeetCheck (2)", FALSE);
  }

  /* disjoints sets stuff; NB "object" cannot appear in a non-redundant meet */
  c1 = ArrayFirst(type->u.ctypes);
  for( i = 1;  i < ArraySize(type->u.ctypes);  i++ )
  {
    c2 = ArrayGet(type->u.ctypes, i);
    ClassDSMerge(c1->class, c2->class);
  }
  db_return(DEBUG5, "TypeMeetCheck", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeLevelThreeValid(TYPE type, CONTEXT cxt)                      */
/*                                                                           */
/*  Check Level 3 validity of this type.  If it is an actual generic         */
/*  parameter, its subtype constraint is checked by the caller, not here.    */
/*  Consequently, if type is a variable there is nothing to do here.         */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeLevelThreeValid(TYPE type, CONTEXT cxt)
{
  CTYPE ctype;  TYPE bc, ag;  int i;  TYPE_VARS c_vars;  COERCION c;
  assert(type != NULL);
  if( DEBUG5 )
    fprintf(stderr, "[ TypeLevelThreeValid(%s)\n", TypeShow(type, cxt));
  if( !IsVar(type) )
  {
    /* if meet type, check consistent genericity and non-redundancy */
    if( ArraySize(type->u.ctypes) > 1 )
      if( !TypeMeetCheck(type, cxt) )
	db_return(DEBUG5, "TypeLevelThreeValid", FALSE);

    /* check Level 3 validity of each actual generic parameter in type */
    ArrayForEach(type->u.ctypes, ctype)
    {
      if( ctype->generics != NULL )
      {
	c_vars = ClassVars(ctype->class);
	assert(ArraySize(ctype->generics) == ArraySize(c_vars));
	for( i = 0;  i < ArraySize(c_vars);  i++ )
	{
	  /* set ag to the ith actual generic and check Level 3 for ag */
	  ag = ArrayGet(ctype->generics, i);
	  if( !TypeLevelThreeValid(ag, cxt) )
	    db_return(DEBUG5, "CTypeLevelThreeValid (3)", FALSE);

	  /* check subtype constraints on actual generic parameter i of ctype */
	  if( ArrayGet(c_vars, i)->upper_constraint != NULL )
	  {
	    /* set bc to the value of the constraint after substitution */
	    TypeVarsResolvedBegin(c_vars, ctype->generics);
	    bc = TypeCopy(ArrayGet(c_vars, i)->upper_constraint);
	    TypeVarsResolvedEnd(c_vars);

	    /* now ag should be a subtype of bc */
	    if( !TypeIsSubType(ag, bc, &c, cxt) )
	    {
	      fprintf(stderr,"%s: constraint violation at parameter %d of %s\n",
		FilePosShow(ctype->file_pos), i+1, CTypeShow(ctype, cxt));
	      fprintf(stderr, "  (%s is defined at %s)\n",
		NameShow(ClassName(ctype->class, cxt)),
		FilePosShow(ClassFilePos(ctype->class)));
	      db_return(DEBUG5, "CTypeLevelThreeValid (4)", FALSE);
	    }
	  }
	}
      }
    }
  }
  db_return(DEBUG5, "TypeLevelThreeValid", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeVarsLevelThreeValid(TYPE_VARS type_vars, CONTEXT cxt)        */
/*                                                                           */
/*  Check that all the constraints on variables of class invocations         */
/*  within the constraints of these variables are satisfied.                 */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeVarsLevelThreeValid(TYPE_VARS type_vars, CONTEXT cxt)
{
  TYPE_VAR bt;
  if( DEBUG30 )
    fprintf(stderr, "[ TypeVarsLevelThreeValid(%s, cxt)\n",
      TypeVarsShow(type_vars, cxt));
  if( type_vars != NULL )  ArrayForEach(type_vars, bt)
  {
    if( bt->lower_constraint != NULL )
      if( !TypeLevelThreeValid(bt->lower_constraint, cxt) )
	db_return(DEBUG30, "TypeVarsLevelThreeValid", FALSE);
    if( bt->upper_constraint != NULL )
      if( !TypeLevelThreeValid(bt->upper_constraint, cxt) )
	db_return(DEBUG30, "TypeVarsLevelThreeValid", FALSE);
  }
  db_return(DEBUG30, "TypeVarsLevelThreeValid", TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeManifest(TYPE t, CONTEXT cxt, BOOLEAN is_public)             */
/*                                                                           */
/*  Instantiate t in the given context.  Unlike the other instantiation      */
/*  calls, this one is assumed to take place after all class types have      */
/*  been created and checked.  It can therefore rely on them and itself      */
/*  carries out a complete check on t.                                       */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeManifest(TYPE t, CONTEXT cxt, BOOLEAN is_public)
{
  if( !TypeLevelOneValid(t, cxt, is_public) )
    return FALSE;
  if( !TypeLevelThreeValid(t, cxt) )
    return FALSE;
  return TRUE;
}

/*****************************************************************************/
/*                                                                           */
/*  BOOL TypeVarsBeginManifest(TYPE_VARS tv, CONTEXT cxt, BOOLEAN is_public) */
/*                                                                           */
/*  Instantiate type variables tv in the given context.  On success, leave   */
/*  the variables in cxt.  On failure, leave cxt as it was before the call.  */
/*                                                                           */
/*  Unlike the other instantiation calls, this one is assumed to take place  */
/*  after all class types have been created and checked.  It can therefore   */
/*  rely on them and itself carries out a complete check.                    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeVarsBeginManifest(TYPE_VARS tv, CONTEXT cxt, BOOLEAN is_public)
{
  if( !TypeVarsBeginLevelOneValid(tv, cxt, is_public) )
    return FALSE;
  if( !TypeVarsLevelThreeValid(tv, cxt) )
  {
    TypeVarsEnd(tv, cxt);
    return FALSE;
  }
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  The "Features" submodule.                                                */
/*                                                                           */
/*  Functions related to handling features.                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeInheritFEFnFeatureSets(CLASS_VIEW cv, ITYPES parent_set,     */
/*    CONTEXT cxt)                                                           */
/*                                                                           */
/*  Orchestrate the pulling of non-corecion features from the parent set     */
/*  into cv.  This involves, for each proper parent class, setting up type   */
/*  copying for that class and calling ClassViewInheritParentFEFnFeatureSets.*/
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeInheritFEFnFeatureSets(CLASS_VIEW cv, ITYPES parent_set,
  CONTEXT cxt)
{
  ITYPE itype;  TYPE_VARS class_vars;
  CLASS parent_c;  CLASS_VIEW parent_cv;
  CLASS c;  BOOLEAN res;
  assert(parent_set != NULL);
  if( DEBUG20 )
    fprintf(stderr, "[ TypeInheritFEFnFeatureSets(%s, %s)\n",
      NameShow(ClassViewName(cv)), TypeITypesShow(parent_set, cxt));
  res = TRUE;
  c = ClassViewClass(cv);
  ArrayForEach(parent_set, itype)
  {
    /* find the parent class and class view denoted by itype */
    parent_c = itype->ctype->class;
    parent_cv = ClassToView(parent_c, cxt);
    assert(parent_cv != cv);

    /* only relevant if not coercion */
    if( itype->coercion == NULL )
    {
      if( DEBUG20 )
	fprintf(stderr, "  inheriting features from %s:\n",
	  NameShow(ClassViewName(parent_cv)));

      /* make sure parent class is done */
      if( !ClassViewInheritFeatures1(parent_cv, cxt) )
	return FALSE; 

      /* prepare for copying out of parent class */
      class_vars = ClassVars(parent_c);
      TypeVarsResolvedBegin(class_vars, itype->ctype->generics);

      /* pull features from parent_cv into all_features */
      if( !ClassViewInheritParentFEFnFeatureSets(cv, parent_cv, cxt) )
	res = FALSE;

      /* end copying out of parent class */
      TypeVarsResolvedEnd(class_vars);
    }
  }
  db_return(DEBUG20, "TypeInheritFEFnFeatureSets", res);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeRetrieveFEFnFeatureSet(TYPE t, CONTEXT cxt,                  */
/*    FILE_POS file_pos, USTRING key, FEFN_FEATURE_SET *res)                 */
/*                                                                           */
/*  Find a feature signature set with name key in type t and return it in    */
/*  *res; or return FALSE if not found.                                      */
/*                                                                           */
/*  The feature signature is retrieved from each element in turn, copied     */
/*  to carry out any type substitutions, and merged with any preceding       */
/*  retrieved feature signatures.  Merging kills any effective feature.      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeRetrieveFEFnFeatureSet(TYPE t, CONTEXT cxt,
  FILE_POS file_pos, USTRING key, FEFN_FEATURE_SET *res)
{
  CTYPE ctype;  FEFN_FEATURE_SET fvs;  TYPE_VARS tv;  CLASS_VIEW cv;

  if( DEBUG12 )
    fprintf(stderr, "[ TypeRetrieveFEFnFeatureSet(%s, %s)\n",
      TypeShow(t, cxt), UStringToUTF8(key));

  /* proceed up the chain of resolved/constrained to a meet type */
  *res = NULL;
  t = TypeResolve(t);
  while( t != NULL && IsVar(t) )
    t = TypeResolve(t->u.variable->upper_constraint);
  if( t == NULL )
    return FALSE;

  /* try each class type in any order, and merge what we find    */
  ArrayForEach(t->u.ctypes, ctype)
  {
    cv = ClassToView(ctype->class, cxt);
    if( ClassViewRetrieveFEFnFeatureSet(cv, key, &fvs) )
    {
      /* meld fvs into *res */
      if( DEBUG12 )
	fprintf(stderr, "  found %s in class %s\n", UStringToUTF8(key),
	  NameShow(ClassViewName(cv)));
      tv = ClassVars(ctype->class);
      TypeVarsResolvedBegin(tv, ctype->generics);
      if( !FEFnFeatureSetMerge(res, fvs, cxt, file_pos) )
	return FALSE;
      TypeVarsResolvedEnd(tv);
    }
  }

  if( DEBUG12 )
  {
    fprintf(stderr, "] TypeRetrieveFEFnFeatureSet returning ");
    if( *res != NULL )
      FEFnFeatureSetDebug(*res, NULL, stderr, SINGLE_LINE);
    else
      fprintf(stderr, "FALSE");
    fprintf(stderr, "\n");
  }
  return *res != NULL;
}


/*****************************************************************************/
/*                                                                           */
/*  The "Swizzle" submodule.                                                 */
/*                                                                           */
/*  This module contains functions needed for swizzling.                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  void TypeSwizzleRegister(TYPE type)                                      */
/*                                                                           */
/*  Record the fact that type is the result type of a creation feature,      */
/*  and hence, if it is of variable type, will need swizzling.               */
/*                                                                           */
/*****************************************************************************/

void TypeSwizzleRegister(TYPE type)
{
  if( DEBUG27 )
    fprintf(stderr, "  TypeSwizzleRegister(%s) (%s)\n", TypeShow(type, NULL),
      !IsVar(type) ? "not var" :
      type->u.variable->swizzle_index == NEED_SWIZZLE ? "already swizzling" :
      "new swizzle");
  if( IsVar(type) )
    type->u.variable->swizzle_index = NEED_SWIZZLE;
}


/*****************************************************************************/
/*                                                                           */
/*  int TypeSwizzleSet(TYPE_VARS type_vars)                                  */
/*                                                                           */
/*  These type_vars are of a class, and some will have been marked as        */
/*  needing swizzling by TypeSwizzleRegister above.  Assign unique indexes   */
/*  starting from 0 to those, and return the number that need swizzling.     */
/*                                                                           */
/*****************************************************************************/

int TypeSwizzleSet(TYPE_VARS type_vars)
{
  int index;  TYPE_VAR tv;
  index = 0;
  if( type_vars != NULL )
  {
    ArrayForEach(type_vars, tv)
      if( tv->swizzle_index == NEED_SWIZZLE )
	tv->swizzle_index = index++;
  }
  return index;
}


/*****************************************************************************/
/*                                                                           */
/*  TYPE_POINTER_STATUS TypeIsPointer(TYPE type, int *swizzle_index)         */
/*                                                                           */
/*  Return the pointer status of type:                                       */
/*                                                                           */
/*    TYPE_POINTER_YES   if type is a pointer                                */
/*    TYPE_POINTER_NO    if type is not a pointer                            */
/*    TYPE_POINTER_VAR   if type is a variable, so may vary.                 */
/*                                                                           */
/*  If the result is TYPE_POINTER_VAR, then a swizzle bit will be needed     */
/*  in object records to say whether a value of this type is a pointer or    */
/*  not in that object; return the index of that bit in *swizzle.            */
/*                                                                           */
/*****************************************************************************/

TYPE_POINTER_STATUS TypeIsPointer(TYPE type, int *swizzle_index)
{
  CTYPE ctype;
  if( IsVar(type) )
  {
    *swizzle_index = type->u.variable->swizzle_index;
    return TYPE_POINTER_VAR;
  }
  else if( ArraySize(type->u.ctypes) > 1 )
    return TYPE_POINTER_YES;
  else
  {
    ctype = ArrayFirst(type->u.ctypes);
    return ClassIsObjRef(ctype->class) ? TYPE_POINTER_YES : TYPE_POINTER_NO;
  }   
}


/*****************************************************************************/
/*                                                                           */
/*  The "CodeGen" submodule.                                                 */
/*                                                                           */
/*  This module contains functions needed for code generation.               */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN TypeIsClassType(TYPE t, CLASS *class)                            */
/*                                                                           */
/*  If t is a class type, return TRUE with *class equal to that class.       */
/*  Otherwise (if t is a variable or meet type), return FALSE.               */
/*                                                                           */
/*****************************************************************************/

BOOLEAN TypeIsClassType(TYPE t, CLASS *class)
{
  CTYPE ctype;
  if( !IsVar(t) && ArraySize(t->u.ctypes) == 1 )
  {
    ctype = ArrayFirst(t->u.ctypes);
    *class = ctype->class;
    return TRUE;
  }
  else
    return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  CASE_TYPE TypeCaseType(TYPE t)                                           */
/*                                                                           */
/*  Return a classification of t needed when implementing case expressions.  */
/*                                                                           */
/*****************************************************************************/

CASE_TYPE TypeCaseType(TYPE t)
{
  return IsVar(t) || ArraySize(t->u.ctypes) > 1 ? CASE_TYPE_OTHER :
    ClassCaseType(ArrayFirst(t->u.ctypes)->class);
}


/*****************************************************************************/
/*                                                                           */
/*  CLASS TypeClass(TYPE t)                                                  */
/*                                                                           */
/*  Return the class that t is an instantiation of (t must be a class type). */
/*                                                                           */
/*****************************************************************************/

CLASS TypeClass(TYPE t)
{
  CTYPE ctype;
  assert(!IsVar(t));
  assert(ArraySize(t->u.ctypes) == 1);
  ctype = ArrayFirst(t->u.ctypes);
  return ctype->class;
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_CLASS TypeClasses(TYPE t)                                          */
/*                                                                           */
/*  Return the classes that make up type t.                                  */
/*                                                                           */
/*****************************************************************************/

ARRAY_CLASS TypeClasses(TYPE t)
{
  ARRAY_CLASS res;  CTYPE ctype;
  ArrayInit(&res);
  if( t != NULL )
  {
    assert(!IsVar(t));
    ArrayForEach(t->u.ctypes, ctype)
      ArrayAddLast(res, ctype->class);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  int TypeStructWidth(TYPE t)                                              */
/*                                                                           */
/*  Here t must be a class type - not a variable, not a meet type.  Return   */
/*  the width in bits of the record holding objects of type t.               */
/*                                                                           */
/*****************************************************************************/

int TypeStructWidth(TYPE t)
{
  CTYPE ctype;
  assert(!IsVar(t) && ArraySize(t->u.ctypes) == 1);
  ctype = ArrayFirst(t->u.ctypes);
  return ClassStructWidth(ctype->class);
}


/*****************************************************************************/
/*                                                                           */
/*  CODEGEN_TYPE TypeBEType(TYPE t, CODEGEN be)                              */
/*                                                                           */
/*  Return the backend type corresponding to this type.                      */
/*  For variables and meet types the backend type is void *; for class       */
/*  types it is the backend type of the class.                               */
/*                                                                           */
/*****************************************************************************/

CODEGEN_TYPE TypeBEType(TYPE t, CODEGEN be)
{
  CTYPE ctype;
  if( IsVar(t) || ArraySize(t->u.ctypes) > 1 )
    return be->voidp_type;
  else
  {
    ctype = ArrayFirst(t->u.ctypes);
    return ClassBEType(ctype->class);
  }
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR TypeMin(TYPE t)                                                     */
/*                                                                           */
/*  Return the minimum value of type t, or fail it it doesn't have one.      */
/*                                                                           */
/*****************************************************************************/

EXPR TypeMin(TYPE t)
{
  CTYPE ctype;
  assert(!IsVar(t) && ArraySize(t->u.ctypes) == 1);
  ctype = ArrayFirst(t->u.ctypes);
  if( DEBUG26 )
  {
    fprintf(stderr, "TypeMin(%s) returning ", TypeShow(t, NULL));
    ExprDebug(ClassMin(ctype->class), NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, "\n");
  }
  return ClassMin(ctype->class);
}


/*****************************************************************************/
/*                                                                           */
/*  EXPR TypeMax(TYPE t)                                                     */
/*                                                                           */
/*  Return the maximum value of type t, or fail it it doesn't have one.      */
/*                                                                           */
/*****************************************************************************/

EXPR TypeMax(TYPE t)
{
  CTYPE ctype;
  assert(!IsVar(t) && ArraySize(t->u.ctypes) == 1);
  ctype = ArrayFirst(t->u.ctypes);
  if( DEBUG26 )
  {
    fprintf(stderr, "TypeMax(%s) returning ", TypeShow(t, NULL));
    ExprDebug(ClassMax(ctype->class), NULL, FALSE, stderr, SINGLE_LINE);
    fprintf(stderr, "\n");
  }
  return ClassMax(ctype->class);
}
