/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         ustring.h                                                  */
/*  DESCRIPTION:  32-bit Unicode strings (header file)                       */
/*                                                                           */
/*  Implementation note.  Strings are represented by arrays terminated by    */
/*  the non-Unicode character 0x110000.  This code assumes that all the      */
/*  calls to memory allocation succeed, but is otherwise secure (no array    */
/*  overruns etc.).                                                          */
/*                                                                           */
/*****************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "ustring.h"
#define DEBUG1 1


/*****************************************************************************/
/*                                                                           */
/*  Submodule "build up UTF8 string char by char".                           */
/*                                                                           */
/*  void UTF8Begin(void)                                                     */
/*  void UTF8Add(UTF8CHAR ch)                                                */
/*  void UTF8Append(UTF8 str)                                                */
/*  UTF8 UTF8End(void)                                                       */
/*                                                                           */
/*  These operations work together to build up a UTF8 string character by    */
/*  character.  At most one construction operation may be afoot at any       */
/*  moment, and this is checked.                                             */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN	utf8_active = FALSE;		/* true when building        */
static int	utf8_max_len;			/* max length without resize */
static int	utf8_curr_len;               	/* current length            */
static UTF8	utf8_val;			/* string being built        */

static void UTF8Begin(void)
{
  assert(!utf8_active);
  utf8_max_len = 10;
  utf8_curr_len = 0;
  utf8_val = (UTF8) malloc((utf8_max_len + 1) * sizeof(UTF8CHAR));
  utf8_active = TRUE;
}

static void UTF8Add(UTF8CHAR ch)
{
  assert(utf8_active);
  if( utf8_curr_len == utf8_max_len )
  {
    utf8_max_len = 2 * utf8_max_len;
    utf8_val = (UTF8) realloc(utf8_val, (utf8_max_len+1)*sizeof(UTF8CHAR));
  }
  utf8_val[utf8_curr_len++] = ch;
}

static UTF8 UTF8End(void)
{
  assert(utf8_active);
  utf8_val[utf8_curr_len++] = '\0';
  utf8_active = FALSE;
  return utf8_val;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "build up UString char by char".                               */
/*                                                                           */
/*  void UStringBegin(void)                                                  */
/*  void UStringAdd(UCHAR ch)                                                */
/*  void UStringAppend(USTRING str)                                          */
/*  USTRING UStringEnd(void)                                                 */
/*                                                                           */
/*  These operations work together to build up a USTRING character by        */
/*  character.  At most one construction operation may be afoot at any       */
/*  moment, and this is checked.                                             */
/*                                                                           */
/*****************************************************************************/

static BOOLEAN	ustring_active = FALSE;		/* true when building        */
static int	ustring_max_len;		/* max length without resize */
static int	ustring_curr_len;               /* current length            */
static USTRING	ustring_val;			/* string being built        */

void UStringBeginWithSize(int size)
{
  assert(!ustring_active);
  ustring_max_len = size;
  ustring_curr_len = 0;
  ustring_val = (USTRING) malloc((ustring_max_len + 1) * sizeof(UCHAR));
  ustring_active = TRUE;
}

void UStringBegin(void)
{
  UStringBeginWithSize(10);
}

void UStringAdd(UCHAR ch)
{
  assert(ustring_active);
  if( ustring_curr_len == ustring_max_len )
  {
    ustring_max_len = 2 * ustring_max_len;
    ustring_val =
      (USTRING) realloc(ustring_val, (ustring_max_len + 1) * sizeof(UCHAR));
  }
  ustring_val[ustring_curr_len++] = ch;
}

USTRING UStringEnd(void)
{
  assert(ustring_active);
  ustring_val[ustring_curr_len++] = UEOF;
  ustring_active = FALSE;
  return ustring_val;
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING UStringEmpty(void)                                               */
/*                                                                           */
/*  Return a shared copy of the empty string.                                */
/*                                                                           */
/*****************************************************************************/

USTRING UStringEmpty(void)
{
  static USTRING res = NULL;
  if( res == NULL )
  {
    UStringBegin();
    res = UStringEnd();
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING UStringMax(void)                                                 */
/*                                                                           */
/*  Return a shared copy of a string which is larger than any legal string.  */
/*  Since the largest Unicode character is 0x10FFFF, this is implemented     */
/*  by a string containing just 0x110000.                                    */
/*                                                                           */
/*****************************************************************************/

USTRING UStringMax(void)
{
  static USTRING res = NULL;
  if( res == NULL )
  {
    UStringBegin();
    UStringAdd(0x110000);
    res = UStringEnd();
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "conversions to ASTRING".                                      */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  ASTRING UStringToDisplayedHex(USTRING str)                               */
/*                                                                           */
/*  Return str as a displayed hexadecimal sequence.                          */
/*                                                                           */
/*****************************************************************************/

ASTRING UStringToDisplayedHex(USTRING str)
{
  int i;  AFACTORY af;
  af = AStringBegin();
  for( i = 0;  str[i] != UEOF;  i++ )
  {
    if( i > 0 )
      AStringAddChar(af, ' ');
    AStringAddHex(af, str[i]);
  }
  return AStringEnd(af);
}


/*****************************************************************************/
/*                                                                           */
/*  ASTRING UStringNumberToAString(USTRING str)                              */
/*                                                                           */
/*  Convert str, which is known to represent a number, into an Ascii string  */
/*  representing the same number.                                            */
/*                                                                           */
/*  Implementation note.  The string, being assumed to contain a number,     */
/*  can contain only the characters x, -, ., e, E, and any Unicode decimal   */
/*  digits.  Only these digits need converting, to their Ascii equivalents.  */
/*                                                                           */
/*****************************************************************************/

ASTRING UStringNumberToAString(USTRING str)
{
  int i, val;  AFACTORY af;
  af = AStringBegin();
  for( i = 0;  str[i] != UEOF;  i++ )
    if( UCharIsDigit(str[i], &val) )
      AStringAddChar(af, '0' + val);
    else
      AStringAddChar(af, str[i]);
  return AStringEnd(af);
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "conversions between numbers and USTRING".                     */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN UStringMakeInteger(USTRING str, int *res)                        */
/*                                                                           */
/*  If str represents an integer, set *res to that integer and return TRUE,  */
/*  otherwise return FALSE.  Beginning with 0x means unsigned hexadecimal.   */
/*                                                                           */
/*****************************************************************************/
#define NOT_HEX		16		/* a non-hexadecimal value           */

#define hex_value(ch)							\
(									\
  ( (ch) >= '0' && ch <= '9' ) ? ch - '0' :				\
  ( (ch) >= 'A' && ch <= 'F' ) ? ch - ('A' - 10) : NOT_HEX		\
)

BOOLEAN UStringMakeInteger(USTRING str, int *res)
{
  int i, val;

  /* skip optional sign */
  i = 0;
  if( str[i] == '-' )
    i++;

  /* find the unsigned result, ordinary or hexadecimal */
  *res = 0;
  if( str[i] == '0' && str[i+1] == 'x' )
  {
    /* hexadecimal */
    i += 2;
    while( str[i] != UEOF )
    {
      val = hex_value(str[i]);
      if( val == NOT_HEX )
	return FALSE;
      *res = *res * 16 + val;
      i++;
    }
  }
  else
  {
    /* make sure there is at least one character */
    if( str[i] == UEOF )
      return FALSE;

    /* accumulate the value, failing if a non-digit occurs */
    do
    {
      if( !UCharIsDigit(str[i], &val) )
	return FALSE;
      *res = *res * 10 + val;
      i++;
    } while( str[i] != UEOF );
  }

  /* attach a sign if present and succeed */
  if( str[0] == '-' )
    *res = - *res;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  int UStringToInteger(USTRING str)                                        */
/*                                                                           */
/*  Assumes str is decimal and optionally signed with at least one digit,    */
/*  or, if str begins with 0x, assumes unsigned hexadecimal.                 */
/*                                                                           */
/*****************************************************************************/

int UStringToInteger(USTRING str)
{
  int res;
  if( !UStringMakeInteger(str, &res) )
  {
    if( DEBUG1 )
      fprintf(stderr, "UStringToInteger failing on %s\n", UStringToUTF8(str));
    assert(FALSE);
  }
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING UStringFromInteger(int i)                                        */
/*                                                                           */
/*  Convert an integer into its representation as a USTRING.                 */
/*                                                                           */
/*****************************************************************************/

USTRING UStringFromInteger(int i)
{
  char buff[30];
  sprintf(buff, "%d", i);
  return AStringToUString(buff);
}


/*****************************************************************************/
/*                                                                           */
/*  UTF8 UStringToUTF8(USTRING str)                                          */
/*                                                                           */
/*  Return the UTF-8 form of USTRING str.                                    */
/*                                                                           */
/*****************************************************************************/

#define last_three_bits(ch)	( (ch) & 0x07 )
#define last_four_bits(ch)	( (ch) & 0x0F )
#define last_five_bits(ch)	( (ch) & 0x1F )
#define last_six_bits(ch)	( (ch) & 0x3F )

#define seven_bit_value(ch)	( ((ch) & ~0x7F    ) == 0 )
#define eleven_bit_value(ch)	( ((ch) & ~0x7FF   ) == 0 )
#define sixteen_bit_value(ch)	( ((ch) & ~0xFFFF  ) == 0 )
#define twentyone_bit_value(ch)	( ((ch) & ~0x1FFFFF) == 0 )

#define non_first_byte(ch)	( (ch) | 0x80 )
#define first_of_two(ch)	( (ch) | 0xC0 )
#define first_of_three(ch)	( (ch) | 0xE0 )
#define first_of_four(ch)	( (ch) | 0xF0 )

UTF8 UStringToUTF8(USTRING str)
{
  int i;  unsigned ch;
  UTF8Begin();
  for( i = 0;  str[i] != UEOF;  i++ )
  {
    ch = str[i];
    if( seven_bit_value(ch) )
    {
      /* seven-bit goes out as is */
      UTF8Add(ch);
    }
    else if( eleven_bit_value(ch) )
    {
      /* eleven-bit goes out in two bytes */
      UTF8Add(first_of_two(ch >> 6));
      UTF8Add(non_first_byte(last_six_bits(ch)));
    }
    else if( sixteen_bit_value(ch) )
    {
      /* sixteen-bit value goes out in three bytes */
      UTF8Add(first_of_three(ch >> 12));
      UTF8Add(non_first_byte(last_six_bits(ch >> 6)));
      UTF8Add(non_first_byte(last_six_bits(ch)));
    }
    else if( twentyone_bit_value(ch) )
    {
      /* twentyone-bit value goes out in four bytes */
      UTF8Add(first_of_four(ch >> 18));
      UTF8Add(non_first_byte(last_six_bits(ch >> 12)));
      UTF8Add(non_first_byte(last_six_bits(ch >> 6)));
      UTF8Add(non_first_byte(last_six_bits(ch)));
    }
    else
    {
      /* illegal value, not Unicode */
      fprintf(stderr, "illegal character (not Unicode) 0x%x after %s\n",
	ch, UTF8End());
      assert(FALSE);
    }
  }
  return UTF8End();
}


/*****************************************************************************/
/*                                                                           */
/*  UTF8 UStringToUTF8Quoted(USTRING str)                                    */
/*                                                                           */
/*  Similar to UStringToUTF8 except that the string is enclosed in quotes,   */
/*  and any " or \ within the string is preceded by \.                       */
/*                                                                           */
/*****************************************************************************/

UTF8 UStringToUTF8Quoted(USTRING str)
{
  int i;  unsigned ch;
  UTF8Begin();
  UTF8Add('"');
  for( i = 0;  str[i] != UEOF;  i++ )
  {
    ch = str[i];
    if( ch == '"' || ch == '\\' )
      UTF8Add('\\');
    if( seven_bit_value(ch) )
    {
      /* seven-bit goes out as is */
      UTF8Add(ch);
    }
    else if( eleven_bit_value(ch) )
    {
      /* eleven-bit goes out in two bytes */
      UTF8Add(first_of_two(ch >> 6));
      UTF8Add(non_first_byte(last_six_bits(ch)));
    }
    else if( sixteen_bit_value(ch) )
    {
      /* sixteen-bit value goes out in three bytes */
      UTF8Add(first_of_three(ch >> 12));
      UTF8Add(non_first_byte(last_six_bits(ch >> 6)));
      UTF8Add(non_first_byte(last_six_bits(ch)));
    }
    else if( twentyone_bit_value(ch) )
    {
      /* twentyone-bit value goes out in four bytes */
      UTF8Add(first_of_four(ch >> 18));
      UTF8Add(non_first_byte(last_six_bits(ch >> 12)));
      UTF8Add(non_first_byte(last_six_bits(ch >> 6)));
      UTF8Add(non_first_byte(last_six_bits(ch)));
    }
    else
    {
      /* illegal value, not Unicode */
      fprintf(stderr, "illegal character (not Unicode) 0x%x after \"%s\"\n",
        ch, UTF8End());
      assert(FALSE);
    }
  }
  UTF8Add('"');
  return UTF8End();
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "UString construction".                                        */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  USTRING UStringCat(USTRING str1, USTRING str2)                           */
/*                                                                           */
/*  Return the concatenation of str1 with str2.                              */
/*                                                                           */
/*****************************************************************************/

USTRING UStringCat(USTRING str1, USTRING str2)
{
  int i, j;
  USTRING res = (USTRING)
    malloc((UStringLength(str1) + UStringLength(str2) + 1)*sizeof(UCHAR));
  for( i = 0, j = 0;  str1[j] != UEOF;  i++, j++ )
    res[i] = str1[j];
  for(        j = 0;  str2[j] != UEOF;  i++, j++ )
    res[i] = str2[j];
  res[i] = UEOF;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING UStringCat3(USTRING str1, USTRING str2, USTRING str3)            */
/*                                                                           */
/*  Return the concatenation of str1 with str2 and str3.                     */
/*                                                                           */
/*****************************************************************************/

USTRING UStringCat3(USTRING str1, USTRING str2, USTRING str3)
{
  int i, j;
  USTRING res = (USTRING)
    malloc( (UStringLength(str1) + UStringLength(str2) +
	     UStringLength(str3) + 1) * sizeof(UCHAR));
  for( i = 0, j = 0;  str1[j] != UEOF;  i++, j++ )
    res[i] = str1[j];
  for(        j = 0;  str2[j] != UEOF;  i++, j++ )
    res[i] = str2[j];
  for(        j = 0;  str3[j] != UEOF;  i++, j++ )
    res[i] = str3[j];
  res[i] = UEOF;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING UStringCat4(USTRING str1, USTRING str2, USTRING str3, str4)      */
/*                                                                           */
/*  Return the concatenation of str1 with str2, str3, and str4.              */
/*                                                                           */
/*****************************************************************************/

USTRING UStringCat4(USTRING str1, USTRING str2, USTRING str3, USTRING str4)
{
  int i, j;
  USTRING res = (USTRING)
    malloc( (UStringLength(str1) + UStringLength(str2) +
       UStringLength(str3) + UStringLength(str4) + 1) * sizeof(UCHAR));
  for( i = 0, j = 0;  str1[j] != UEOF;  i++, j++ )
    res[i] = str1[j];
  for(        j = 0;  str2[j] != UEOF;  i++, j++ )
    res[i] = str2[j];
  for(        j = 0;  str3[j] != UEOF;  i++, j++ )
    res[i] = str3[j];
  for(        j = 0;  str4[j] != UEOF;  i++, j++ )
    res[i] = str4[j];
  res[i] = UEOF;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING UStringCopy(USTRING str)                                         */
/*                                                                           */
/*****************************************************************************/

USTRING UStringCopy(USTRING str)
{
  int i;
  USTRING res = (USTRING) malloc( (UStringLength(str) + 1) * sizeof(UCHAR));
  for( i = 0;  str[i] != UEOF;  i++ )
    res[i] = str[i];
  res[i] = UEOF;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  USTRING UStringSubstring(USTRING s, int start_pos, int len)              */
/*                                                                           */
/*  Return the substring of s starting at start_pos of length len.           */
/*  If this implies going off the end of s, truncate at the end of s.        */
/*                                                                           */
/*****************************************************************************/

USTRING UStringSubstring(USTRING s, int start_pos, int len)
{
  USTRING res;  int i, j, slen;
  slen = UStringLength(s);
  res = (USTRING) malloc( (len + 1) * sizeof(UCHAR));
  for( i = 0, j = start_pos;  i < len && j < slen;  i++, j++ )
    res[i] = s[j];
  res[i] = UEOF;
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  ARRAY_USTRING UStringSplit(USTRING s, UCHAR split_char)                  */
/*                                                                           */
/*  Split s at split_char into an array of ustrings.  There will be one      */
/*  more ustring than there are occurrences of split_char; hence there       */
/*  will always be at least one ustring in the result.                       */
/*                                                                           */
/*****************************************************************************/

ARRAY_USTRING UStringSplit(USTRING s, UCHAR split_char)
{
  ARRAY_USTRING res;  int start_pos, i;  USTRING str;
  ArrayInit(&res);
  start_pos = 0;
  for( i = 0;  s[i] != UEOF;  i++ )
  {
    if( s[i] == split_char )
    {
      str = UStringSubstring(s, start_pos, i - start_pos);
      ArrayAddLast(res, str);
      start_pos = i + 1;
    }
  }
  str = UStringSubstring(s, start_pos, i - start_pos);
  ArrayAddLast(res, str);
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  Submodule "string query and searching".                                  */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*  int UStringLength(USTRING s)                                             */
/*                                                                           */
/*****************************************************************************/

int UStringLength(USTRING s)
{
  int res;
  for( res = 0;  s[res] != UEOF;  res++ );
  return res;
}


/*****************************************************************************/
/*                                                                           */
/*  unsigned int UStringGet(USTRING s, int i)                                */
/*                                                                           */
/*  Return element i of s, where 0 <= i <= UStringLength(s); in the          */
/*  latter case the concluding UEOF will be returned.  The string bounds     */
/*  are not checked.                                                         */
/*                                                                           */
/*****************************************************************************/

UCHAR UStringGet(USTRING s, int i)
{
  return s[i];
}


/*****************************************************************************/
/*                                                                           */
/*  UCHAR UStringGetLast(USTRING s)                                          */
/*                                                                           */
/*  Return the last element of s.  Naturally, s must be non-empty.           */
/*                                                                           */
/*****************************************************************************/

UCHAR UStringGetLast(USTRING s)
{
  int len = UStringLength(s);
  assert(len > 0);
  return s[len - 1];
}


/*****************************************************************************/
/*                                                                           */
/*  int UStringCmp(USTRING s1, USTRING s2)                                   */
/*                                                                           */
/*  Function for comparing two strings as required by sorting; but note      */
/*  that this can't be called directly by the sorter, because the level      */
/*  of indirection is wrong.                                                 */
/*                                                                           */
/*****************************************************************************/

int UStringCmp(USTRING s1, USTRING s2)
{
  int i;
  for( i = 0;  s1[i] != UEOF && s2[i] != UEOF;  i++ )
    if( s1[i] != s2[i] )
      return s1[i] - s2[i];
  return s1[i] - s2[i];
}


/*****************************************************************************/
/*                                                                           */
/*  int UStringLexicalCmp(const void *t1, const void *t2)                    */
/*                                                                           */
/*  Comparison function for sorting UStrings into increasing order.          */
/*                                                                           */
/*****************************************************************************/

int UStringLexicalCmp(const void *t1, const void *t2)
{
  USTRING s1 = *(USTRING *) t1;
  USTRING s2 = *(USTRING *) t2;
  return UStringCmp(s1, s2);
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN UStringContains(USTRING s, UCHAR ch)                             */
/*                                                                           */
/*  Return TRUE if s contains ch.                                            */
/*                                                                           */
/*****************************************************************************/

BOOLEAN UStringContains(USTRING s, UCHAR ch)
{
  int i;
  for( i = 0;  s[i] != UEOF;  i++ )
    if( s[i] == ch )
      return TRUE;
  return FALSE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN UStringBeginsWith(USTRING str1, USTRING str2)                    */
/*                                                                           */
/*  Return TRUE if str1 begins with str2.                                    */
/*                                                                           */
/*****************************************************************************/

BOOLEAN UStringBeginsWith(USTRING str1, USTRING str2)
{
  int i, len;
  len = UStringLength(str2);
  if( UStringLength(str1) < len )
    return FALSE;
  for( i = 0;  i < len;  i++ )
    if( str1[i] != str2[i] )
      return FALSE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN UStringEndsWith(USTRING str1, USTRING str2)                      */
/*                                                                           */
/*  Return TRUE if str1 ends with str2.                                      */
/*                                                                           */
/*****************************************************************************/

BOOLEAN UStringEndsWith(USTRING str1, USTRING str2)
{
  int i, j, len1, len2;
  len1 = UStringLength(str1);
  len2 = UStringLength(str2);
  if( len1 < len2 )
    return FALSE;
  for( i = len1 - len2, j = 0;  j < len2;  i++, j++ )
    if( str1[i] != str2[j] )
      return FALSE;
  return TRUE;
}


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN UStringEqual(USTRING str1, USTRING str2)                         */
/*                                                                           */
/*  Returns true if str1 and str2 are equal.                                 */
/*                                                                           */
/*****************************************************************************/

BOOLEAN UStringEqual(USTRING str1, USTRING str2)
{
  USTRING p1 = str1, p2 = str2;
  while( *p1 == *p2 && *p1 != UEOF )
    p1++, p2++;
  return (*p1 == *p2);
}


/*****************************************************************************/
/*                                                                           */
/*  int UStringHash(USTRING s)                                               */
/*                                                                           */
/*  This hash function adds the characters together, but progressively       */
/*  shifts to the left one place.  If the result is negative it is negated.  */
/*                                                                           */
/*  We don't do % table_size here, that varies from table to table so is     */
/*  done in the symbol table code.                                           */
/*                                                                           */
/*****************************************************************************/

int UStringHash(USTRING s)
{
  int i, res;
  res = 0;
  for( i = 0;  s[i] != UEOF;  i++ )
    res = (res << 1) + s[i];
  if( res < 0 )  res = - res;
  return res;
}
