/*****************************************************************************/
/*                                                                           */
/*  THE NONPAREIL DOCUMENT FORMATTING SYSTEM                                 */
/*  COPYRIGHT (C) 2002, 2005 Jeffrey H. Kingston                             */
/*                                                                           */
/*  Jeffrey H. Kingston (jeff@it.usyd.edu.au)                                */
/*  School of Information Technologies                                       */
/*  The University of Sydney 2006                                            */
/*  AUSTRALIA                                                                */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either Version 2, or (at your option)      */
/*  any later version.                                                       */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place, Suite 330, Boston MA 02111-1307 USA   */
/*                                                                           */
/*  FILE:         symtab.h                                                   */
/*  DESCRIPTION:  Symbol table (linear probing hash table) (header file).    */
/*                                                                           */
/*  The keys of these symbol tables are always USTRING, but the values can   */
/*  be any type.  To create a symbol table with values of type VALUE_TYPE:   */
/*                                                                           */
/*       typedef ARRAY(VALUE_TYPE)          ARRAY_VALUE_TYPE;                */
/*       typedef SYMTAB(ARRAY_VALUE_TYPE)   SYMTAB_VALUE_TYPE;               */
/*                                                                           */
/*  Notice that the type parameter of SYMTAB is an array type; this is       */
/*  necessary owing to problems behind the scenes.  Use SYMTAB_VALUE_TYPE    */
/*  wherever a symbol table of this type is required.                        */
/*                                                                           */
/*****************************************************************************/
#ifndef SYMTAB_HEADER_FILE
#define SYMTAB_HEADER_FILE

#include <stdio.h>
#include <stdlib.h>
#include "boolean.h"
#include "ustring.h"
#include "array.h"

/*****************************************************************************/
/*                                                                           */
/*  SYMTAB type                                                              */
/*                                                                           */
/*  Implementation note.  The values corresponding to NULL keys are          */
/*  undefined.  This is slightly dangerous but it saves time.                */
/*                                                                           */
/*****************************************************************************/

#define SYMTAB(ARRAY_TYPE)						\
  struct {								\
    USTRING empty_string;	/* for marking deleted slots	 */	\
    int count;			/* number of entries in table	 */	\
    int position;		/* internal position marker      */	\
    int cursor;			/* internal cursor		 */	\
    int enlarge_threshold;	/* threshold for enlarging table */	\
    ARRAY_USTRING  keys;	/* array of keys		 */	\
    ARRAY_TYPE    values;	/* parallel array of values	 */	\
  } *

typedef SYMTAB(ARRAY_VOIDP) SYMTAB_VOIDP;
typedef SYMTAB(ARRAY_USTRING) SYMTAB_USTRING;
typedef SYMTAB(ARRAY_INT) SYMTAB_INT;
typedef SYMTAB(ARRAY_BOOLEAN) SYMTAB_BOOLEAN;


/*****************************************************************************/
/*                                                                           */
/*  void SymInit(SYMTAB(TYPE) *s)                                            */
/*                                                                           */
/*  Initialize *s to a new, empty symbol table.                              */
/*                                                                           */
/*****************************************************************************/

#define SymInit(s)	SymInitKey( (SYMTAB_VOIDP *) (s))


/*****************************************************************************/
/*                                                                           */
/*  void SymClear(SYMTAB(TYPE) s)                                            */
/*                                                                           */
/*  Clear out this symbol table.                                             */
/*                                                                           */
/*****************************************************************************/

#define SymClear(s)	SymClearKey( (SYMTAB_VOIDP) (s))


/*****************************************************************************/
/*                                                                           */
/*  void SymFree(SYMTAB(TYPE) s)                                             */
/*                                                                           */
/*  Free memory used by s.                                                   */
/*                                                                           */
/*****************************************************************************/

#define SymFree(s)	SymFreeKey( (SYMTAB_VOIDP) (s))


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SymInsert(SYMTAB(TYPE) s, USTRING key, TYPE v, TYPE *cv)         */
/*                                                                           */
/*  If key is not already present in s, insert (key, v) and return TRUE.     */
/*  If key is already present in s, set *cv to the corresponding vaue, do    */
/*  no insertion, and return FALSE.                                          */
/*                                                                           */
/*****************************************************************************/

#define SymInsert(s, key, v, cv)					\
(									\
  SymInsertKey((SYMTAB_VOIDP) (s), (key)) ?				\
  ( ArrayPut((s)->values, (s)->position, (v)), TRUE ) : 		\
  ( *(cv) = ArrayGet((s)->values, (s)->position), FALSE )		\
)


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SymRetrieve(SYMTAB(TYPE) s, USTRING str, TYPE *v)                */
/*  BOOLEAN SymRetrieveWithString(SYMTAB(TYPE) s, USTRING str, *vs, TYPE *v) */
/*                                                                           */
/*  If key is present, set *v to its value and return TRUE.                  */
/*  If key is not present, set nothing and return FALSE.                     */
/*                                                                           */
/*  SymRetrieveWithString also returns the search string, in *vs.  This is   */
/*  useful when str is in stack memory but the table's vs is in heap memory. */
/*                                                                           */
/*****************************************************************************/

#define SymRetrieve(s, str, v)						\
(									\
  SymRetrieveKey((SYMTAB_VOIDP) (s), (str)) ?				\
  ( *(v) = ArrayGet((s)->values, (s)->position), TRUE) : FALSE		\
)

#define SymRetrieveWithString(s, str, vs, v)				\
(									\
  SymRetrieveKey((SYMTAB_VOIDP) (s), (str)) ?				\
  ( *(vs) = ArrayGet((s)->keys, (s)->position),				\
    *(v) = ArrayGet((s)->values, (s)->position), TRUE) : FALSE		\
)


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SymDelete(SYMTAB(TYPE) s, USTRING str)                           */
/*                                                                           */
/*  If key is present, delete it and return TRUE; else return FALSE.         */
/*                                                                           */
/*****************************************************************************/

#define SymDelete(s, str)	SymDeleteKey((SYMTAB_VOIDP) (s), (str))


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SymChangeKey(SYMTAB(TYPE) s, USTRING oldkey, USTRING newkey)     */
/*                                                                           */
/*  Update the name of an object in s from oldkey to newkey, returning       */
/*  TRUE if successful, else FALSE.                                          */
/*                                                                           */
/*  Having oldkey = newkey is allowed and is a no-op.                        */
/*                                                                           */
/*****************************************************************************/

#define SymChangeKey(s, oldkey, newkey)					\
  SymChangeSymKey((SYMTAB_VOIDP) (s), oldkey, newkey)


/*****************************************************************************/
/*                                                                           */
/*  void SymFirst(SYMTAB(TYPE) s)                                            */
/*                                                                           */
/*  Set internal cursor to the first entry of s.                             */
/*                                                                           */
/*****************************************************************************/

#define SymFirst(s)	SymFirstKey( (SYMTAB_VOIDP) (s) )


/*****************************************************************************/
/*                                                                           */
/*  void SymNext(SYMTAB(TYPE) s)                                             */
/*                                                                           */
/*  Set internal cursor to the next entry of s.                              */
/*                                                                           */
/*****************************************************************************/

#define SymNext(s)	SymNextKey( (SYMTAB_VOIDP) (s) )


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SymOffEnd(SYMTAB(TYPE) s)                                        */
/*                                                                           */
/*****************************************************************************/

#define SymOffEnd(s) ((s)->cursor == -1)


/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SymRetrieveCurrent(SYMTAB(TYPE) s, USTRING *key, TYPE *v)        */
/*                                                                           */
/*  Retrieve the current (key, value) pair of the traversal.                 */
/*                                                                           */
/*****************************************************************************/

#define SymRetrieveCurrent(s, key, v)					\
(									\
  *(key) = ArrayGet((s)->keys, (s)->cursor),				\
  *(v) = ArrayGet((s)->values, (s)->cursor)				\
)

/*****************************************************************************/
/*                                                                           */
/*  BOOLEAN SymNotOffEndAndCurrent(SYMTAB(TYPE) s, USTRING *key, TYPE *v)    */
/*                                                                           */
/*  If not off the end, return TRUE and set *key and *v to the current pair. */
/*  If off the end, return FALSE and set nothing.                            */
/*                                                                           */
/*****************************************************************************/

#define SymNotOffEndAndCurrent(s, key, v)				\
(SymOffEnd(s) ? FALSE : (SymRetrieveCurrent((s), (key), (v)), TRUE))


/*****************************************************************************/
/*                                                                           */
/*  SymForEach(SYMTAB(TYPE) s, USTRING *key, TYPE *v)                        */
/*                                                                           */
/*  Equivalent to "for( each (*key, *v) in s )"                              */
/*                                                                           */
/*****************************************************************************/

#define SymForEach(s, key, v)						\
for( SymFirst(s); SymNotOffEndAndCurrent((s), (key), (v)); SymNext(s) )


/*****************************************************************************/
/*                                                                           */
/*  void SymDebugPrint(SYMTAB(TYPE) s)                                       */
/*                                                                           */
/*  Debug print of symbol table s.                                           */
/*                                                                           */
/*****************************************************************************/

#define SymDebugPrint(s)  SymDebugPrintKey((SYMTAB_VOIDP) (s))


/*****************************************************************************/
/*                                                                           */
/*  External functions - not intended to be called directly.                 */
/*                                                                           */
/*****************************************************************************/

extern void SymInitKey(SYMTAB_VOIDP *s);
extern void SymClearKey(SYMTAB_VOIDP s);
extern void SymFreeKey(SYMTAB_VOIDP s);
extern BOOLEAN SymInsertKey(SYMTAB_VOIDP s, USTRING str);
extern BOOLEAN SymRetrieveKey(SYMTAB_VOIDP s, USTRING str);
extern BOOLEAN SymDeleteKey(SYMTAB_VOIDP s, USTRING str);
extern BOOLEAN SymChangeSymKey(SYMTAB_VOIDP s, USTRING oldkey, USTRING newkey);
extern void SymFirstKey(SYMTAB_VOIDP s);
extern void SymNextKey(SYMTAB_VOIDP s);
extern void SymDebugPrintKey(SYMTAB_VOIDP s);

#endif
